# Copyright 2008 Canonical Ltd.  All rights reserved.

"""Test for the WADL generation."""

__metaclass__ = type

from cStringIO import StringIO
from operator import attrgetter
import unittest

from zope.component import getGlobalSiteManager, getUtility
from zope.interface import implements, Interface
from zope.publisher.browser import TestRequest
from zope.schema import Date, Datetime, TextLine
from zope.security.management import (
    endInteraction, newInteraction, queryInteraction)
from zope.traversing.browser.interfaces import IAbsoluteURL

from lazr.restful.fields import Reference
from lazr.restful.interfaces import (
    ICollection, IEntry, IEntryResource, IResourceGETOperation,
    IServiceRootResource, IWebServiceConfiguration,
    IWebServiceClientRequest, IWebServiceVersion)
from lazr.restful import EntryResource, ResourceGETOperation
from lazr.restful.declarations import exported, export_as_webservice_entry
from lazr.restful.testing.webservice import (
    create_web_service_request, IGenericCollection, IGenericEntry,
    WebServiceTestCase, WebServiceTestPublication)
from lazr.restful.testing.tales import test_tales
from lazr.restful.utils import (
    get_current_browser_request, get_current_web_service_request,
    tag_request_with_version_name)


def get_resource_factory(model_interface, resource_interface):
    """Return the autogenerated adapter class for a model_interface.

    :param model_interface: the annnotated interface for which we are looking
        for the web service resource adapter
    :param resource_interface: the method provided by the resource, usually
        `IEntry` or `ICollection`.
    :return: the resource factory (the autogenerated adapter class.
    """
    request_interface = getUtility(IWebServiceVersion, name='2.0')
    return getGlobalSiteManager().adapters.lookup(
        (model_interface, request_interface), resource_interface)


def get_operation_factory(model_interface, name):
    """Find the factory for a GET operation adapter.

    :param model_interface: the model interface on which the operation is
        defined.
    :param name: the name of the exported method.
    :return: the factory (autogenerated class) that implements the operation
        on the webservice.
    """
    request_interface = getUtility(IWebServiceVersion, name='2.0')
    return getGlobalSiteManager().adapters.lookup(
        (model_interface, request_interface),
        IResourceGETOperation, name=name)


class IHas_getitem(Interface):
    pass


class Has_getitem:
    implements(IHas_getitem)
    def __getitem__(self, item):
        return "wibble"


class ResourceOperationTestCase(unittest.TestCase):
    """A test case for resource operations."""

    def test_object_with_getitem_should_not_batch(self):
        """Test ResourceOperation.should_batch().

        Custom operations returning a Reference to objects that
        implement __getitem__ should not batch the results (iter() on
        such objects does not fail).
        """
        return_type = Reference(IHas_getitem)
        result = Has_getitem()

        operation = ResourceGETOperation("fake context", "fake request")
        operation.return_type = return_type

        self.assertFalse(
            operation.should_batch(result),
            "Batching should not happen for Reference return types.")

class InterfaceRestrictedField(TextLine):
    """A field that must be exported from one kind of interface."""

    def __init__(self, restrict_to_interface, *args, **kwargs):
        self.restrict_to_interface = restrict_to_interface
        super(InterfaceRestrictedField, self).__init__(*args, **kwargs)

    def bind(self, context):
        if not self.restrict_to_interface.providedBy(context):
            raise AssertionError(
                "InterfaceRestrictedField can only be used with %s"
                % self.restrict_to_interface.__name__)
        return super(InterfaceRestrictedField, self).bind(context)


class IHasRestrictedField(Interface):
    """An entry with an InterfaceRestrictedField."""
    export_as_webservice_entry()
    a_field = exported(InterfaceRestrictedField(Interface))


class HasRestrictedField:
    """An implementation of IHasRestrictedField."""
    implements(IHasRestrictedField)
    def __init__(self, value):
        self.a_field = value


class DummyAbsoluteURL:
    """Implements IAbsoluteURL for when you don't care what the URL is."""
    def __init__(self, *args):
        pass

    def __str__(self):
        return 'http://dummyurl/'

    __call__ = __str__


class EntryTestCase(WebServiceTestCase):
    """A test suite for entries."""

    testmodule_objects = [HasRestrictedField, IHasRestrictedField]

    def setUp(self):
        super(EntryTestCase, self).setUp()
        getGlobalSiteManager().registerAdapter(
            DummyAbsoluteURL, [IHasRestrictedField, IWebServiceClientRequest],
            IAbsoluteURL)

    def test_applyChanges_binds_to_resource_context(self):
        """Make sure applyChanges binds fields to the resource context.

        This case verifies that applyChanges binds fields to the entry
        resource's context, not the resource itself. If an
        InterfaceRestrictedField is bound to an object that doesn't
        expose the right interface, it will raise an exception.
        """
        entry_class = get_resource_factory(IHasRestrictedField, IEntry)
        request = getUtility(IWebServiceConfiguration).createRequest(
            StringIO(""), {})

        entry = entry_class(HasRestrictedField(""), request)
        resource = EntryResource(HasRestrictedField(""), request)
        entry.schema['a_field'].restrict_to_interface = IHasRestrictedField
        self.assertEquals(resource.entry.a_field, '')
        resource.applyChanges({'a_field': u'a_value'})
        self.assertEquals(resource.entry.a_field, 'a_value')

        # Make sure that IHasRestrictedField itself works correctly.
        class IOtherInterface(Interface):
            """An interface not provided by IHasRestrictedField."""
            pass
        entry.schema['a_field'].restrict_to_interface = IOtherInterface
        self.assertRaises(AssertionError, resource.applyChanges,
                          {'a_field': u'a_new_value'})
        self.assertEquals(resource.entry.a_field, 'a_value')

class WadlAPITestCase(WebServiceTestCase):
    """Test the docstring generation."""

    # This one is used to test when docstrings are missing.
    class IUndocumentedEntry(Interface):
        export_as_webservice_entry()

        a_field = exported(TextLine())

    testmodule_objects = [
        IGenericEntry, IGenericCollection, IUndocumentedEntry]

    def test_wadl_field_type(self):
        """Test the generated XSD field types for various fields."""
        self.assertEquals(test_tales("field/wadl:type", field=TextLine()),
                          None)
        self.assertEquals(test_tales("field/wadl:type", field=Date()),
                          "xsd:date")
        self.assertEquals(test_tales("field/wadl:type", field=Datetime()),
                          "xsd:dateTime")

    def test_wadl_entry_doc(self):
        """Test the wadl:doc generated for an entry adapter."""
        entry = get_resource_factory(IGenericEntry, IEntry)
        doclines = test_tales(
            'entry/wadl_entry:doc', entry=entry).splitlines()
        self.assertEquals([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            '<p>A simple, reusable entry interface for use in tests.</p>',
            '<p>The entry publishes one field and one named operation.</p>',
            '',
            '</wadl:doc>'], doclines)

    def test_empty_wadl_entry_doc(self):
        """Test that no docstring on an entry results in no wadl:doc."""
        entry = get_resource_factory(self.IUndocumentedEntry, IEntry)
        self.assertEquals(
            None, test_tales('entry/wadl_entry:doc', entry=entry))

    def test_wadl_collection_doc(self):
        """Test the wadl:doc generated for a collection adapter."""
        collection = get_resource_factory(IGenericCollection, ICollection)
        doclines = test_tales(
            'collection/wadl_collection:doc', collection=collection
            ).splitlines()
        self.assertEquals([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            'A simple collection containing IGenericEntry, for use in tests.',
            '</wadl:doc>'], doclines)

    def test_field_wadl_doc (self):
        """Test the wadl:doc generated for an exported field."""
        entry = get_resource_factory(IGenericEntry, IEntry)
        field = entry.schema['a_field']
        doclines = test_tales(
            'field/wadl:doc', field=field).splitlines()
        self.assertEquals([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            '<p>A &quot;field&quot;</p>',
            '<p>The only field that can be &lt;&gt; 0 in the entry.</p>',
            '',
            '</wadl:doc>'], doclines)

    def test_field_empty_wadl_doc(self):
        """Test that no docstring on a collection results in no wadl:doc."""
        entry = get_resource_factory(self.IUndocumentedEntry, IEntry)
        field = entry.schema['a_field']
        self.assertEquals(None, test_tales('field/wadl:doc', field=field))

    def test_wadl_operation_doc(self):
        """Test the wadl:doc generated for an operation adapter."""
        operation = get_operation_factory(IGenericEntry, 'greet')
        doclines = test_tales(
            'operation/wadl_operation:doc', operation=operation).splitlines()
        # Only compare the first 2 lines and the last one.
        # we dont care about the formatting of the parameters table.
        self.assertEquals([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            '<p>Print an appropriate greeting based on the message.</p>',],
            doclines[0:2])
        self.assertEquals('</wadl:doc>', doclines[-1])
        self.failUnless(len(doclines) > 3,
            'Missing the parameter table: %s' % "\n".join(doclines))


class DuplicateNameTestCase(WebServiceTestCase):
    """Test AssertionError when two resources expose the same name.

    This class contains no tests of its own. It's up to the subclass
    to define IDuplicate and call doDuplicateTest().
    """

    def doDuplicateTest(self, expected_error_message):
        """Try to generate a WADL representation of the root.

        This will fail due to a name conflict.
        """
        resource = getUtility(IServiceRootResource)
        request = create_web_service_request('/2.0')
        request.traverse(resource)
        try:
            resource.toWADL()
            self.fail('Expected toWADL to fail with an AssertionError')
        except AssertionError, e:
            self.assertEquals(str(e), expected_error_message)


class DuplicateSingularNameTestCase(DuplicateNameTestCase):
    """Test AssertionError when resource types share a singular name."""

    class IDuplicate(Interface):
        """An entry that reuses the singular name of IGenericEntry."""
        export_as_webservice_entry('generic_entry')

    testmodule_objects = [IGenericEntry, IDuplicate]

    def test_duplicate_singular(self):
        self.doDuplicateTest("Both IDuplicate and IGenericEntry expose the "
                             "singular name 'generic_entry'.")


class DuplicatePluralNameTestCase(DuplicateNameTestCase):
    """Test AssertionERror when resource types share a plural name."""

    class IDuplicate(Interface):
        """An entry that reuses the plural name of IGenericEntry."""
        export_as_webservice_entry(plural_name='generic_entrys')

    testmodule_objects = [IGenericEntry, IDuplicate]

    def test_duplicate_plural(self):
        self.doDuplicateTest("Both IDuplicate and IGenericEntry expose the "
                             "plural name 'generic_entrys'.")


class GetCurrentWebServiceRequestTestCase(WebServiceTestCase):
    """Test the get_current_web_service_request utility function."""

    testmodule_objects = [IGenericEntry]

    def test_web_service_request_is_versioned(self):
        """Ensure the get_current_web_service_request() result is versioned.

        When a normal browser request is turned into a web service
        request, it needs to have a version associated with it.
        lazr.restful associates the new request with the latest
        version of the web service: in this case, version 2.0.
        """

        # When there's no interaction setup, get_current_web_service_request()
        # returns None.
        self.assertEquals(None, queryInteraction())
        self.assertEquals(None, get_current_web_service_request())

        # Set up an interaction.
        request = TestRequest()
        newInteraction(request)

        # A normal web browser request isn't associated with any version.
        website_request = get_current_browser_request()
        self.assertRaises(AttributeError,
                          attrgetter('version', website_request), None)

        # But the result of get_current_web_service_request() is
        # associated with version 2.0.
        webservice_request = get_current_web_service_request()
        self.assertEquals("2.0", webservice_request.version)
        marker_20 = getUtility(IWebServiceVersion, "2.0")
        self.assertTrue(marker_20.providedBy(webservice_request))

        # We can use tag_request_with_version_name to change the
        # version of a request object.
        tag_request_with_version_name(webservice_request, '1.0')
        self.assertEquals("1.0", webservice_request.version)
        marker_10 = getUtility(IWebServiceVersion, "1.0")
        self.assertTrue(marker_10.providedBy(webservice_request))

        tag_request_with_version_name(webservice_request, '2.0')
        self.assertEquals("2.0", webservice_request.version)
        self.assertTrue(marker_20.providedBy(webservice_request))

def additional_tests():
    return unittest.TestLoader().loadTestsFromName(__name__)
