/*
 *  Copyright 1994-2011 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "erlang.h"


#define PROTOCOL     '7'
#define MAX_PROTOCOL PROTOCOL

char proto = PROTOCOL;

/*
 * Packet format:
 *
 * [ packet size ]        4 bytes
 * [ protocol version ]   4 bytes, should be enough ;)
 * [ protocol specific ]  N bytes
 *
 * See the v?.c files for protocol specific format
 */

u_long id = 1216486988;
u_long options = BE_NONE;

/* erlang port plugin */
#define IN_FD      3
#define OUT_FD     4

FILE *in = NULL, *out = NULL;


extern void v1(const Context_t *);
extern void v2(const Context_t *);
extern void v3(const Context_t *);
extern void v4(const Context_t *);
extern void v5(const Context_t *);
extern void v6(const Context_t *);
extern void v7(const Context_t *);


uint32_t last_colormap;
uint32_t last_picture;
uint32_t last_sequence;

char send_colormap_update;
char send_picture_update;
char send_sequence_update;


static int set_nonblocking(int);

void
create(__attribute__ ((unused)) Context_t *ctx)
{
  int flags;
  char *env;
  int ret;

#if 0
  in = fdopen(IN_FD, "r");
  if (NULL == in)
    xperror("fdopen");
  if (-1 == (flags = fcntl(IN_FD, F_GETFL, 0)))
    flags = 0;
  if (-1 == fcntl(IN_FD, F_SETFL, flags | O_NONBLOCK))
    xperror("fcntl");
#else
  ret = set_nonblocking(IN_FD); /* FIXME do something with ret */
#endif

#if 0
  out = fdopen(OUT_FD, "w");
  if (NULL == out)
    xperror("fdopen");
#endif

  env = getenv("BINIOU_ERLANG_PROTO");
  if (NULL != env) {
    char v = env[0];
    if ((v < '1') || (v > MAX_PROTOCOL)) {
      printf("[!] Unknown protocol version '%c', setting to %d\n", v, PROTOCOL);
      proto = PROTOCOL;
    } else {
      printf("[i] erlang: setting protocol to %c\n", v);
      proto = v;
    }
  }

  last_colormap = last_picture = last_sequence = 0;
}


void
destroy(__attribute__ ((unused)) Context_t *ctx)
{
  if (fclose(in) != 0)
    xperror("fclose");
#if 0
  if (fclose(out) != 0)
    xperror("fclose");
#endif
}


void
xfwrite(const void *ptr, size_t size, size_t nmemb, FILE *stream)
{
  size_t res;

  res = fwrite(ptr, size, nmemb, stream);
  if (res != nmemb) {
    fprintf(stderr, "[!] xfwrite: short write (%d vs %d)\n", (int)res, (int)nmemb);
    /* TODO reopen stream ? */
    exit(1);
  }
}


static int
set_nonblocking(int fd)
{
  int flags;

  /* If they have O_NONBLOCK, use the Posix way to do it */
#if defined(O_NONBLOCK)
  /* Fixme: O_NONBLOCK is defined but broken on SunOS 4.1.x and AIX 3.2.5. */
  if (-1 == (flags = fcntl(fd, F_GETFL, 0)))
    flags = 0;
  return fcntl(fd, F_SETFL, flags | O_NONBLOCK);
#else
  /* Otherwise, use the old way of doing it */
  flags = 1;
  return ioctl(fd, FIOBIO, &flags);
#endif
}


int
read_exact(u_char *buf, const int len)
{
  int i, got = 0;

  do {
    if ((i = read(IN_FD, buf+got, len-got)) <= 0)
      return i;
    got += i;
  } while (got < len);

  return len;
}


int
write_exact(const u_char *buf, const int len)
{
  int i, wrote = 0;

  do {
    if ((i = write(OUT_FD, buf+wrote, len-wrote)) <= 0)
      return i;
    wrote += i;
  } while (wrote < len);

  return len;
}


static void
create_or_skip_sequence(const uint32_t id)
{
  Sequence_t *seq = Sequences_find(id);
}


/* Process any command coming from the erlang side */
static int
command(Context_t *ctx)
{
  uint32_t pkt_size, pkt_size2;
  u_char cmd[BUFLEN+1];
  size_t res;

  res = read_exact((u_char *)&pkt_size, sizeof(uint32_t));
  if (-1 == res)
    return 0;

  pkt_size2 = ntohl(pkt_size);
  if (pkt_size2 > BUFLEN)
    xerror("%s:%d buffer overflow attempt\n", __FILE__, __LINE__);

  memset((void *)&cmd, 0, BUFLEN*sizeof(u_char));
  res = read_exact((u_char *)&cmd, sizeof(u_char)*pkt_size2);

  if (1 == res) /* got one byte */
    if (STOP_CHAR == cmd[0]) {
      printf("[i] Erlang port: got STOP_CHAR\n");
      return -1;
    } else
      xerror("Unknown 1-byte command %d\n", cmd[0]);

  if (4 == res) /* got command */
    if (CMD_CHAR == cmd[0]) {
      Event_t e;
      
      e.to = (enum RcptTo)cmd[1];
      e.cmd = (enum Command)cmd[2];
      e.arg0 = (enum Arg)cmd[3];
      
      Context_event(ctx, &e);
      return 1;
    } else
      xerror("Unknown 4-bytes command %d\n", cmd[0]);

  if (6 == res) /* set picture/colormap/sequence */
    if (UPDATE_CHAR == cmd[0]) {
      uint32_t *id, id2;

      id = (uint32_t *)&cmd[2];
      id2 = ntohl(*id);

      switch (cmd[1]) {
      case UPDATE_COLORMAP_CHAR:
	ctx->sm->next->cmap_id = id2;
	Context_set_colormap(ctx);
	last_colormap = id2;
	send_colormap_update = 0;
	break;

      case UPDATE_PICTURE_CHAR:
	ctx->sm->next->picture_id = id2;
	Context_set_picture(ctx);
	last_picture = id2;
	send_picture_update = 0;
	break;

      case UPDATE_SEQUENCE_CHAR:
	if (id2 != 0) { /* sent by auto-scheme mode */
	  create_or_skip_sequence(id2);
	  Context_set_sequence(ctx, id2);
	  last_sequence = id2;
	  send_sequence_update = send_colormap_update = send_picture_update = 0;
	}
	break;

      default:
	xerror("Wrong 6-byte command\n");
	break;
      }

      return 1;
    } else
      xerror("Unknown 6-byte command %d\n", cmd[0]);

  xerror("Got wrong packet length from erlang: %li\n", pkt_size2);

  return 0; /* not reached */
}


void
run(Context_t *ctx)
{
  uint32_t total, total2;
  int res;

  send_colormap_update =
    send_picture_update =
    send_sequence_update = 1;

  do
    res = command(ctx);
  while (res == 1);

  if (res == -1) {
    u_char c = STOP_CHAR;
    int ret;
    
    ctx->running = 0;

    /* ACK stop */
    /* send packet size */
    total = sizeof(char);
    total2 = htonl(total);
    ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

    /* send STOP_CHAR */
    ret = write_exact((const u_char *)&c, sizeof(u_char));
  } else {
    switch (proto) {
    case '1':
      v1(ctx);
      break;

    case '2':
      v2(ctx);
      break;

    case '3':
      v3(ctx);
      break;

    case '4':
      v4(ctx);
      break;

    case '5':
      v5(ctx);
      break;

    case '6':
      v6(ctx);
      break;

    case '7':
      v7(ctx);
      break;

    default:
      xerror("Unknown protocol version\n");
      break;
    }
  }
}
