/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* file: write_mesh_gmv.c                                                   */
/*                                                                          */
/*                                                                          */
/* description: This program converts ALBERTA meshes and dof_real_[d_]vecs  */
/*              to a GMV 4.0 format (ASCII or "iecxi?r?", one of the binary */
/*              format versions, NOT NECESSARILY PORTABLE)                  */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by D. Koester (2004)                                                */
/*--------------------------------------------------------------------------*/

#include <ctype.h>
#include "alberta.h"

#define VERT_IND(dim,i,j) ((i)*N_VERTICES(dim)+(j))


/***************************************************************************/
/* gmv_open_ascii(filenam, mesh, sim_time): open a file for ASCII GMV      */
/* output.                                                                 */
/***************************************************************************/

static FILE * gmv_open_ascii(const char *filenam, MESH *mesh, REAL sim_time)
{
  /* FUNCNAME("gmv_open_ascii"); */
  FILE       *file = nil;

  file = fopen(filenam, "w");

  /*  Write header.  */ 
  fprintf(file, "gmvinput ascii\n");
  
  if(mesh->name) {
    fprintf(file, "comments\n");
    fprintf(file, "Mesh '%s'\n", mesh->name);
    fprintf(file, "endcomm\n");
  }

  fprintf(file, "probtime %.6E\n", sim_time);
  fprintf(file, "codename ALBERTA \n");
  fprintf(file, "codever 2.0      \n");

  return file;
}

/***************************************************************************/
/* gmv_open_bin(filenam,isize,rsize,time): open a file for binary output   */
/***************************************************************************/

static FILE * gmv_open_bin(const char *filenam, int isize, int rsize,
			   REAL sim_time)
{
  FUNCNAME("gmv_open_bin");
  FILE *fp = nil;

  /*  Check if this machine can address 64bit integers (isize = 8).  */
  if (isize == 8)
    TEST_EXIT (sizeof(long) >= 8,
      "Cannot write 8 byte integers on this machine.\n");

  fp = fopen(filenam, "w");

  /*  Write header.  */ 
  fwrite("gmvinput", sizeof(char), 8, fp);
  if (isize == 4 && rsize == 8) {
    fwrite("iecxi4r8", sizeof(char), 8, fp);
  }
  else if (isize == 8 && rsize == 4) {
    fwrite("iecxi8r4", sizeof(char), 8, fp);
  }
  else if (isize == 8 && rsize == 8) {
    fwrite("iecxi8r8", sizeof(char), 8, fp);
  }
  else  /* just do the normal "iecxi4r4" thing */ {
    fwrite("iecxi4r4", sizeof(char), 8, fp);
  }

  fwrite("probtime", sizeof(char), 8, fp);
  fwrite(&sim_time, sizeof(REAL), 1, fp);
  fwrite("codenameALBERTA codever 2.0     ", sizeof(char), 32, fp);

  return fp;
}


/***************************************************************************/
/* convert_string(in): Replace all white-space characters in "string"      */
/*                     with underscores.                                   */
/***************************************************************************/

static void convert_string(char *pos)
{
  for(; *pos; pos++)
    if(!isgraph((int)*pos)) *pos = '_';
}


/***************************************************************************/
/* write_coord_array(file, n_vert, coords, write_ascii): dump the given    */
/* coordinate array to the GMV file.                                       */
/***************************************************************************/

static void write_coord_array(FILE *file, int n_vert, REAL_D *coords, 
			      int write_ascii)
{
  REAL *tempxyz;
  int   i, j;

  if(write_ascii) {
    fprintf(file, "nodev %d\n", n_vert);

    for(i = 0; i < n_vert; i++) {
      for(j = 0; j < 3; j++) {
	if(j < DIM_OF_WORLD)
	  fprintf(file, "%.10E ", coords[i][j]);
	else
	  fprintf(file, "0.0 ");
      }
      fprintf(file, "\n");
    }
  }
  else {
    fwrite("nodev   ",sizeof(char),8, file);

    fwrite(&n_vert, sizeof(int), 1, file);

#if DIM_OF_WORLD == 3
    tempxyz = (REAL *) coords;
#else
    tempxyz = MEM_CALLOC(n_vert * 3, REAL);

    for (i = 0; i < n_vert; i++)
      for(j = 0; j < DIM_OF_WORLD; j++) 
	tempxyz[3*i + j] = coords[i][j]; 
#endif
    
    fwrite(tempxyz, sizeof(REAL), 3*n_vert, file);

#if DIM_OF_WORLD < 3
    MEM_FREE(tempxyz, n_vert * 3, REAL);
#endif
  }

  return;
}  


/***************************************************************************/
/* write_elem_array(file, dim, n_elem, elem, write_ascii): dump the given  */
/* element array to the GMV file.                                          */
/***************************************************************************/

static void write_elem_array(FILE *file, int dim, int n_elem, int *elem, 
			     int write_ascii)
{
  char        *cell_types[4] = {"general ", "line    ", 
				"tri     ", "tet     "};
  int          i, j, dim_1 = dim + 1, one = 1, tmp;

  if(write_ascii) {
    fprintf(file, "cells %d\n", n_elem);
  
    for(i = 0; i < n_elem; i++) {
      fprintf(file, "%s%d\n", cell_types[dim], dim+1);
      
      for(j = 0; j < N_VERTICES(dim); j++) {
	if(dim == 0)
	  fprintf(file, "1 ");
	
	fprintf(file, "%d ", 1 + elem[VERT_IND(dim,i,j)]);
      }
      fprintf(file, "\n");
    }
  }
  else {
    fwrite("cells   ", sizeof(char), 8, file);
    fwrite(&n_elem, sizeof(int), 1, file);

    for(i = 0; i < n_elem; i++) {
      fwrite(cell_types[dim], sizeof(char), 8, file);
      fwrite(&dim_1, sizeof(int), 1, file);
      
      for(j = 0; j < N_VERTICES(dim); j++) {
	if(dim == 0)
	  fwrite(&one, sizeof(int), 1, file);

	tmp = 1 + elem[VERT_IND(dim, i, j)];
	fwrite(&tmp, sizeof(int), 1, file);
      }
    }
  }

  return;
}


/***************************************************************************/
/* write_mat_array(file, n_elem, mat_array, write_ascii): dump the given   */
/* material array to the GMV file.                                         */
/***************************************************************************/

static void write_mat_array(FILE *file, int n_elem, int *mat, int write_ascii)
{
  char        *mat_types[2] = {"affine  ",
			       "parametr"};
  int          i, tmp;

  if(write_ascii) {
    fprintf(file, "material 2 0\n");
    fprintf(file, "%s\n", mat_types[0]);
    fprintf(file, "%s\n", mat_types[1]);
    
    for(i = 0; i < n_elem; i++)
      fprintf(file, "%d\n", mat[i]);
  }
  else {
    fwrite("material", sizeof(char), 8, file);
    tmp = 2;
    fwrite(&tmp, sizeof(int), 1, file);
    tmp = 0;
    fwrite(&tmp, sizeof(int), 1, file);
    fwrite(mat_types[0], sizeof(char), 32, file);
    fwrite(mat_types[1], sizeof(char), 32, file);

    fwrite(mat, sizeof(int), n_elem, file);
  }

  return;
}


/***************************************************************************/
/* add_mesh(file, mesh, write_ascii): Convert given ALBERTA mesh to GMV    */
/* format and add the result to the output file.                           */
/***************************************************************************/

static void add_mesh(FILE *file, MESH *mesh, int write_ascii)
{
  FUNCNAME("add_mesh");
  MACRO_DATA  *data;

/* Step 1: Convert the current mesh to a macro triangulation, which        */
/* has index-based information about the current triangulation.            */
/* Some checks on mesh sanity are also done by this routine.               */

  TEST_EXIT(data = mesh2macro_data(mesh),
    "Could not get macro data from mesh!\n");

/* Step 2: Write the coordinate array.                                     */
/*         If DIM_OF_WORLD < 3, then pad the remaining coords with 0.0.    */

  write_coord_array(file, data->n_total_vertices, data->coords, write_ascii);

/* Step 3: Write the element array. The offset is 1, not 0!                */

  write_elem_array(file, mesh->dim, 
		   data->n_macro_elements, data->mel_vertices,
		   write_ascii);

/* Step 4: If we have parametric simplices, mark these as a special        */
/* material array.                                                         */

  if(mesh->parametric) {
    int             *mat_array = MEM_CALLOC(data->n_macro_elements, int);
    TRAVERSE_STACK  *stack = get_traverse_stack();
    const EL_INFO   *el_info;
    FLAGS            fill_flag = CALL_LEAF_EL|FILL_COORDS|FILL_PROJECTION;
    int              elem = 0;

    el_info = traverse_first(stack, mesh, -1, fill_flag);

    while (el_info) {
      if(mesh->parametric->init_element(el_info, mesh->parametric))
	mat_array[elem] = 2;
      else
	mat_array[elem] = 1;

      el_info = traverse_next(stack, el_info);
      elem++;
    }
    free_traverse_stack(stack);

    write_mat_array(file, data->n_macro_elements, mat_array, write_ascii);
    MEM_FREE(mat_array, data->n_macro_elements, int);
  }

/* Step 5: Clean up.                                                        */

  free_macro_data(data);

  return;
}


/***************************************************************************/
/* add_drv(file, mesh, drv, write_ascii): Convert ALBERTA                  */
/* DOF_REAL_VEC drv to GMV format and append the result to the output file.*/
/***************************************************************************/

static void add_drv(FILE *file, MESH *mesh, DOF_REAL_VEC *drv, int write_ascii)
{
  FUNCNAME("add_drv");
  REAL            *new_vec;
  int              i, n0, index = 0, dim = mesh->dim;
  int              is_node_centered;
  int             *vert_ind = nil;
  char             drv_name[32] = {};
  const EL_INFO   *el_info = nil;
  DOF            **local_dof_ptr;
  DOF_INT_VEC     *dof_vert_ind = nil;
  const DOF_ADMIN *admin = drv->fe_space->admin;
  TRAVERSE_STACK  *stack = get_traverse_stack();


/* Step 1: We decide whether the values are node-centered or zone-centered */
/* based on the value of drv->fe_space->bas_fcts->ndof.                    */

  if(drv->fe_space->bas_fcts->n_dof[VERTEX])
    is_node_centered = 1;
  else if(drv->fe_space->bas_fcts->n_dof[CENTER])
    is_node_centered = 0;
  else
    ERROR_EXIT("Could not determine centering type of data (n_dof[VERTEX] and n_dof[CENTER] both nil).\n");

  snprintf(drv_name, 32, drv->name);
  convert_string((char *)drv_name);

  if(write_ascii)
    fprintf(file, "%s %d\n", drv_name, is_node_centered);
  else {
    fwrite(drv_name, sizeof(char), 32, file);
    fwrite(&is_node_centered, sizeof(int), 1, file);
  }

  if(is_node_centered) {
    new_vec = MEM_ALLOC(mesh->n_vertices, REAL);
    n0 = admin->n0_dof[VERTEX];
    
    dof_vert_ind = get_dof_int_vec("vertex indices", drv->fe_space);
    GET_DOF_VEC(vert_ind, dof_vert_ind);
    FOR_ALL_DOFS(admin, vert_ind[dof] = 0);
  }
  else {
    new_vec = MEM_ALLOC(mesh->n_elements, REAL);
    n0 = admin->n0_dof[CENTER];
  }

/* Step 2: We need to copy the correct values of drv->vec into new_vec.     */

  for (el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL);
       el_info;
       el_info=traverse_next(stack,el_info)) {

    local_dof_ptr = el_info->el->dof;

    if(is_node_centered) {
      for (i = 0; i < N_VERTICES(dim); i++)
	if (!vert_ind[local_dof_ptr[mesh->node[VERTEX] + i][n0]]) {
/* assign a global index to each vertex in the same way as above.           */
	  vert_ind[local_dof_ptr[i][n0]] = -1;    
	  
	  new_vec[index] = drv->vec[local_dof_ptr[i][n0]];

	  index++;
	}
    }
    else {
/* just use the first CENTER dof of each element.                           */
      new_vec[index] = drv->vec[local_dof_ptr[mesh->node[CENTER]][n0]];

      index++;
    }
  }

/* Just another cheap check to see if everything is alright.               */
  if(is_node_centered)
    TEST_EXIT(index == mesh->n_vertices,"Wrong no. of vertices counted!\n");
  else
    TEST_EXIT(index == mesh->n_elements,"Wrong no. of elements counted!\n");

/* Step 3: Write the values to the file.                                    */

  if(write_ascii)
    for(i = 0; i < index; i++)
      fprintf(file, "%.10E\t", new_vec[i]);
  else
    fwrite(new_vec, sizeof(REAL), index, file);

/* Step 4: Clean up.                                                        */

  MEM_FREE(new_vec, index, REAL);
  free_traverse_stack(stack);
  if(dof_vert_ind)
    free_dof_int_vec(dof_vert_ind);

  return;
}


/***************************************************************************/
/* add_drdv(file, mesh, drdv, write_ascii, as_velocity): Convert given     */
/* DOF_REAL_D_VEC to GMV format and add the result to the output file. If  */
/* the parameter "as_velocity" is set, then "drdv" will be written using   */
/* GMVs "velocity" keyword. This implies the useful automatic generation   */
/* of a field storing vector magnitude within the GMV viewer. The downside */
/* is that only one "velocity" vector is possible within a GMV file, and   */
/* that "velocity" vectors always have three components.                   */
/*                                                                         */
/* Please refer to the GMV documentation for details.                      */
/***************************************************************************/

static void add_drdv(FILE *file, MESH *mesh, DOF_REAL_D_VEC *drdv,
		     int write_ascii, int as_velocity)
{
  FUNCNAME("add_drdv");
  REAL            *new_vec[3] = {};
  int              i, j, n0, index = 0, dim = mesh->dim;
  int              is_node_centered, n_components;
  int             *vert_ind = nil;
  char             drdv_name[32] = {};
  const EL_INFO   *el_info = nil;
  DOF            **local_dof_ptr;
  DOF_INT_VEC     *dof_vert_ind = nil;
  const DOF_ADMIN *admin = drdv->fe_space->admin;
  TRAVERSE_STACK  *stack = get_traverse_stack();


/* Step 1: We decide whether the values are node-centered or zone-centered */
/* based on the value of drdv->fe_space->bas_fcts->ndof.                   */

  if(drdv->fe_space->bas_fcts->n_dof[VERTEX])
    is_node_centered = true;
  else if(drdv->fe_space->bas_fcts->n_dof[CENTER])
    is_node_centered = false;
  else
    ERROR_EXIT("Could not determine centering type of data (n_dof[VERTEX] and n_dof[CENTER] both nil).\n");

  if(as_velocity) {
    n_components = 3;

    if(write_ascii)
      fprintf(file, "velocity %d\n",is_node_centered);
    else {
      fwrite("velocity", sizeof(char), 8, file);
      fwrite(&is_node_centered, sizeof(int), 1, file);
    }
  }
  else {
    n_components = DIM_OF_WORLD;

    snprintf(drdv_name, 32, drdv->name);
    convert_string((char *)drdv_name);

    if(write_ascii) {
      fprintf(file, "%s %d %d %d\n", drdv_name, is_node_centered, 
	      n_components, 0);
    }
    else {
      i = 0;

      fwrite(drdv_name, sizeof(char), 32, file);
      fwrite(&is_node_centered, sizeof(int), 1, file);
      fwrite(&n_components, sizeof(int), 1, file);
      fwrite(&i, sizeof(int), 1, file);
    }
  }


  if(is_node_centered) {
    for(i = 0; i < n_components; i++)
      new_vec[i] = MEM_CALLOC(mesh->n_vertices, REAL);

    n0 = admin->n0_dof[VERTEX];
    
    dof_vert_ind = get_dof_int_vec("vertex indices", drdv->fe_space);
    GET_DOF_VEC(vert_ind, dof_vert_ind);
    FOR_ALL_DOFS(admin, vert_ind[dof] = 0);
  }
  else {
    for(i = 0; i < n_components; i++)
      new_vec[i] = MEM_CALLOC(mesh->n_elements, REAL);

    n0 = admin->n0_dof[CENTER];
  }

/* Step 2: We need to copy the correct values of drdv->vec into new_vec.    */

  for (el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL);
       el_info;
       el_info=traverse_next(stack,el_info)) {

    local_dof_ptr = el_info->el->dof;

    if(is_node_centered) {
      for (i = 0; i < N_VERTICES(dim); i++)
	if (!vert_ind[local_dof_ptr[mesh->node[VERTEX] + i][n0]]) {
/* assign a global index to each vertex in the same way as above.           */
	  vert_ind[local_dof_ptr[i][n0]] = -1;    
	  
	  for (j = 0; j < DIM_OF_WORLD; j++) 
	    new_vec[j][index] = drdv->vec[local_dof_ptr[i][n0]][j];

	  index++;
	}
    }
    else {
/* just use the first CENTER dof of each element.                           */
      for (j = 0; j < DIM_OF_WORLD; j++) 
	new_vec[j][index] =
	  drdv->vec[local_dof_ptr[mesh->node[CENTER]][n0]][j];

      index++;
    }
  }

/* Just another cheap check to see if everything is alright.               */
  if(is_node_centered)
    TEST_EXIT(index == mesh->n_vertices,"Wrong no. of vertices counted!\n");
  else
    TEST_EXIT(index == mesh->n_elements,"Wrong no. of elements counted!\n");

/* Step 3: Write the values to the file.                                   */

  if(write_ascii) {
    for(i = 0; i < n_components; i++)
      for(j = 0; j < index; j++)
	fprintf(file, "%.10E\t", new_vec[i][j]);
  }
  else
    for(i = 0; i < n_components; i++)
      fwrite(new_vec[i], sizeof(REAL), index, file);

/* Step 4: Clean up.                                                        */

  for (j = 0; j < n_components; j++) 
    MEM_FREE(new_vec[j], index, REAL);
  free_traverse_stack(stack);
  free_dof_int_vec(dof_vert_ind);
}


/***************************************************************************/
/* add_refined_data(file, mesh, write_ascii, write_mesh_date,  n_drv,      */
/*                  drv_ptr, n_drdv, drdv_ptr, velocity):                  */
/*                                                                         */
/* This routine enables us to output more information for higher order     */
/* Lagrange finite elements. Since GMV only displays linear data we first  */
/* perform a virtual refinement of all elements, see the array             */
/* "n_sub_elements" below. The refined mesh is then output to GMV as usual.*/
/***************************************************************************/

static const int n_sub_elements[/*dimension*/ 3][/*degree*/ 3] = {{2,3,4},
								  {4,9,16},
								  {4,10,20}};

/* sub_elements[dimension][degree][max_n_elements == 20][vertex] */
static const int sub_elements[3][3][20][4] =
  {/*dim=1*/ {/*p=2*/ {{0,2}, {2,1}},
	      /*p=3*/ {{0,2}, {2,3}, {3,1}},
	      /*p=4*/ {{0,2}, {2,3}, {3,4}, {4,1}}},
   /*dim=2*/ {/*p=2*/ {{0,5,4},{5,3,4},
		       {5,1,3},{4,3,2}},
	      /*p=3*/ {{0,7,6},{7,9,6},{7,8,9},
		       {6,9,5},{8,3,9},{9,4,5},
		       {8,1,3},{9,3,4},{5,4,2}},
	      /*p=4*/ {{ 0, 9, 8},{ 9,12, 8},{ 9,10,12},{ 8,12, 7},
		       {10,13,12},{12,14, 7},{10,11,13},{12,13,14},
		       { 7,14, 6},{11, 3,13},{13, 4,14},{14, 5, 6},
		       {11, 1, 3},{13, 3, 4},{14, 4, 5},{ 6, 5, 2}}},
   /*dim=3*/ {/*p=2*/ {{ 0, 4, 5, 6},{ 4, 1, 7, 8},
		       { 5, 7, 2, 9},
		       { 6, 8, 9, 3}},
	      /*p=3*/ {{ 0, 4, 6, 8},{ 4, 5,19,18},{ 5, 1,10,12},
		       { 6,19, 7,17},{19,10,11,16},
		       { 7,11, 2,14},
		       { 8,18,17, 9},{18,12,16,13},
		       {17,16,14,15},
		       { 9,13,15, 3}},
	      /*p=4*/ {{ 0, 4, 7,10},{ 4, 5,31,28},{ 5, 6,32,29},{ 6, 1,13,16},
		       { 7,31, 8,25},{31,32,33,34},{32,13,14,22},
		       { 8,33, 9,26},{33,14,15,23},
		       { 9,15, 2,19},
		       {10,28,25,11},{28,29,34,30},{29,16,22,17},
		       {25,34,26,27},{34,22,23,24},
		       {26,23,19,20},
		       {11,30,27,12},{30,17,24,18},
		       {27,24,20,21},
		       {12,18,21, 3}}}};

static const REAL *identity(const REAL_D x, REAL_D buffer)
{
  static REAL_D buffer2 = {};
  REAL *result = buffer ? buffer : buffer2;

  COPY_DOW(x, result);
  return result;
}

static int add_refined_data(FILE *file, MESH *mesh, int write_ascii,
			    int write_mesh_data,
			    const int n_drv, DOF_REAL_VEC **drv_ptr,
			    const int n_drdv, DOF_REAL_D_VEC **drdv_ptr,
			    DOF_REAL_D_VEC *velocity)
{
  FUNCNAME("add_refined_data");
  int              count, max_degree = 0, ne, nv, n_bas_fcts, i, j, k, n_sub;
  int              dim = mesh->dim, index;
  DOF_REAL_VEC    *drv = nil;
  DOF_REAL_D_VEC  *drdv = nil;
  const FE_SPACE  *max_fe_space = nil;
  DOF_INT_VEC     *dof_vert_ind;
  DOF             *local_dofs = nil;
  DOF            **local_dof_ptr;
  const DOF     *(*get_dof_indices)(const EL *, const DOF_ADMIN *, int *);
  const REAL_D  *(*interpol_d)(const EL_INFO *, int, const int *,
			      const REAL *(*)(const REAL_D, REAL_D),
			      const REAL *(*f_loc)(const EL_INFO *,
						   const REAL [N_LAMBDA],
						   REAL_D),
			      REAL_D *);
  int             *vert_ind = NULL, *elements;
  REAL_D          *vertices, *local_coords, buffer;
  int             *mat_array = nil;
  REAL            *new_vecs[250] = {};
  REAL            *new_vecs_d[250][DIM_OF_WORLD] = {};
  REAL            *new_vel[3] = {};
  REAL            *local_new_vec;
  REAL_D          *local_new_vec_d;
  const BAS_FCTS  *new_bas_fcts;
  const DOF_ADMIN *new_admin;
  TRAVERSE_STACK  *stack;
  const EL_INFO   *el_info;
  char             name[32] = {};

  if(dim == 0) {
    WARNING("mesh->dim == 0, using standard output mechanism.\n");
    return false;
  }

/***************************************************************************/
/* Step 1: Look for the largest Lagrange degree of vectors on mesh.        */
/* Remember the corresponding FE_SPACE, since we need a mapping from local */
/* DOFs to global indices.                                                 */
/***************************************************************************/

  for(count = 0; count < n_drv; count++) {
    drv = drv_ptr[count];
    
    if(!drv) {
      ERROR("Could not find DOF_REAL_VEC %d!\n", count);
      break;
    }
    if(!strstr(drv->fe_space->bas_fcts->name, "lagrange")) {
      WARNING("Only implemented for Lagrange Finite Elements!\n");
      WARNING("Using standard output mechanism.\n");
      return false;
    }
    if(drv->fe_space->bas_fcts->degree > max_degree) {
      max_fe_space = drv->fe_space;
      max_degree  = max_fe_space->bas_fcts->degree;
    }
  }
  for(count = 0; count < n_drdv; count++) {
    drdv = drdv_ptr[count];
    
    if(!drdv) {
      ERROR("Could not find DOF_REAL_D_VEC %d!\n", count);
      break;
    }
    if(!strstr(drdv->fe_space->bas_fcts->name, "lagrange")) {
      WARNING("Only implemented for Lagrange Finite Elements!\n");
      WARNING("Using standard output mechanism.\n");
      return false;
    }
    if(drdv->fe_space->bas_fcts->degree > max_degree) {
      max_fe_space = drdv->fe_space;
      max_degree  = max_fe_space->bas_fcts->degree;
    }
  }
  if(velocity) {
    if(!strstr(velocity->fe_space->bas_fcts->name, "lagrange")) {
      WARNING("Only implemented for Lagrange Finite Elements!\n");
      WARNING("Using standard output mechanism.\n");
      return false;
    }

    if(velocity->fe_space->bas_fcts->degree > max_degree) {
      max_fe_space = velocity->fe_space;
      max_degree  = max_fe_space->bas_fcts->degree;
    }
  }

  MSG("Maximal degree found:%d\n", max_degree);
  if(max_degree <= 1) {
    MSG("Using standard output mechanism.\n");
    return false;
  }


/***************************************************************************/
/* Step 2: Count the number of needed vertices and elements. Fill element  */
/* and vertex arrays.                                                      */
/***************************************************************************/
  n_sub           = n_sub_elements[dim-1][max_degree-2];
  n_bas_fcts      = max_fe_space->bas_fcts->n_bas_fcts;
  local_dofs      = MEM_ALLOC(n_bas_fcts, DOF);
  get_dof_indices = max_fe_space->bas_fcts->get_dof_indices;
  dof_vert_ind    = get_dof_int_vec("vertex indices", max_fe_space);

  GET_DOF_VEC(vert_ind, dof_vert_ind);
  FOR_ALL_DOFS(max_fe_space->admin, vert_ind[dof] = -1);

  elements        = MEM_ALLOC(mesh->n_elements * n_sub * N_VERTICES(dim), int);

  ne = nv = 0;
  stack = get_traverse_stack();
  for(el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL);
      el_info;
      el_info = traverse_next(stack, el_info)) {

    get_dof_indices(el_info->el, max_fe_space->admin, local_dofs);

    for(i = 0; i < n_bas_fcts; i++)
      if (vert_ind[local_dofs[i]] == -1) {
/* Assign a global index to each vertex.                                    */
        vert_ind[local_dofs[i]] = nv;    
	
        nv++;
      }

/* Assign element indices.                                                  */
    for(i = 0; i < n_sub; i++) {
      for(j = 0; j < N_VERTICES(dim); j++)
	elements[VERT_IND(dim,ne,j)] =
	  vert_ind[local_dofs[sub_elements[dim-1][max_degree-2][i][j]]];
     
      ne++;
    }
  }

/* Do a quick check on the element number.                                 */
  TEST_EXIT(ne == mesh->n_elements * n_sub,
    "Wrong no. of elements counted!\n");  

/***************************************************************************/
/* Step 3: Allocate and fill the vertex array, as well as the DOF_REAL_VEC */
/* and DOF_REAL_D_VEC arrays.                                              */
/***************************************************************************/
  vertices        = MEM_ALLOC(nv, REAL_D);
  local_coords    = MEM_ALLOC(n_bas_fcts, REAL_D);
  interpol_d      = max_fe_space->bas_fcts->interpol_d;
  local_new_vec   = MEM_ALLOC(n_bas_fcts, REAL);
  local_new_vec_d = MEM_ALLOC(n_bas_fcts, REAL_D);

  for(i = 0; i < n_drv; i++) {
    if(!drv_ptr[i]) {
      ERROR("Could not find DOF_REAL_VEC %d!\n", i);
      break;
    }
    if(drv_ptr[i]->fe_space->bas_fcts->degree == 0)
      new_vecs[i] = MEM_ALLOC(ne, REAL);
    else
      new_vecs[i] = MEM_ALLOC(nv, REAL);
  }
  for(i = 0; i < n_drdv; i++) {
    if(!drdv_ptr[i]) {
      ERROR("Could not find DOF_REAL_D_VEC %d!\n", i);
      break;
    }
    if(drdv_ptr[i]->fe_space->bas_fcts->degree == 0)
      index = ne;
    else
      index = nv;

    for(j = 0; j < DIM_OF_WORLD; j++)
      new_vecs_d[i][j] = MEM_ALLOC(index, REAL);
  }
  if(velocity) {
    if(velocity->fe_space->bas_fcts->degree == 0)
      index = ne;
    else
      index = nv;
    for(i = 0; i < 3; i++)
      new_vel[i] = MEM_CALLOC(index, REAL);
  }

  stack = get_traverse_stack();
  for(index = 0,
	el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL | FILL_COORDS);
      el_info;
      index++,
	el_info = traverse_next(stack, el_info)) {

    get_dof_indices(el_info->el, max_fe_space->admin, local_dofs);
    interpol_d(el_info, 0, nil, identity, nil, local_coords);

    for(i = 0; i < n_bas_fcts; i++) 
      for(j = 0; j < DIM_OF_WORLD; j++)
	vertices[vert_ind[local_dofs[i]]][j] = local_coords[i][j];

    for(i = 0; i < n_drv; i++)
      if(drv_ptr[i]) {
	new_bas_fcts = drv_ptr[i]->fe_space->bas_fcts;
	new_admin = drv_ptr[i]->fe_space->admin;

	if(new_bas_fcts->degree == 0) {
	  int n0 = new_admin->n0_dof[CENTER];
	  local_dof_ptr = el_info->el->dof;

	  for(j = 0; j < n_sub; j++) 
	    new_vecs[i][index*n_sub + j] = 
	      drv_ptr[i]->vec[local_dof_ptr[mesh->node[CENTER]][n0]];
	}
	else {
	  new_bas_fcts->get_real_vec(el_info->el,
				     drv_ptr[i], 
				     local_new_vec);

	  for(j = 0; j < n_bas_fcts; j++)
	    new_vecs[i][vert_ind[local_dofs[j]]] =
	      eval_uh(LAGRANGE_NODES(max_fe_space->bas_fcts)[j], local_new_vec,
		      new_bas_fcts);
	}
      }

    for(i = 0; i < n_drdv; i++)
      if(drdv_ptr[i]) {
	new_bas_fcts = drdv_ptr[i]->fe_space->bas_fcts;
	new_admin = drdv_ptr[i]->fe_space->admin;

	if(new_bas_fcts->degree == 0) {
	  int n0 = new_admin->n0_dof[CENTER];
	  local_dof_ptr = el_info->el->dof;

	  for(j = 0; j < n_sub; j++) 
	    for(k = 0; k < DIM_OF_WORLD; k++)
	      new_vecs_d[i][k][index*n_sub + j] = 
		drdv_ptr[i]->vec[local_dof_ptr[mesh->node[CENTER]][n0]][k];
	}
	else {
	  new_bas_fcts->get_real_d_vec(el_info->el,
				       drdv_ptr[i], 
				       local_new_vec_d);

	  for(j = 0; j < n_bas_fcts; j++) {
	    eval_uh_d(LAGRANGE_NODES(max_fe_space->bas_fcts)[j], 
		      (const REAL_D *)local_new_vec_d,
		      new_bas_fcts, buffer);
	    for(k = 0; k < DIM_OF_WORLD; k++)
	      new_vecs_d[i][k][vert_ind[local_dofs[j]]] = buffer[k];
	  }
	}
      }
    
    if(velocity) {
      new_bas_fcts = velocity->fe_space->bas_fcts;
      new_admin = velocity->fe_space->admin;
      
      if(new_bas_fcts->degree == 0) {
	int n0 = new_admin->n0_dof[CENTER];
	local_dof_ptr = el_info->el->dof;  

	for(i = 0; i < n_sub; i++) 
	  for(j = 0; j < DIM_OF_WORLD; j++)
	    new_vel[j][index*n_sub + i] = 
	      velocity->vec[local_dof_ptr[mesh->node[CENTER]][n0]][j];
      }
      else {
	new_bas_fcts->get_real_d_vec(el_info->el,
				     velocity, 
				     local_new_vec_d);

	for(i = 0; i < n_bas_fcts; i++) {
	  eval_uh_d(LAGRANGE_NODES(max_fe_space->bas_fcts)[i],
		    (const REAL_D *)local_new_vec_d,
		    new_bas_fcts, buffer);
	  for(j = 0; j < DIM_OF_WORLD; j++)
	    new_vel[j][vert_ind[local_dofs[i]]] = buffer[j];
	}
      }
    }

  }

/***************************************************************************/
/* Step 5: Write a material array in the parametric case.                  */
/***************************************************************************/

  if(mesh->parametric) {
    mat_array = MEM_CALLOC(ne, int);
    ne = 0;

    el_info = traverse_first(stack, mesh, 
			     -1, CALL_LEAF_EL|FILL_COORDS|FILL_PROJECTION);
    while (el_info) {
      if(mesh->parametric->init_element(el_info, mesh->parametric))
	for(i = 0; i < n_sub; i++)
	  mat_array[ne + i] = 2;
      else
	for(i = 0; i < n_sub; i++)
	  mat_array[ne + i] = 1;

      ne += n_sub;
      el_info = traverse_next(stack, el_info);
    }

    ne = mesh->n_elements * n_sub;
  }


/***************************************************************************/
/* Step 6: Dump all vectors to the file.                                   */
/***************************************************************************/

  if(write_mesh_data) {
    write_coord_array(file, nv, vertices, write_ascii);
    write_elem_array(file, dim, ne, elements, write_ascii);
    if(mesh->parametric) 
      write_mat_array(file, ne, mat_array, write_ascii);
  }

  if(n_drv && drv_ptr) {
    if(write_ascii)
      fprintf(file, "variable\n");
    else
      fwrite("variable",sizeof(char),8,file);
  }

  for(i = 0; i < n_drv; i++)
    if(new_vecs[i]) {
      snprintf(name, 32, drv_ptr[i]->name);
      convert_string(name);

      if(drv_ptr[i]->fe_space->bas_fcts->degree == 0) {
	j = 0;
	index = ne*n_sub;
      }
      else {
	j = 1;
	index = nv;
      }

      if(write_ascii) {
	fprintf(file, "%s %d\n", name, j);

	for(j = 0; j < index; j++) 
	  fprintf(file, "%.10E\t", new_vecs[i][j]);
      }
      else {
	fwrite(name, sizeof(char), 32, file);
	fwrite(&j, sizeof(int), 1, file);
	fwrite(new_vecs[i], sizeof(REAL), index, file);
      }
    }

  if(n_drv && drv_ptr) {
    if(write_ascii)
      fprintf(file, "endvars\n");
    else
      fwrite("endvars ",sizeof(char),8,file);
  }

  if(n_drdv && drdv_ptr) {
    if(write_ascii)
      fprintf(file, "vectors \n");
    else
      fwrite("vectors ",sizeof(char),8,file);
  }

  for(i = 0; i < n_drdv; i++)
    if(new_vecs_d[i][0]) {
      snprintf(name, 32, drdv_ptr[i]->name);
      convert_string(name);

      if(drdv_ptr[i]->fe_space->bas_fcts->degree == 0) {
	j = 0;
	index = ne*n_sub;
      }
      else {
	j = 1;
	index = nv;
      }

      if(write_ascii) {
	fprintf(file, "%s %d %d %d\n", name, j, DIM_OF_WORLD, 0);

	for(j = 0; j < DIM_OF_WORLD; j++)
	  for(k = 0; k < index; k++) 
	    fprintf(file, "%.10E\t", new_vecs_d[i][j][k]);
      }
      else {
	fwrite(name, sizeof(char), 32, file);
	fwrite(&j, sizeof(int), 1, file);
	j = DIM_OF_WORLD;
	fwrite(&j, sizeof(int), 1, file);
	j = 0;
	fwrite(&j, sizeof(int), 1, file);
	for(j = 0; j < DIM_OF_WORLD; j++)
	  fwrite(new_vecs_d[i][j], sizeof(REAL), index, file);
      }
    }

  if(n_drdv && drdv_ptr) {
    if(write_ascii)
      fprintf(file, "endvect\n");
    else
      fwrite("endvect ",sizeof(char),8,file);
  }
  
  if(velocity) {
    if(velocity->fe_space->bas_fcts->degree == 0) {
      j = 0;
      index = ne;
    }
    else {
      j = 1;
      index = nv;
    }

    if(write_ascii) { 
      fprintf(file, "velocity %d\n", j);

      for(i = 0; i < 3; i++)
	for(j = 0; j < index; j++)
	  fprintf(file, "%.10E\t", new_vel[i][j]);
    }
    else {
      fwrite("velocity", sizeof(char), 8, file);
      fwrite(&j, sizeof(int), 1, file);

      for(i = 0; i < 3; i++)
	fwrite(new_vel[i], sizeof(REAL), index, file);
    }
  }

/* Free all allocated memory.   .                                          */
  MEM_FREE(elements, ne * N_VERTICES(dim), int);
  MEM_FREE(vertices, nv, REAL_D);
  MEM_FREE(local_dofs, n_bas_fcts, DOF);
  MEM_FREE(local_coords, n_bas_fcts, REAL_D);
  MEM_FREE(local_new_vec, n_bas_fcts, REAL);
  MEM_FREE(local_new_vec_d, n_bas_fcts, REAL_D);
  if(mesh->parametric) 
    MEM_FREE(mat_array, ne, int);

  for(i = 0; i < n_drv; i++)
    if(new_vecs[i]) {
      if(drv_ptr[i]->fe_space->bas_fcts->degree == 0)
	MEM_FREE(new_vecs[i], ne, REAL);
      else
	MEM_FREE(new_vecs[i], nv, REAL);
    }
  for(i = 0; i < n_drdv; i++)
    if(new_vecs_d[i][0]) {
      for(j = 0; j < DIM_OF_WORLD; j++)
	if(drdv_ptr[i]->fe_space->bas_fcts->degree == 0)
	  MEM_FREE(new_vecs_d[i][j], ne, REAL);
	else
	  MEM_FREE(new_vecs_d[i][j], nv, REAL);
    }
  for(i = 0; i < 3; i++)
    if(new_vel[i]) {
      if(velocity->fe_space->bas_fcts->degree == 0)
	MEM_FREE(new_vel[i], ne, REAL);
      else
	MEM_FREE(new_vel[i], nv, REAL);
    }

  free_dof_int_vec(dof_vert_ind);
  free_traverse_stack(stack);
  
  return true;
}


/***************************************************************************/
/* User interface                                                          */
/***************************************************************************/

/***************************************************************************/
/* write_mesh_gmv():                                                       */
/* Write mesh and vectors in GMV ASCII or binary format.                   */
/* We may output up to 250 DOF_REAL_VECs and 250 DOF_REAL_D_VECs into the  */
/* GMV file. These are passed as lists. The "velocity" argument is treated */
/* in a special way; GMV will automatically generate a field storing the   */
/* velocity magnitudes. Only the mesh and file names are mandatory. See    */
/* the description of "add_refined_data()" above for an explanation of     */
/* "use_refined_grid". The entry "sim_time" represents the numerical time  */
/* for instationary problems. GMV can display this value together with     */
/* the FEM data.                                                           */
/*                                                                         */
/* Return value is 0 for OK and 1 for ERROR.                               */
/***************************************************************************/

int write_mesh_gmv(MESH *mesh, const char *file_name, int write_ascii,
		   int use_refined_grid,
		   const int n_drv,
		   DOF_REAL_VEC **drv_ptr,
		   const int n_drdv,
		   DOF_REAL_D_VEC **drdv_ptr,
		   DOF_REAL_D_VEC *velocity, 
		   REAL sim_time)
{
  FUNCNAME("write_mesh_gmv");
  FILE           *file = nil;
  DOF_REAL_VEC   *drv = nil;
  DOF_REAL_D_VEC *drdv = nil;
  int             count;

  if(!mesh) {
    ERROR("no mesh - no file created!\n");
    return(1);
  }

  if(n_drv < 0 || n_drv > 250) {
    ERROR("n_drv must be an int between 0 and 250!\n");
    return(1);
  }

  if(n_drdv < 0 || n_drdv > 250) {
    ERROR("n_drdv must be an int between 0 and 250!\n");
    return(1);
  }

  if(write_ascii)
    file = gmv_open_ascii(file_name, mesh, sim_time);
  else
    file = gmv_open_bin(file_name, sizeof(int), sizeof(REAL), sim_time);

  if(!file) {
    ERROR("cannot open file %s\n",file_name);
    return(1);
  }
  
  dof_compress(mesh);

/* If we are using a refined Lagrange grid, then a special routine is called */
/* which takes care of mesh and vectors.                                     */

  if(!use_refined_grid ||
     !add_refined_data(file, mesh, write_ascii, true, n_drv, drv_ptr,
		       n_drdv, drdv_ptr, velocity)) {
    add_mesh(file, mesh, write_ascii);

    if(n_drv && drv_ptr) {
      if(write_ascii)
	fprintf(file, "variable\n");
      else {
	fwrite("variable",sizeof(char),8,file);
      }
      
      for(count = 0; count < n_drv; count++) {
	drv = drv_ptr[count];
	
	if(!drv) {
	  ERROR("Could not find DOF_REAL_VEC!\n");
	  break;
	}
	add_drv(file, mesh, drv, write_ascii);
      }
      
      if(write_ascii)
	fprintf(file, "endvars\n");
      else {
	fwrite("endvars ",sizeof(char),8,file);
      }
    }

    if(n_drdv && drdv_ptr) {
      if(write_ascii)
	fprintf(file, "vectors\n");
      else {
	fwrite("vectors ",sizeof(char),8,file);
      }
      
      for(count = 0; count < n_drdv; count++) {
	drdv = drdv_ptr[count];
	
	if(!drdv) {
	  ERROR("Could not find DOF_REAL_D_VEC!\n");
	  break;
	}
	add_drdv(file, mesh, drdv, write_ascii, false);
      }
      
      if(write_ascii)
	fprintf(file, "endvect\n");
      else {
	fwrite("endvect ",sizeof(char),8,file);
      }
    }

    if(velocity)
      add_drdv(file, mesh, velocity, write_ascii, true);
  }

  if(write_ascii)
    fprintf(file, "endgmv");
  else {
    fwrite("endgmv  ",sizeof(char),8,file);
  }

  fclose(file);

  return(0);
}


/***************************************************************************/
/* write_dof_vec_gmv():                                                    */
/* This function is similar to the one above. However, we do not output    */
/* the mesh node and element data - instead we direct GMV to access this   */
/* data from the "mesh_file". This file must have been created by a        */
/* previous call to write_mesh_gmv(). This way we avoid wasting time and   */
/* disk space when writing full GMV files including redundant mesh         */
/* information during each step of an instationary simulation, see GMV     */
/* documentation.                                                          */
/*                                                                         */
/* PLEASE NOTE: When using the "use_refined_grid" feature, the prior       */
/* write_mesh_gmv() call must also be given DOF_REAL_[D_]VECS to           */
/* determine the needed virtual refinement of the output mesh.             */
/*                                                                         */
/* Return value is 0 for OK and 1 for ERROR.                               */
/***************************************************************************/

int write_dof_vec_gmv(MESH *mesh,
		      const char *mesh_file, 
		      const char *file_name, int write_ascii,
		      int use_refined_grid,
		      const int n_drv,
		      DOF_REAL_VEC **drv_ptr,
		      const int n_drdv,
		      DOF_REAL_D_VEC **drdv_ptr,
		      DOF_REAL_D_VEC *velocity,
		      REAL sim_time)
{
  FUNCNAME("write_mesh_gmv");
  FILE           *file = nil;
  DOF_REAL_VEC   *drv = nil;
  DOF_REAL_D_VEC *drdv = nil;
  int             count;

  if(n_drv < 0 || n_drv > 250) {
    ERROR("n_drv must be an int between 0 and 250!\n");
    return(1);
  }

  if(n_drdv < 0 || n_drdv > 250) {
    ERROR("n_drdv must be an int between 0 and 250!\n");
    return(1);
  }


  if(write_ascii)
    file = gmv_open_ascii(file_name, mesh, sim_time);
  else
    file = gmv_open_bin(file_name, sizeof(int), sizeof(REAL), sim_time);

  if(!file) {
    ERROR("cannot open file %s\n",file_name);
    return(1);
  }
  
  dof_compress(mesh);

/* At this point we use the "fromfile" keyword to tell GMV about the other   */
/* file storing the mesh nodes and elements.                                 */

  if(write_ascii) {
    fprintf(file, "nodev fromfile \"%s\"\n", mesh_file);
    fprintf(file, "cells fromfile \"%s\"\n", mesh_file);

    /* Material is only written for parametric meshes. */
    if(mesh->parametric)
      fprintf(file, "material fromfile \"%s\"\n", mesh_file);
  }
  else {
    char filename[1024] = {};

    TEST_EXIT(strlen(mesh_file) < 1024, "Sorry, the filename is too long, please use less than 1024 characters.\n");
    
    snprintf(filename, 1024, "\"%s\"", mesh_file);

    fwrite("nodev   fromfile",sizeof(char),16,file);
    fwrite(filename,sizeof(char),strlen(filename),file);
    fwrite("cells   fromfile",sizeof(char),16,file);
    fwrite(filename,sizeof(char),strlen(filename),file);
    /* Material is only written for parametric meshes. */
    if(mesh->parametric) {
      fwrite("materialfromfile",sizeof(char),16,file);
      fwrite(filename,sizeof(char),strlen(filename),file);
    }
  }

/* If we are using a refined Lagrange grid, then a special routine is called */
/* which takes care of mesh and vectors.                                     */

  if(!use_refined_grid ||
     !add_refined_data(file, mesh, write_ascii, false, n_drv, drv_ptr,
		       n_drdv, drdv_ptr, velocity)) {

    if(n_drv && drv_ptr) {
      if(write_ascii)
	fprintf(file, "variable\n");
      else {
	fwrite("variable",sizeof(char),8,file);
      }
      
      for(count = 0; count < n_drv; count++) {
	drv = drv_ptr[count];
	
	if(!drv) {
	  ERROR("Could not find DOF_REAL_VEC!\n");
	  break;
	}
	add_drv(file, mesh, drv, write_ascii);
      }
      
      if(write_ascii)
	fprintf(file, "endvars\n");
      else {
	fwrite("endvars ",sizeof(char),8,file);
      }
    }

    if(n_drdv && drdv_ptr) {
      if(write_ascii)
	fprintf(file, "vectors\n");
      else {
	fwrite("vectors ",sizeof(char),8,file);
      }
      
      for(count = 0; count < n_drdv; count++) {
	drdv = drdv_ptr[count];
	
	if(!drdv) {
	  ERROR("Could not find DOF_REAL_D_VEC!\n");
	  break;
	}
	add_drdv(file, mesh, drdv, write_ascii, false);
      }
      
      if(write_ascii)
	fprintf(file, "endvect\n");
      else {
	fwrite("endvect ",sizeof(char),8,file);
      }
    }

    if(velocity)
      add_drdv(file, mesh, drdv, write_ascii, true);
  }

  if(write_ascii)
    fprintf(file, "endgmv");
  else {
    fwrite("endgmv  ",sizeof(char),8,file);
  }

  fclose(file);

  return(0);
}
