# The super class of HTTP Request-Response classes.
class CKMessage
	# HTML content.
	attr_accessor :content

	# Hash of HTTP headers.
	attr_accessor :headers

	# HTTP version. The default value is "1.1".
	attr_accessor :http_version

	# The encoding used for the content.
	attr_accessor :encoding

	# Array of CKCookie objects.
	attr_reader   :cookies

	EOL = "\r\n"

	def initialize( headers = nil )
		@headers      = headers || {}
		@http_version = "1.1"
		@cookies      = []
	end

	# Adds a cookie object.
	def add_cookie( cookie )
		@cookies << cookie
	end

	# Removes the specified cookie in the cookies.
	def remove_cookie( cookie )
		@cookies.delete cookie
	end

	# Returns HTTP version.
	def http_version_line
		"HTTP/#@http_version"
	end
end


# CKRequest represents requests of HTTP headers.
#
# CKRequest returns form parameters and cookies.
# But, in many cases, this class isn't used directly to get parameters 
# because CGIKit sets these parameters to components 
# automatically in the process of instanciating components.
#
# == Getting a request object
# You can get a request object by CKApplication#request or CKComponent#request.
#
# == Parameters
# CKRequest has some methods to access parameters.
#
# <b>form_values[key]</b>::       Returns an array of parameters.
#                                 If key is missing, returns an empty array.
# <b>form_value(key), [key]</b>:: This method returns a first object of
#                                 an array of parameters.
#                                 If key is missing, returns nil.
#
# == Cookies
# Cookies is an array which includes some CKCookie objects.
# See CKCookie section about details of cookie.
#
# <b>cookie(key)</b>::        Returns a CKCookie object whose key is the 
#                             same as the argument.
# <b>cookies</b>::            Returns an array of CKCookie objects.
# <b>cookie_value(key)</b>::  Returns the value of CKCookie object
#                             whose key is the the same as the argument.
# <b>cookie_values(key)</b>:: If the argument is nil (by default, argument is
#                             nil.), this method returns an array which has
#                             all the values of cookies. Otherwise, it returns
#                             an array which has the values of cookies
#                             specified by the argument.
#
# == HTTP Headers
# Some of HTTP headers are defined by instance methods.
# Call CKRequest#headers to get other CGI environment variables.
#
# <b>headers[key]</b>::      Returns a value of HTTP header.
# <b>accept</b>::            HTTP_ACCEPT
# <b>accept_charset</b>::    HTTP_ACCEPT_CHARSET 
# <b>accept_language</b>::   HTTP_ACCEPT_LANGUAGE
# <b>auth_type</b>::         AUTH_TYPE
# <b>content_length</b>::    CONTENT_LENGTH
# <b>content_type</b>::      CONTENT_TYPE
# <b>from</b>::              HTTP_FROM    
# <b>gateway_interface</b>:: GATEWAY_INTERFACE
# <b>path_info</b>::         PATH_INFO
# <b>path_translated</b>::   PATH_TRANSLATED
# <b>query_string</b>::      QUERY_STRING
# <b>raw_cookie</b>::        HTTP_COOKIE
# <b>referer</b>::           HTTP_REFERER
# <b>remote_addr</b>::       REMOTE_ADDR
# <b>remote_host</b>::       HTTP_HOST
# <b>remote_ident</b>::      REMOTE_IDENT
# <b>remote_user</b>::       REMOTE_USER
# <b>request_method</b>::    REQUEST_METHOD
# <b>script_name</b>::       SCRIPT_NAME
# <b>server_name</b>::       SERVER_NAME
# <b>server_port</b>::       SERVER_PORT
# <b>server_protocol</b>::   SERVER_PROTOCOL
# <b>server_software</b>::   SERVER_SOFTWARE
# <b>uri</b>, <b>url</b>::   REQUEST_URI
# <b>user_agent</b>::        HTTP_USER_AGENT
class CKRequest < CKMessage
	# Array of parameters.
	attr_accessor :form_values

	class << self
		# Parse query string and return a hash of parameters.
		def parse_query_string( query )
			params = Hash.new([])
			query.split(/[&;]/n).each do |pairs|
				key, value = pairs.split('=',2).collect{|v| CKUtilities.unescape_url(v) }
				if params.has_key?(key)
					params[key].push(value)
				else
					params[key] = [value]
				end
			end
			params
		end
	end

	def initialize( headers = nil, form_values = nil )
		super headers
		@cookies     = CKCookie.parse_raw_cookie @headers['HTTP_COOKIE']
		@form_values = form_values || Hash.new([])
	end

	# Returns a first object of an array of parameters. If key is missing, returns nil.
	def form_value( key )
		if @form_values.key? key
			@form_values[key].first
		else
			nil
		end
	end

	alias [] form_value

	# Returns a CKCookie object whose key is the same as the argument.
	def cookie( key )
		@cookies.each { | cookie |
			if cookie.name == key
				return cookie
			end
		}
		nil
	end

	# Returns the value of CKCookie object whose key is the the same as the argument.
	def cookie_value( key )
		@cookies.each { | cookie |
			if cookie.name == key
				return cookie.value
			end
		}
		nil
	end

	# If the argument is nil (by default, argument is nil.),
	# this method returns an array which has all the values of cookies. Otherwise,
	# it returns an array which has the values of cookies specified by the argument.
	def cookie_values( key = nil )
		if key then
			_cookie_values_for_key( @cookies, key )
		else
			_cookie_values @cookies
		end
	end

	# Returns an array of languages for localization.
	# The order is the preferred order of languages.
	def languages
		langs   = []
		quality = {}

		if accept_language then
			accept_language.scan(/\s*([^,;]*)\s*;\s*q\s*=\s*([^,]*),|([^,]*),/) do |match|
				lang = match.pop  || match.shift
				q    = (match.pop || 1).to_f

				if lang =~ /([^-]*)-(.*)/ then lang = $1 end

				if quality[lang] then
					if q > quality[lang] then
						quality[lang] = q
					end
				else
					quality[lang] = q
				end
			end

			sorted = quality.sort do |a,b|
				if    a.last <  b.last then  1
				elsif a.last == b.last then  0
				elsif a.last >  b.last then -1
				end
			end

			sorted.each do |lang|
				langs << lang.first
			end
		end

		langs
	end

	def accept;            self.headers['HTTP_ACCEPT']          end
	def accept_charset;    self.headers['HTTP_ACCEPT_CHARSET']  end
	def accept_language;   self.headers['HTTP_ACCEPT_LANGUAGE'] end
	def auth_type;         self.headers['AUTH_TYPE']            end
	def content_length;    self.headers['CONTENT_LENGTH']       end
	def content_type;      self.headers['CONTENT_TYPE']         end
	def from;              self.headers['HTTP_FROM']            end
	def gateway_interface; self.headers['GATEWAY_INTERFACE']    end
	def path_info;         self.headers['PATH_INFO']            end
	def path_translated;   self.headers['PATH_TRANSLATED']      end
	def query_string;      self.headers['QUERY_STRING']         end
	def raw_cookie;        self.headers['HTTP_COOKIE']          end
	def referer;           self.headers['HTTP_REFERER']         end
	def remote_addr;       self.headers['REMOTE_ADDR']          end
	def remote_host;       self.headers['HTTP_HOST']            end
	def remote_ident;      self.headers['REMOTE_IDENT']         end
	def remote_user;       self.headers['REMOTE_USER']          end
	def request_method;    self.headers['REQUEST_METHOD']       end
	def script_name;       self.headers['SCRIPT_NAME']          end
	def server_name;       self.headers['SERVER_NAME']          end
	def server_port;       self.headers['SERVER_PORT']          end
	def server_protocol;   self.headers['SERVER_PROTOCOL']      end
	def server_software;   self.headers['SERVER_SOFTWARE']      end
	def uri;               self.headers['REQUEST_URI']          end
	def user_agent;        self.headers['HTTP_USER_AGENT']      end

	alias url uri

	private

	def _cookie_values( cookies )
		values = {}
		cookies.each { | cookie |
			values[ cookie.name ] = cookie.value
		}
		values
	end

	def _cookie_values_for_key( cookies, key )
		values = []
		cookies.each { | cookie |
			if cookie.name == key
				values << cookie.value
			end
		}
		values
	end
end


# A response object that is sent to a browser by a CKAdapter object.
#
# == Getting a response object
# You can get a response object by CKApplication#response or CKComponent#response.
#
# == Setting headers for a response object
# To send HTTP response headers, append a pair of key and value to headers.
# For example,
#
#  application.response.headers['Content-Type'] = 'text/html'
#
class CKResponse < CKMessage
	# Status code in HTTP. Default status is 200 ( OK ).
	attr_accessor :status

	STATUS = { 
		100 => 'Continue',
		101 => 'Switching Protocols',
		200 => 'OK',
		201 => 'Created',
		202 => 'Accepted',
		203 => 'Non-Authoritative Information',
		204 => 'No Content',
		205 => 'Reset Content',
		206 => 'Partial Content',
		300 => 'Multiple Choices',
		301 => 'Moved Permanently',
		302 => 'Found',
		303 => 'See Other',
		304 => 'Not Modified',
		305 => 'Use Proxy',
		307 => 'Temporary Redirect',
		400 => 'Bad Request',
		401 => 'Unauthorized',
		402 => 'Payment Required',
		403 => 'Forbidden',
		404 => 'Not Found',
		405 => 'Method Not Allowed',
		406 => 'Not Acceptable',
		407 => 'Proxy Authentication Required',
		408 => 'Request Timeout',
		409 => 'Conflict',
		410 => 'Gone',
		411 => 'Length Required',
		412 => 'Precondition Failed',
		413 => 'Request Entity Too Large',
		414 => 'Request-URI Too Long',
		415 => 'Unsupported Media Type',
		416 => 'Requested Range Not Satisfiable',
		417 => 'Expectation Failed',
		500 => 'Internal Server Error',
		501 => 'Not Implemented',
		502 => 'Bad Gateway',
		503 => 'Service Unavailable',
		504 => 'Gateway Timeout',
		505 => 'HTTP Version Not Supported'
	}

	def initialize( headers = nil )
		super
		@status = 200
		@headers['Content-Type'] = 'text/html'
	end

	def to_s
		response = ''
		response << ( __cookie || '' )
		response << ( _general_header  || '' )
		response << ( _response_header || '' )
		response << _entity_header
		response << EOL
		response << ( self.content     || '' ) unless redirect?
		response
	end

	private

	def _general_header
		header = ''
		header << ( __header('Cache-Control') || '' )
		header << ( __header('Connection') || '' )
		header << ( __header('Date') || '' )
		header << ( __header('Pragma') || '' )
		header << ( __header('Trailer') || '' )
		header << ( __header('Transfet-Encoding') || '' )
		header << ( __header('Upgrade') || '' )
		header << ( __header('Via') || '' )
		header << ( __header('Warning') || '' )
	end

	def _response_header
		header = ''
		header << ( __header('Accept-Ranges') || '' )
		header << ( __header('Age') || '' )
		header << ( __header('ETag') || '' )
		header << ( __header('Location') || '' )
		header << ( __header('Proxy-Authenticate') || '' )
		header << ( __header('Retry-After') || '' )
		header << ( __header('Server') || '' )
		header << ( __header('Vary') || '' )
		header << ( __header('WWW-Authenticate') || '' )
	end

	def _entity_header
		header = ''
		header << ( __header('Allow') || '' )
		header << ( __header('Content-Encoding') || '' )
		header << ( __header('Content-Language') || '' )
		header << ( __header('Content-Length') || '' )
		header << ( __header('Content-Location') || '' )
		header << ( __header('Content-MD5') || '' )
		header << ( __header('Content-Range') || '' )
		header << __content_type
		header << ( __header('Content-Disposition') || '' )
		header << ( __header('Expires') || '' )
		header << ( __header('Last-Modified') || '' )
	end

	def __header( header )
		"#{header}: #{self.headers[header]}#{EOL}" if self.headers[header]
	end

 	def __content_type
		header = "Content-Type: #{self.headers['Content-Type']}"
		header << "; charset=#{self.encoding}" if self.encoding
		header << EOL
	end

	def __cookie
		return if @cookies.empty?

		header = ''
		@cookies.each { | cookie |
			header << "Set-Cookie: #{cookie.to_s}"
			header << EOL
		}
		header
	end

	public

	def status_line
		"#{http_version_line} #@status #{STATUS[@status]}#{EOL}"
	end

	# Sends a temporary redirect response to the client using the specified URL.
	def set_redirect( url )
		@status = 302
		@headers['Location'] = url
	end

	# Returns true if the response is setted redirect.
	def redirect?
		if ( @status == 302 ) or ( @status == 307 ) then
			true
		else
			false
		end
	end
end


# CKCookie is a class for cookie.
# To send cookies to a browser needs to create cookie objects 
# and set them to a response object. Instead of creating cookie objects,
# you can also get cookie objects from a request object.
#
# CKCookie objects have a pair of a cookie name and value.
# If you make the objects have multiple values for one name, 
# you must write code by yourself. 
# 
# == Controlling cookie objects
#
# === Creating cookies
# Give arguments of initialize() a name or a pair of name/value.
# The value of cookie is omittable.
#
#  cookie = CKCookie.new( name, value )
#
# === Getting cookies from a request object
# CKRequest has some methods for getting cookies.
# The methods are cookie(key), cookies, cookie_value(key), cookie_values(key).
# See also CKRequest.
#
# === Setting cookies to a response object
# CKResponse has methods for setting cookies. These methods are 
# defined in CKMessage, the superclass of CKResponse.
# Use add_cookie(cookie) and remove_cookie(cookie).
class CKCookie
	# Name of the cookie.
	attr_accessor :name

	# Value of the cookie.
	attr_accessor :value

	# Restricts the cookie in the site.
	attr_accessor :path

	# Domain that can receive the cookie.
	attr_accessor :domain

	# Expiry date. You set Time object to the cookie object.
	# The value is formatted when the cookie is returned.
	attr_accessor :expires

	# Decides whether the cookie is encrypted or not.
	attr_accessor :secure

	class << self
		# Parse raw cookie string and return an array of cookies.
 		def parse_raw_cookie( raw_cookie )
			cookies = []
			return cookies unless raw_cookie

			raw_cookie.split('; ').each do |pairs|
				name, value = pairs.split('=',2)
				name  = CKUtilities.unescape_url name
				value = CKUtilities.unescape_url value

				cookies << CKCookie.new( name, value )
			end

			cookies
		end
	end

	def initialize( name, value = nil, domain = nil, path = nil, secure = false )
		@name   = name
		@value  = value
		@domain = domain
		@path   = path
		@secure = secure
	end

	def to_s
		buf = "#@name="
		if @value   then buf << CKUtilities.escape_url(@value.to_s) end
		if @domain  then buf << "; domain=#@domain" end
		if @path	   then buf << "; path=#@path" end
		if @expires then buf << "; expires=#{CKUtilities.date(@expires)}" end
		if @secure == true then buf << '; secure' end
		buf
	end
end


