/*--------------------------------------------------------------------------+
$Id: UnmodifiableCollection.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.collections;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;

/**
 * This is a wrapper for a {@link Collection} prohibiting all calls which would
 * modify its contents. As the construction of this class is performed in
 * constant time it is preferred over copying the collection (which takes linear
 * time). Using this class is also preferred to using the
 * <code>unmodifiableX()</code> in class {@link Collections} as they return
 * the collection base type that does not signal, that the object is
 * unmodifiable. Using the classes in this package makes unmodifiability more
 * explicit.
 * <p>
 * All prohibited methods throw an {@link UnsupportedOperationException}. The
 * class is nearly the same as the one returned by
 * {@link Collections#unmodifiableCollection(Collection)}, but by making it a
 * public class we can make the return value of some methods more explicit.
 * <p>
 * This collection is serializable if the wrapped collection is serializable.
 * 
 * @author Benjamin Hummel
 * @author $Author: juergens $
 * 
 * @version $Revision: 26283 $
 * @levd.rating GREEN Hash: E7479B41B1D5AFEB227EFF86689F0DAA
 */
public class UnmodifiableCollection<E> implements Collection<E>, Serializable {

	/** The underlying collection. */
	private final Collection<E> c;

	/**
	 * Creates a new unmodifiable collection from another collection. All
	 * modifications to the underlying collection will directly be visible in
	 * this wrapper.
	 */
	public UnmodifiableCollection(Collection<E> c) {
		if (c == null) {
			throw new IllegalArgumentException(
					"Underlying collection may not be null!");
		}
		this.c = c;
	}

	/** {@inheritDoc} */
	public boolean isEmpty() {
		return c.isEmpty();
	}

	/** {@inheritDoc} */
	public int size() {
		return c.size();
	}

	/** {@inheritDoc} */
	public boolean contains(Object o) {
		return c.contains(o);
	}

	/** {@inheritDoc} */
	public boolean containsAll(Collection<?> other) {
		return c.containsAll(other);
	}

	/** {@inheritDoc} */
	public UnmodifiableIterator<E> iterator() {
		return new UnmodifiableIterator<E>(c.iterator());
	}

	/** {@inheritDoc} */
	public Object[] toArray() {
		return c.toArray();
	}

	/** {@inheritDoc} */
	public <T> T[] toArray(T[] a) {
		return c.toArray(a);
	}

	/**
	 * Operation is not supported.
	 * 
	 * @throws UnsupportedOperationException
	 */
	public boolean add(E arg0) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Operation is not supported.
	 * 
	 * @throws UnsupportedOperationException
	 */
	public boolean addAll(Collection<? extends E> arg0) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Operation is not supported.
	 * 
	 * @throws UnsupportedOperationException
	 */
	public void clear() {
		throw new UnsupportedOperationException();
	}

	/**
	 * Operation is not supported.
	 * 
	 * @throws UnsupportedOperationException
	 */
	public boolean remove(Object arg0) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Operation is not supported.
	 * 
	 * @throws UnsupportedOperationException
	 */
	public boolean removeAll(Collection<?> arg0) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Operation is not supported.
	 * 
	 * @throws UnsupportedOperationException
	 */
	public boolean retainAll(Collection<?> arg0) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Returns a string representation of this collection.
	 */
	@Override
	public String toString() {
		return c.toString();
	}
}