/*--------------------------------------------------------------------------+
$Id: TreeUtils.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.tree;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * Utility class for tree construction.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: 3CA3EEE13CB7C2935D6E01EE3AD9DA30
 */
public class TreeUtils {

	/**
	 * This method creates a tree from a set of paths.
	 * 
	 * @param <T>
	 *            the node type to be created
	 * 
	 * @param <K>
	 *            the key type used by the nodes
	 * @param paths
	 *            a set of paths where each path is a list of path elements
	 *            (keys)
	 * @param handler
	 *            the handler used for creating the tree.
	 * @return The root node of the tree. The root node (as generated by
	 *         {@link ITreeNodeHandler#createRoot()} is always returned, even if
	 *         the set of paths is empty.
	 */
	public static <T, K> T createTree(Set<List<K>> paths,
			ITreeNodeHandler<T, K> handler) {
		T root = handler.createRoot();

		for (List<K> path : paths) {
			insert(root, path, handler);
		}

		return root;
	}

	/**
	 * Utility method for creating a tree from paths described by strings.
	 * 
	 * @param <T>
	 *            the node type to be created
	 * @param paths
	 *            a set of paths where each path is described by a path
	 *            expression string, e.g. node1/node2/node3
	 * @param separator
	 *            regular expression that defines the separator between path
	 *            elements.
	 * @param handler
	 *            the handler used for creating the tree.
	 * @return The root node of the tree. The root node (as generated by
	 *         {@link ITreeNodeHandler#createRoot()} is always returned, even if
	 *         the set of paths is empty.
	 */
	public static <T> T createTreeFromStrings(Set<String> paths,
			String separator, ITreeNodeHandler<T, String> handler) {
		T root = handler.createRoot();

		for (String pathDescriptor : paths) {
			String[] pathElements = pathDescriptor.split(separator);
			insert(root, Arrays.asList(pathElements), handler);
		}

		return root;
	}

	/**
	 * Create a node from a path.
	 * 
	 * @param <T>
	 *            the node type to be created
	 * @param <K>
	 *            the key type used by the nodes
	 * @param node
	 *            the reference node the new node is added to
	 * @param path
	 *            the path is a list of path elements (keys)
	 * @param handler
	 *            the handler used for creating the tree
	 */
	private static <T, K> void insert(T node, List<K> path,
			ITreeNodeHandler<T, K> handler) {
		for (K pathElement : path) {
			node = handler.getOrCreateChild(node, pathElement);
		}
	}
}