/*--------------------------------------------------------------------------+
$Id: XMLUtilsTest.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.xml;

import java.io.File;
import java.io.IOException;

import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import edu.tum.cs.commons.filesystem.FileSystemUtils;
import edu.tum.cs.commons.string.StringUtils;
import edu.tum.cs.commons.test.CCSMTestCaseBase;

/**
 * Test class for {@link XMLUtils}.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: 91BEAF95C49B93787A8DACC9B5B6D49B
 */
public class XMLUtilsTest extends CCSMTestCaseBase {

	/** Name of the XML element (contains German umlauts). */
	private final static String ELEMENT_NAME = "x\u00c4\u00d6\u00dc";

	/** Attribute name */
	private final static String ATTRIBUTE_NAME = "attr";

	/** Attribute value (contains German umlauts). */
	private final static String ATTRIBUTE_VALUE = "hallo\u00e4\u00f6\u00fc";

	/** Test string for encoding tests. */
	private final String ENCODING_TEST_INPUT = "<?xml version='1.0' %s ?>"
			+ StringUtils.CR + "<" + ELEMENT_NAME + " " + ATTRIBUTE_NAME
			+ " ='" + ATTRIBUTE_VALUE + "'/>";

	/** Schema file. */
	private final File schema = useTestFile("test.xsd");

	/** Test with well-formed but invalid file. */
	public void testInvalidXML() throws IOException {
		File file = useTestFile("test01.xml");

		try {
			XMLUtils.parse(file, schema);
			fail("expected exception");
		} catch (SAXException e) {
			// expected
		}

	}

	/** Test with well-formed and valid file. */
	public void testValidXML() throws IOException, SAXException {
		File file = useTestFile("test02.xml");

		Document document = XMLUtils.parse(file, schema);

		assertEquals(21, document.getDocumentElement().getChildNodes()
				.getLength());
	}

	/** Test with non-well-formed file. */
	public void testNonWellFormedXML() throws IOException {
		File file = useTestFile("test03.xml");

		try {
			XMLUtils.parse(file, schema);
			fail("expected exception");
		} catch (SAXException e) {
			// expected
		}
	}

	/**
	 * Test with a file that has spaces in its path name.
	 */
	public void testFileInFolderWithSpaces() throws IOException, SAXException {
		File file = useTestFile("folder with spaces/test02.xml");

		assertTrue(file.exists());

		Document document = XMLUtils.parse(file, schema);
		assertEquals(21, document.getDocumentElement().getChildNodes()
				.getLength());

	}

	/** Test non-validating parse. */
	public void testNonValidating() throws SAXException, IOException {
		File file = useTestFile("test01.xml");

		Document document = XMLUtils.parse(file);

		assertEquals(3, document.getDocumentElement().getChildNodes()
				.getLength());

		file = useTestFile("test02.xml");

		document = XMLUtils.parse(file);

		assertEquals(21, document.getDocumentElement().getChildNodes()
				.getLength());

		file = useTestFile("test03.xml");

		try {
			XMLUtils.parse(file, schema);
			fail("expected exception");
		} catch (SAXException e) {
			// expected
		}
	}

	/**
	 * Our assumption is that we do not need to deal with encodings explicitly
	 * when working with xml. This test method is meant to document the
	 * assumptions we made.
	 */
	public void testEncoding() throws IOException, SAXException {
		assertEncodingWorks("UTF-8", "UTF-8");

		// UTF-8 is guessed correctly
		assertEncodingWorks("UTF-8", null);

		assertEncodingWorks("UTF-16", "UTF-16");
		// UTF-16 is guessed correctly
		assertEncodingWorks("UTF-16", null);

		assertEncodingWorks("UTF-16BE", "UTF-16BE");

		// UTF-16 is guessed correctly even without a byte order mark
		assertEncodingWorks("UTF-16BE", null);
		assertEncodingWorks("UTF-16LE", null);

		assertEncodingWorks("ISO-8859-1", "ISO-8859-1");

		// ISO-8859-1 is not guessed correctly
		assertEncodingError("ISO-8859-1", null);

		assertEncodingError("UTF-8", "UTF-16");
		assertEncodingError("UTF-16", "UTF-8");

	}

	/**
	 * Check if the encoding works as expected
	 * 
	 * @param fileEncoding
	 *            encoding used for the file
	 * @param xmlEncoding
	 *            encoding stored in the XML file
	 */
	public void assertEncodingWorks(String fileEncoding, String xmlEncoding)
			throws IOException, SAXException {
		File file = writeFile(fileEncoding, xmlEncoding);
		Document doc = XMLUtils.parse(file);
		// apparently the XML parser uses either UTF-8, UTF-16BE or UTF-16LE but
		// no other encodings
		String encoding = doc.getInputEncoding();
		assertTrue(encoding.equals("UTF-8") || encoding.equals("UTF-16BE")
				|| encoding.equals("UTF-16LE"));
		assertEquals(ELEMENT_NAME, doc.getDocumentElement().getNodeName());
		assertEquals(ATTRIBUTE_VALUE, doc.getDocumentElement().getAttribute(
				ATTRIBUTE_NAME));

	}

	/**
	 * Check if XML parser creates the expected exception.
	 */
	public void assertEncodingError(String fileEncoding, String xmlEncoding) {

		try {
			File file = writeFile(fileEncoding, xmlEncoding);
			XMLUtils.parse(file);
			fail();
		} catch (SAXException e) {
			// expected
		} catch (IOException e) {
			// expected
		}
	}

	/** Write the test file to disk. */
	private File writeFile(String fileEncoding, String xmlEncoding)
			throws IOException {
		File file = new File(getTmpDirectory(), "encoding-test_" + fileEncoding
				+ "_" + xmlEncoding + ".txt");

		FileSystemUtils.writeFile(file, createContent(xmlEncoding),
				fileEncoding);
		return file;
	}

	/**
	 * Create test content.
	 * 
	 * @param xmlEncoding
	 *            the encoding to be used in the XML header. If this is
	 *            <code>null</code>, encoding is left unspecified.
	 */
	private String createContent(String xmlEncoding) {
		String encodingSpecifier = StringUtils.EMPTY_STRING;
		if (xmlEncoding != null) {
			encodingSpecifier = " encoding='" + xmlEncoding + "'";
		}
		return String.format(ENCODING_TEST_INPUT, encodingSpecifier);
	}

}