/*
 * @(#)DirClassMetaDataWriter.java
 *
 * Copyright (C) 2002-2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;


/**
 * Knows how to store the meta-data in a repository.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 16, 2002
 */
public class DirClassMetaDataWriter implements IClassMetaDataWriter
{
    private DirMetaDataIO store;
    private IAnalysisModule am;
    private AnalysisModuleSet ams;
    
    DirClassMetaDataWriter( IAnalysisModule module, DirMetaDataIO data )
            throws IOException
    {
        if (data == null || module == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        this.store = data;
        this.am = module;
        this.ams = data.getAnalysisModuleSet();
    }
    
    
    /**
     * Writes the given class record and its associated marks for this
     * Analysis Module to the repository.  It will overwrite any existing
     * data.
     */
    public void writeClassRecord( ClassRecord cr )
            throws IOException
    {
        if (cr == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        checkClose();
        
        // create the record for this particular module
        cr = getAMClassRecord( cr );
        
        this.store.putClassRecord( this.am, cr );
    }
    
    
    /**
     * Closes off the writer's connection to the store.  Mark writers may
     * still have open connections that are still able to be used.
     */
    public void close()
            throws IOException
    {
        checkClose();
        // do not close the store!!!
        this.store = null;
    }
    
    
    /**
     * Creates a class record for this specific module.
     */
    private ClassRecord getAMClassRecord( ClassRecord cr )
            throws IOException
    {
        ClassRecord c2 = new ClassRecord( cr.getClassName(), cr.getClassCRC(),
            cr.getSourceFileName(), cr.getMethods(), this.ams );
        
        // get only the marks for this analysis module
        MarkRecord mr[] = cr.getMarksForAnalysisModule( this.am );
        for (int i = 0; i < mr.length; ++i)
        {
            // this guarantees a unique addition of marks
            c2.addMark( mr[i] );
        }
            
        return c2;
    }
    
    
    private void checkClose()
            throws IOException
    {
        if (this.store == null)
        {
            throw new IOException( "Writer has already been closed." );
        }
    }
    
    
    protected void finalize() throws Throwable
    {
        super.finalize();
        
        // class-based post condition
        if (this.store != null)
        {
            throw new IllegalStateException("Did not close writer.");
        }
    }
}

