#ifndef _MSV_H
#define _MSV_H 1

/** @file msv/msv.h
 *  @brief main public libmsv header file
 */

/**
 * @def LIBMSV_ERROR_SUCCESS
 * @brief Success or absence of error
 */
#define LIBMSV_ERROR_SUCCESS 0
/**
 * @def LIBMSV_ERROR_INVALID
 * @brief Use of certificate could not be determined valid or was determined invalid
 */
#define LIBMSV_ERROR_INVALID 1
/**
 * @def LIBMSV_ERROR_NOENVVAR
 * @brief Environment variable was needed but was not set
 */
#define LIBMSV_ERROR_NOENVVAR 2
/**
 * @def LIBMSV_ERROR_CURLINIT_FAILED
 * @brief curl_easy_init() failed for unknown reason
 */
#define LIBMSV_ERROR_CURLINIT_FAILED 3
/**
 * @def LIBMSV_ERROR_CURLCODE
 * @brief an unspecified curl function failed
 *
 * Until the next libmsv call, the curl error code should
 * be retained by libmsv for retrieval by msv_strerror(int)
 */
#define LIBMSV_ERROR_CURLCODE 4
/**
 * @def LIBMSV_ERROR_INCOMPATIBLE_AGENT
 * @brief MSVA was perceived as incompatible
 */
#define LIBMSV_ERROR_INCOMPATIBLE_AGENT 5
/**
 * @def LIBMSV_ERROR_BADARG
 * @brief An argument to the function was invalid
 */
#define LIBMSV_ERROR_BADARG 6

/**
 * @struct msv_response
 * @brief a response from the MSVA
 */
struct msv_response
{
  int valid; /**< 0 for invalid, 1 for valid */
  char *message; /**< text response to be displayed to the user */
};

/**
 * @fn const char *msv_strerror (int error_code)
 * @brief Return string describing libmsv error code
 * @param error_code return value of libmsv function
 * @return string describing error
 */
extern const char *msv_strerror (int error_code);
/**
 * @fn int msv_check_msva (char *url)
 * @brief Check suitability of Monkeysphere Validation Agent
 * @param url URL for MSVA or NULL to use MONKEYSPHERE_VALIDATION_AGENT_SOCKET environment variable
 * @return 0 for success, or libmsv error code
 */
extern int msv_check_msva (char *url);
/**
 * @fn int msv_query_agent (char *url, char *context, char *peertype, char *peername, char *pkctype, char *pkcdata, struct msv_response *response)
 * @brief Query validation agent for certificate validity
 * @param url URL for MSVA or NULL to use MONKEYSPHERE_VALIDATION_AGENT_SOCKET environment variable
 * @param context certificate use context, such as "https", "ssh", or "ike"
 * @param peertype type of peer on remote side, such as "client", "server", or "peer"
 * @param peername name of peer, such as "web.monkeysphere.info"
 * @param pkctype format of public key carrier data, such as "x509der", "x509pem", or "opensshpubkey"
 * @param pkcdata public key carrier data encoded in the format specified by \a pkctype
 * @param response pointer to msv_response struct that will be populated with response
 * @return 0 for valid use of certificate, 1 for invalid use of certificate, or libmsv error code
 */
extern int msv_query_agent (char *url, char *context, char *peertype,
			    char *peername, char *pkctype, char *pkcdata,
			    struct msv_response *response);

#endif /* _MSV_H */
