/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.core.output2;

import java.awt.Color;
import java.io.IOException;
import java.io.Reader;
import java.util.Random;
import static junit.framework.Assert.assertEquals;
import junit.framework.TestCase;
import org.openide.util.Exceptions;
import org.openide.windows.IOColorLines;
import org.openide.windows.IOColors;
import org.openide.windows.OutputEvent;
import org.openide.windows.OutputListener;
import org.openide.windows.OutputWriter;

/**
 *
 * @author mkleint
 */
public class NbIOTest extends TestCase {

    public NbIOTest(String testName) {
        super(testName);
    }

    public void test54117() throws Exception {
        NbIO io = new NbIO("test");
        assertFalse(io.isClosed());
        Reader str = io.getIn();
        assertNotNull(str);
        assertEquals(NbIO.IOReader.class, str.getClass());
        writeText(str);
        int read = str.read(new char[100]);
        // not eof..
        assertTrue(read != -1);
        writeEof(str);
        read = str.read(new char[100]);
        assertTrue(read == -1);
        //reseting
        io.getOut().close();
        io.getErr().close();
        io.dispose();
        io.getOut().reset();
        io.getErr().reset();
        
        str = io.getIn();
        writeText(str);
        read = str.read(new char[100]);
        // not eof..
        assertTrue(read != -1);
        writeEof(str);
        read = str.read(new char[100]);
        assertTrue(read == -1);
        
    }
    
    private void writeText(final Reader reader) {
              NbIO.IOReader rdr = (NbIO.IOReader)reader;
              rdr.pushText("hello");

    }
    private void writeEof(final Reader reader) {
              NbIO.IOReader rdr = (NbIO.IOReader)reader;
              rdr.eof();
    }
    
    public void testSynchronization223370SeveralTimes() {
        for (int i = 0; i < 10; i++) {
            try {
                checkSynchronization223370();
            } catch (InterruptedException ex) {
                Exceptions.printStackTrace(ex);
            }
        }
    }

    public void testLimit() throws IOException {
        NbIO io = new NbIO("test");
        io.getOut().println("first");
        NbWriter nbWriter = (NbWriter) io.getOut();
        OutWriter ow = nbWriter.out();
        AbstractLines lines = (AbstractLines) ow.getLines();
        lines.setOutputLimits(new OutputLimits(5, 1024, 3));
        assertEquals(2, lines.getLineCount());
        io.getOut().println("second");
        io.getOut().println("third");
        OutputListener ol = new OutputListener() {
            @Override
            public void outputLineSelected(OutputEvent ev) {
            }

            @Override
            public void outputLineAction(OutputEvent ev) {
            }

            @Override
            public void outputLineCleared(OutputEvent ev) {
            }
        };
        IOColorLines.println(io, "fourth", ol, true, Color.yellow);
        assertNotNull(lines.getLineInfo(3).getFirstListener(new int[]{1, 2}));

        IOColorLines.println(io, "fifth", ol, false, Color.red);

        assertEquals(3, lines.getLineCount());

        assertNotNull("fourth\n", lines.getLine(0));
        assertNotNull(lines.getLineInfo(0).getFirstListener(new int[2]));
        assertTrue(lines.isImportantLine(0));
        assertEquals(Color.yellow, lines.getLineInfo(0).getLineSegments()
                .iterator().next().getColor());


        assertEquals("fifth\n", lines.getLine(1));
        assertNotNull(lines.getLineInfo(1).getFirstListener(new int[2]));
        assertFalse(lines.isImportantLine(1));
        assertEquals(Color.red, lines.getLineInfo(1).getLineSegments()
                .iterator().next().getColor());

        IOColorLines.println(io, "sixth", ol, true, Color.green);
        assertEquals("sixth\n", lines.getLine(2));
        assertNotNull(lines.getLineInfo(2).getFirstListener(new int[2]));
        assertTrue(lines.isImportantLine(2));
        assertEquals(Color.green, lines.getLineInfo(2).getLineSegments()
                .iterator().next().getColor());

        assertEquals("", lines.getLine(3));
    }

    private void checkSynchronization223370() throws InterruptedException {
        final NbIO nbio = new NbIO("test223370");
        final int[] nullOuts = new int[1];
        final int[] nullIns = new int[1];
        Thread t1 = new Thread(new Runnable() {
            @Override
            public void run() {
                for (int i = 0; i < 10000; i++) {
                    OutputWriter out = nbio.getOut();
                    Reader in = nbio.getIn();
                    if (out == null) {
                        synchronized (nullOuts) {
                            nullOuts[0]++;
                        }
                    }
                    if (in == null) {
                        synchronized (nullIns) {
                            nullIns[0]++;
                        }
                    }
                }
            }
        });
        Thread t2 = new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    Thread.sleep(1);
                } catch (InterruptedException ex) {
                    Exceptions.printStackTrace(ex);
                }
                for (int i = 0; i < 10000; i++) {
                    nbio.closeInputOutput();
                }
            }
        });
        Thread t3 = new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    Thread.sleep(2);
                } catch (InterruptedException ex) {
                    Exceptions.printStackTrace(ex);
                }
                for (int i = 0; i < 10000; i++) {
                    nbio.dispose();
                }
            }
        });

        t1.start();
        t2.start();
        t3.start();

        t1.join();
        t2.join();
        t3.join();
        synchronized (nullIns) {
            assertEquals(0, nullIns[0]);
        }
        synchronized (nullOuts) {
            assertEquals(0, nullOuts[0]);
        }
    }

    public void testIOColorsImpl() {
        NbIO io = new NbIO("test");
        io.getOptions().setColorStandard(Color.RED);
        io.getOptions().setColorError(Color.WHITE);
        io.getOptions().setColorInput(Color.YELLOW);
        io.getOptions().setColorLink(Color.PINK);
        io.getOptions().setColorLinkImportant(Color.LIGHT_GRAY);
        io.getOptions().setColorDebug(Color.CYAN);
        io.getOptions().setColorWarning(Color.BLACK);
        io.getOptions().setColorFailure(Color.MAGENTA);
        io.getOptions().setColorSuccess(Color.BLUE);
        assertEquals(Color.RED,
                IOColors.getColor(io, IOColors.OutputType.OUTPUT));
        assertEquals(Color.WHITE,
                IOColors.getColor(io, IOColors.OutputType.ERROR));
        assertEquals(Color.YELLOW,
                IOColors.getColor(io, IOColors.OutputType.INPUT));
        assertEquals(Color.PINK,
                IOColors.getColor(io, IOColors.OutputType.HYPERLINK));
        assertEquals(Color.LIGHT_GRAY,
                IOColors.getColor(io, IOColors.OutputType.HYPERLINK_IMPORTANT));
        assertEquals(Color.CYAN,
                IOColors.getColor(io, IOColors.OutputType.LOG_DEBUG));
        assertEquals(Color.BLACK,
                IOColors.getColor(io, IOColors.OutputType.LOG_WARNING));
        assertEquals(Color.MAGENTA,
                IOColors.getColor(io, IOColors.OutputType.LOG_FAILURE));
        assertEquals(Color.BLUE,
                IOColors.getColor(io, IOColors.OutputType.LOG_SUCCESS));
    }

    public void testIOColorsImplSetting() {
        NbIO io = new NbIO("test");
        io.getOptions().setColorStandard(Color.ORANGE);
        io.getOptions().setColorDebug(Color.WHITE);
        assertEquals(Color.ORANGE,
                IOColors.getColor(io, IOColors.OutputType.OUTPUT));
        assertEquals(Color.WHITE,
                IOColors.getColor(io, IOColors.OutputType.LOG_DEBUG));
        IOColors.setColor(io, IOColors.OutputType.OUTPUT, Color.BLUE);
        IOColors.setColor(io, IOColors.OutputType.LOG_DEBUG, Color.GREEN);
        assertEquals(Color.BLUE,
                IOColors.getColor(io, IOColors.OutputType.OUTPUT));
        assertEquals(Color.GREEN,
                IOColors.getColor(io, IOColors.OutputType.LOG_DEBUG));
    }

    public void testBug201450() {
        NbIO io = new NbIO("test201450");
        io.getOut().println(randomString(1024, 1024));
        io.getOut().println(randomString(1024, 1024));
        io.dispose();
        io.closeInputOutput();
    }

    private String randomString(int lines, int lineLength) {
        StringBuilder sb = new StringBuilder();
        char lastChar = ' ';
        Random r = new Random(System.currentTimeMillis());
        for (int i = 0; i < lines; i++) {
            for (int j = 0; j < lineLength; j++) {
                char c = (char) ((r.nextDouble() * 127) + 1);
                if (lastChar == '\u001B' && c == '[') { // prevent ANSI seqs
                    continue;
                }
                sb.append(c);
                lastChar = c;
            }
            sb.append('\t');
            sb.append('\b');
            sb.append('\n');
        }
        return sb.toString();
    }

    public void testBug201450AndMultipleErases() {
        NbIO io = new NbIO("test201450bb");
        io.getOut().println("abcdef\t\t\b\b\b\b\th");
        io.dispose();
        io.closeInputOutput();
    }
}
