/* midgameEval.t.cc
 */
#include "osl/eval/ml/openMidEndingEval.h"
#include "osl/progress/ml/newProgress.h"
#include "osl/eval/ml/kingTable.h"
#include "osl/state/numEffectState.h"
#include "../consistencyTest.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>
#include <fstream>

class OpenMidEndingEvalTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(OpenMidEndingEvalTest);
  CPPUNIT_TEST(testInit);
  CPPUNIT_TEST(testBasic);
  CPPUNIT_TEST(testConsistentUpdate);
  CPPUNIT_TEST(testSymmetry);
  CPPUNIT_TEST(testIcc);
  CPPUNIT_TEST(testKing25EmptySquareNoEffect);
  CPPUNIT_TEST(testCaptureValue);
  CPPUNIT_TEST(testArray);
  CPPUNIT_TEST_SUITE_END();
public:
  void testInit();
  void testConsistentUpdate();
  void testSymmetry();
  void testIcc();
  void testBasic();
  void testKing25EmptySquareNoEffect();
  void testCaptureValue();
  void testArray();
};

CPPUNIT_TEST_SUITE_REGISTRATION(OpenMidEndingEvalTest);

using namespace osl;
using namespace osl::eval;
using namespace osl::eval::ml;

void OpenMidEndingEvalTest::testInit()
{
  CPPUNIT_ASSERT(OpenMidEndingEval::setUp());
  CPPUNIT_ASSERT(osl::progress::ml::NewProgress::setUp());
  CPPUNIT_ASSERT(OpenMidEndingEval::infty() < EvalTraits<BLACK>::MAX_VALUE - 10); 
}

void OpenMidEndingEvalTest::testConsistentUpdate()
{
  OpenMidEndingEval::setRandom();
  consistencyTestUpdate<OpenMidEndingEval>();
}

void OpenMidEndingEvalTest::testSymmetry()
{
  using namespace osl;
  OpenMidEndingEval::setRandom();

  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;
  for (int i=0;i<(OslConfig::inUnitTestShort() ? 10 : 200) && (ifs >> file_name) ; i++)
  {
    if (file_name == "") 
      break;
    file_name = OslConfig::testCsaFile(file_name);

    const Record rec=CsaFile(file_name).getRecord();
    const vector<osl::Move> moves=rec.getMoves();

    NumEffectState state(rec.getInitialState());
    OpenMidEndingEval eval(state);
    for (unsigned int i=0; i<moves.size(); i++) {
      const Move m = moves[i];
      state.makeMove(m);
      eval.update(state, m);

      NumEffectState state_r(state.rotate180());
      OpenMidEndingEval eval_r(state_r);
      if (abs(eval.value() - -eval_r.value()) > 2)
      {
	OpenMidEndingEval eval_l(state);
	std::cerr << state << state_r
		  << m << std::endl
		  << eval.value() << " " << eval_r.value() << " " << eval_l.value() <<std::endl;
      }
      CPPUNIT_ASSERT(abs(eval.value() - -eval_r.value()) <= 2);
#if 0
      // King8 is not horizontally symmetrical.
      NumEffectState state_h(state.flipHorizontal());
      OpenMidEndingEval eval_h(state_h);
      if (eval.value() != eval_h.value())
      {
	std::cerr << state << state_h << "\n";
      }
      CPPUNIT_ASSERT_EQUAL(eval.value(), eval_h.value());
#endif
    }
  }
}

void OpenMidEndingEvalTest::testIcc()
{
  OpenMidEndingEval::setRandom();
  const NumEffectState state(CsaString(
			       "P1-KY-KE * -KI *  * -GI+RY-KY\n"
			       "P2 *  *  * -OU *  * +NK *  * \n"
			       "P3-FU * -GI-FU-FU-FU *  * -FU\n"
			       "P4 *  * -FU *  *  *  *  *  * \n"
			       "P5 *  *  *  * +KA *  *  *  * \n"
			       "P6 * -KE *  *  *  *  *  *  * \n"
			       "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
			       "P8 *  * -TO+KI *  *  *  *  * \n"
			       "P9+KY+KE *  * +OU+KI+GI * +KY\n"
			       "P+00KA00KI00FU00FU\n"
			       "P-00HI00GI00FU00FU\n"
			       "+\n").getInitialState());
  const NumEffectState state_r(CsaString(
				 "P1-KY * -GI-KI-OU *  * -KE-KY\n"
				 "P2 *  *  *  *  * -KI+TO *  * \n"
				 "P3-FU * -FU-FU-FU-FU-FU * -FU\n"
				 "P4 *  *  *  *  *  *  * +KE * \n"
				 "P5 *  *  *  * -KA *  *  *  * \n"
				 "P6 *  *  *  *  *  * +FU *  * \n"
				 "P7+FU *  * +FU+FU+FU+GI * +FU\n"
				 "P8 *  * -NK *  * +OU *  *  * \n"
				 "P9+KY-RY+GI *  * +KI * +KE+KY\n"
				 "P+00HI00GI00FU00FU\n"
				 "P-00KA00KI00FU00FU\n"
				 "-\n").getInitialState());
  OpenMidEndingEval eval(state), eval_r(state_r);
  CPPUNIT_ASSERT(abs(eval.value() - -eval_r.value()) <= 2);
}

void OpenMidEndingEvalTest::testBasic()
{
  {
    const NumEffectState state(CsaString(
				 "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
				 "P2 *  *  *  *  *  *  * -KA * \n"
				 "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
				 "P4 *  *  *  *  *  *  *  *  * \n"
				 "P5 *  *  *  *  *  *  *  *  * \n"
				 "P6 *  *  *  *  *  *  *  *  * \n"
				 "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
				 "P8 * +KA *  *  *  *  * +HI * \n"
				 "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
				 "P+00HI\n"
				 "+\n").getInitialState());
    OpenMidEndingEval eval(state);
    CPPUNIT_ASSERT(eval.value() > 0);
  }
  {
    const NumEffectState state(CsaString(
				 "P1-KY-KE *  * -OU *  * -KE-KY\n"
				 "P2 *  *  *  *  *  *  * -KA * \n"
				 "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
				 "P4 *  *  *  *  *  *  *  *  * \n"
				 "P5 *  *  *  *  *  *  *  *  * \n"
				 "P6 * +GI *  *  *  *  * +GI * \n"
				 "P7+FU+KI+FU+FU+FU+FU+FU+KI+FU\n"
				 "P8 * +GI+KA * +OU * +HI+KI * \n"
				 "P9+KY+KI+KE *  *  * +KE+GI+KY\n"
				 "P+00HI00FU00FU\n"
				 "+\n").getInitialState());
    OpenMidEndingEval eval(state);
    CPPUNIT_ASSERT(eval.value() > 0);
  }
}

void OpenMidEndingEvalTest::testKing25EmptySquareNoEffect()
{
  OpenMidEndingEval::setUp();
  
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;
  for (int i=0;i<(OslConfig::inUnitTestShort() ? 10 : 200) && (ifs >> file_name) ; i++)
  {
    if (file_name == "") 
      break;
    file_name = OslConfig::testCsaFile(file_name);

    const Record rec=CsaFile(file_name).getRecord();
    const vector<osl::Move> moves=rec.getMoves();

    NumEffectState state(rec.getInitialState());
    CArray<int,2> last_opening, last_ending;
    for (unsigned int i=0; i<moves.size(); i++) {
      const Move m = moves[i];
      state.makeMove(m);

      if (i == 0) 
      {
	std::pair<CArray<int,2>, CArray<int,2> > oe = King25EmptySquareNoEffect::eval
	  (state, King25EmptySquareNoEffectOpening::weights(),
	   King25EmptySquareNoEffectEnding::weights());
	CPPUNIT_ASSERT(oe.first == King25EmptySquareNoEffectOpening::eval(state));
	CPPUNIT_ASSERT(oe.second == King25EmptySquareNoEffectEnding::eval(state));
	last_opening = oe.first;
	last_ending = oe.second;
      } 
      else 
      {
	std::pair<CArray<int,2>, CArray<int,2> > oe = King25EmptySquareNoEffect::evalWithUpdate
	  (state, m, 
	   King25EmptySquareNoEffectOpening::weights(),
	   King25EmptySquareNoEffectEnding::weights(),
	   last_opening, last_ending);
	CPPUNIT_ASSERT(oe.first == King25EmptySquareNoEffectOpening::eval(state));
	CPPUNIT_ASSERT(oe.second == King25EmptySquareNoEffectEnding::eval(state));
	last_opening = oe.first;
	last_ending = oe.second;
      }
    }
  }
}

void OpenMidEndingEvalTest::testCaptureValue()
{
  OpenMidEndingEval::setUp();
  for (int ptype=PTYPE_PIECE_MIN; ptype<=PTYPE_MAX; ++ptype) {
    PtypeO ptypeo = newPtypeO(BLACK, static_cast<Ptype>(ptype));
    CPPUNIT_ASSERT_EQUAL(OpenMidEndingEval::Piece_Value.captureValue(ptypeo)
			 * OpenMidEndingEval::seeScale(),
			 OpenMidEndingEval::captureValue(ptypeo));
    ptypeo = newPtypeO(WHITE, static_cast<Ptype>(ptype));
    CPPUNIT_ASSERT_EQUAL(OpenMidEndingEval::Piece_Value.captureValue(ptypeo)
			 * OpenMidEndingEval::seeScale(),
			 OpenMidEndingEval::captureValue(ptypeo));
  }
}

void OpenMidEndingEvalTest::testArray()
{
  OpenMidEndingEval evals[3];
  (void)evals[0];
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
