/* JListLogHandler.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.logging;

import java.awt.Frame;
import java.awt.Toolkit;
import java.awt.Window;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.ErrorManager;
import java.util.logging.Filter;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;

import javax.swing.AbstractListModel;
import javax.swing.JDialog;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;

/**
 * Log handler which outputs its log records to a {@link JList}, where
 * new records end up at the top.<p>
 * The simplest way to use this class is by adding the following statement
 * to the start of the application
 * <pre>
 *    JListLogHandler.createDialog ("Log window", "");
 * </pre>
 * This will make a window pop up with all logged messages, as soon as
 * a message of a sufficiently high level (a warning, by default) is sent to
 * the loggin system.<p>
 *
 * <b>Configuration:</b>
 * By default the handler is initialized using the following
 * {@link LogManager} configuration properties.
 * <ul>
 * <li>   be.ugent.caagt.swirl.logging.JListLogHandler.level
 * specifies the default level for the handler (defaults to {@code java.util.logging.Level.INFO}).
 * <li>   be.ugent.caagt.swirl.logging.JListLogHandler.filter
 * specifies the name of a {@link Filter} class to use (defaults to none).
 * <li>   be.ugent.caagt.swirl.logging.JListLogHandler.formatter
 * specifies the name of a {@link Formatter} class to use
 * (defaults to {@link SimpleFormatter}).
 * <li>   be.ugent.caagt.swirl.logging.JListLogHandler.popupLevel specifies
 * the level for which the handler should try to draw the attention of the user
 * by showing the window which contains the list and bringing it to the front
 * </ul>
 **/

public class JListLogHandler extends Handler {
    // TODO: make a default constructor which launches its own window
    
    private final JList jlist;
    
    private Level popupLevel;
    
    /**
     * Retrieve the popup log level, i.e., the minimum level
     * for which the handler will try to draw the attention of the user
     * by sounding a beep, making the parent window of the list visible
     * and moving it to the front.
     */
    public Level getPopupLevel() {
        return popupLevel;
    }
    
    /**
     * Set the popup log level.
     */
    public final void setPopupLevel(Level popupLevel) {
        this.popupLevel = popupLevel;
    }
    
    // convert a string to a level name
    private Level toLevel(String str, Level defaultValue) {
        if (str == null)
            return defaultValue;
        else {
            try {
                return Level.parse(str.trim());
            } catch (IllegalArgumentException ex) {
                return defaultValue;
            }
        }
    }
    
    // convert a class name to a filter instance
    private Filter toFilter(String className) {
        if (className == null)
            return null;
        try {
            return (Filter)Class.forName(className).newInstance();
        } catch (Exception ex) {
            return null;
        }
    }
    
    // convert a class name to a formatter instance
    private Formatter toFormatter(String className) {
        if (className == null)
            return new SimpleFormatter();
        try {
            return (Formatter)Class.forName(className).newInstance();
        } catch (Exception ex) {
            return new SimpleFormatter();
        }
    }
    
    //
    private final LogModel model;
    
    /**
     * Create a log handler which shall send its output to the given list.
     */
    public JListLogHandler(JList jlist) {
        this.jlist = jlist;
        this.closed = false;
        this.model = new LogModel();
        
        // configure
        LogManager manager = LogManager.getLogManager();
        String className = getClass().getName();
        setLevel(toLevel(manager.getProperty(className + "level"), Level.INFO));
        setPopupLevel(toLevel(manager.getProperty(className + "popupLevel"),
                Level.WARNING));
        setFilter(toFilter(manager.getProperty(className + "filter")));
        setFormatter(toFormatter(manager.getProperty(className + "formatter")));
        
        // initialize list
        jlist.setModel(model);
        jlist.setCellRenderer(new JListLogCellRenderer(getFormatter()));
    }
    
    /**
     * Draw attention to the logging window.
     */
    protected void drawAttention() {
        Window window = SwingUtilities.getWindowAncestor(jlist);
        if (window != null) {
            window.setVisible(true);
            window.toFront();
            Toolkit.getDefaultToolkit().beep();
        }
    }
    
    /**
     * Create a dialog which contains a list which is handled by a new handler,
     * registered to the named logger of the application (use the empty string
     * for the root logger). Initially this dialog is not visible to the application.
     */
    public static JDialog createLoggingDialog(String title, String loggerName) {
        JDialog dialog = new JDialog((Frame)null, title, false);
        dialog.setLocationByPlatform(true);
        JList list = new JList();
        Handler handler = new JListLogHandler(list);
        LogManager.getLogManager().getLogger(loggerName).addHandler(handler);
        dialog.add(new JScrollPane(list));
        dialog.pack();
        dialog.setSize(500, 300); // reasonable default size
        return dialog;
    }
    
    //
    private boolean closed;
    
    public void publish(LogRecord record) {
        
        if (closed) {
            reportError("Handler already closed", null, ErrorManager.WRITE_FAILURE);
            return ;
        }
        
        model.add(record);
        if (record.getLevel().intValue() >= getPopupLevel().intValue())
            drawAttention();
    }
    
    public void close() throws SecurityException {
        closed = true;
    }
    
    public void flush() {
        // ignore
    }
    
    
    /**
     * List model used by this handler.
     */
    private class LogModel extends AbstractListModel {
        
        // stores the records in reverse order
        private final List<LogRecord> records;
        
        public LogModel() {
            records = new ArrayList<LogRecord> ();
        }
        
        public Object getElementAt(int index) {
            return records.get(records.size()-index-1);
        }
        
        public int getSize() {
            return records.size();
        }
        
        public void add(LogRecord record) {
            records.add(record);
            fireIntervalAdded(this, 0, 0);
        }
        
    }
    
    
}

