# Copyright 2012 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Tests for `maastest.report`."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = []


import base64
from datetime import datetime
import email
import io
import logging
import os.path
import urllib2

from fixtures import TempDir
from maastest import report
import mock
import testtools


class TestMakeFilePayload(testtools.TestCase):

    def test_returns_MIMEApplication(self):
        payload = report.make_file_payload(
            self.getUniqueString(), self.getUniqueString())
        self.assertIsInstance(
            payload, email.mime.application.MIMEApplication)

    def test_sets_content_disposition_header(self):
        filename = self.getUniqueString()
        payload = report.make_file_payload(
            filename, self.getUniqueString())
        expected_disposition = (
            'attachment; name="%(name)s"; filename="%(name)s"' %
            {'name': filename})
        self.assertEqual(
            expected_disposition, payload['Content-Disposition'])

    def test_sets_payload(self):
        content = self.getUniqueString()
        filename = self.getUniqueString()
        payload = report.make_file_payload(filename, content)
        encoded_content = base64.b64encode(content)
        self.assertEqual(
            encoded_content,
            payload.get_payload().replace("\n", ""))


class TestCreateLaunchpadBlob(testtools.TestCase):

    def test_returns_MIMEMultipart(self):
        blob = report.create_launchpad_blob(self.getUniqueString(), False)
        self.assertIsInstance(blob, email.mime.multipart.MIMEMultipart)

    def test_sets_tags_header_for_success(self):
        blob = report.create_launchpad_blob(
            self.getUniqueString(), test_succeeded=True)
        self.assertEqual("success", blob['tags'])

    def test_sets_tags_header_for_failure(self):
        blob = report.create_launchpad_blob(
            self.getUniqueString(), test_succeeded=False)
        self.assertEqual("failure", blob['tags'])

    def test_sets_subject_header_for_success(self):
        blob = report.create_launchpad_blob(
            self.getUniqueString(), test_succeeded=True)
        self.assertEqual("maas-test success", blob['subject'])

    def test_sets_subject_header_for_failure(self):
        blob = report.create_launchpad_blob(
            self.getUniqueString(), test_succeeded=False)
        self.assertEqual("maas-test failure", blob['subject'])

    def test_sets_private_header(self):
        blob = report.create_launchpad_blob(self.getUniqueString(), False)
        self.assertEqual("yes", blob['private'])

    def test_sets_subscribers_header(self):
        blob = report.create_launchpad_blob(self.getUniqueString(), False)
        self.assertEqual("private-canonical-maas", blob['subscribers'])

    def test_attaches_results_as_payload(self):
        results = self.getUniqueString()
        blob = report.create_launchpad_blob(results, False)
        expected_payload = report.make_file_payload("maas-test.log", results)
        actual_payload = blob.get_payload()[0]
        self.assertEqual(
            expected_payload['Content-Disposition'],
            actual_payload['Content-Disposition'])
        self.assertEqual(
            expected_payload.get_payload(), actual_payload.get_payload())


class TestBuildUploadMIMEMultipart(testtools.TestCase):

    def setUp(self):
        super(TestBuildUploadMIMEMultipart, self).setUp()
        self.launchpad_blob = report.create_launchpad_blob(
            self.getUniqueString(), False)

    def test_returns_MIMEMultipart(self):
        upload_multipart = report.build_upload_mime_multipart(
            self.launchpad_blob)
        self.assertIsInstance(
            upload_multipart,
            email.mime.multipart.MIMEMultipart)

    def test_adds_submit_payload(self):
        upload_multipart = report.build_upload_mime_multipart(
            self.launchpad_blob)
        submit_payload = upload_multipart.get_payload()[0]
        self.assertIsInstance(submit_payload, email.mime.Text.MIMEText)
        self.assertEqual(
            submit_payload['Content-Disposition'],
            'form-data; name="FORM_SUBMIT"')
        self.assertEqual('1', submit_payload.get_payload())

    def test_adds_blob_as_payload(self):
        upload_multipart = report.build_upload_mime_multipart(
            self.launchpad_blob)
        blob_payload = upload_multipart.get_payload()[1]
        self.assertEqual(
            blob_payload['Content-Disposition'],
            'form-data; name="field.blob"; filename="x"')
        self.assertEqual(
            self.launchpad_blob.as_string().encode('ascii'),
            blob_payload.get_payload())


class TestBuildUploadRequest(testtools.TestCase):

    def setUp(self):
        super(TestBuildUploadRequest, self).setUp()
        self.launchpad_blob = report.create_launchpad_blob(
            self.getUniqueString(), False)
        self.launchpad_form_data = report.build_upload_mime_multipart(
            self.launchpad_blob)

    def test_returns_http_request(self):
        request = report.build_upload_request(
            "http://example.com", self.launchpad_form_data)
        self.assertIsInstance(request, urllib2.Request)

    def test_sets_request_content_type_header(self):
        request = report.build_upload_request(
            "http://example.com", self.launchpad_form_data)
        self.assertRegexpMatches(
            request.get_header("Content-type"),
            'multipart/form-data; boundary=.*')

    def test_sets_request_data(self):
        flat_buffer = io.BytesIO()
        generator = email.generator.Generator(flat_buffer, mangle_from_=False)
        generator.flatten(self.launchpad_form_data)
        flattened_data = flat_buffer.getvalue()

        request = report.build_upload_request(
            "http://example.com", self.launchpad_form_data)
        self.assertEqual(flattened_data, request.data)


class TestUploadBlob(testtools.TestCase):

    def setUp(self):
        super(TestUploadBlob, self).setUp()
        self.mock_opener = mock.MagicMock()
        self.mock_result = self.mock_opener.open.return_value
        self.patch(
            report, 'build_opener',
            mock.MagicMock(return_value=self.mock_opener))

    def test_builds_opener(self):
        results = self.getUniqueString()
        blob = report.create_launchpad_blob(results, False)
        report.upload_blob(blob)
        report.build_opener.assert_has_calls(
            [mock.call(urllib2.HTTPSHandler, report.NO_PROXY_HANDLER)])

    def test_opens_url(self):
        results = self.getUniqueString()
        blob = report.create_launchpad_blob(results, False)
        mock_request = mock.MagicMock()
        mock_build_upload_request = mock.MagicMock(
            return_value=mock_request)
        self.patch(report, 'build_upload_request', mock_build_upload_request)
        report.upload_blob(blob)
        self.mock_opener.open.assert_has_calls([mock.call(mock_request)])

    def test_returns_token(self):
        token = self.getUniqueString()
        mock_info = mock.MagicMock(
            return_value={'X-Launchpad-Blob-Token': token})
        self.mock_result.info = mock_info

        blob = report.create_launchpad_blob(self.getUniqueString(), False)
        self.assertEqual(token, report.upload_blob(blob))

    def test_returns_none_if_request(self):
        self.mock_opener.open.return_value = None
        self.mock_opener.open.side_effect = Exception(self.getUniqueString())
        mock_error = mock.MagicMock()
        self.patch(logging, 'error', mock_error)

        blob = report.create_launchpad_blob(self.getUniqueString(), False)
        expected_message = ('Unable to connect to Launchpad.')
        token = report.upload_blob(blob)
        mock_error.assert_has_calls([mock.call(expected_message, exc_info=1)])
        self.assertEqual(None, token)


class TestReportTestResults(testtools.TestCase):

    def test_uploads_blob(self):
        token = self.getUniqueString()
        blob = email.mime.multipart.MIMEMultipart()
        mock_logger = mock.MagicMock()
        mock_create_launchpad_blob = mock.MagicMock(return_value=blob)
        mock_upload_blob = mock.MagicMock(return_value=token)
        mock_write_test_results = mock.MagicMock()

        self.patch(logging, 'info', mock_logger)
        self.patch(report, 'create_launchpad_blob', mock_create_launchpad_blob)
        self.patch(report, 'upload_blob', mock_upload_blob)
        self.patch(report, 'write_test_results', mock_write_test_results)

        report.report_test_results(
            self.getUniqueString(), test_succeeded=False)
        mock_upload_blob.assert_has_calls([mock.call(blob)])
        mock_logger.assert_has_calls([
            mock.call("Test results uploaded to Launchpad."),
            mock.call(
                "Visit "
                "https://bugs.launchpad.net/maas-test-reports/+filebug/%s "
                "to file a bug and complete the maas-test reporting process."
                % token)
            ])


class TestWriteTestResults(testtools.TestCase):

    def test_writes_results_to_file(self):
        temp_dir = self.useFixture(TempDir()).path
        results = self.getUniqueString()
        log_file_name = self.getUniqueString()

        report.write_test_results(results, log_file_name, temp_dir)

        with open(os.path.join(temp_dir, log_file_name), 'rb') as log_file:
            self.assertEqual(results, log_file.read().strip())

    def test_creates_log_dir(self):
        temp_dir = self.useFixture(TempDir()).path
        results = self.getUniqueString()
        log_dir = os.path.join(temp_dir, self.getUniqueString())
        log_file_name = self.getUniqueString()

        self.assertFalse(os.path.exists(log_dir))
        report.write_test_results(results, log_file_name, log_dir)
        self.assertTrue(os.path.exists(log_dir))

    def test_saves_log_with_timestamp(self):
        temp_dir = self.useFixture(TempDir()).path
        results = self.getUniqueString()
        mock_logger = mock.MagicMock()

        self.patch(logging, 'info', mock_logger)

        results = self.getUniqueString()
        now = datetime(2013, 12, 17, 7, 18, 0)
        now_string = now.strftime("%Y-%m-%d_%H:%M:%S")
        expected_filename = 'maas-test.%s.log' % now_string
        expected_file_path = os.path.join(temp_dir, expected_filename)
        report.write_test_results(results, log_dir=temp_dir, now=now)
        self.assertIn(
            mock.call("Test log saved to %s." % expected_file_path),
            mock_logger.mock_calls
            )
        self.assertTrue(os.path.exists(expected_file_path))
