/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_SYNCMANAGER_H_
#define MCLOUD_API_SYNCMANAGER_H_

#include <mcloud/api/taskqueue.h>
#include <mcloud/api/downloadtask.h>
#include <mcloud/api/uploadtask.h>
#include <mcloud/api/visibility.h>

#include <memory>
#include <vector>

namespace mcloud {
namespace api {

class ClientPriv;
class SyncManagerPriv;

/*!
    \class SyncManager
    \brief SyncManager handles all the requests to upload and download. 
    It's constructed around two threads for content synchronization. One for content uploading to cloud, another for cloud item downloading to local. 
    Both thread are running asynchronously.
    Calls \sa add_download_tasks() by passing a content id list for cloud content download,
    Calls \sa add_upload_tasks() by passing a local file list path for local content upload.
 */

class MCLOUD_API_DLL_PUBLIC SyncManager {
  public:
    typedef std::shared_ptr<SyncManager> Ptr;

    typedef TaskQueue<DownloadTask::Ptr> DownloadList;

    typedef TaskQueue<UploadTask::Ptr>  UploadList;

    typedef std::vector<std::string>    Stringlist;

    virtual ~SyncManager() = default;

    SyncManager(const SyncManager&) = delete;

    SyncManager& operator=(const SyncManager &) = delete;

    /*!
     * \brief Starts to run download or upload tasks from sync-up thread.
     * The tasks can be resumed by sync-up thread if sync manager is paused or canceled before.
     * \sa pause(), cancel()
     */
    void start();

    /*!
     * \brief Cancels all download or upload tasks from sync-up thread if sync manager is running.
     * \note A canceled download or upload task has no chance to run again by sync-up thread.
     * \sa start(), pause()
     */
    void cancel();

    /*!
     * \brief Pauses download or upload task from sync-up thread if sync manager is running.
     * Sync-up thread can continue to run previous paused task by calling \sa start().
     * \sa start(), cancel()
     */
    void pause();

    /*!
     * \brief Add a new task with a given \a content id into download queue.
     * \return a download task object that is pushed in sync-up manager.
     * \throw std::runtime_error if error occurs.
     * \sa add_download_tasks(), DownloadTask
     */
    DownloadTask::Ptr add_download_task(const std::string &content_id);

    /*!
     * \brief Pushes a download buffer callback item \a buffer_cb to sync manager and add regarding upload task into download queue.
     * A download buffer object consists of the following fields
     *      - `write_cb` (std::function<size_t(void *dest, size_t buf_size)>)
     *         buffer reading callback function.
     *         The buffer area pointed at by the pointer \a dest should be filled up with at most \a buf_size number of bytes.
     *      - `content_id` (string)
     *         The id of content on mcloud.
     * \return a download task object that is pushed in sync-up manager.
     * \throw std::runtime_error if error occurs.
     * \sa add_download_tasks(), DownloadTask
     */
    DownloadTask::Ptr add_download_task(const DownloadBufferCb &buffer_cb);

    /*!
     * \brief Pushes a upload request item \a request_item to sync manager and add regarding upload task into upload queue.
     * A upload request object consists of the following fields
     *      - `file_path` (string)
     *         local file path.
     *      - `folder_id` (string)
     *         mcloud folder id which stores content in uploads folder.
     *      - `content_name` (string)
     *         uploaded content name.
     * If content_name is empty, local file name will be used as a content name displayed on mcloud.
     * If folder_id is empty, content will be uploaded into root folder.
     * \return a download task object that are pushed in sync-up manager.
     * \throw std::runtime_error if error occurs.
     * \note A file name can not contain any of following characters:
     *      '\', '/', ':', '*', '?', '"', '<', '>', '|'
     * \note Sync manager will detect if the uploading file is already existing on cloud, 
     * if so, it simply sets the status of upload task completed to avoid duplicated upload.
     * \sa add_download_tasks(), UploadRequest, UploadTask,
     */
    UploadTask::Ptr add_upload_task(const UploadRequest &request_item);

    /*!
     * \brief Pushes a new upload request object \a buffer_cb to sync manager and add regarding upload task into upload queue.
     * A upload buffer request object consists of the following fields
     *      - `read_cb` (std::function<size_t(void *dest, size_t buf_size)>)
     *         buffer reading callback function.
     *         The buffer area pointed at by the pointer \a dest should be filled up with at most \a buf_size number of bytes.
     *      - `buffer_size` (size_t)
     *         total size of data buffers.
     *      - `folder_id` (string)
     *         mcloud folder id which stores content in uploads folder.
     *      - `content_name` (string)
     *         uploaded content name, should not be empty.
     * If folder_id is empty, content will be uploaded into root folder.
     * \return a upload task object that are pushed in sync-up manager.
     * \throw std::runtime_error if error occurs.
     * \note A file name can not contain any of following characters:
     *      '\', '/', ':', '*', '?', '"', '<', '>', '|'
     * \sa add_download_tasks(), UploadRequest, UploadTask,
     */
    UploadTask::Ptr add_upload_task(const UploadBufferCb &buffer_cb);

    /*!
     * \brief Returns a download task list hold by sync up manager.
     * Each item in the list contains basic information of associated task.
     * \sa upload_queue(), DownloadTask
     */
    DownloadList download_queue();

    /*!
     * \brief Returns a upload task list hold by sync up manager. 
     * Each item in the list contains basic information of associated task.
     * \sa download_queue(), UploadTask
     */
    UploadList   upload_queue();

private:
    SyncManager(ClientPriv* client_priv);

    friend class ClientPriv;

    std::shared_ptr<SyncManagerPriv> p_;
};

}
}

#endif // MCLOUD_API_SYNCMANAGER_H_
