/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#include "client_priv.h"
#include "downloadtask_priv.h"
#include "config.h"

#include <mcloud/api/cloudfolder.h>
#include <mcloud/api/cloudcontent.h>
#include <mcloud/api/diskinfo.h>
#include <mcloud/api/downloadtask.h>
#include <mcloud/api/uploadtask.h>
#include <mcloud/api/cloudresource.h>
#include <mcloud/api/outlink.h>
#include <mcloud/api/syncmanager.h>
#include <mcloud/api/exceptions.h>

#include <core/net/error.h>
#include <core/net/uri.h>
#include <core/net/http/client.h>
#include <core/net/http/request.h>
#include <core/net/http/response.h>

#include <tinyxml2.h>

#include <boost/archive/iterators/base64_from_binary.hpp>
#include <boost/archive/iterators/binary_from_base64.hpp>
#include <boost/archive/iterators/transform_width.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>
#include <boost/regex.hpp>

#include <openssl/md5.h>
#include <iomanip>
#include <iostream>

namespace http = core::net::http;
namespace net = core::net;

namespace fs = boost::filesystem;
namespace alg = boost::algorithm;

using namespace mcloud::api;
using namespace std;

namespace {

void log(const std::string &info) {
#ifndef NDEBUG
    cout << info << endl;
#else
    (void)(info);
#endif
}

static constexpr const char * content_type = "text/xml;UTF-8";
static constexpr const char * register_app_name = "和彩云ubuntu";
static constexpr const char * thirdparty_anonymous_account = "thirdparty_anonymous_account";

std::string encode64(const string& val) {
    using namespace boost::archive::iterators;
    typedef base64_from_binary<transform_width<std::string::const_iterator, 6, 8>> iter;
    auto tmp = std::string(iter(std::begin(val)), iter(std::end(val)));
    return tmp.append((3 - val.size() % 3) % 3, '=');
}

static std::string md5sum(std::string filepath) {
    std::ifstream  ifs(filepath, std::ios::binary);

    if (!ifs.good()) {
        std::cerr << "file is invalid: " << filepath << std::endl;
        return std::string();
    }

    unsigned char hash[MD5_DIGEST_LENGTH];
    MD5_CTX mdContext;
    char data[1024];

    MD5_Init(&mdContext);
    while(ifs.good()) {
        ifs.read(data, sizeof(data));
        MD5_Update (&mdContext, data, ifs.gcount());
    }
    MD5_Final (hash,&mdContext);

    std::stringstream stream;
    for(int i = 0; i < MD5_DIGEST_LENGTH; i++) {
        stream << std::hex << std::setw(2) << std::setfill('0') << (int)hash[i];
    }

    return stream.str();
}

static void append_child(tinyxml2::XMLPrinter &printer,
                         const std::string & field,
                         const std::string & value) {
    printer.OpenElement(field.c_str());
    printer.PushText(value.c_str());
    printer.CloseElement();
}

static string generate_diskinfo_body() {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("getDiskInfo");
    append_child(printer, "MSISDN", thirdparty_anonymous_account);
    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_content_list_body(int start_index = -1,
                                         int count = 200,
                                         CloudContent::Type content_type = CloudContent::Type::All,
                                         const std::string & shared_folder_id = std::string()) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("getDisk");

    //index on mcloud starts from 1, not 0
    if (start_index == 0)
        start_index = 1;

    append_child(printer, "MSISDN", thirdparty_anonymous_account);
    append_child(printer, "catalogID", shared_folder_id.empty() ? "root" : shared_folder_id.c_str());
    append_child(printer, "filterType", "0");
    append_child(printer, "catalogSortType", "0");
    append_child(printer, "contentType", std::to_string((int)content_type).c_str());
    append_child(printer, "contentSortType", "0");
    append_child(printer, "sortDirection", "1");
    append_child(printer, "startNumber",std::to_string(start_index).c_str());
    append_child(printer, "endNumber",  std::to_string(start_index + count - 1).c_str());
    append_child(printer, "catalogType", "-1");

    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_content_body(const std::string &content_id) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("getContentInfo");
    append_child(printer, "ContentID", content_id);
    append_child(printer, "CatalogID", "");
    append_child(printer, "ownerMSISDN", thirdparty_anonymous_account);
    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_delete_body(const std::vector<std::string> &item_ids) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("delCatalogContent");

    append_child(printer, "MSISDN", thirdparty_anonymous_account);
    printer.OpenElement("catalogIDs");
    printer.PushAttribute("length", std::to_string(item_ids.size()).c_str());
    for (auto & item_id: item_ids) {
        append_child(printer, "ID", item_id.c_str());
    }
    printer.CloseElement();

    printer.OpenElement("contentIDs");
    printer.PushAttribute("length", std::to_string(item_ids.size()).c_str());
    for (auto & content_id: item_ids) {
        append_child(printer, "ID", content_id.c_str());
    }
    printer.CloseElement();

    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_copy_body(const std::vector<std::string> &dir_ids,
                                 const std::vector<std::string> &content_ids,
                                 const std::string &new_folder_id) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("copyContentCatalog");

    append_child(printer, "MSISDN", thirdparty_anonymous_account);
    printer.OpenElement("contentInfoList");
    printer.PushAttribute("length", std::to_string(content_ids.size()).c_str());
    for (auto & content_id: content_ids) {
        append_child(printer, "ID", content_id.c_str());
    }
    printer.CloseElement();

    printer.OpenElement("catalogInfoList");
    printer.PushAttribute("length", std::to_string(dir_ids.size()).c_str());
    for (auto & dir_id: dir_ids) {
        append_child(printer, "ID", dir_id.c_str());
    }
    printer.CloseElement();

    append_child(printer, "newCatalogID", new_folder_id);

    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_update_folder_body(const std::string &folder_id,
                                          const std::string &new_folder_name) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("updateCatalogInfo");
    append_child(printer, "MSISDN", thirdparty_anonymous_account);
    append_child(printer, "catalogID", folder_id);
    append_child(printer, "catalogName", new_folder_name);
    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_create_folder_body(const std::string &folder_name,
                                          const std::string &folder_id) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("createCatalogExt");
    printer.OpenElement("createCatalogExtReq");

    append_child(printer, "parentCatalogID", folder_id);
    append_child(printer, "newCatalogName", folder_name);
    append_child(printer, "ownerMSISDN", thirdparty_anonymous_account);
    append_child(printer, "catalogType", "0");

    printer.CloseElement();
    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_look_up_body(const std::string &name,
                                    const std::string &folder_id,
                                    CloudResource::Property property) {
#if 1
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("advanceSearch");
    printer.OpenElement("advanceSearchReq");

    append_child(printer, "account", thirdparty_anonymous_account);
    append_child(printer, "scope", std::to_string((int)property));

    printer.OpenElement("fileCond");

    append_child(printer, "prtCtlgID", folder_id);
    append_child(printer, "name", name);

    printer.CloseElement();

    printer.CloseElement();
    printer.CloseElement();
#else
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("simpleSearch");
    printer.OpenElement("simpleSearchReq");

    append_child(printer, "account", thirdparty_anonymous_account);
    append_child(printer, "scope", "1100000000000000");
    append_child(printer, "keyword", name);

    printer.CloseElement();
    printer.CloseElement();
#endif
    return string(printer.CStr());
}

static string generate_move_body(const Client::Stringlist &dir_ids,
                                 const Client::Stringlist &content_ids,
                                 const std::string &folder_id) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("moveContentCatalog");

    append_child(printer, "MSISDN", thirdparty_anonymous_account);

    printer.OpenElement("contentInfoList");
    printer.PushAttribute("length", std::to_string(content_ids.size()).c_str());
    for (auto & content_id: content_ids) {
        append_child(printer, "ID", content_id.c_str());
    }
    printer.CloseElement();

    printer.OpenElement("catalogInfoList");
    printer.PushAttribute("length", std::to_string(dir_ids.size()).c_str());
    for (auto & dir_id: dir_ids) {
        append_child(printer, "ID", dir_id.c_str());
    }
    printer.CloseElement();

    append_child(printer, "newCatalogID", folder_id);

    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_extranet_body(const Client::Stringlist &dir_ids,
                                     const Client::Stringlist &content_ids) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("getOutLink");
    printer.OpenElement("getOutLinkReq");

    append_child(printer, "account", thirdparty_anonymous_account);
    printer.OpenElement("caIDLst");
    printer.PushAttribute("length", std::to_string(dir_ids.size()).c_str());
    for (auto & dir_id: dir_ids) {
        append_child(printer, "item", dir_id.c_str());
    }
    printer.CloseElement();

    printer.OpenElement("coIDLst");
    printer.PushAttribute("length", std::to_string(content_ids.size()).c_str());
    for (auto & content_id: content_ids) {
        append_child(printer, "item", content_id.c_str());
    }
    printer.CloseElement();

    append_child(printer, "pubType", "1");

    printer.CloseElement();
    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_download_body(const std::string &content_id) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("downloadRequest");
    append_child(printer, "appName", register_app_name);
    append_child(printer, "MSISDN", thirdparty_anonymous_account);

    vector<string> content_ids = {{content_id}};
    string content_list = alg::join(content_ids, "|");

    append_child(printer, "contentID", content_list.c_str());
    append_child(printer, "OwnerMSISDN", "");
    append_child(printer, "entryShareCatalogID", "");
    append_child(printer, "operation", "0");
    append_child(printer, "fileVersion", "-1");

    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_upload_body(const UploadRequest & request_item) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("pcUploadFileRequest");
    append_child(printer, "ownerMSISDN", thirdparty_anonymous_account);
    append_child(printer, "fileCount", "1");

    int64_t file_size = 0;
    if (fs::exists(request_item.file_path)) {
        file_size = fs::file_size(request_item.file_path);
    } else {
        throw std::runtime_error("file not exists");
    }
    append_child(printer, "totalSize", std::to_string(file_size).c_str());

    printer.OpenElement("uploadContentList");
    printer.PushAttribute("length", "1");

    printer.OpenElement("uploadContentInfo");
    append_child(printer, "contentName", request_item.content_name.empty() ?
                            fs::path(request_item.file_path).filename().c_str() : request_item.content_name);
    append_child(printer, "contentSize", std::to_string(file_size).c_str());
    append_child(printer, "contentDesc", "");
    append_child(printer, "contentTAGList", "");
    
    //The parameter to check upload files' md5sum 
    //"isNeedupload" field in response body will indicate if the file exists in mcloud.
    append_child(printer, "digest", md5sum(request_item.file_path)); 

    printer.CloseElement();
    printer.CloseElement();

    append_child(printer, "newCatalogName", "");
    append_child(printer, "parentCatalogID", request_item.folder_id);
    printer.CloseElement();

    return string(printer.CStr());
}

static string generate_upload_body(const UploadBufferCb & buffer_cb) {
    tinyxml2::XMLPrinter printer;
    printer.OpenElement("pcUploadFileRequest");
    append_child(printer, "ownerMSISDN", thirdparty_anonymous_account);
    append_child(printer, "fileCount", "1");

    append_child(printer, "totalSize", std::to_string(buffer_cb.buffer_size).c_str());

    printer.OpenElement("uploadContentList");
    printer.PushAttribute("length", "1");

    printer.OpenElement("uploadContentInfo");
    append_child(printer, "contentName", buffer_cb.content_name); 
    append_child(printer, "contentSize", std::to_string(buffer_cb.buffer_size).c_str());
    append_child(printer, "contentDesc", "");
    append_child(printer, "contentTAGList", "");

    printer.CloseElement();

    printer.CloseElement();

    append_child(printer, "newCatalogName", "");
    append_child(printer, "parentCatalogID", buffer_cb.folder_id);
    printer.CloseElement();

    return string(printer.CStr());
}

template<typename T>
static void set_auth_exception(std::shared_ptr<std::promise<T>> prom, 
                               const std::string json_data) {

    std::string error_info = json_data;
	boost::cmatch cap;
    const boost::regex exp("{\"error\":\"(.*?)\".*");
	if(boost::regex_match(json_data.c_str(), cap, exp)){ 
        error_info = cap[1];
    }

    prom->set_exception(make_exception_ptr(CredentialException(error_info)));
}

template<typename T>
static void set_exception(std::shared_ptr<std::promise<T>> prom,
                          const tinyxml2::XMLElement *root) {
    ClientPriv::ServerCode err_code= (ClientPriv::ServerCode)stoi(root->Attribute("resultCode"));
    switch(err_code) {
        case ClientPriv::ServerCode::CATALOGID_INVALID:
        case ClientPriv::ServerCode::CONTENTID_INVALID:
            prom->set_exception(make_exception_ptr(InvalidIDException(root->Attribute("desc"))));
        break;
        case ClientPriv::ServerCode::CATALOG_NOT_EXIST:
        case ClientPriv::ServerCode::CONTENT_AND_CATALOG_NOT_EXIST:
        case ClientPriv::ServerCode::CONTENT_NOT_EXIST:
            prom->set_exception(make_exception_ptr(NonExistentException(root->Attribute("desc"))));
        break;
        case ClientPriv::ServerCode::USER_SPACE_LACKED:
            prom->set_exception(make_exception_ptr(OutofSpaceException(root->Attribute("desc"))));
        break;
        case ClientPriv::ServerCode::PARAMETER_INVALID:
            prom->set_exception(make_exception_ptr(ParameterInvalidException(root->Attribute("desc"))));
        break;
        case ClientPriv::ServerCode::CREDENTIAL_FAILED:
            prom->set_exception(make_exception_ptr(CredentialException(root->Attribute("desc"))));
        break;
        default:
            prom->set_exception(make_exception_ptr(runtime_error(root->Attribute("desc"))));
        break;
    }
}

}

class ClientPriv::HttpClient {
public:
    HttpClient(int request_timeout) :
        client_(http::make_client()),
        worker_ { [this]() {client_->run();} },
        cancelled_(false) {
        config_.request_timeout = request_timeout;
    }

    ~HttpClient() {
        client_->stop();
        if (worker_.joinable()) {
            worker_.join();
        }
    }

    std::shared_ptr<core::net::http::Client> client_;

    std::thread worker_;

    Config config_;

    std::mutex config_mutex_;

    std::atomic<bool> cancelled_;

    void get(const net::Uri::Path &path,
            const net::Uri::QueryParameters &parameters,
            http::Request::Handler &handler) {
        auto configuration = net_config(path, parameters);
        configuration.header.add("User-Agent", config_.user_agent);

        auto request = client_->head(configuration);
        request->async_execute(handler);
    }

    void post(const net::Uri::Path &path,
            const net::Uri::QueryParameters &parameters,
            const std::string &postbody,
            const std::string &content_type,
            http::Request::Handler &handler) {
        std::lock_guard<std::mutex> lock(config_mutex_);
        http::Request::Configuration configuration = net_config(path, parameters);
        configuration.header.add("User-Agent", config_.user_agent);
        configuration.header.add("Content-Length", std::to_string(postbody.size()));
        configuration.header.add("Content-Type", content_type);

        auto request = client_->post(configuration, postbody, content_type);
        request->async_execute(handler);
    }

    http::Request::Configuration net_config(const net::Uri::Path &path,
                                            const net::Uri::QueryParameters &parameters) {
        http::Request::Configuration configuration;
        net::Uri::QueryParameters complete_parameters(parameters);
        if (!config_.access_token.empty()) {
            configuration.header.add("Authorization",
                    "Bearer " + config_.access_token);
        } else {
            configuration.header.add("Authorization",
                    "Basic " + encode64(config_.client_id + ":" + config_.client_password));
        }

        if (getenv("MCLOUD_LOCAL_SERVER_URL")) {
            config_.apiroot = string(getenv("MCLOUD_LOCAL_SERVER_URL"));
            configuration.header.set("Authorization", "Bearer " + config_.access_token);
        }

        net::Uri uri = net::make_uri(config_.apiroot, path,
                complete_parameters);
        configuration.uri = client_->uri_to_string(uri);

        return configuration;
    }

    http::Request::Progress::Next progress_changed(
            const http::Request::Progress&) {
        return cancelled_ ?
                http::Request::Progress::Next::abort_operation :
                http::Request::Progress::Next::continue_operation;
    }

    template<typename T>
    future<T> async_get(const net::Uri::Path &path,
            const net::Uri::QueryParameters &parameters,
            const function<T(const tinyxml2::XMLElement *root)> &func) {
        auto prom = make_shared<promise<T>>();

        http::Request::Handler handler;
        handler.on_progress(
                bind(&ClientPriv::HttpClient::progress_changed, this, placeholders::_1));
        handler.on_error([prom](const net::Error& e)
        {
            prom->set_exception(make_exception_ptr(runtime_error(e.what())));
        });
        handler.on_response(
            [prom,func](const http::Response& response)
            {
                tinyxml2::XMLDocument doc;
                tinyxml2::XMLError error = doc.Parse(response.body.c_str());
                const tinyxml2::XMLElement *root = doc.FirstChildElement();

                if (error != tinyxml2::XML_SUCCESS) {
                    prom->set_exception(make_exception_ptr(runtime_error("xml parse failed")));
                } else if (response.status != http::Status::ok) {
                    prom->set_exception(make_exception_ptr(runtime_error(root->GetText())));
                } else {
                    prom->set_value(func(root));
                }
            }
        );

        get(path, parameters, handler);

        return prom->get_future();
    }

    template<typename T>
    future<T> async_post(const net::Uri::Path &path,
            const net::Uri::QueryParameters &parameters,
            const std::string &postmsg,
            const std::string &content_type,
            const function<T(const tinyxml2::XMLElement *root)> &func) {
        auto prom = make_shared<promise<T>>();

        http::Request::Handler handler;
        handler.on_progress(
                bind(&ClientPriv::HttpClient::progress_changed, this, placeholders::_1));
        handler.on_error([prom](const net::Error& e)
        {
            prom->set_exception(make_exception_ptr(runtime_error(e.what())));
        });
        handler.on_response(
            [path, prom,func](const http::Response& response)
            {
                auto iter = std::find_if(path.cbegin(), path.cend(), [=](const string & pathitem){
                    return (pathitem == "OAuth2");
                });

                //mcloud seperate authentication server and content server,
                //1.if authentication failed, responses server gives it back are json-based data. 
                //2.for generaic content access, responses server gives it back are xml-based data,
                //just as API document mentioned.
                tinyxml2::XMLDocument doc;
                tinyxml2::XMLError error = tinyxml2::XML_SUCCESS;
                if (iter != path.end()) {
                    std::string xml_response = "<result resultCode=\"0\"><![CDATA[" + response.body + "]]></result>";
                    error = doc.Parse(xml_response.c_str());
                } else {
                    error = doc.Parse(response.body.c_str());
                }
                const tinyxml2::XMLElement *result = doc.FirstChildElement();

                if (error != tinyxml2::XML_SUCCESS) {
                    set_auth_exception(prom, response.body);
                } else if ((response.status != http::Status::created &&
                       response.status != http::Status::ok &&
                       response.status != http::Status::no_content ) ||
                       strcmp(result->Attribute("resultCode"), "0") != 0) {
                    set_exception(prom, result);
                } else {
                    prom->set_value(func(result));
                }
            }
        );

        post(path, parameters, postmsg, content_type, handler);

        return prom->get_future();
    }
};

ClientPriv::ClientPriv(int request_timeout)
    : httpclient_(std::make_shared<HttpClient>(request_timeout))
    , sync_manager_(std::shared_ptr<SyncManager>(new SyncManager(this))) {
}

ClientPriv::~ClientPriv() {
    httpclient_->cancelled_ = true;
}

string ClientPriv::cloud_sync_folder_id() {
    Client::ResourceList folderlist = cloud_root_folder_list();
    for (auto &folder: folderlist) {
        mcloud::api::CloudFolder::Ptr cloudfolder = std::static_pointer_cast<mcloud::api::CloudFolder>(folder);
        if (cloudfolder->folder_type() == mcloud::api::CloudFolder::Type::Sync){
            return cloudfolder->id();
        }
    }

    return std::string();
}

void ClientPriv::set_access_token(const string &access_token) {
    httpclient_->config_.access_token = access_token;
}

bool ClientPriv::exist_on_cloud(const std::string &file_path,
                                const std::string &folder_id) {
    if (!fs::exists(file_path))
        throw std::runtime_error(string("file not exists: ").append(file_path));

    UploadRequest request_item{file_path, folder_id, ""};
    string postbody = generate_upload_body(request_item);

    return exist_on_cloud_internal(postbody);
}

DiskInfo ClientPriv::disk_info() {
    std::string postbody = generate_diskinfo_body();

    log(postbody);

    auto disk_future = httpclient_->async_post<DiskInfo>(
    { "richlifeApp", "devapp", "IUser" }, { }, postbody, content_type,
                [](const tinyxml2::XMLElement *root) {
        return DiskInfo(root);
    });

    return get_or_throw(disk_future);
}

Client::ResourceList ClientPriv::cloud_content_list(int start_index,
                                                int count,
                                                CloudContent::Type type,
                                                const std::string & folder_id) {
    std::string postbody = generate_content_list_body(start_index,
                                                      count,
                                                      type,
                                                      folder_id);
    return cloud_content_list_internal(postbody);
}

CloudContent::Ptr ClientPriv::content_info(const string &content_id) {
    string postbody = generate_content_body(content_id);
    return content_item_internal(postbody);
}

CloudFolder::Ptr ClientPriv::create_folder(const string &folder_name,
                                           const string &folder_id) {
    string postbody = generate_create_folder_body(folder_name, folder_id);
    return create_folder_internal(postbody);
}

Client::ResourceList ClientPriv::look_up(const string &name,
                                         const string &folder_id,
                                         CloudResource::Property property) {
    string postbody = generate_look_up_body(name, folder_id, property);
    return look_up_internal(postbody);
}

bool ClientPriv::move_items(const Client::Stringlist &folder_ids,
                            const Client::Stringlist &content_ids,
                            const std::string &folder_id) {
    string postbody = generate_move_body(folder_ids, content_ids,
                                         folder_id);
    return move_items_internal(postbody);
}

bool ClientPriv::update_folder(const string &folder_id,
                               const string &new_folder_name) {
    string postbody = generate_update_folder_body(folder_id, new_folder_name);
    return update_internal(postbody);
}

bool ClientPriv::delete_contents(const Client::Stringlist &content_ids) {
    string postbody = generate_delete_body(content_ids);
    return delete_internal(postbody);
}

Client::Stringlist ClientPriv::copy_folders(const Client::Stringlist &folder_ids,
                                            const std::string &folder_id) {
    string postbody = generate_copy_body(folder_ids, vector<string>{},
                                         folder_id);
    return copy_internal(postbody);
}

Client::Stringlist ClientPriv::copy_contents(const Client::Stringlist &content_ids,
                                             const std::string &folder_id) {
    string postbody = generate_copy_body(vector<string>{}, content_ids,
                                         folder_id);
    return copy_internal(postbody);
}

Client::OutlinkList ClientPriv::create_folder_extranet_link(
        const Client::Stringlist &folder_ids) {
    string postbody = generate_extranet_body(folder_ids, Client::Stringlist{});
    return create_extranet_internal(postbody);
}

Client::OutlinkList ClientPriv::create_content_extranet_link(
        const Client::Stringlist &content_ids) {
    string postbody = generate_extranet_body(Client::Stringlist{}, content_ids);
    return create_extranet_internal(postbody);
}

DownloadTaskPriv::Ptr ClientPriv::create_download_link(const string &content_id) {
    string postbody = generate_download_body({{content_id}});

    log(postbody);

    auto future = httpclient_->async_post<DownloadTaskPriv::Ptr>(
            { "richlifeApp", "devapp", "IUploadAndDownload" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {

        auto link_list = root->FirstChildElement("String");
        if (link_list) {
            auto link = link_list->FirstChildElement();
            while(link) {
                return std::make_shared<DownloadTaskPriv>(link);
            }

            if (link_list->GetText()) {
                return std::make_shared<DownloadTaskPriv>(link_list);
            }
        }
        return DownloadTaskPriv::Ptr();
    });

    return get_or_throw(future);
}

DownloadTaskPriv::Ptr ClientPriv::create_download_link(const DownloadBufferCb &buffer_cb) {
    string postbody = generate_download_body(buffer_cb.content_id);

    log(postbody);

    auto future = httpclient_->async_post<DownloadTaskPriv::Ptr>(
            { "richlifeApp", "devapp", "IUploadAndDownload" }, { }, postbody, content_type,
            [buffer_cb](const tinyxml2::XMLElement *root) {

        auto link_list = root->FirstChildElement("String");
        if (link_list) {
            auto link = link_list->FirstChildElement();
            while(link) {
                return std::make_shared<DownloadTaskPriv>(link, buffer_cb.write_cb);
            }

            if (link_list->GetText()) {
                return std::make_shared<DownloadTaskPriv>(link_list, buffer_cb.write_cb);
            }
        }
        return DownloadTaskPriv::Ptr();
    });

    return get_or_throw(future);
}

UploadTaskPriv::Ptr ClientPriv::create_upload_link(const UploadRequest &request_item) {
    if (!fs::exists(request_item.file_path)) 
        throw std::runtime_error(string("file not exists: ").append(request_item.file_path));
                            
    string postbody = generate_upload_body(request_item);

    log(postbody);

    auto future = httpclient_->async_post<UploadTaskPriv::Ptr>(
            { "richlifeApp", "devapp", "IUploadAndDownload" }, { }, postbody, content_type,
            [request_item](const tinyxml2::XMLElement *root) {
        auto upload_res = root->FirstChildElement("uploadResult");
        std::string task_id = upload_res->FirstChildElement("uploadTaskID")->GetText();
        std::string redirection_url;
        if (upload_res->FirstChildElement("redirectionUrl")) {
            redirection_url = upload_res->FirstChildElement("redirectionUrl")->GetText();
        }

        auto content_list = upload_res->FirstChildElement("newContentIDList");
        if (content_list) {
            auto content = content_list->FirstChildElement();
            while(content) {
                return std::make_shared<UploadTaskPriv>(content,
                                                        task_id,
                                                        redirection_url,
                                                        request_item.file_path);
            }
        }

        return UploadTaskPriv::Ptr();
    });

    return get_or_throw(future);
}

UploadTaskPriv::Ptr ClientPriv::create_upload_link(const UploadBufferCb &buffer_cb) {
    string postbody = generate_upload_body(buffer_cb);

    log(postbody);

    auto future = httpclient_->async_post<UploadTaskPriv::Ptr>(
            { "richlifeApp", "devapp", "IUploadAndDownload" }, { }, postbody, content_type,
            [buffer_cb](const tinyxml2::XMLElement *root) {
        SyncManager::UploadList results;

        auto upload_res = root->FirstChildElement("uploadResult");
        std::string task_id = upload_res->FirstChildElement("uploadTaskID")->GetText();

        std::string redirection_url;
        if (upload_res->FirstChildElement("redirectionUrl")) {
            redirection_url = upload_res->FirstChildElement("redirectionUrl")->GetText();
        }

        auto content_list = upload_res->FirstChildElement("newContentIDList");
        if (content_list) {
            auto content = content_list->FirstChildElement();
            while(content) {
                return std::make_shared<UploadTaskPriv>(content,
                                                       task_id,
                                                       redirection_url,
                                                       buffer_cb.buffer_size,
                                                       buffer_cb.read_cb);
            }
        }

        return UploadTaskPriv::Ptr();
    });

    return get_or_throw(future);
}

Client::ResourceList ClientPriv::cloud_root_folder_list() {
    std::string postbody = generate_content_list_body();

    log(postbody);

    auto future = httpclient_->async_post<Client::ResourceList>(
            { "richlifeApp", "devapp", "ICatalog" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *res) {
        auto root = res->FirstChildElement();
        Client::ResourceList results;

        auto folder_list = root->FirstChildElement("catalogList");
        if (folder_list) {
            auto folder = folder_list->FirstChildElement();
            while(folder) {
                results.emplace_back(std::shared_ptr<CloudFolder>(new CloudFolder(folder)));
                folder = folder->NextSiblingElement();
            }
        }

        return  results;
    });

    return get_or_throw(future);
}

Client::ResourceList ClientPriv::cloud_content_list_internal(const string &postbody) {
    log(postbody);

    auto future = httpclient_->async_post<Client::ResourceList>(
            { "richlifeApp", "devapp", "ICatalog" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {
        Client::ResourceList results;

        auto res = root->FirstChildElement();
        auto folder_list = res->FirstChildElement("catalogList");
        if (folder_list) {
            auto folder_ele = folder_list->FirstChildElement();
            while(folder_ele) {
                results.emplace_back(std::shared_ptr<CloudFolder>(
                                         new CloudFolder(folder_ele)));
                folder_ele = folder_ele->NextSiblingElement();
            }
        }

        auto content_list = res->FirstChildElement("contentList");
        if (content_list) {
            auto content = content_list->FirstChildElement();
            while(content) {
                results.emplace_back(std::shared_ptr<CloudContent>(
                                         new CloudContent(content)));
                content = content->NextSiblingElement();
            }
        }

        return results;
    });

    return get_or_throw(future);
}

CloudContent::Ptr ClientPriv::content_item_internal(const string &postbody) {
    log(postbody);

    auto future = httpclient_->async_post<CloudContent::Ptr>(
            { "richlifeApp", "devapp", "IContent" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) -> CloudContent::Ptr {
        if (strcmp(root->Attribute("resultCode"), "0"))
            return nullptr;

        auto content = root->FirstChildElement("contentInfo");
        return std::shared_ptr<CloudContent>(new CloudContent(content));
    });

    return get_or_throw(future);
}

CloudFolder::Ptr ClientPriv::create_folder_internal(const string &postbody) {
    log(postbody);

    auto future =  httpclient_->async_post<CloudFolder::Ptr>(
            { "richlifeApp", "devapp", "ICatalog" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) -> CloudFolder::Ptr {
        if (strcmp(root->Attribute("resultCode"), "0"))
            return nullptr;

        auto folder = root->FirstChildElement("catalogInfo");
        return std::shared_ptr<CloudFolder>(new CloudFolder(folder));
    });

    return get_or_throw(future);
}

Client::ResourceList ClientPriv::look_up_internal(const string &postbody) {
    log(postbody);

    auto future =  httpclient_->async_post<Client::ResourceList>(
            { "richlifeApp", "devapp", "ISearch" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {
        Client::ResourceList results;

        auto res = root->FirstChildElement();
        auto folder_list = res->FirstChildElement("srchCtlgList");
        if (folder_list) {
            auto folder_ele = folder_list->FirstChildElement();
            while(folder_ele) {
                results.emplace_back(std::shared_ptr<CloudFolder>(
                                         new CloudFolder(folder_ele)));
                folder_ele = folder_ele->NextSiblingElement();
            }
        }

        auto content_list = res->FirstChildElement("srchCntList");
        if (content_list) {
            auto content_ele = content_list->FirstChildElement();
            while(content_ele) {
                results.emplace_back(std::shared_ptr<CloudContent>(
                                         new CloudContent(content_ele)));
                content_ele = content_ele->NextSiblingElement();
            }
        }

        return results;
    });

    return get_or_throw(future);
}

bool ClientPriv::exist_on_cloud_internal(const string &postbody) {
    log(postbody);

    auto future = httpclient_->async_post<bool>(
            { "richlifeApp", "devapp", "IUploadAndDownload" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {
        bool is_exists = false;

        auto upload_ele = root->FirstChildElement("uploadResult");
        auto new_content_list = upload_ele->FirstChildElement("newContentIDList");
        if (new_content_list) {
            auto new_content_ele = new_content_list->FirstChildElement();
            if(new_content_ele) {
                is_exists = strcmp(new_content_ele->FirstChildElement("isNeedUpload")->GetText(), "1");
            }
        }

        return is_exists;
    });

    return get_or_throw(future);
}

bool ClientPriv::move_items_internal(const string &postbody) {
    return update_internal(postbody);
}

bool ClientPriv::update_internal(const string &postbody) {
    log(postbody);

    auto future =  httpclient_->async_post<bool>(
            { "richlifeApp", "devapp", "ICatalog" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {
        return !strcmp(root->Attribute("resultCode"), "0");
    });

    return get_or_throw(future);
}

bool ClientPriv::delete_internal(const string &postbody) {
    log(postbody);

    auto future =  httpclient_->async_post<bool>(
            { "richlifeApp", "devapp", "IContent" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {
        return !strcmp(root->Attribute("resultCode"), "0");
    });

    return get_or_throw(future);
}

Client::Stringlist ClientPriv::copy_internal(const string &postbody) {
    log(postbody);

    auto future =  httpclient_->async_post<Client::Stringlist>(
            { "richlifeApp", "devapp", "ICatalog" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {
        Client::Stringlist results;
        if (strcmp(root->Attribute("resultCode"), "0"))
            return results;

        auto new_item = root->FirstChildElement("array")->FirstChildElement("item");
        while (new_item) {
            results.emplace_back(new_item->GetText());
            new_item = new_item->NextSiblingElement();
        }

        return results;
    });

    return get_or_throw(future);
}

Client::OutlinkList ClientPriv::create_extranet_internal(const string &postbody) {
    log(postbody);

    auto future = httpclient_->async_post<Client::OutlinkList>(
            { "richlifeApp", "devapp", "IOutLink" }, { }, postbody, content_type,
            [](const tinyxml2::XMLElement *root) {
        Client::OutlinkList results;

        auto link_list = root->FirstChildElement("getOutLinkRes")->FirstChildElement("getOutLinkResSet");
        if (link_list) {
            auto link = link_list->FirstChildElement();
            while(link) {
                results.emplace_back(new Outlink(link));
                link = link->NextSiblingElement();
            }
        }

        return results;
    });

    return get_or_throw(future);
}

bool ClientPriv::refersh_token(const string &refresh_token) {
    const std::string content_type = "application/x-www-form-urlencoded";
    const std::string postbody = "grant_type=refresh_token&refresh_token="
                                    + refresh_token;
    log(postbody);

    auto future = httpclient_->async_post<bool>(
            { "oauthApp", "OAuth2", "refreshToken" }, { }, postbody, content_type,
            [this](const tinyxml2::XMLElement *root) {
        boost::cmatch cap;
        const boost::regex exp(".*\"(.*?)\"\\}");
        if (boost::regex_match(root->GetText(), cap, exp)) {
            httpclient_->config_.access_token = cap[1];
            return true;
        }
        return false;
    });

    return get_or_throw(future);
}

void ClientPriv::cancel() {
    httpclient_->cancelled_ = true;
}

SyncManager::Ptr ClientPriv::sync_manager() {
    return sync_manager_;
}

template<typename T> T ClientPriv::get_or_throw(std::future<T> &f) {
    if (f.wait_for(std::chrono::seconds(httpclient_->config_.request_timeout)) != std::future_status::ready) {
        throw HttpTimeoutException("HTTP request timeout");
    }

    return f.get();
}
