/*
 * Copyright (C) 2016 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of version 3 of the GNU Lesser General Public License as published
 * by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by:  Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_CLIENTPRIV_H_
#define MCLOUD_API_CLIENTPRIV_H_

#include <mcloud/api/client.h>
#include <mcloud/api/visibility.h>
#include <mcloud/api/cloudresource.h>

#include <future>
#include <string>
#include <memory>

#include "downloadtask_priv.h"
#include "uploadtask_priv.h"

namespace mcloud {
namespace api {

class MCLOUD_API_DLL_LOCAL ClientPriv {
  public:

    enum class ServerCode {
        CATALOGID_INVALID = 9101,
        CONTENTID_INVALID = 9102,
        CONTENT_AND_CATALOG_NOT_EXIST = 9149,
        CATALOG_NOT_EXIST = 9406,
        USER_SPACE_LACKED = 9424,
        CONTENT_NOT_EXIST = 9438,
        PARAMETER_INVALID = 200000400,
        CREDENTIAL_FAILED = 200000401
    };

    typedef std::shared_ptr<ClientPriv> Ptr;

    ClientPriv(int request_timeout = 10);

    virtual ~ClientPriv();

    DiskInfo disk_info();

    std::string cloud_sync_folder_id();

    void set_access_token(const std::string &access_token);

    CloudContent::Ptr   content_info(const std::string &content_id);

    CloudFolder::Ptr    create_folder(const std::string &folder_name,
                                      const std::string &folder_id);

    Client::ResourceList look_up(const std::string &name,
                                 const std::string &folder_id,
                                 CloudResource::Property property = CloudResource::Property::Content);

    bool refersh_token(const std::string &refresh_token);

    bool move_items(const Client::Stringlist &folder_ids,
                    const Client::Stringlist &content_ids,
                    const std::string &folder_id);

    bool exist_on_cloud(const std::string &file_path, const std::string &folder_id);

    bool update_folder(const std::string &folder_id,
                       const std::string &new_name);

    bool delete_contents(const Client::Stringlist &content_ids);

    Client::Stringlist copy_folders(const Client::Stringlist &folder_ids,
                                    const std::string &folder_id);

    Client::Stringlist copy_contents(const Client::Stringlist &content_ids,
                                     const std::string &folder_id);

    Client::ResourceList cloud_content_list(int start_index,
                                            int count,
                                            CloudContent::Type content_type = CloudContent::Type::All,
                                            const std::string &folder_id = std::string());

    Client::OutlinkList create_folder_extranet_link(const Client::Stringlist &folder_ids);

    Client::OutlinkList create_content_extranet_link(const Client::Stringlist &content_ids);

    DownloadTaskPriv::Ptr create_download_link(const std::string  &content_id);

    DownloadTaskPriv::Ptr create_download_link(const DownloadBufferCb  &buffer_cb);

    //Same name files in different directories are not allowed to upload to mcloud because of file names conflicts.
    UploadTaskPriv::Ptr create_upload_link(const UploadRequest &reqeust_item);

    UploadTaskPriv::Ptr create_upload_link(const UploadBufferCb &buffer_cb);

    void cancel();

    std::shared_ptr<SyncManager> sync_manager();

  private:
    Client::ResourceList cloud_root_folder_list();

    Client::ResourceList cloud_content_list_internal(const std::string &postbody);

    CloudContent::Ptr content_item_internal(const std::string &postbody);

    CloudFolder::Ptr create_folder_internal(const std::string &postbody);

    Client::ResourceList look_up_internal(const std::string &postbody);

    bool exist_on_cloud_internal(const std::string &postbody);

    bool move_items_internal(const std::string &postbody);

    bool update_internal(const std::string &postbody);

    bool delete_internal(const std::string &postbody);

    Client::Stringlist copy_internal(const std::string &postbody);

    Client::OutlinkList create_extranet_internal(const std::string &postbody);

    template<typename T> T get_or_throw(std::future<T> &f);

  private:

    class HttpClient;

    std::shared_ptr<HttpClient>  httpclient_;

    std::shared_ptr<SyncManager> sync_manager_;
};

}
}

#endif // MCLOUD_API_CLIENTPRIV_H_
