/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#include <mcloud/api/cloudfolder.h>

#include <tinyxml2.h>

#include <time.h>
#include <iostream>
#include <sstream>

using namespace mcloud::api;
using namespace std;

std::time_t string_to_time(const std::string dt) {
    struct std::tm tm;
    std::istringstream ss(dt);
    strptime(dt.c_str(), "%Y%m%d%H%M%S",&tm); // std::get std::put only available in GCC 5
    return std::mktime(&tm);
}

class CloudFolder::Priv {
public:
    Priv(const tinyxml2::XMLElement *root) {
        name_= root->FirstChildElement("catalogName")->GetText();
        id_ = root->FirstChildElement("catalogID")->GetText();
        created_date_ = string_to_time(root->FirstChildElement("createTime")->GetText());
        update_date_ = string_to_time(root->FirstChildElement("updateTime")->GetText());

        //Buggy on MCloud server rest api design, "parentCatalogID" and "parentCatalogId" are both
        //existing in some API response.
        auto parent_catalog_ele = root->FirstChildElement("parentCatalogID");
        if (parent_catalog_ele && parent_catalog_ele->GetText()) {
            parent_catalog_id_ = parent_catalog_ele->GetText();
        }

        parent_catalog_ele = root->FirstChildElement("parentCatalogId");
        if (parent_catalog_ele && parent_catalog_ele->GetText()) {
            parent_catalog_id_ = parent_catalog_ele->GetText();
        }

        auto folder_ele = root->FirstChildElement("path");
        if (folder_ele && folder_ele->GetText()) {
            folder_path_ = folder_ele->GetText();
        }

        auto etag_ele = root->FirstChildElement("dirEtag");
        if (etag_ele && etag_ele->GetText()) {
            etag_ = etag_ele->GetText();
        }

        auto owner_ele = root->FirstChildElement("owner");
        if (owner_ele && owner_ele->GetText()) {
            owner_ = owner_ele->GetText();
        }

        folder_type_ = (Type)atoi(root->FirstChildElement("catalogType")->GetText());


    #ifndef NDEBUG
        cout << "id: "<< id_ << " name: " << name_ << "  etag: "<< etag_ << endl;
    #endif

    }

    std::string id_;

    std::string name_;

    std::time_t created_date_;

    std::time_t update_date_;

    std::string parent_catalog_id_;

    std::string etag_;

    std::string owner_;

    CloudFolder::Type folder_type_;

    std::string folder_path_;
};

CloudFolder::CloudFolder(const tinyxml2::XMLElement *root)
    :p_(std::make_shared<Priv>(root)){
}

const string &CloudFolder::id() const {
    return p_->id_;
}

const string &CloudFolder::name() const {
    return p_->name_;
}

const time_t &CloudFolder::created_date() const {
    return p_->created_date_;
}

const time_t &CloudFolder::updated_date() const {
    return p_->update_date_;
}

const string &CloudFolder::parent_catalog_id() const {
    return p_->parent_catalog_id_;
}

const string &CloudFolder::folder_path() const {
    return p_->folder_path_;
}

const string &CloudFolder::etag() const {
    return p_->etag_;
}

const string &CloudFolder::owner() const {
    return p_->owner_;
}

CloudResource::Property CloudFolder::property() const {
    return CloudResource::Property::Folder;
}

CloudFolder::Type CloudFolder::folder_type() const {
    return p_->folder_type_;
}

