#include <core/posix/exec.h>

#include <mcloud/api/client.h>
#include <mcloud/api/diskinfo.h>
#include <mcloud/api/uploadtask.h>
#include <mcloud/api/downloadtask.h>
#include <mcloud/api/outlink.h>
#include <boost/filesystem.hpp>
#include <gtest/gtest.h>

#include <iostream>
#include <fstream>
#include <random>
#include <thread>
#include <fcntl.h>

using namespace mcloud::api;
using namespace core::posix;
using namespace std;

namespace {
    static const std::string token = "valid_token";
    static const std::string TMPDIR = "/tmp";
    static const char * Status[] = { "unstart", "running", "canceled", "paused", "broken", "complete"};

    std::string tmpdir() {
        const char *v = getenv("TMPDIR");
        return v == NULL ? TMPDIR : std::string(v);
    }

    std::string status_to_string(Task::Status status) {
        return Status[(int)status];
    }

    void create_file(std::string file_path, size_t size) {
        std::ofstream ofs(file_path, std::ios::binary | std::ios::out);
        ofs.seekp(size - 1);
        ofs.write("", 1);
        ofs.close();
    }

    class McloudAPI : public ::testing::Test {
        protected:
            virtual void SetUp() override {
                fake_mcloud_server_ = exec(FAKE_MCLOUD_SERVER, { }, { },
                        StandardStream::stdout);

                ASSERT_GT(fake_mcloud_server_.pid(), 0);
                string port;
                fake_mcloud_server_.cout() >> port;

                string apiroot = "http://127.0.0.1:" + port;
                setenv("MCLOUD_LOCAL_SERVER_URL", apiroot.c_str(), true);

                string download_folder = tmpdir();
                setenv("MCLOUD_DOWNLOAD_FOLDER", download_folder.c_str(), true);
            }

            virtual void TearDown() override {
            }

            ChildProcess fake_mcloud_server_ = ChildProcess::invalid();
    };

    TEST_F(McloudAPI, sync_download_files)
    {
        Client::Stringlist content_id_list{{"1811asktx23a054201604201417321zo"},  //mp3
                                           {"1811asktx23a05620160425125424fle"},  //mp4
                                           {"1811asktx23a0572016042014173129t"},  //gif
                                           {"1811asktx23a05620160420134213zx4"},  //pdf
                                           {"1811asktx23a058201604201519046ec"}}; //png

        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);

        SyncManager::DownloadList task_list;
        SyncManager::Ptr syncManager = c.syncmanager();
        for (const auto & content_id: content_id_list) {
            try {
                DownloadTask::Ptr task = syncManager->add_download_task(content_id);
                task_list.push(task);

                task->status_changed() = [task](Task::Status status) {
                    cout<< "status : "<< status_to_string(status) << "  " << task->content_name() << endl;
                };

                task->progress_changed() = [task](float percent) {
                    cout<< "downloading: " << task->content_name() << "  " << percent << endl;
                };
            } catch (...) { }
        }

        std::this_thread::sleep_for(std::chrono::milliseconds(3000 * content_id_list.size()));

        EXPECT_EQ(5, task_list.size());
        for (const auto & task: task_list) {
            EXPECT_FALSE(task->task_id().empty());
            EXPECT_EQ(Task::Status::Complete, task->status());
            EXPECT_TRUE(boost::filesystem::exists(task->file_path()));
        }
    }

    TEST_F(McloudAPI, sync_download_files_with_buffer)
    {
        int fd = open("/dev/null", O_WRONLY);
        DownloadBufferCb buffer_cb{{"1811asktx23a054201604201417321zo"},
                                [fd](void *dest, size_t buf_size) -> size_t {
                                     //data writing callback.
                                     return write(fd, dest, buf_size);
                                }};
        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);

        SyncManager::DownloadList task_list;
        SyncManager::Ptr syncManager = c.syncmanager();
        try {
            DownloadTask::Ptr task = syncManager->add_download_task(buffer_cb);
            task_list.push(task);

            task->status_changed() = [task](Task::Status status) {
                cout<< "status : "<< status_to_string(status) << "  " << task->content_name() << endl;
            };

            task->progress_changed() = [task](float percent) {
                cout<< "downloading: " << task->content_name() << "  " << percent << endl;
            };
        } catch (...) { }

        std::this_thread::sleep_for(std::chrono::milliseconds(3000));
        
        close(fd);

        EXPECT_EQ(1, task_list.size());
        EXPECT_EQ(Task::Status::Complete, task_list[0]->status());
    }

    TEST_F(McloudAPI, sync_upload_files)
    {
        static constexpr const char *mp3_upload_content = "song.mp3";
        static constexpr const char *mp4_upload_content = "movie.mp4";
        static constexpr const char *gif_upload_content = "funny.gif";
        static constexpr const char *pdf_upload_content = "ubuntu.pdf";
        static constexpr const char *png_upload_content = "screenshot.png";

        UploadRequestList request_list{{tmpdir() + "/" + pdf_upload_content, "", ""},
                                    {tmpdir() + "/" + mp4_upload_content, "", ""},
                                    {tmpdir() + "/" + mp3_upload_content, "", "my_song.mp3"},
                                    {tmpdir() + "/" + gif_upload_content, "", "my_funny.gif"},
                                    {tmpdir() + "/" + png_upload_content, "", "my_screenshot_1.png"}};

        std::default_random_engine engine;
        std::uniform_int_distribution<int> dis(1024, 1024 * 1024 * 50);
        for (const auto &request_item: request_list) {
            size_t file_size = dis(engine);
            create_file(request_item.file_path, file_size);
        }

        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);

        SyncManager::UploadList task_list;
        for (const auto &request: request_list) {
            try {
                UploadTask::Ptr task = c.syncmanager()->add_upload_task(request);
                task_list.push(task);

                task->progress_changed() = [task](float percent) {
                    cout<< "uploading: " << task->content_name() << "  " << percent << endl;
                };

                task->status_changed() = [task](Task::Status status) {
                    cout<< "status : "<< status_to_string(status) << "  "<< task->content_name() << endl;
                };
            } catch (...) { }
        }

        std::this_thread::sleep_for(std::chrono::milliseconds(2000 * task_list.size()));

        EXPECT_EQ(5, task_list.size());
        for (const auto &task: task_list) {
            EXPECT_GT(task->file_size(), 0);
            EXPECT_TRUE(task->is_need_upload());
            EXPECT_FALSE(task->task_id().empty());
            EXPECT_EQ(Task::Status::Complete, task->status());
        }
    }

    TEST_F(McloudAPI, sync_upload_pause_and_resume)
    {
        static constexpr const char *mp4_upload_content = "movie.mp4";

        UploadRequest request{tmpdir() + "/" + mp4_upload_content, "", ""};

        std::default_random_engine engine;
        std::uniform_int_distribution<int> dis(1024, 1024 * 1024 * 50);
        size_t file_size = dis(engine);
        create_file(request.file_path, file_size);

        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);
        SyncManager::Ptr syncManager = c.syncmanager();
        try {
            UploadTask::Ptr task = syncManager->add_upload_task(request);

            task->progress_changed() = [task](float percent) {
                cout<< "uploading: " << task->content_name() << "  " << percent << endl;
            };

            task->status_changed() = [task](Task::Status status) {
                cout<< "status : "<< status_to_string(status) << "  "<< task->content_name() << endl;
            };

            cout << "sync manager pause"<< endl;
            syncManager->pause();

            std::this_thread::sleep_for(std::chrono::milliseconds(1000));

            cout << "sync manager resume"<< endl;
            syncManager->start();

            std::this_thread::sleep_for(std::chrono::milliseconds(3000));

            EXPECT_GT(task->file_size(), 0);
            EXPECT_TRUE(task->is_need_upload());
            EXPECT_FALSE(task->task_id().empty());
            EXPECT_EQ(Task::Status::Complete, task->status());
        } catch (...) { }
    }

    TEST_F(McloudAPI, sync_download_pause_and_resume)
    {
        static constexpr const char *content_id  = "1811asktx23a054201604201417321zo";

        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);
        SyncManager::Ptr syncManager = c.syncmanager();
        try {
            DownloadTask::Ptr task = syncManager->add_download_task(content_id);

            task->progress_changed() = [task, syncManager](float percent) {
                cout<< "downloading: " << task->content_name() << "  " << percent << endl;
            };

            task->status_changed() = [task](Task::Status status) {
                cout<< "status : "<< status_to_string(status) << "  "<< task->content_name() << endl;
            };

            cout << "sync manager pause"<< endl;
            syncManager->pause();

            std::this_thread::sleep_for(std::chrono::milliseconds(1000));

            cout << "sync manager resume"<< endl;
            syncManager->start();

            std::this_thread::sleep_for(std::chrono::milliseconds(3000));

            EXPECT_EQ(Task::Status::Complete, task->status());
            EXPECT_TRUE(boost::filesystem::exists(task->file_path()));
        } catch (...) { }
    }

    TEST_F(McloudAPI, sync_manager_download_cancel)
    {
        Client::Stringlist content_id_list{{"1811asktx23a054201604201417321zo"},  //mp3
                                        {"1811asktx23a05620160425125424fle"},  //mp4
                                        {"1811asktx23a058201604201519046ec"}}; //png

        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);

        SyncManager::DownloadList task_list;
        SyncManager::Ptr syncManager = c.syncmanager();
        for(const auto &content_id: content_id_list) {
            try {
                DownloadTask::Ptr task = syncManager->add_download_task(content_id);
                task_list.push(task);

                task->progress_changed() = [task](float percent) {
                    cout<< "downloading: " << task->content_name() << "  " << percent << endl;
                };

                task->status_changed() = [task](Task::Status status) {
                    cout<< "status : "<< status_to_string(status) << "  "<< task->content_name() << endl;
                };

                cout << "sync manager cancel"<< endl;
                syncManager->cancel();

            } catch (...) { }
        }

        std::this_thread::sleep_for(std::chrono::milliseconds(2000));

        EXPECT_EQ(3, task_list.size());
        EXPECT_NE(Task::Status::Unstart, task_list[0]->status());
        EXPECT_EQ(Task::Status::Canceled, task_list[1]->status());
        EXPECT_EQ(Task::Status::Canceled, task_list[2]->status());
    }

    TEST_F(McloudAPI, task_upload_cancel)
    {
        static constexpr const char *pdf_upload_content = "ubuntu.pdf";

        UploadRequest request{tmpdir() + "/" + pdf_upload_content, "", ""};

        std::default_random_engine engine;
        std::uniform_int_distribution<int> dis(1024, 1024 * 1024 * 50);
        size_t file_size = dis(engine);
        create_file(request.file_path, file_size);

        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);
        SyncManager::Ptr syncmanager = c.syncmanager();
        try {
            UploadTask::Ptr task = syncmanager->add_upload_task(request);

            task->progress_changed() = [task](float percent) {
                cout<< "uploading: " << task->content_name() << "  " << percent << endl;
            };

            task->status_changed() = [task](Task::Status status) {
                cout<< "status : "<< status_to_string(status) << "  "<< task->content_name() << endl;
            };

            std::this_thread::sleep_for(std::chrono::milliseconds(500));

            task->cancel();

            EXPECT_EQ(Task::Status::Canceled, task->status());
        } catch (...) { }
    }

    TEST_F(McloudAPI, task_download_cancel)
    {
        static constexpr const char *content_id  = "1811asktx23a054201604201417321zo"; //mp3

        const int time_out = 10;
        Client c(time_out);
        c.set_access_token(token);
        SyncManager::Ptr syncManager = c.syncmanager();
        try {
            DownloadTask::Ptr task = syncManager->add_download_task(content_id);

            task->progress_changed() = [task](float percent) {
                cout<< "downloading: " << task->content_name() << "  " << percent << endl;
            };

            task->status_changed() = [task](Task::Status status) {
                cout<< "status : "<< status_to_string(status) << "  "<< task->content_name() << endl;
            };

            std::this_thread::sleep_for(std::chrono::milliseconds(500));

            task->cancel();

            EXPECT_EQ(Task::Status::Canceled, task->status());
        } catch (...) { }
    }
}
