# Handling Callbacks in the Messaging Framework

The Messaging framework aims to ease the task of announcing state changes and general events for integrators. That is, its aim is to provide a type- and thread-safe mechanism to implementors that is minimally intrusive and as easy to use as possible. For that, integrators/implementors are offered with explicit functions for announcing all events of interest, with the function parameters specifying the required parameters. At runtime, the Messaging framework takes care of dispatching the method calls to observer instances, bridging any sort of thread boundary (which is a concept brought to us by leveraging Qt). 

A note for people contributing to the core Messaging framework development:
  - Observer classes should be inner classes of the interfaces that they monitor.
  - Interface classes should take a ready-made instance of the respective Observer class as a managed std::shared_ptr.
  - Bridging over to Qt/TelepathyQt should happen via messaging::qt::Runtime::enter_with_task

An example to illustrate the guidelines introduced before:

~~~~~~~~~~~~~~~~{.cpp}
class Interface
{
public:
    class Observer
    {
    public:
        virtual void on_something_happened(int a) = 0;
    };

protected:
    explicit Interface(const std::shared_ptr<Observer>& observer);

    // This should be called by subclasses.
    void announce_something_happened(int a);
};
~~~~~~~~~~~~~~~~

With that, integrators would have to write the following code:

~~~~~~~~~~~~~~~~{.cpp}
class MyImplementation : public Interface
{
public:
    MyImplementation(const std::shared_ptr<Interface::Observer>& observer)
        : Interface{observer}
    {
    }

    void run()
    {
        while (true)
        {
            announce_something_happened(42);
        }
    }
};
~~~~~~~~~~~~~~~~

Within the Messaging framework core implementation, bridgning over to Qt/TelepathyQt looks like:

~~~~~~~~~~~~~~~~{.cpp}
class ABridgingObserver : public Interface::Observer
{
public:
    ABridgingObserver(const std::shared_ptr<messaging::qt::Runtime>& runtime)
        : runtime{runtime}
    {
    }

    void on_something_happened(int a) override
    {
        runtime->enter_with_task([a]()
        {
            // Safe to access anything Qt in this functor.
        });
    }

private:
    std::shared_ptr<messaging::qt::Runtime> runtime;
};
}
~~~~~~~~~~~~~~~~