/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_FLAGS_H
#define MESSAGING_FLAGS_H

#include <glog/logging.h>

#include <type_traits>

namespace messaging
{

template <typename T>
class Flags
{
public:
    typedef typename std::underlying_type<T>::type UnderlyingType;

    static const T NoFlag = static_cast<T>(0);

    Flags(const T& flags = static_cast<T>(0))
        : flags_(static_cast<UnderlyingType>(flags))
    {
        if (not std::is_enum<T>::value)
        {
            LOG(ERROR) << "Cannot create flags from a not enum underlaying type";
        }
    }

    inline UnderlyingType operator & (T flag) const
    {
        return flags_ & static_cast<UnderlyingType>(flag);
    }

    inline UnderlyingType operator | (T flag) const
    {
        return flags_ | static_cast<UnderlyingType>(flag);
    }

    inline Flags<T> operator |= (T flag)
    {
        flags_ = flags_ | static_cast<UnderlyingType>(flag);
        return *this;
    }

    inline bool operator == (const Flags<T> &other) const
    {
        return flags_ == other.underlying_type();
    }

    inline bool operator != (const Flags<T> &other) const
    {
        return flags_ != other.underlying_type();
    }

    inline bool is_set(T flag) const
    {
        return *this & flag;
    }

    inline UnderlyingType underlying_type() const
    {
        return flags_;
    }

private:
    UnderlyingType flags_;
};

template<typename T>
T operator | (T lhs, T rhs)
{
    return static_cast<T>(static_cast<typename messaging::Flags<T>::UnderlyingType>(lhs) |
                          static_cast<typename messaging::Flags<T>::UnderlyingType>(rhs));
}

}


#endif // MESSAGING_FLAG_H

