/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_PLUGIN_CONNECTOR_H_
#define MESSAGING_PLUGIN_CONNECTOR_H_

#include <messaging/connector.h>
#include <messaging/connector_factory.h>
#include <messaging/plugin.h>
#include <messaging/visibility.h>

#include <memory>

namespace messaging
{
// A handy forward decl.
class NamedSymbolLoader;

typedef messaging::Connector* (*Creator)();

/// @brief PluginConnector loads a messaging::Connector implementation.
class MESSAGING_FW_PUBLIC PluginConnector : public Connector
{
public:
    /// @brief Descriptor bundles the name and parameter parser generation
    /// for inclusion of PluginConnector with the Connector creation infrastructure.
    ///
    /// TODO(tvoss): This could very well be an interface.
    struct Descriptor
    {
        /// @brief The name of the PluginConnector.
        static constexpr const char* name{"messaging::PluginConnector"};

        /// @brief generator_for_dict returns a functor that allows for
        /// creating a PluginConnector instance.
        static ConnectorFactory::Generator generator();
    };

    /// @brief Create bundles the create function signature and its name
    /// as being resolved from a dynamic library.
    struct Create
    {
        typedef messaging::PluginCreateFunction Type;
        static constexpr const char* name = MESSAGING_FW_CREATE_FUNCTION_NAME;

        Type call;
    };

    /// @brief Destroy bundles the destroy function signature and its name
    /// as being resolved from a dynamic library.
    struct Destroy
    {
        typedef messaging::PluginDestroyFunction Type;
        static constexpr const char* name = MESSAGING_FW_DESTROY_FUNCTION_NAME;

        Type call;
    };

    /// @brief Creates a new instance, trying to load required symbols
    /// from the given NamedSymbolLoader.
    /// @throws std::out_of_range if any required symbol is missing.
    PluginConnector(const std::shared_ptr<NamedSymbolLoader>& named_symbol_loader);

    const Enumerator<Parameter>& parameters() const override;

    /// @brief run hands a thread of execution to the connector implementation.
    void run() override;

    /// @brief stop requests the connector to stop executing.
    void stop() override;

    std::shared_ptr<Connection> request_connection(const std::shared_ptr<messaging::Connection::Observer>& connection_observer,
                                                   const std::shared_ptr<messaging::Messenger::Observer>& messenger_observer,
                                                   const std::shared_ptr<messaging::PresenceManager::Observer>& presence_observer,
                                                   const messaging::Dictionary<std::string, messaging::Variant>& dict) override;

private:
    std::shared_ptr<NamedSymbolLoader> named_symbol_loader;
    std::shared_ptr<messaging::Connector> connector;
};
}

#endif  // MESSAGING_PLUGIN_CONNECTOR_H_
