/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_VARIANT_H_
#define MESSAGING_VARIANT_H_

#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>
#include <messaging/visibility.h>

#include <iosfwd>
#include <memory>
#include <string>
#include <deque>

namespace messaging
{
/// @brief Variant models a discriminated union type.
class MESSAGING_FW_PUBLIC Variant : NonCopyable, NonMovable
{
public:
    /// @brief Type enumerates all types that can be stored in the variant.
    enum class Type
    {
        boolean,
        integer,
        floating_point,
        string,
        data,
        recursive
    };

    /// @brief TypeMismatch is thrown if trying to access
    /// the value of the variant as a type that is not
    /// contained in the variant.
    struct TypeMismatch : public std::runtime_error
    {
        TypeMismatch(Type expected, Type actual);
    };

    /// @brief throw_if_type_mismatch throws TypeMismatch if expected != actual.
    template <Type expected>
    static void throw_if_type_mismatch(Type actual)
    {
        if (expected == actual)
        {
            return;
        }

        throw TypeMismatch{expected, actual};
    }

    /// type returns the type of the variant.
    virtual Type type() const = 0;

    /// @brief as_bool tries to extract a boolean value from this instance.
    /// @throw TypeMismatch in case of issues.
    virtual bool as_bool() const = 0;

    /// @brief as_int tries to extract an integer value from this instance.
    /// @throw TypeMismatch in case of issues.
    virtual std::int64_t as_int() const = 0;

    /// @brief as_double tries to extract a double value from this instance.
    /// @throw TypeMismatch in case of issues.
    virtual double as_double() const = 0;

    /// @brief as_string tries to extract a string value from this instance.
    /// @throw TypeMismatch in case of issues.
    virtual std::string as_string() const = 0;

    /// @brief as_data tries to extract an array of bytes from this instance.
    virtual const char* as_data() const = 0;

    /// @brief data_size returns the size of the array returned by as_data().
    virtual std::size_t data_size() const = 0;

    /// @brief size returns the number of children of this variant.
    virtual std::size_t size() const = 0;

    /// @brief keys returns a deque of the keys of the Variant if it's a map
    /// a logic error will be thrown if not a map type
    virtual std::deque<std::string> keys() const = 0;

    /// @brief value returns the value related with the received key.
    /// This operation only has sense with maps
    virtual std::shared_ptr<Variant> at(const std::string& key) const = 0;

    /// @brief at returns the variant instance at the given idx.
    /// @throw std::out_of_range if the idx >= size().
    virtual std::shared_ptr<Variant> at(std::size_t idx) const = 0;

protected:
    Variant() = default;
};

/// @brief operator<< pretty-prints the given type to the given stream.
MESSAGING_FW_PUBLIC std::ostream& operator<<(std::ostream&, Variant::Type);
}

#endif  // MESSAGING_VARIANT_H_
