/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/dynamic_library.h>

#include <functional>
#include <stdexcept>

#include <dlfcn.h>

namespace
{
void* throw_if_null(void* handle)
{
    if (not handle)
    {
        throw std::runtime_error(dlerror());
    }

    return handle;
}

std::function<void(void*)> the_closing_deleter()
{
    return [](void* handle)
    {
        if (handle)
        {
            if (::dlclose(handle) != 0)
            {
                // TODO(tvoss): How do we handle errors here?
                // We cannot easily throw, and we would like to avoid
                // accessing global logger objects.
            }
        }
    };
}

int the_default_open_flags()
{
    return RTLD_NOW | RTLD_LOCAL;
}
}

/// @brief Constructs a new instance, trying to load the dynamic library pointed to by path.
/// @throws std::runtime_error in case of errors.
messaging::DynamicLibrary::DynamicLibrary(const boost::filesystem::path& path)
    : handle(throw_if_null(::dlopen(path.c_str(), the_default_open_flags())), the_closing_deleter())
{
}

void* messaging::DynamicLibrary::load_symbol_for_name(const std::string& name) const
{
    auto symbol = ::dlsym(handle.get(), name.c_str());

    if (not symbol)
    {
        if (auto err = ::dlerror())
        {
            throw std::runtime_error(err);
        }

        throw std::out_of_range("Could not resolve symbol for name: " + name);
    }

    return symbol;
}
