/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/messenger.h>
#include <glog/logging.h>

using namespace std;

class messaging::Messenger::Private
{
public:
    Private(const shared_ptr<messaging::Messenger::Observer>& obs)
        : observer{obs} {}
    ~Private() {}

    shared_ptr<messaging::Messenger::Observer> observer;
    map<string, shared_ptr<messaging::Chat>> chats_;
};

messaging::Messenger::Messenger(const shared_ptr<messaging::Messenger::Observer>& observer)
        : impl{new Private{observer}}
{
}

void messaging::Messenger::announce_message_without_chat_received(const Recipient::shared_ptr &recipient, const messaging::Message& message)
{
    impl->observer->on_message_without_chat_received(recipient, message);
}

void messaging::Messenger::announce_new_group_invitation_received(const std::shared_ptr<Group>& new_group)
{
    impl->observer->on_new_group_invitation_received(new_group);
}

messaging::Chat::SPtr messaging::Messenger::initiate_chat_with(const Recipient::shared_ptr &recipient,
                                                                     const shared_ptr<Chat::Observer> &observer)
{
    if (!recipient->id().empty())
    {
        auto it = chats().find(recipient->id());
        if (it != chats().end())
        {
            LOG(WARNING) << "Returning already existing chat for recipient " + recipient->id();
            return it->second;
        }
    }

    auto chat = create_chat_with(recipient, observer);
    chats()[recipient->id()] = chat;
    return chat;
}

messaging::Chat::SPtr messaging::Messenger::get_chat(const string &recipient_id) {
    auto it = chats().find(recipient_id);
    return it != chats().end() ? it->second : nullptr;
}

map<string, messaging::Chat::SPtr>& messaging::Messenger::chats() {
    return impl->chats_;
}


