/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/qt/tp/connection.h>

#include <messaging/qt/runtime.h>
#include <messaging/qt/tp/text_channel.h>

#include <messaging/qt/tp/interfaces/base_channel_destroyable.h>
#include <messaging/qt/tp/interfaces/base_channel_roles.h>

#include <messaging/broadcast.h>
#include <messaging/message.h>
#include <messaging/presence_status_not_supported.h>
#include <messaging/user.h>
#include <messaging/group.h>
#include <messaging/flags.h>

#include <TelepathyQt/Constants>

#include <QCryptographicHash>
#include <cstdint>

#include <iostream>

#include <glog/logging.h>

#define MQT_TP_RECIPIENT "messaging-framework-qt-tp-recipient"

Q_DECLARE_METATYPE(messaging::Recipient::shared_ptr)
Q_DECLARE_METATYPE(std::shared_ptr<messaging::Group>)

namespace msg = messaging;
namespace mqt = messaging::qt;

messaging::Recipient::shared_ptr recipient_from_identifier(const std::string identifier)
{
    return messaging::Recipient::shared_ptr{new messaging::User{identifier}};
}

mqt::tp::Connection::Observer::Observer(const std::shared_ptr<qt::Runtime>& runtime, const Tp::SharedPtr<mqt::tp::Connection>& connection)
    : runtime{runtime}
    , connection{connection}
{
}

void mqt::tp::Connection::Observer::on_status_changed(messaging::Connection::Status new_status, messaging::Connection::StatusChangedReason reason)
{
    auto sp = shared_from_this();
    runtime->enter_with_task([sp, new_status, reason]()
                             {
                                 sp->connection->on_status_changed(new_status, reason);
                             });
}

void mqt::tp::Connection::Observer::on_message_without_chat_received(const Recipient::shared_ptr& recipient, const messaging::Message &message)
{
    auto sp = shared_from_this();
    runtime->enter_with_task([sp, recipient, message]()
                             {
                                 sp->connection->on_message_without_chat_received(recipient, message);
                             });
}

void mqt::tp::Connection::Observer::on_presence_changed(const Recipient::shared_ptr& recipient, const messaging::Presence& presence)
{
    auto sp = shared_from_this();
    runtime->enter_with_task([sp, recipient, presence]()
                             {
                                 sp->connection->on_presence_changed(recipient, presence);
                             });
}

void mqt::tp::Connection::Observer::on_new_group_invitation_received(const messaging::Group::shared_ptr& new_group)
{
    auto sp = shared_from_this();
    runtime->enter_with_task([sp, new_group]()
                             {
                                 sp->connection->on_new_group_invitation_received(new_group);
                             });
}

Tp::SharedPtr<mqt::tp::Connection> mqt::tp::Connection::create(const std::shared_ptr<msg::Connector>& connector,
                                                               const std::shared_ptr<mqt::Runtime>& runtime,
                                                               const QString& connection_manager_name,
                                                               const QString& protocol_name,
                                                               const QVariantMap& parameters)
{
    return Tp::SharedPtr<mqt::tp::Connection>{new mqt::tp::Connection{connector, runtime, connection_manager_name, protocol_name, parameters}};
}

Tp::ConnectionPresenceType mqt::tp::Connection::messaging_presence_type_to_telepathy(const messaging::PresenceType& presence_type)
{
    Tp::ConnectionPresenceType type;
    switch(presence_type)
    {
        case messaging::PresenceType::available:
            type = Tp::ConnectionPresenceTypeAvailable;
            break;
        case messaging::PresenceType::away:
            type = Tp::ConnectionPresenceTypeAway;
            break;
        case messaging::PresenceType::busy:
            type = Tp::ConnectionPresenceTypeBusy;
            break;
        case messaging::PresenceType::error:
            type = Tp::ConnectionPresenceTypeError;
            break;
        case messaging::PresenceType::extended_away:
            type = Tp::ConnectionPresenceTypeAway;
            break;
        case messaging::PresenceType::hidden:
            type = Tp::ConnectionPresenceTypeHidden;
            break;
        case messaging::PresenceType::offline:
            type = Tp::ConnectionPresenceTypeOffline;
            break;
        case messaging::PresenceType::unset:
            type = Tp::ConnectionPresenceTypeUnset;
            break;
        case messaging::PresenceType::unknown:
        default:
            type = Tp::ConnectionPresenceTypeUnknown;
            break;
    }

    return type;
}

mqt::tp::Connection::Connection(const std::shared_ptr<msg::Connector>& connector,
                                const std::shared_ptr<mqt::Runtime>& runtime,
                                const QString& connection_manager_name,
                                const QString& protocol_name,
                                const QVariantMap& parameters)
    : Tp::BaseConnection{runtime->dbus_connection(), connection_manager_name, protocol_name, parameters}
    , runtime{runtime}
    , connector{connector}
    , observer{std::make_shared<mqt::tp::Connection::Observer>(runtime, Tp::SharedPtr<mqt::tp::Connection>{this})}
    , connection{connector->request_connection(observer, observer, observer, mqt::VariantMapFacade{parameters})}
    , network_monitor_{connection}
{
    qRegisterMetaType<messaging::Recipient::shared_ptr>();

    setSelfHandle(1);
    setSelfID(QString::fromStdString(connection->normalize_identifier(connection->self_identifier())));
    handles_.insert(HandleIdPair(selfHandle(), selfID()));

    // Wire up the actual request for connecting.
    setConnectCallback(Tp::memFun(this, &tp::Connection::connect));
    setInspectHandlesCallback(Tp::memFun(this, &tp::Connection::inspect_handles));
    setRequestHandlesCallback(Tp::memFun(this, &tp::Connection::request_handles));
    setCreateChannelCallback(Tp::memFun(this, &tp::Connection::create_channel));
    /* Connection.Interface.Requests */
    requests = Tp::BaseConnectionRequestsInterface::create(this);

    /* Fill requestableChannelClasses */
    Tp::RequestableChannelClass personalChat;
    personalChat.fixedProperties[TP_QT_IFACE_CHANNEL + QLatin1String(".ChannelType")] = TP_QT_IFACE_CHANNEL_TYPE_TEXT;
    personalChat.fixedProperties[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandleType")] = Tp::HandleTypeContact;
    personalChat.allowedProperties.append(TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandle"));
    personalChat.allowedProperties.append(TP_QT_IFACE_CHANNEL + QLatin1String(".TargetID"));

    Tp::RequestableChannelClass existingGroupChat;
    existingGroupChat.fixedProperties[TP_QT_IFACE_CHANNEL + QLatin1String(".ChannelType")] = TP_QT_IFACE_CHANNEL_TYPE_TEXT;
    existingGroupChat.fixedProperties[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandleType")]  = Tp::HandleTypeRoom;
    existingGroupChat.allowedProperties.append(TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandle"));
    existingGroupChat.allowedProperties.append(TP_QT_IFACE_CHANNEL + QLatin1String(".TargetID"));

    Tp::RequestableChannelClass newGroupChat;
    newGroupChat.fixedProperties[TP_QT_IFACE_CHANNEL + QLatin1String(".ChannelType")] = TP_QT_IFACE_CHANNEL_TYPE_TEXT;
    newGroupChat.fixedProperties[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandleType")]  = Tp::HandleTypeNone;
    newGroupChat.allowedProperties.append(TP_QT_IFACE_CHANNEL_INTERFACE_CONFERENCE + QLatin1String(".InitialInviteeIDs"));
    newGroupChat.allowedProperties.append(TP_QT_IFACE_CHANNEL_INTERFACE_ROOM + QLatin1String(".RoomName"));

    requests->requestableChannelClasses << personalChat << existingGroupChat << newGroupChat;

    plugInterface(Tp::AbstractConnectionInterfacePtr::dynamicCast(requests));

    contacts = Tp::BaseConnectionContactsInterface::create();
    contacts->setGetContactAttributesCallback(Tp::memFun(this, &tp::Connection::get_contact_attributes));
    contacts->setContactAttributeInterfaces(QStringList()
                                                 << TP_QT_IFACE_CONNECTION
                                                 << TP_QT_IFACE_CONNECTION_INTERFACE_SIMPLE_PRESENCE);
    plugInterface(Tp::AbstractConnectionInterfacePtr::dynamicCast(contacts));

    simplePresenceIface = Tp::BaseConnectionSimplePresenceInterface::create();

    // read the statuses supported by the plugin, and expose them to telepathy
    Tp::SimpleStatusSpecMap statusMap;

    try
    {
        auto statuses = connection->presence_manager()->get_valid_statuses();
        for (auto it = statuses.begin(); it != statuses.end(); ++it)
        {
            statusMap.insert(QString::fromStdString(it->first),
                             Tp::SimpleStatusSpec{messaging_presence_type_to_telepathy(it->second.type),
                                                  (it->second.flags & StatusSpec::Flags::may_set_on_self) == StatusSpec::Flags::may_set_on_self,
                                                  (it->second.flags & StatusSpec::Flags::can_have_message) == StatusSpec::Flags::can_have_message});
        }
    }
    catch (...)
    {
        LOG(ERROR) << "PresenceManager::get_valid_statuses() implementation throws exception. Please review. " \
                      "Providing empty valid statuses map to presence interface";
    }


    simplePresenceIface->setStatuses(statusMap);
    plugInterface(Tp::AbstractConnectionInterfacePtr::dynamicCast(simplePresenceIface));

    // generate an unique id for this connection
    QString timestamp(QString::number(QDateTime::currentMSecsSinceEpoch()));
    QString md5(QCryptographicHash::hash(timestamp.toLatin1(), QCryptographicHash::Md5).toHex());
    connection_id = QString(QLatin1String("connection_%1")).arg(md5);

    QObject::connect(this, SIGNAL(disconnected()), this, SLOT(onDisconnected()));
}

mqt::tp::Connection::~Connection()
{
    try
    {
        dbusConnection().unregisterObject(objectPath(), QDBusConnection::UnregisterTree);
        dbusConnection().unregisterService(busName());
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when unregistering telepathy connection object and service";
    }
}

void mqt::tp::Connection::onDisconnected()
{
    try
    {
        // request the api a disconnection. The requested reason is because this case comes externally
        // by a telepathy disconnection of the plugin
        connection->disconnect(messaging::Connection::StatusChangedReason::requested);
        network_monitor_.stop_monitoring();
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when disconnecting framework connection implementation";
    }
}
 
void mqt::tp::Connection::on_message_without_chat_received(const Recipient::shared_ptr& recipient, const messaging::Message &message)
{
    if (recipient->type() != messaging::RecipientType::user) {
        LOG(ERROR) << "recipient type should be of type RecipientType::user to create a new 1-1 conversation with a user";
        return;
    }

    Tp::DBusError error;
    int handle = request_handles(Tp::HandleTypeContact, QStringList() << QString::fromStdString(recipient->id()), &error)[0];
    QVariantMap request;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".ChannelType")] = TP_QT_IFACE_CHANNEL_TYPE_TEXT;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandleType")] = Tp::HandleTypeContact;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandle")] = handle;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".InitiatorHandle")] = handle;

    Tp::BaseChannelPtr channel = Tp::BaseConnection::createChannel(request, false, &error);

    mqt::tp::TextChannel::Ptr text_channel = mqt::tp::TextChannel::Ptr::dynamicCast(channel);
    if (text_channel.isNull()) {
        LOG(ERROR) << "mqt::tp::Connection text channel is not a mqt::tp::TextChannel";
        return;
    }

    text_channel->on_message_received(message);
}

void mqt::tp::Connection::on_status_changed(messaging::Connection::Status new_status, messaging::Connection::StatusChangedReason reason)
{
    setStatus(static_cast<uint>(new_status), static_cast<uint>(reason));
    Tp::SimpleContactPresences presences;
    Tp::SimplePresence presence;
    PresenceManager::StatusMap statuses;

    try
    {
        statuses = connection->presence_manager()->get_valid_statuses();
    }
    catch (...)
    {
        LOG(ERROR) << "PresenceManager::get_valid_statuses() implementation throws exception. Please review it. " \
                      "Providing empty valid statuses map to presence interface";
    }

    if (static_cast<Tp::ConnectionStatus>(new_status) == Tp::ConnectionStatusConnected)
    {
        if (statuses.find("available") == statuses.end())
        {
            LOG(ERROR) << "presence status not supported: available";
            return;
        }

        presence.status = QLatin1String("available");
        presence.type = Tp::ConnectionPresenceTypeAvailable;

        network_monitor_.set_connection_ready(true);
    }
    else
    {
        if (statuses.find("offline") == statuses.end())
        {
            LOG(ERROR) << "presence status not supported: offline";
            return;
        }

        presence.status = QLatin1String("offline");
        presence.type = Tp::ConnectionPresenceTypeOffline;
    }

    presences[selfHandle()] = presence;
    presences_[selfHandle()] = presence;
    simplePresenceIface->setPresences(presences);
}

void mqt::tp::Connection::on_presence_changed(const Recipient::shared_ptr &recipient, const messaging::Presence& presence)
{
    if (recipient->type() != messaging::RecipientType::user) {
        LOG(ERROR) << "only RecipientType::User is supported for now.";
        return;
    }

    try
    {
        auto valid_statuses = connection->presence_manager()->get_valid_statuses();
        if (valid_statuses.find(presence.status) == valid_statuses.end()) {
            LOG(ERROR) << "trying to set presence to " << presence.status << " that is not a valid presence status";
            return;
        }
    }
    catch (...)
    {
        LOG(ERROR) << "PresenceManager::get_valid_statuses() implementation throws exception. Please review it. " \
                      "Providing empty valid statuses map to presence interface";
    }

    Tp::DBusError error;
    int handle = request_handles(Tp::HandleTypeContact, QStringList() << QString::fromStdString(recipient->id()), &error)[0];
    Tp::SimpleContactPresences presences;
    Tp::SimplePresence tpPresence = Tp::SimplePresence{messaging_presence_type_to_telepathy(presence.type),
                                           QString::fromStdString(presence.status),
                                           QString::fromStdString(presence.status_message)};
    presences[handle] = tpPresence;
    presences_[handle] = tpPresence;
    simplePresenceIface->setPresences(presences);
}

void mqt::tp::Connection::on_new_group_invitation_received(const messaging::Group::shared_ptr& group)
{
    Tp::DBusError error;
    int handle = request_handles(Tp::HandleTypeRoom, QStringList() << QString::fromStdString(group->id()), &error)[0];

    if (group->creator()->id().empty()) {
        LOG(ERROR) << "Group creator cannot be empty";
        error.set(TP_QT_ERROR_CANCELLED, "Group creator is empty");
        return;
    }

    QString initiator_normalized_id = QString::fromStdString(group->creator()->id());
    try
    {
        initiator_normalized_id = QString::fromStdString(connection->normalize_identifier(group->creator()->id()));
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when normalizing group initiator identifier. Using not normalized one";
    }
    int initiator_handle = request_handles(Tp::HandleTypeContact, QStringList() << initiator_normalized_id, &error)[0];

    // if there is an error requesting handles, reject the group creation
    if (error.isValid()) {
        std::shared_ptr<GroupStarter> group_starter = connection->group_starter();
        try
        {
            group_starter->reject_group(group);
        }
        catch (const std::exception &e)
        {
            LOG(ERROR) << "Group rejection failed, " << e.what();
            error.set(TP_QT_ERROR_CANCELLED, "Group rejection failed");
        }
        catch (...)
        {
            LOG(ERROR) << "Group rejection failed";
            error.set(TP_QT_ERROR_CANCELLED, "Group rejection failed");
        }
        return;
    }

    QVariantMap request;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".ChannelType")] = TP_QT_IFACE_CHANNEL_TYPE_TEXT;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandleType")] = Tp::HandleTypeRoom;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandle")] = handle;
    request[TP_QT_IFACE_CHANNEL + QLatin1String(".InitiatorHandle")] = initiator_handle;
    request[MQT_TP_RECIPIENT] = QVariant::fromValue(group);
    Tp::BaseChannelPtr channel = Tp::BaseConnection::createChannel(request, false, &error);

    mqt::tp::TextChannel::Ptr text_channel = mqt::tp::TextChannel::Ptr::dynamicCast(channel);
    if (text_channel.isNull())
    {
        LOG(ERROR) << "mqt::tp::Connection text channel is not a mqt::tp::TextChannel";
    }
}

void mqt::tp::Connection::connect(Tp::DBusError* error) noexcept(true)
{
    try
    {
        connection->connect();
    }
    catch (const std::exception& e)
    {
        error->set(QLatin1String{"std::exception"}, QString::fromStdString(e.what()));
    }
    catch (...)
    {
        error->set(QLatin1String{"unknown"}, QLatin1String{"unknown"});
    }
}

QStringList mqt::tp::Connection::inspect_handles(uint handleType, const Tp::UIntList& handles, Tp::DBusError* error)
{
    if (handleType != Tp::HandleTypeContact && handleType != Tp::HandleTypeRoom)
    {
        error->set(TP_QT_ERROR_INVALID_ARGUMENT, "Not supported");
        return QStringList();
    }

    QStringList identifiers;

    HandleIdMap &handles_ref = handleType == Tp::HandleTypeContact ? handles_ : group_handles_;

    for (auto handle : handles)
    {
        if (handles_ref.left.count(handle) > 0)
        {
            identifiers.append(handles_ref.left.at(handle));
        }
        else
        {
            error->set(TP_QT_ERROR_INVALID_HANDLE, "Handle not found");
            return QStringList();
        }
    }

    return identifiers;
}

Tp::UIntList mqt::tp::Connection::request_handles(uint handleType, const QStringList& identifiers, Tp::DBusError* error)
{
    static std::uint32_t counter{2};

    Tp::UIntList handles;

    if (handleType == Tp::HandleTypeContact)
    {
        for (auto id : identifiers)
        {
            QString normalizedId = id;
            try
            {
                normalizedId = connection->normalize_identifier(id.toStdString()).c_str();
            }
            catch (...)
            {
                LOG(ERROR) << "An exception has been thrown when normalizing identifier. Using not normalized one";
                return Tp::UIntList();
            }


            if (handles_.right.count(normalizedId))
            {
                handles.append(handles_.right.at(normalizedId));
            }
            else
            {
                try
                {
                    // before creating the handle, check that the id is valid
                    if (!connection->is_valid_identifier(normalizedId.toStdString())) {
                        if (error) {
                            error->set(TP_QT_ERROR_INVALID_HANDLE, "Identifier not valid.");
                        }
                        return Tp::UIntList();
                    }
                }
                catch (...)
                {
                    LOG(ERROR) << "An exception has been thrown when evaluating messaging::Connection::is_valid_identifier() " \
                                  "for identifier: " << normalizedId.toStdString() << " ,please review its implementation";
                    return Tp::UIntList();
                }

                auto sample = ++counter;
                handles_.insert(HandleIdPair(sample, normalizedId));
                handles.append(sample);

                // request the presence status of the new handle
                try
                {
                    connection->presence_manager()->request_presence(recipient_from_identifier(normalizedId.toStdString()));
                }
                catch (...)
                {
                    LOG(ERROR) << "An exception has been thrown when requesting presence for " << normalizedId.toStdString();
                    return Tp::UIntList();
                }
            }
        }
    }
    else if (handleType == Tp::HandleTypeRoom)
    {
        for (auto id : identifiers)
        {
            if (group_handles_.right.count(id))
            {
                handles.append(group_handles_.right.at(id));
            }
            else
            {
                auto sample = ++counter;
                group_handles_.insert(HandleIdPair(sample, id));
                handles.append(sample);
            }
        }
    }
    else
    {
        if (error) {
            error->set(TP_QT_ERROR_INVALID_ARGUMENT, "Not supported");
        }
        return Tp::UIntList();
    }

    return handles;
}

Tp::BaseChannelPtr mqt::tp::Connection::create_channel(const QVariantMap& request,
                                                       Tp::DBusError* error)
{
    const QString channel_type = request.value(TP_QT_IFACE_CHANNEL + QLatin1String(".ChannelType")).toString();
    uint target_handle_type = request.value(TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandleType")).toUInt();
    uint final_handle = request.value(TP_QT_IFACE_CHANNEL + QLatin1String(".TargetHandle")).toUInt();
    if (channel_type != TP_QT_IFACE_CHANNEL_TYPE_TEXT ||
            (target_handle_type != Tp::HandleTypeNone && target_handle_type != Tp::HandleTypeContact && target_handle_type != Tp::HandleTypeRoom))
    {
        error->set(TP_QT_ERROR_NOT_IMPLEMENTED, "Channel type not available");
        return Tp::BaseChannelPtr();
    }

    if (request.contains(TP_QT_IFACE_CHANNEL + QLatin1String(".TargetID")))
    {
        QString identifier = request[TP_QT_IFACE_CHANNEL + QLatin1String(".TargetID")].toString();
        Tp::DBusError error;
        final_handle = request_handles(target_handle_type, QStringList() <<  identifier, &error)[0];
    }

    return create_text_channel(target_handle_type, final_handle, request, error);
}

Tp::BaseChannelPtr mqt::tp::Connection::create_text_channel(uint target_handle_type,
                                                            uint target_handle,
                                                            const QVariantMap& hints,
                                                            Tp::DBusError* error)
{
    Q_UNUSED(error)
    messaging::Recipient::shared_ptr recipient;
    std::shared_ptr<GroupManager> group_manager;

    // channel is created by a received group invitation
    if (hints.contains(MQT_TP_RECIPIENT))
    {
        auto group = hints[MQT_TP_RECIPIENT].value<std::shared_ptr<messaging::Group>>();
        if (target_handle_type == Tp::HandleTypeRoom)
        {
            try
            {
                std::shared_ptr<GroupStarter> group_starter = connection->group_starter();
                group_manager = group_starter->accept_group(group);
                if (not group_manager || group_manager->group_id().empty())
                {
                    LOG(ERROR) << "Group creation rejected";
                    error->set(TP_QT_ERROR_CANCELLED, "Group creation rejected");
                    return Tp::BaseChannelPtr{};
                }

                // update recipient with id, title, creator, members and additional properties
                // FIXME(rmescandon): a refactor is needed to move all this to a group_manager->group() object
                // and create a copy constructor for Group which can be populated with cached data when needed
                auto additional_properties = group->additional_properties();
                if (not group_manager->group_subject().empty()) {
                    additional_properties[Group::SUBJECT] = std::make_shared<messaging::BoostVariant>(group_manager->group_subject());
                }
                auto members = group_manager->members();
                recipient = std::make_shared<messaging::Group>(
                            group_manager->group_id(),
                            members,
                            group_manager->group_title(),
                            group_manager->group_creator(),
                            additional_properties);
            }
            catch (const std::exception &e)
            {
                LOG(ERROR) << "Group creation rejected, " << e.what();
                error->set(TP_QT_ERROR_CANCELLED, "Group creation rejected");
                return Tp::BaseChannelPtr{};
            }
            catch (...)
            {
                LOG(ERROR) << "Group creation rejected";
                error->set(TP_QT_ERROR_CANCELLED, "Group creation rejected");
                return Tp::BaseChannelPtr{};
            }
        }
    }
    // channel is created from app or by receiving from plugin a message without chat
    else
    {
        switch (target_handle_type)
        {
        case Tp::HandleTypeContact:
            recipient = std::make_shared<messaging::User>(
                        handles_.left.find(target_handle)->second.toLatin1().data());
            break;
        case Tp::HandleTypeRoom: {
            recipient = std::make_shared<messaging::Group>(
                     group_handles_.left.find(target_handle)->second.toLatin1().data());

            try
            {
                std::shared_ptr<GroupStarter> group_starter = connection->group_starter();
                messaging::Group::shared_ptr group = std::dynamic_pointer_cast<messaging::Group>(recipient);
                group_manager = group_starter->rejoin_group(group);

                if (not group_manager || group_manager->group_id().empty())
                {
                    LOG(ERROR) << "Group rejoin rejected";
                    error->set(TP_QT_ERROR_CANCELLED, "Group rejoin rejected");
                    return Tp::BaseChannelPtr{};
                }

                // update recipient with id, title, creator, members and additional properties
                // FIXME(rmescandon): a refactor is needed to move all this to a group_manager->group() object
                // and create a copy constructor for Group which can be populated with cached data when needed
                auto additional_properties = group->additional_properties();
                if (not group_manager->group_subject().empty()) {
                    additional_properties[Group::SUBJECT] = std::make_shared<messaging::BoostVariant>(group_manager->group_subject());
                }
                auto members = group_manager->members();
                recipient = std::make_shared<messaging::Group>(
                            group_manager->group_id(),
                            members,
                            group_manager->group_title(),
                            group_manager->group_creator(),
                            additional_properties);
            }
            catch (const std::exception &e)
            {
                LOG(ERROR) << "Group rejoin rejected, " << e.what();
                error->set(TP_QT_ERROR_CANCELLED, "Group rejoin rejected");
                return Tp::BaseChannelPtr{};
            }
            catch (...)
            {
                LOG(ERROR) << "Group rejoin rejected";
                error->set(TP_QT_ERROR_CANCELLED, "Group rejoin rejected");
                return Tp::BaseChannelPtr{};
            }
            break;
        }
        case Tp::HandleTypeNone: {
            bool is_room = hints.contains(TP_QT_IFACE_CHANNEL_INTERFACE_ROOM + QLatin1String(".RoomName"));

            // There are two possible cases in which handle type is going to be none: chat rooms and broadcast channels.
            // For broadcast channels the handle type will remain as none during the entire channel lifetime. For chat rooms
            // once the creation is finished, the type will change to room.
            // For the chat room case, this code will be called when creating a group from the app.
            // There is not type for the handle of the channel to be created when
            // received the request. We have to switch to Room type and get the provided information
            messaging::Members initial_invitees;
            if (hints.contains(TP_QT_IFACE_CHANNEL_INTERFACE_CONFERENCE + QLatin1String(".InitialInviteeIDs")))
            {
                QStringList initial_invitees_ids =
                        hints[TP_QT_IFACE_CHANNEL_INTERFACE_CONFERENCE + QLatin1String(".InitialInviteeIDs")].toStringList();
                for (QString initial_invitee_id : initial_invitees_ids)
                {
                    std::string normalized_id = initial_invitee_id.toStdString();
                    try
                    {
                        normalized_id = connection->normalize_identifier(normalized_id);
                    }
                    catch (...)
                    {
                        LOG(ERROR) << "An exception has been thrown when normalizing identifier " << normalized_id
                                   << " using not normalized version";
                    }

                    //FIXME: get contact attribute display name for the user second param
                    //FIXME: consider here if needed to provide initial roles to members
                    initial_invitees.push_back(std::make_shared<Member>(normalized_id,
                                                                        is_room ? PendingStatus::Remote : PendingStatus::None,
                                                                        Flags<Role>(Role::Member),
                                                                        std::string{}/* display_name*/ ));
                }
            }

            // check if the client requested a chat room
            // if so, create the group related stuff
            if (is_room)
            {
                target_handle_type = Tp::HandleTypeRoom;

                std::string title;
                auto additional_properties = VariantMap{};

                if (hints.contains(TP_QT_IFACE_CHANNEL_INTERFACE_ROOM_CONFIG + QLatin1String(".Title")))
                {
                    title = hints[TP_QT_IFACE_CHANNEL_INTERFACE_ROOM_CONFIG + QLatin1String(".Title")].toString().toStdString();
                }

                std::string initiator_handle;
                if (hints.contains(TP_QT_IFACE_CHANNEL + QLatin1String(".InitiatorHandle")))
                {
                    initiator_handle = hints[TP_QT_IFACE_CHANNEL + QLatin1String(".InitiatorHandle")].toString().toStdString();
                }

                if (hints.contains(TP_QT_IFACE_CHANNEL_INTERFACE_SUBJECT + QLatin1String(".Subject")))
                {
                    additional_properties[Group::SUBJECT] =
                            std::make_shared<messaging::BoostVariant>(
                                hints[TP_QT_IFACE_CHANNEL_INTERFACE_SUBJECT + QLatin1String(".Subject")].toString().toStdString());
                }

                // create initial group recipient with no identifier for plugin to create the group
                recipient = std::make_shared<messaging::Group>(
                            std::string{} /* no id exists yet */,
                            initial_invitees,
                            title,
                            std::make_shared<messaging::Member>(initiator_handle),
                            additional_properties);
                try
                {
                    std::shared_ptr<GroupStarter> group_starter = connection->group_starter();
                    messaging::Group::shared_ptr group = std::dynamic_pointer_cast<messaging::Group>(recipient);
                    group_manager = group_starter->create_group(group);

                    if (not group_manager || group_manager->group_id().empty())
                    {
                        LOG(ERROR) << "Group creation rejected";
                        error->set(TP_QT_ERROR_CANCELLED, "Group creation rejected");
                        return Tp::BaseChannelPtr{};
                    }

                    // update recipient with id, title, members and additional properties
                    auto additional_properties = group->additional_properties();
                    if (not group_manager->group_subject().empty()) {
                        additional_properties[Group::SUBJECT] = std::make_shared<messaging::BoostVariant>(group_manager->group_subject());
                    }
                    recipient = std::make_shared<messaging::Group>(
                                group_manager->group_id(),
                                group_manager->members(),
                                group_manager->group_title(),
                                group_manager->group_creator(),
                                additional_properties);
                }
                catch (const std::exception &e)
                {
                    LOG(ERROR) << "Group creation rejected, " << e.what();
                    error->set(TP_QT_ERROR_CANCELLED, "Group creation rejected");
                    return Tp::BaseChannelPtr{};
                }
                catch (...)
                {
                    LOG(ERROR) << "Group creation rejected";
                    error->set(TP_QT_ERROR_CANCELLED, "Group creation rejected");
                    return Tp::BaseChannelPtr{};
                }

                // request handle from group_manager id, filled with the group id after calling start_group()
                Tp::DBusError error;
                target_handle = request_handles(Tp::HandleTypeRoom, QStringList() << QString::fromStdString(group_manager->group_id()), &error)[0];
            } else {
                // if not, consider this a broadcast channel
                std::shared_ptr<GroupStarter> group_starter = connection->group_starter();
                recipient = group_starter->create_broadcast(initial_invitees);
            }
            break;
        }
        default:
            LOG(ERROR) << "Target handle type for the channel to be created is not valid";
            return Tp::BaseChannelPtr{};
        }
    }

    Tp::BaseChannelPtr text_channel;

    try
    {
        text_channel = Tp::BaseChannelPtr{mqt::tp::TextChannel::create(this,
                                                                       target_handle,
                                                                       target_handle_type,
                                                                       runtime,
                                                                       connection->messenger(),
                                                                       recipient,
                                                                       group_manager)};
    }
    catch (...)
    {
        LOG(ERROR) << "An exception has been thrown when creating the text channel";
    }

    return text_channel;
}

Tp::ContactAttributesMap  mqt::tp::Connection::get_contact_attributes(const Tp::UIntList &handles, const QStringList &ifaces, Tp::DBusError *error)
{
    Tp::ContactAttributesMap attributesMap;
    QVariantMap attributes;
    for(auto handle : handles) {
        QStringList inspectedHandles = inspect_handles(Tp::HandleTypeContact, Tp::UIntList() << handle, error);
        if (inspectedHandles.size() > 0) {
            attributes[TP_QT_IFACE_CONNECTION+"/contact-id"] = inspectedHandles.at(0);
        } else {
            continue;
        }
        if (ifaces.contains(TP_QT_IFACE_CONNECTION_INTERFACE_SIMPLE_PRESENCE)) {
            Tp::SimplePresence presence = presences_[handle];
            attributes[TP_QT_IFACE_CONNECTION_INTERFACE_SIMPLE_PRESENCE+"/presence"] = QVariant::fromValue(presence);
        }
        attributesMap[handle] = attributes;
    }
    return attributesMap;
}

QString mqt::tp::Connection::uniqueName() const
{
    return connection_id;
}
