/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <TelepathyQt/DBusObject>
#include <messaging/qt/tp/interfaces/base_channel_subject.h>
#include <messaging/qt/tp/interfaces/base_channel_subject_internal.h>

namespace mqti = messaging::qt::tp::interfaces;

// Chan.I.Subject
mqti::BaseChannelSubjectInterface::Adaptee::Adaptee(BaseChannelSubjectInterface *interface)
    : QObject(interface),
      mInterface(interface)
{
}

mqti::BaseChannelSubjectInterface::Adaptee::~Adaptee()
{
}

struct TP_QT_NO_EXPORT mqti::BaseChannelSubjectInterface::Private {
    Private(BaseChannelSubjectInterface *parent)
        :adaptee(new BaseChannelSubjectInterface::Adaptee(parent)) {
    }

    QString subject;
    QString actor;
    uint actorHandle;
    qlonglong timestamp;
    bool canSet;
    SetSubjectCallback setSubjectCB;
    BaseChannelSubjectInterface::Adaptee *adaptee;
};

QString mqti::BaseChannelSubjectInterface::Adaptee::subject() const
{
    return mInterface->subject();
}

QString mqti::BaseChannelSubjectInterface::Adaptee::actor() const
{
    return mInterface->actor();
}

uint mqti::BaseChannelSubjectInterface::Adaptee::actorHandle() const
{
    return mInterface->actorHandle();
}

qlonglong mqti::BaseChannelSubjectInterface::Adaptee::timestamp() const
{
    return mInterface->timestamp();
}

bool mqti::BaseChannelSubjectInterface::Adaptee::canSet() const
{
    return mInterface->canSet();
}

void mqti::BaseChannelSubjectInterface::Adaptee::setSubject(const QString &subject, const Tp::Service::ChannelInterfaceSubjectAdaptor::SetSubjectContextPtr &context)
{
    if (!mInterface->mPriv->setSubjectCB.isValid()) {
        context->setFinishedWithError(TP_QT_ERROR_NOT_IMPLEMENTED, QLatin1String("Not implemented"));
        return;
    }

    Tp::DBusError error;
    mInterface->mPriv->setSubjectCB(subject, &error);
    if (error.isValid()) {
        context->setFinishedWithError(error.name(), error.message());
        return;
    }
    context->setFinished();
}

/**
 * \class BaseChannelSubjectInterface
 *
 * \brief Base class for implementations of Channel.Interface.Subject
 *
 */

/**
 * Class constructor.
 */
mqti::BaseChannelSubjectInterface::BaseChannelSubjectInterface()
    : Tp::AbstractChannelInterface(TP_QT_IFACE_CHANNEL_INTERFACE_SUBJECT),
      mPriv(new Private(this))
{
}

void mqti::BaseChannelSubjectInterface::setSetSubjectCallback(const SetSubjectCallback &cb)
{
    mPriv->setSubjectCB = cb;
}

/**
 * Class destructor.
 */
mqti::BaseChannelSubjectInterface::~BaseChannelSubjectInterface()
{
    delete mPriv;
}

/**
 * Return the immutable properties of this interface.
 *
 * Immutable properties cannot change after the interface has been registered
 * on a service on the bus with registerInterface().
 *
 * \return The immutable properties of this interface.
 */
QVariantMap mqti::BaseChannelSubjectInterface::immutableProperties() const
{
    QVariantMap map;
    return map;
}

void mqti::BaseChannelSubjectInterface::createAdaptor()
{
    (void) new Tp::Service::ChannelInterfaceSubjectAdaptor(dbusObject()->dbusConnection(),
            mPriv->adaptee, dbusObject());
}

void mqti::BaseChannelSubjectInterface::setSubject(const QString &subject, Tp::DBusError *error)
{
    if (!mPriv->setSubjectCB.isValid()) {
        error->set(TP_QT_ERROR_NOT_IMPLEMENTED, QLatin1String("Not implemented"));
        return;
    }

    if (!canSet()) {
        error->set(TP_QT_ERROR_PERMISSION_DENIED, QLatin1String("Operation not allowed; canSet flag set to false"));
    }

    return mPriv->setSubjectCB(subject, error);
}

QString mqti::BaseChannelSubjectInterface::subject() const
{
    return mPriv->subject;
}

void mqti::BaseChannelSubjectInterface::setSubject(const QString &subject)
{
    mPriv->subject = subject;
    notifyPropertyChanged(QLatin1String("Subject"), QVariant::fromValue(subject));
}

QString mqti::BaseChannelSubjectInterface::actor() const
{
    return mPriv->actor;
}

void mqti::BaseChannelSubjectInterface::setActor(const QString &actor)
{
    mPriv->actor = actor;
    notifyPropertyChanged(QLatin1String("Actor"), QVariant::fromValue(actor));
}

uint mqti::BaseChannelSubjectInterface::actorHandle() const
{
    return mPriv->actorHandle;
}

void mqti::BaseChannelSubjectInterface::setActorHandle(uint handle)
{
    mPriv->actorHandle = handle;
    notifyPropertyChanged(QLatin1String("ActorHandle"), QVariant::fromValue(handle));
}

qlonglong mqti::BaseChannelSubjectInterface::timestamp() const
{
    return mPriv->timestamp;
}

void mqti::BaseChannelSubjectInterface::setTimestamp(qlonglong timestamp)
{
    mPriv->timestamp = timestamp;
    notifyPropertyChanged(QLatin1String("Timestamp"), QVariant::fromValue(timestamp));
}

bool mqti::BaseChannelSubjectInterface::canSet() const
{
    return mPriv->canSet;
}

void mqti::BaseChannelSubjectInterface::setCanSet(bool canSet)
{
    mPriv->canSet = canSet;
    notifyPropertyChanged(QLatin1String("CanSet"), QVariant::fromValue(canSet));
}
