/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/boost/variant.h>

#include <gtest/gtest.h>

TEST(BoostVariant, construction_with_boolean_works)
{
    messaging::BoostVariant bv{true};
    EXPECT_EQ(messaging::Variant::Type::boolean, bv.type());
    EXPECT_TRUE(bv.as_bool());
}

TEST(BoostVariant, construction_with_integer_works)
{
    messaging::BoostVariant bv{std::int64_t{42}};
    EXPECT_EQ(messaging::Variant::Type::integer, bv.type());
    EXPECT_EQ(42, bv.as_int());
}

TEST(BoostVariant, construction_with_double_works)
{
    messaging::BoostVariant bv{double{42}};
    EXPECT_EQ(messaging::Variant::Type::floating_point, bv.type());
    EXPECT_EQ(42, bv.as_double());
}

TEST(BoostVariant, construction_with_string_works)
{
    messaging::BoostVariant bv{std::string{"42"}};
    EXPECT_EQ(messaging::Variant::Type::string, bv.type());
    EXPECT_EQ("42", bv.as_string());
}

TEST(BoostVariant, construction_with_vector_works)
{
    messaging::BoostVariant bv{std::vector<messaging::BoostVariant::Value>{{std::string{"42"}, std::string{"42"}}}};
    EXPECT_EQ(messaging::Variant::Type::recursive, bv.type());
    EXPECT_EQ(2, bv.size());
    EXPECT_EQ("42", bv.at(0)->as_string());
    EXPECT_EQ("42", bv.at(1)->as_string());
}

TEST(BoostVariant, throws_for_type_mismatch)
{
    messaging::BoostVariant bv{std::vector<messaging::BoostVariant::Value>{{std::string{"42"}, std::string{"42"}}}};
    EXPECT_THROW(bv.as_bool(), messaging::Variant::TypeMismatch);
}

TEST(BoostVariant, construction_with_char_array_reply_as_data)
{
    const char *literal = "abc\0defghi";
    messaging::BoostVariant bv{literal, 10};

    EXPECT_EQ(messaging::Variant::Type::data, bv.type());
    EXPECT_EQ(10, bv.data_size());

    for (int i=0; i<10; i++)
        EXPECT_EQ(literal[i], bv.as_data()[i]);
}

TEST(BoostVariant, construction_with_char_vector_reply_as_data)
{
    const char *literal = "abc\0defghi";
    std::vector<char> data{literal, literal + 10};

    messaging::BoostVariant bv{data};

    EXPECT_EQ(messaging::Variant::Type::data, bv.type());
    EXPECT_EQ(10, bv.data_size());

    for (int i=0;i<10;i++)
        EXPECT_EQ(data.data()[i], std::vector<char>(bv.as_data(), bv.as_data() + bv.data_size()).data()[i]);
}

TEST(BoostVariant, construction_with_string_reply_as_data)
{
    messaging::BoostVariant bv{std::string{"abc\0defghi", 10}};
    EXPECT_EQ(messaging::Variant::Type::string, bv.type());
    EXPECT_THROW(bv.as_data(), messaging::Variant::TypeMismatch);
}

