/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "utils.h"

#include <TelepathyQt/Account>
#include <TelepathyQt/AccountFactory>
#include <TelepathyQt/AccountManager>
#include <TelepathyQt/ChannelFactory>
#include <TelepathyQt/ConnectionFactory>
#include <TelepathyQt/PendingAccount>
#include <TelepathyQt/PendingChannelRequest>
#include <TelepathyQt/PendingReady>
#include <TelepathyQt/SimpleObserver>
#include <TelepathyQt/TextChannel>

#include <QtDBus/QDBusServiceWatcher>
#include <QtTest/QSignalSpy>

#include <chrono>
#include <thread>

namespace tp = testing::qt::tp;

namespace
{
/// @brief creates an AccountManager stub, connecting to the other side via the given bus instance
Tp::AccountManagerPtr make_am_stub(const QDBusConnection& bus)
{
    return Tp::AccountManager::create(bus,
                                      Tp::AccountFactory::create(bus, Tp::Account::FeatureCore),
                                      Tp::ConnectionFactory::create(bus, Tp::Features() << Tp::Connection::FeatureCore
                                                                                        << Tp::Connection::FeatureConnected
                                                                                        << Tp::Connection::FeatureSelfContact
                                                                                        << Tp::Connection::FeatureSimplePresence),
                                      Tp::ChannelFactory::create(bus),
                                      Tp::ContactFactory::create(Tp::Features() << Tp::Contact::FeatureAlias
                                                                                << Tp::Contact::FeatureSimplePresence));

}

/// @brief finished helps us in saving some typing when expressing long call-chains.
auto finished = &Tp::PendingOperation::finished;
}

testing::AssertionResult tp::pending_operation_finished_successfully(Tp::PendingOperation *op)
{
    if (not op->isFinished())
    {
        return testing::AssertionFailure() << "PendingOperation is not finished, despite having been reported as such";
    }

    if (op->isError())
    {
        return testing::AssertionFailure() << "PendingOperation completed with error: " << op->errorName().toStdString()
                                           << ", " << op->errorMessage().toStdString();
    }

    if (not op->isValid())
    {
        return testing::AssertionFailure() << "PendingOperation instance is invalid.";
    }

    return testing::AssertionSuccess();
}

void tp::when_finished(Tp::PendingOperation* op, const std::function<void(Tp::PendingOperation* op)>& then)
{
    QObject::connect(op, finished, [then](Tp::PendingOperation* op) {
            ASSERT_TRUE(pending_operation_finished_successfully(op));
            then(op);
        });
}

void tp::when_finished(Tp::PendingOperation* op, const std::function<void()>& then)
{
    QObject::connect(op, finished, [then](Tp::PendingOperation* op) {
            ASSERT_TRUE(pending_operation_finished_successfully(op));
            then();
        });
}

testing::AssertionResult tp::wait_for_connection_manager_with_name_and_protocol_on_bus(const std::string& name, const std::string& protocol, const QDBusConnection& bus)
{
    QString cm_service("org.freedesktop.Telepathy.ConnectionManager");
    auto cm = QString("%1.%2").arg(cm_service).arg(QString::fromStdString(name));
    auto obj_path = QString("/org/freedesktop/Telepathy/ConnectionManager/%1").arg(QString::fromStdString(name));
    auto proto = QString::fromStdString(protocol);

    // if the service is already registered, we can jump straight to checking if the protocol is available
    if (not bus.interface()->isServiceRegistered(cm))
    {
        QDBusServiceWatcher watcher(cm, bus, QDBusServiceWatcher::WatchForRegistration);
        QSignalSpy spy{&watcher, &QDBusServiceWatcher::serviceRegistered};

        if (not spy.wait() && not bus.interface()->isServiceRegistered(cm))
            return testing::AssertionFailure() << "Wait for connection manager timed out.";
    }

    QDBusInterface cm_iface(cm, obj_path, cm_service);
    bool found_protocol = false;
    int tries = 0;
    while (!found_protocol && tries < 5)
    {
        tries++;
        QDBusReply<QStringList> reply = cm_iface.call("ListProtocols");
        if (not reply.isValid())
            return testing::AssertionFailure() << "Could not list available protocols";

        if (reply.value().contains(proto))
        {
            found_protocol = true;
            break;
        }
        std::this_thread::sleep_for(std::chrono::milliseconds(500));
    }

    if (found_protocol)
        return testing::AssertionSuccess();
    return testing::AssertionFailure() << "Protocol not found";
}

void tp::create_account_and_then(const QString &cm_name, const QString &protocol_name, const std::shared_ptr<messaging::qt::Runtime> &rt, const std::function<void (const Tp::AccountPtr &)> &then)
{
    rt->enter_with_task([cm_name, protocol_name, rt, then]() {
        auto am = make_am_stub(rt->dbus_connection());
        when_finished(am->becomeReady(), [cm_name, protocol_name, rt, then, am]() {
            when_finished(am->createAccount(cm_name, protocol_name, "Testing", QVariantMap{}), [cm_name, protocol_name, rt, then, am](Tp::PendingOperation* op) {
                auto account = static_cast<Tp::PendingAccount*>(op)->account();
                when_finished(account->becomeReady(), [then, account, rt](){
                    when_finished(account->setEnabled(true), [then, account, rt]() {
                        then(account);
                    });
                });
            });
        });
    });
}

void tp::create_online_account_and_then(const QString &cm_name,
                                        const QString &protocol_name,
                                        const std::shared_ptr<messaging::qt::Runtime> &rt,
                                        const std::function<void (const Tp::AccountPtr &)> &then)
{
    create_account_and_then(cm_name, protocol_name, rt, [cm_name, protocol_name, rt, then] (const Tp::AccountPtr &account) {
        when_finished(account->setRequestedPresence(Tp::Presence::available()), [then, account, rt]() {
            then(account);
        });
    });
}

void tp::disconnect_account_and_then(const Tp::AccountPtr &account, const std::shared_ptr<messaging::qt::Runtime> &rt, const std::function<void ()> &then)
{
    rt->enter_with_task([account, then](){
        when_finished(account->setRequestedPresence(Tp::Presence::offline()), [then]{
            then();
        });
    });
}

void tp::ensure_text_channel_and_then(const QString& cm_name, const QString& protocol_name, const std::shared_ptr<messaging::qt::Runtime>& rt, const std::function<void(const Tp::TextChannelPtr&)>& then)
{
    rt->enter_with_task([cm_name, protocol_name, rt, then](){
        create_account_and_then(cm_name, protocol_name, rt, [then, rt](const Tp::AccountPtr &account) {
            when_finished(account->ensureTextChat("42"), [then, account, rt](Tp::PendingOperation* op) {
                auto pc = static_cast<Tp::PendingChannelRequest*>(op);
                auto tc = Tp::TextChannelPtr::dynamicCast(pc->channelRequest()->channel());
                when_finished(tc->becomeReady(Tp::Features() << Tp::TextChannel::FeatureCore << Tp::TextChannel::FeatureMessageQueue), [account, then, tc, rt] () {
                    then(tc);
                });
            });
        });
    });
}

void tp::wait_for_text_channel_and_then(const Tp::AccountPtr &account, const std::shared_ptr<messaging::qt::Runtime> &rt, const std::function<void (const Tp::TextChannelPtr &)> &then)
{
    rt->enter_with_task([account, rt, then](){
        // FIXME: switch over to use this constructor once we update to telepathy-qt 0.9.6 or later.
        // In the current version (0.9.3) it causes a crash
        //Tp::SimpleObserverPtr observer = Tp::SimpleObserver::create(account, Tp::ChannelClassSpecList() << Tp::ChannelClassSpec::textChat());
        Tp::SimpleObserverPtr observer = Tp::SimpleObserver::create(account, Tp::ChannelClassSpecList() << Tp::ChannelClassSpec::textChat() << Tp::ChannelClassSpec::textChatroom(), Tp::ContactPtr());
        QObject::connect(observer.data(), &Tp::SimpleObserver::newChannels, [account, observer, rt, then](const QList<Tp::ChannelPtr>& channels){
            for(auto channel : channels)
            {
                Tp::TextChannelPtr tc = Tp::TextChannelPtr::dynamicCast(channel);
                ASSERT_FALSE(tc.isNull());
                when_finished(tc->becomeReady(Tp::Features() << Tp::TextChannel::FeatureCore << Tp::TextChannel::FeatureMessageQueue), [then, tc, rt] () {
                    then(tc);
                });
            }
        });
    });
}
