/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/dictionary.h>
#include <messaging/enumerator.h>
#include <messaging/parameter.h>
#include <messaging/variant.h>
#include <messaging/user.h>
#include <messaging/group.h>

#include <messaging/qt/runtime.h>
#include <messaging/qt/variant.h>
#include <messaging/qt/variant_map_facade.h>

#include <messaging/qt/tp/adapter.h>
#include <messaging/qt/tp/initializer.h>
#include <messaging/qt/tp/connection.h>

#include <messaging/raii.h>

#include "dbus_monitor.h"
#include "did_finish.h"
#include "mission_control.h"
#include "mock_chat.h"
#include "mock_connector.h"
#include "mock_connection.h"
#include "mock_group_manager.h"
#include "mock_messenger.h"
#include "mock_presence_manager.h"

#include "qt/tp/client.h"
#include "qt/tp/utils.h"

#include <QtDBus/QDBusServiceWatcher>
#include <QtTest/QSignalSpy>

#include <TelepathyQt/Account>
#include <TelepathyQt/AccountManager>
#include <TelepathyQt/Channel>
#include <TelepathyQt/ChannelClassSpecList>
#include <TelepathyQt/ChannelClassSpec>
#include <TelepathyQt/ConnectionCapabilities>
#include <TelepathyQt/ConnectionManager>
#include <TelepathyQt/ConnectionManagerLowlevel>
#include <TelepathyQt/ContactManager>
#include <TelepathyQt/PendingAccount>
#include <TelepathyQt/PendingChannelRequest>
#include <TelepathyQt/PendingConnection>
#include <TelepathyQt/PendingContacts>
#include <TelepathyQt/PendingOperation>
#include <TelepathyQt/PendingReady>
#include <TelepathyQt/PendingVariantMap>
#include <TelepathyQt/ReceivedMessage>
#include <TelepathyQt/TextChannel>

#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>

#include <core/dbus/fixture.h>
#include <core/posix/exit.h>
#include <core/posix/fork.h>
#include <core/posix/wait.h>
#include <core/testing/cross_process_sync.h>

#include <gtest/gtest.h>

#include <csignal>
#include <fstream>
#include <thread>


// All tests in this file go over the wire, relying on a private dbus connection setup and torn down
// for every individual testcase to ensure maximum test isolation. In addition, it eases with running
// the tests in CI reliably (modulo very infrequent timing issues.) In general, the setup looks like:
//   * skeleton creates a qt::tp::Adapter exposing a tp::ConnectionManager.
//   * stub sets up tp::* proxies to communicate with the tp::* instances setup by qt::tp::Adapter
namespace
{
/// @brief failure_to_exit_status maps Google Test HasFailure() to a process exit status.
/// We introduce the int argument as a way to be able to pass the result qt::Runtime::run
/// to the function.
core::posix::exit::Status failure_to_exit_status(int)
{
    return testing::Test::HasFailure() ? core::posix::exit::Status::failure : core::posix::exit::Status::success;
}

/// @brief AdapterWithPrivateBus is a fixture setting up a private bus instance,
/// providing the address of the session and system bus to test cases.
struct AdapterWithPrivateBus : public core::dbus::testing::Fixture
{
    // Our custom protocol name used in testing.
    static constexpr const char* protocol_name{"protocol"};
    // Our custom connection manager name used in testing.
    static constexpr const char* connection_manager_name{"cm"};

    // Env variables affecting all test cases using this fixture go here.
    struct Env
    {
        // Only for grouping purposes.
        Env() = delete;

        // Controls whether we enable debug output from telepathy qt within test cases.
        // The debug output is disabled by default, can be enabled for debugging issues in test cases.
        struct MessagingQtTpEnableDebugOutput
        {
            static bool get()
            {
                return core::posix::this_process::env::get("MESSAGING_QT_TP_ENABLE_DEBUG_OUTPUT", "0") == "1";
            }
        };

        // Controls whether we enable warning output from telepathy qt within test cases.
        // The warning output is disabled by default, but can be enabled for debugging issues in test cases.
        struct MessagingQtTpEnableWarningOutput
        {
            static bool get()
            {
                return core::posix::this_process::env::get("MESSAGING_QT_TP_ENABLE_WARNING_OUTPUT", "0") == "1";
            }
        };
    };

    // Creates a new instance.
    AdapterWithPrivateBus()
            : sig_chld_trap{core::posix::trap_signals_for_all_subsequent_threads({core::posix::Signal::sig_chld})},
              dbus_monitor{session_bus_address()}
    {
        tp_qt_configuration.enable_debug_output = Env::MessagingQtTpEnableDebugOutput::get();
        tp_qt_configuration.enable_warning_output = Env::MessagingQtTpEnableWarningOutput::get();
    }

    std::shared_ptr<core::posix::SignalTrap> sig_chld_trap; ///< Trap sigchld to avoid races with process dying away.
    testing::DBusMonitor dbus_monitor;  ///< dbus_monitor monitors testing-specific bus traffic.
    testing::tp::MissionControl mc;  ///< Our private mission control instance we use in testing.
    messaging::qt::tp::Initializer::Configuration tp_qt_configuration;  ///< Central configuration to be used in all
                                                                        ///test cases.
};

// make_cm_stub creates a ConnectionManager stub, connecting to the other side via the
// given bus instance with the given name.
Tp::ConnectionManagerPtr make_cm_stub(const QDBusConnection& bus, const QString& name)
{
    // We have to rely on this overload as the simpler (bus, name) overload
    // creates the proxy for the default session bus and does *not* use the
    // connection passed in for testing.
    return Tp::ConnectionManager::create(
        bus, name, Tp::ConnectionFactory::create(bus), Tp::ChannelFactory::create(bus), Tp::ContactFactory::create());
}
}

TEST_F(AdapterWithPrivateBus, construction_and_deconstruction_works)
{
    using namespace ::testing;
    auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
    auto connector = std::make_shared<NiceMock<MockConnector>>();
    messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};
}

TEST_F(AdapterWithPrivateBus, registers_a_connection_manager_with_correct_name)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cps;

    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [this, &cps]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            rt->enter_with_task(
                [this, rt, &cps]()
                {
                    // trigger cross process signal
                    EXPECT_NO_THROW(cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                    ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));
                    rt->stop(EXIT_SUCCESS);
                });

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    // Skeleton instantiates a qt::tp::Adapter instance, exposing it to the bus.
    auto skeleton = core::posix::fork(
        [this]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            auto connector = std::make_shared<NiceMock<MockConnector>>();

            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
}

TEST_F(AdapterWithPrivateBus, exposes_a_protocol_with_correct_name_supporting_text_chat)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cps;

    auto stub = core::posix::fork(
        [this, &cps]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));

            rt->enter_with_task(
                [rt, &cps]()
                {
                    // trigger cross process signal
                    EXPECT_NO_THROW(cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                    ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));
                    auto cm = make_cm_stub(rt->dbus_connection(), QString::fromStdString(connection_manager_name));
                    testing::qt::tp::when_finished(cm->becomeReady(), [cm, rt]() {
                        EXPECT_TRUE(cm->hasProtocol(QString::fromStdString(protocol_name)));
                        EXPECT_TRUE(cm->protocol(QString::fromStdString(protocol_name)).capabilities().textChats());
                        rt->stop(EXIT_SUCCESS);
                    });
                });

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    // Skeleton instantiates a qt::tp::Adapter instance, exposing it to the bus.
    auto skeleton = core::posix::fork(
        [this]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            auto connector = std::make_shared<NiceMock<MockConnector>>();

            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
}

TEST_F(AdapterWithPrivateBus, exposes_correct_set_of_protocol_parameters)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cps;

    static const std::vector<messaging::Parameter> params{
        messaging::Parameter{"test1", "s", messaging::Parameter::Flags::required},
        messaging::Parameter{"test2", "i", messaging::Parameter::Flags::secret},
        messaging::Parameter{"test3", "b", messaging::Parameter::Flags::required_for_registration}};

    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [this, &cps]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            rt->enter_with_task(
                [rt, &cps]()
                {
                    // trigger cross process signal
                    EXPECT_NO_THROW(cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                    ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));

                    auto cm = make_cm_stub(rt->dbus_connection(), QString::fromStdString(connection_manager_name));
                    testing::qt::tp::when_finished(cm->becomeReady(), [cm, rt]() {
                        auto pi = cm->protocol(QString::fromStdString(protocol_name));
                        for (const auto& param : params)
                        {
                            EXPECT_TRUE(pi.hasParameter(QString::fromStdString(param.name)));
                        }
                        rt->stop(EXIT_SUCCESS);
                    });
                });

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    // Skeleton instantiates a qt::tp::Adapter instance, exposing it to the bus.
    auto skeleton = core::posix::fork(
        [this]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            auto connector = std::make_shared<NiceMock<MockConnector>>();
            messaging::StdVectorEnumerator<messaging::Parameter> enumerator{params};
            ON_CALL(*connector, parameters()).WillByDefault(ReturnRef(enumerator));
            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
}

namespace
{
MATCHER_P(ParametersAreEqual, ref, "")
{
    for (const auto& pair : ref.data())
    {
        if (not arg.has_value_for_key(pair.first))
        {
            return false;
        }
    }

    return true;
}
}

TEST_F(AdapterWithPrivateBus, can_create_connections_via_available_protocols)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cps;

    static const QVariantMap connect_parameters{{"entry1", 42}, {"entry2", "42"}};

    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [this, &cps]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            rt->enter_with_task(
                [rt, &cps]()
                {
                    // trigger cross process signal
                    EXPECT_NO_THROW(cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                    ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));
                    auto cm = make_cm_stub(rt->dbus_connection(), QString::fromStdString(connection_manager_name));
                    testing::qt::tp::when_finished(cm->becomeReady(), [cm, rt]() {
                        testing::qt::tp::when_finished(cm->lowlevel()->requestConnection(QString::fromStdString(protocol_name),
                                                                                         connect_parameters),
                                                       [=]() {
                            rt->stop(EXIT_SUCCESS);
                        });
                    });
                });

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    auto skeleton = core::posix::fork(
        [this]()
        {
            messaging::qt::VariantMapFacade vmf{connect_parameters};

            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            auto connector = std::make_shared<NiceMock<MockConnector>>();
            EXPECT_CALL(*connector, request_connection(_, _, _, ParametersAreEqual(vmf))).Times(1);
            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
}

TEST_F(AdapterWithPrivateBus, can_create_text_channel_and_send_message)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cps;

    auto client = core::posix::fork(
        [this]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            testing::qt::tp::Client client{rt->dbus_connection(),
                                           Tp::ChannelClassSpecList() << Tp::ChannelClassSpec::textChat()};
            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [this, &cps]()
        {
            auto trap = core::posix::trap_signals_for_all_subsequent_threads({core::posix::Signal::sig_term});
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            trap->signal_raised().connect([trap](core::posix::Signal) { trap->stop(); });
            messaging::raii::AutoJoiningThread ajt{[trap]() { trap->run(); }};

            rt->enter_with_task([rt, &cps]() {
                    // trigger cross process signal
                    EXPECT_NO_THROW(cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                    ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));
                    testing::qt::tp::ensure_text_channel_and_then(connection_manager_name, protocol_name, rt, [rt](const Tp::TextChannelPtr& tc) {
                            testing::qt::tp::when_finished(tc->send("This is just a test"), [rt](Tp::PendingOperation* op) {
                                  EXPECT_TRUE(testing::qt::tp::pending_operation_finished_successfully(op));
                                  rt->stop(EXIT_SUCCESS);
                              });
                        });
                });

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    // Skeleton instantiates a qt::tp::Adapter instance, exposing it to the bus.
    auto skeleton = core::posix::fork(
        [this]()
        {
            auto trap = core::posix::trap_signals_for_all_subsequent_threads({core::posix::Signal::sig_term});
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));

            trap->signal_raised().connect([trap](core::posix::Signal) { trap->stop(); });
            messaging::raii::AutoJoiningThread ajt{[trap]() { trap->run(); }};

            auto connector = std::make_shared<NiceMock<MockConnector>>();

            // The liftime of the connector and connection instance is not govered by this scope.
            // To prevent test erros, we let the mocking framework know about this niggle.

            ON_CALL(*connector, request_connection(_, _, _, _)).WillByDefault(
                        Invoke([rt](const std::shared_ptr<messaging::Connection::Observer>& connection_observer,
                               const std::shared_ptr<messaging::Messenger::Observer>& messenger_observer,
                               const std::shared_ptr<messaging::PresenceManager::Observer>& presence_observer,
                               const messaging::Dictionary<std::string, messaging::Variant>& )
                        {
                            auto connection = std::make_shared<NiceMock<MockConnection>>(connection_observer, messenger_observer, presence_observer);
                            auto messenger = std::dynamic_pointer_cast<NiceMock<MockMessenger>>(connection->messenger());

                            Mock::AllowLeak(connection.get());
                            Mock::AllowLeak(messenger.get());

                            ON_CALL(*messenger, create_chat_with(_, _)).WillByDefault(Invoke([rt](const messaging::Recipient::shared_ptr&, const std::shared_ptr<messaging::Chat::Observer>& observer) {
                                        auto chat = std::make_shared<NiceMock<MockChat>>(observer);
                                        Mock::AllowLeak(chat.get());
                                        EXPECT_CALL(*chat, send_message(_)).Times(1).WillRepeatedly(
                                            DoAll(InvokeWithoutArgs([rt]()
                                                                    {
                                                                        rt->stop(EXIT_SUCCESS);
                                                                    }),
                                                  Return("42")));
                                        return chat;
                                    }));


                            return connection;
                        }));


            // The liftime of the connector and connection instance is not govered by this scope.
            // To prevent test erros, we let the mocking framework know about this niggle.
            Mock::AllowLeak(connector.get());

            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    stub.send_signal_or_throw(core::posix::Signal::sig_term);
    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
    skeleton.send_signal_or_throw(core::posix::Signal::sig_term);
    EXPECT_TRUE(testing::did_finish_successfully(skeleton.wait_for(core::posix::wait::Flags::untraced)));
}

TEST_F(AdapterWithPrivateBus, message_received_without_chat_creates_a_channel)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cm_cps;
    core::testing::CrossProcessSync account_cps;

    std::string recipient_id{"12345"};
    std::string message_text{"Hi there!"};

    auto client = core::posix::fork(
        [this]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            testing::qt::tp::Client client{rt->dbus_connection(),
                                           Tp::ChannelClassSpecList() << Tp::ChannelClassSpec::textChat()};
            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);


    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [this, &cm_cps, &account_cps, recipient_id, message_text]()
        {
            auto trap = core::posix::trap_signals_for_all_subsequent_threads({core::posix::Signal::sig_term});
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            trap->signal_raised().connect([trap](core::posix::Signal) { trap->stop(); });
            messaging::raii::AutoJoiningThread ajt{[trap]() { trap->run(); }};

            rt->enter_with_task([rt, &cm_cps, &account_cps, recipient_id, message_text]() {
                // trigger cross process signal
                EXPECT_NO_THROW(cm_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));

                // wait for cross process signal
                std::uint32_t counter = 0;
                EXPECT_NO_THROW(counter = account_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
                EXPECT_EQ(counter, 1);

                testing::qt::tp::create_online_account_and_then(connection_manager_name, protocol_name, rt,
                                                         [rt, recipient_id, message_text](const Tp::AccountPtr& account) {
                    testing::qt::tp::wait_for_text_channel_and_then(account, rt, [account, rt, recipient_id, message_text](const Tp::TextChannelPtr& channel){
                        ASSERT_FALSE(channel->isRequested());
                        ASSERT_FALSE(channel->targetContact().isNull());
                        Tp::ContactPtr contact = channel->targetContact();
                        ASSERT_EQ(contact->id().toStdString(), recipient_id);
                        // the channel needs to contain the message that triggered its creation
                        ASSERT_EQ(channel->messageQueue().count(), 1);
                        Tp::ReceivedMessage message = channel->messageQueue().first();
                        ASSERT_EQ(message.sender(), contact);
                        ASSERT_EQ(message.text().toStdString(), message_text);
                        testing::qt::tp::disconnect_account_and_then(account, rt, [rt]{
                            rt->stop(EXIT_SUCCESS);
                        });
                    });
                });
            });

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cm_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    // Skeleton instantiates a qt::tp::Adapter instance, exposing it to the bus.
    auto skeleton = core::posix::fork(
        [this, &account_cps, recipient_id, message_text]()
        {
            auto trap = core::posix::trap_signals_for_all_subsequent_threads({core::posix::Signal::sig_term});
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));

            trap->signal_raised().connect([trap](core::posix::Signal) { trap->stop(); });
            messaging::raii::AutoJoiningThread ajt{[trap]() { trap->run(); }};

            auto connector = std::make_shared<NiceMock<MockConnector>>();

            // The lifetime of the connector and connection instance is not govered by this scope.
            // To prevent test erros, we let the mocking framework know about this niggle.
            Mock::AllowLeak(connector.get());

            ON_CALL(*connector, request_connection(_, _, _, _)).WillByDefault(
                Invoke([rt, recipient_id, message_text](const std::shared_ptr<messaging::Connection::Observer>& connection_observer,
                       const std::shared_ptr<messaging::Messenger::Observer>& messenger_observer,
                       const std::shared_ptr<messaging::PresenceManager::Observer>& presence_observer,
                       const messaging::Dictionary<std::string, messaging::Variant>& )
                {
                    auto connection = std::make_shared<NiceMock<MockConnection>>(connection_observer, messenger_observer, presence_observer);
                    auto messenger = std::dynamic_pointer_cast<NiceMock<MockMessenger>>(connection->messenger());

                    Mock::AllowLeak(connection.get());
                    Mock::AllowLeak(messenger.get());

                    ON_CALL(*connection, normalize_identifier(_)).WillByDefault(Return(recipient_id));
                    ON_CALL(*connection, connect()).WillByDefault(Invoke([rt, connection, messenger, recipient_id, message_text](){
                        connection->announce_status_connected();
                        // announce the incoming messaging

                        auto user = std::make_shared<messaging::User>(recipient_id);
                        messaging::Message message{"some_id",
                                                   recipient_id,
                                                   message_text,
                                                   std::chrono::system_clock::now()};
                        messenger->announce_message_without_chat_received(user, message);
                    }));
                    ON_CALL(*connection, disconnect()).WillByDefault(Invoke([rt]{
                        rt->stop(EXIT_SUCCESS);
                    }));
                    return connection;
                }));

            // The lifetime of the connector and connection instance is not govered by this scope.
            // To prevent test erros, we let the mocking framework know about this niggle.
            Mock::AllowLeak(connector.get());

            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            EXPECT_NO_THROW(account_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    stub.send_signal_or_throw(core::posix::Signal::sig_term);
    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
    skeleton.send_signal_or_throw(core::posix::Signal::sig_term);
    EXPECT_TRUE(testing::did_finish_successfully(skeleton.wait_for(core::posix::wait::Flags::untraced)));
}

TEST_F(AdapterWithPrivateBus, new_group_invitation_creates_a_channel)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cm_cps;
    core::testing::CrossProcessSync account_cps;

    std::string group_id{"12345"};
    std::string initiator_id{"6798"};
    std::string title{"My Awesome Group"};
    std::string message_text{"Hi there!"};

    auto client = core::posix::fork(
        [=]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            testing::qt::tp::Client client{rt->dbus_connection(),
                                           Tp::ChannelClassSpecList() << Tp::ChannelClassSpec::textChatroom()};
            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);


    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [=, &cm_cps, &account_cps]()
        {
            auto trap = core::posix::trap_signals_for_all_subsequent_threads({core::posix::Signal::sig_term});
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            trap->signal_raised().connect([trap](core::posix::Signal) { trap->stop(); });
            messaging::raii::AutoJoiningThread ajt{[trap]() { trap->run(); }};

            rt->enter_with_task([=, &cm_cps, &account_cps]() {
                // trigger cross process signal
                EXPECT_NO_THROW(cm_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));

                // wait for cross process signal
                std::uint32_t counter = 0;
                EXPECT_NO_THROW(counter = account_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
                EXPECT_EQ(counter, 1);

                testing::qt::tp::create_online_account_and_then(connection_manager_name, protocol_name, rt,
                                                         [=](const Tp::AccountPtr& account) {
                    testing::qt::tp::wait_for_text_channel_and_then(account, rt, [=](const Tp::TextChannelPtr& channel){
                        ASSERT_FALSE(channel->isRequested());
                        ASSERT_TRUE(channel->targetContact().isNull());
                        ASSERT_EQ(channel->targetId().toStdString(), group_id);
                        // the channel doesn't contain any initial message
                        ASSERT_EQ(channel->messageQueue().count(), 0);

                        // check that the channel has the room related interfaces
                        ASSERT_TRUE(channel->hasInterface(TP_QT_IFACE_CHANNEL_INTERFACE_ROOM));
                        ASSERT_TRUE(channel->hasInterface(TP_QT_IFACE_CHANNEL_INTERFACE_ROOM_CONFIG));
                        ASSERT_TRUE(channel->hasInterface(TP_QT_IFACE_CHANNEL_INTERFACE_GROUP));
                        ASSERT_TRUE(channel->hasInterface(TP_QT_IFACE_CHANNEL_INTERFACE_DESTROYABLE));

                        // check that the group has the invitee contact
                        ASSERT_EQ(channel->groupContacts(false).count(), 1);
                        ASSERT_EQ(channel->groupContacts(false).toList()[0]->id().toStdString(), initiator_id);

                        auto room_interface = channel->optionalInterface<Tp::Client::ChannelInterfaceRoomInterface>();
                        auto pending_variant_map = room_interface->requestAllProperties();
                        testing::qt::tp::when_finished(pending_variant_map, [=](){
                            ASSERT_TRUE(pending_variant_map->isValid());
                            QVariantMap properties = pending_variant_map->result();
                            ASSERT_EQ(properties["RoomName"].toString().toStdString(), group_id);

                            auto room_config_interface = channel->optionalInterface<Tp::Client::ChannelInterfaceRoomConfigInterface>();
                            auto pending_variant_map2 = room_config_interface->requestAllProperties();
                            testing::qt::tp::when_finished(pending_variant_map2, [=](){
                                ASSERT_TRUE(pending_variant_map2->isValid());
                                QVariantMap properties = pending_variant_map2->result();
                                ASSERT_EQ(properties["Title"].toString().toStdString(), title);
                                testing::qt::tp::disconnect_account_and_then(account, rt, [=]{
                                    rt->stop(EXIT_SUCCESS);
                                });
                            });
                        });
                    });
                });
            });

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cm_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    // Skeleton instantiates a qt::tp::Adapter instance, exposing it to the bus.
    auto skeleton = core::posix::fork(
        [=, &account_cps]()
        {
            auto trap = core::posix::trap_signals_for_all_subsequent_threads({core::posix::Signal::sig_term});
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));

            trap->signal_raised().connect([trap](core::posix::Signal) { trap->stop(); });
            messaging::raii::AutoJoiningThread ajt{[trap]() { trap->run(); }};

            auto connector = std::make_shared<NiceMock<MockConnector>>();

            // The lifetime of the connector and connection instance is not govered by this scope.
            // To prevent test erros, we let the mocking framework know about this niggle.
            Mock::AllowLeak(connector.get());

            ON_CALL(*connector, request_connection(_, _, _, _)).WillByDefault(
                Invoke([=](const std::shared_ptr<messaging::Connection::Observer>& connection_observer,
                       const std::shared_ptr<messaging::Messenger::Observer>& messenger_observer,
                       const std::shared_ptr<messaging::PresenceManager::Observer>& presence_observer,
                       const messaging::Dictionary<std::string, messaging::Variant>& )
                {
                    auto connection = std::make_shared<NiceMock<MockConnection>>(connection_observer, messenger_observer, presence_observer);
                    auto messenger = std::dynamic_pointer_cast<NiceMock<MockMessenger>>(connection->messenger());

                    Mock::AllowLeak(connection.get());
                    Mock::AllowLeak(messenger.get());

                    ON_CALL(*connection, normalize_identifier(_)).WillByDefault(ReturnArg<0>());
                    ON_CALL(*connection, connect()).WillByDefault(Invoke([=](){
                        connection->announce_status_connected();

                        auto initiator = std::make_shared<messaging::User>(initiator_id);
                        messaging::Users invitees;
                        invitees.push_back(initiator);
                        auto new_group = std::make_shared<messaging::Group>(group_id, invitees, title);

                        messenger->announce_new_group_invitation_received(new_group, initiator);
                    }));
                    ON_CALL(*connection, disconnect()).WillByDefault(Invoke([rt]{
                        rt->stop(EXIT_SUCCESS);
                    }));

                    ON_CALL(*messenger, create_chat_with(_, _)).WillByDefault(Invoke([rt](const messaging::Recipient::shared_ptr& recipient, const std::shared_ptr<messaging::Chat::Observer>& observer) {
                        Q_UNUSED(recipient)
                        auto chat = std::make_shared<testing::NiceMock<testing::MockChat>>(observer);
                        auto group_manager = std::make_shared<testing::NiceMock<testing::MockGroupManager>>();

                        chat->plug_interface(group_manager);

                        Mock::AllowLeak(chat.get());

                        return chat;
                    }));
                    return connection;
                }));

            // The lifetime of the connector and connection instance is not govered by this scope.
            // To prevent test erros, we let the mocking framework know about this niggle.
            Mock::AllowLeak(connector.get());

            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            EXPECT_NO_THROW(account_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    stub.send_signal_or_throw(core::posix::Signal::sig_term);
    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
    skeleton.send_signal_or_throw(core::posix::Signal::sig_term);
    EXPECT_TRUE(testing::did_finish_successfully(skeleton.wait_for(core::posix::wait::Flags::untraced)));
}

TEST_F(AdapterWithPrivateBus, exposes_available_presences_correctly)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cm_cps;
    core::testing::CrossProcessSync account_cps;

    messaging::PresenceManager::StatusMap status_map{
        {"available", messaging::StatusSpec{messaging::PresenceType::available,
                                            messaging::StatusSpec::Flags::can_have_message &
                                            messaging::StatusSpec::Flags::may_set_on_self}},
        {"offline", messaging::StatusSpec{messaging::PresenceType::offline,
                                          messaging::StatusSpec::Flags::can_have_message &
                                          messaging::StatusSpec::Flags::may_set_on_self}},
        {"custom1", messaging::StatusSpec{messaging::PresenceType::away,
                                          messaging::StatusSpec::Flags::can_have_message}},
        {"custom2", messaging::StatusSpec{messaging::PresenceType::hidden,
                                          messaging::StatusSpec::Flags::none}}
    };

    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [this, &cm_cps, &account_cps, status_map]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            rt->enter_with_task(
                [rt, &cm_cps, &account_cps, status_map]()
                {
                    // trigger cross process signal
                    EXPECT_NO_THROW(cm_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                    ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));

                    // wait for cross process signal
                    std::uint32_t counter = 0;
                    EXPECT_NO_THROW(counter = account_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
                    EXPECT_EQ(counter, 1);

                    testing::qt::tp::create_account_and_then(connection_manager_name, protocol_name, rt,
                                                             [rt, status_map](const Tp::AccountPtr& account) {

                    QObject::connect(account.data(), &Tp::Account::connectionChanged, [rt, account, status_map] {
                        // set the account online
                        EXPECT_FALSE(account->connection().isNull());
                        EXPECT_TRUE(account->connection()->isReady(Tp::Connection::FeatureSimplePresence));
                        auto available_statuses = account->allowedPresenceStatuses(true).toMap();
                        ASSERT_EQ(available_statuses.count(), status_map.size());
                        for (auto status : available_statuses.keys())
                        {
                            Tp::SimpleStatusSpec tp_spec = available_statuses[status].bareSpec();
                            messaging::PresenceManager::StatusMap::const_iterator it = status_map.find(status.toStdString());
                            ASSERT_TRUE(it != status_map.end());
                            ASSERT_EQ(tp_spec.maySetOnSelf,
                                (it->second.flags & messaging::StatusSpec::Flags::may_set_on_self) == messaging::StatusSpec::Flags::may_set_on_self);
                            ASSERT_EQ(tp_spec.canHaveMessage,
                                (it->second.flags & messaging::StatusSpec::Flags::can_have_message) == messaging::StatusSpec::Flags::can_have_message);
                            ASSERT_EQ(tp_spec.type, messaging::qt::tp::Connection::messaging_presence_type_to_telepathy(it->second.type));
                        }
                        rt->stop(EXIT_SUCCESS);
                    });
                    account->setRequestedPresence(Tp::Presence::available());
                });
            });
            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cm_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    auto skeleton = core::posix::fork(
        [this, &account_cps, status_map]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));

            auto connector = std::make_shared<testing::NiceMock<MockConnector>>();
            ON_CALL(*connector, request_connection(_, _, _, _)).WillByDefault(
                Invoke([rt, status_map](const std::shared_ptr<messaging::Connection::Observer>& connection_observer,
                       const std::shared_ptr<messaging::Messenger::Observer>& messenger_observer,
                       const std::shared_ptr<messaging::PresenceManager::Observer>& presence_observer,
                       const messaging::Dictionary<std::string, messaging::Variant>& )
                {
                    auto connection = std::make_shared<NiceMock<MockConnection>>(connection_observer, messenger_observer, presence_observer);
                    auto messenger = std::dynamic_pointer_cast<NiceMock<MockMessenger>>(connection->messenger());
                    auto presence_manager = std::dynamic_pointer_cast<NiceMock<MockPresenceManager>>(connection->presence_manager());
                    Mock::AllowLeak(connection.get());
                    Mock::AllowLeak(messenger.get());
                    Mock::AllowLeak(presence_manager.get());
                    ON_CALL(*presence_manager, get_valid_statuses()).WillByDefault(
                        Return(status_map));

                    return connection;
                }));
            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            EXPECT_NO_THROW(account_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
}

TEST_F(AdapterWithPrivateBus, contact_presence_changes_correctly)
{
    using namespace ::testing;

    core::testing::CrossProcessSync cm_cps;
    core::testing::CrossProcessSync account_cps;

    messaging::PresenceManager::StatusMap status_map{
        {"available", messaging::StatusSpec{messaging::PresenceType::available,
                                            messaging::StatusSpec::Flags::can_have_message &
                                            messaging::StatusSpec::Flags::may_set_on_self}},
        {"offline", messaging::StatusSpec{messaging::PresenceType::offline,
                                          messaging::StatusSpec::Flags::can_have_message &
                                          messaging::StatusSpec::Flags::may_set_on_self}},
        {"away", messaging::StatusSpec{messaging::PresenceType::away,
                                       messaging::StatusSpec::Flags::can_have_message &
                                       messaging::StatusSpec::Flags::may_set_on_self}},
        {"hidden", messaging::StatusSpec{messaging::PresenceType::hidden,
                                         messaging::StatusSpec::Flags::can_have_message &
                                         messaging::StatusSpec::Flags::may_set_on_self}}
    };

    std::string contact_id{"12345"};

    auto client = core::posix::fork(
        [this]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            testing::qt::tp::Client client{rt->dbus_connection(),
                                           Tp::ChannelClassSpecList() << Tp::ChannelClassSpec::textChat()};
            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);


    // Stub connects to the bus instance local to this test case and watches for service owner changes.
    // The watching times out after 5000ms.
    auto stub = core::posix::fork(
        [this, &cm_cps, &account_cps, status_map, contact_id]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            messaging::qt::tp::Initializer initializer{tp_qt_configuration};

            rt->enter_with_task(
                [rt, &cm_cps, &account_cps, status_map, contact_id]()
                {
                    // trigger cross process signal
                    EXPECT_NO_THROW(cm_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

                    ASSERT_TRUE(testing::qt::tp::wait_for_connection_manager_with_name_and_protocol_on_bus(connection_manager_name, protocol_name, rt->dbus_connection()));

                    // wait for cross process signal
                    std::uint32_t counter = 0;
                    EXPECT_NO_THROW(counter = account_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
                    EXPECT_EQ(counter, 1);

                    testing::qt::tp::create_account_and_then(connection_manager_name, protocol_name, rt,
                                                             [rt, status_map, contact_id](const Tp::AccountPtr& account) {
                    QObject::connect(account.data(), &Tp::Account::connectionChanged, [rt, account, status_map, contact_id]() {
                        // set the account online
                        ASSERT_FALSE(account->connection().isNull());
                        ASSERT_TRUE(account->connection()->isReady(Tp::Connection::FeatureSimplePresence));

                        testing::qt::tp::when_finished(account->connection()->contactManager()->contactsForIdentifiers(QStringList() << QString::fromStdString(contact_id)),
                            [rt, account, contact_id](Tp::PendingOperation *op) {
                                ASSERT_FALSE(op->isError());
                                auto *pc = qobject_cast<Tp::PendingContacts*>(op);
                                ASSERT_TRUE(pc);
                                ASSERT_FALSE(pc->contacts().isEmpty());
                                auto contact = pc->contacts().first();
                                ASSERT_FALSE(contact.isNull());
                                ASSERT_TRUE(contact->requestedFeatures().contains(Tp::Contact::FeatureSimplePresence));
                                QObject::connect(contact.data(), &Tp::Contact::presenceChanged, [rt, account, contact](const Tp::Presence &presence) {
                                    ASSERT_EQ(presence.type(), messaging::qt::tp::Connection::messaging_presence_type_to_telepathy(messaging::PresenceType::away));
                                    ASSERT_EQ(presence.status().toStdString(), "away");
                                    testing::qt::tp::disconnect_account_and_then(account, rt, [rt](){
                                        rt->stop(EXIT_SUCCESS);
                                    });
                                });

                                // request a text channel only to inform the CM that we are already prepared and listening to presence changes
                                account->ensureTextChat(contact);
                            });
                    });
                    account->setRequestedPresence(Tp::Presence::available());
                });
            });
            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    // wait for cross process signal
    std::uint32_t counter = 0;
    EXPECT_NO_THROW(counter = cm_cps.wait_for_signal_ready_for(std::chrono::milliseconds{30 * 1000}));
    EXPECT_EQ(counter, 1);

    // workaround to pass this test on jenkins. This shouldn't be needed in other cases
    std::this_thread::sleep_for(std::chrono::milliseconds{1000});

    auto skeleton = core::posix::fork(
        [this, &account_cps, status_map]()
        {
            auto rt = messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(session_bus_address()));
            auto connector = std::make_shared<testing::NiceMock<MockConnector>>();

            Mock::AllowLeak(connector.get());

            ON_CALL(*connector, request_connection(_, _, _, _)).WillByDefault(
                Invoke([rt, status_map](const std::shared_ptr<messaging::Connection::Observer>& connection_observer,
                       const std::shared_ptr<messaging::Messenger::Observer>& messenger_observer,
                       const std::shared_ptr<messaging::PresenceManager::Observer>& presence_observer,
                       const messaging::Dictionary<std::string, messaging::Variant>& )
                {
                    auto connection = std::make_shared<NiceMock<MockConnection>>(connection_observer, messenger_observer, presence_observer);
                    auto messenger = std::dynamic_pointer_cast<NiceMock<MockMessenger>>(connection->messenger());
                    auto presence_manager = std::dynamic_pointer_cast<NiceMock<MockPresenceManager>>(connection->presence_manager());
                    Mock::AllowLeak(connection.get());
                    Mock::AllowLeak(messenger.get());
                    Mock::AllowLeak(presence_manager.get());
                    ON_CALL(*presence_manager, get_valid_statuses()).WillByDefault(Return(status_map));
                    ON_CALL(*messenger, create_chat_with(_, _)).WillByDefault(Invoke([rt, presence_manager](const messaging::Recipient::shared_ptr& recipient, const std::shared_ptr<messaging::Chat::Observer>& observer) {
                        auto chat = std::make_shared<testing::NiceMock<testing::MockChat>>(observer);
                        Mock::AllowLeak(chat.get());

                        // a requested text channel is the signal of the client side that it is listening to presence changes for the given contact
                        presence_manager->announce_presence_changed(recipient, messaging::Presence{messaging::PresenceType::away, "away", ""});
                        return chat;
                    }));

                    ON_CALL(*connection, disconnect()).WillByDefault(Invoke([rt]{
                        rt->stop(EXIT_SUCCESS);
                    }));

                    return connection;
                }));

            messaging::qt::tp::Adapter adapter{rt, connector, connection_manager_name, protocol_name};

            // trigger cross process signal
            EXPECT_NO_THROW(account_cps.try_signal_ready_for(std::chrono::milliseconds{500}));

            return failure_to_exit_status(rt->run());
        },
        core::posix::StandardStream::empty);

    EXPECT_TRUE(testing::did_finish_successfully(stub.wait_for(core::posix::wait::Flags::untraced)));
}
