﻿// Copyright (c) AlphaSierraPapa for the SharpDevelop Team
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy of this
// software and associated documentation files (the "Software"), to deal in the Software
// without restriction, including without limitation the rights to use, copy, modify, merge,
// publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons
// to whom the Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all copies or
// substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

using System;
using System.Collections.Generic;
using System.Diagnostics.Contracts;

namespace ICSharpCode.NRefactory.TypeSystem
{
	#if WITH_CONTRACTS
	[ContractClass(typeof(IEntityContract))]
	#endif
	public interface IEntity : INamedElement, IFreezable
	{
		EntityType EntityType { get; }
		
		/// <summary>
		/// Gets the complete entity region (including header+body)
		/// </summary>
		DomRegion Region { get; }
		
		/// <summary>
		/// Gets the entity body region.
		/// </summary>
		DomRegion BodyRegion { get; }
		
		/// <summary>
		/// Gets the declaring class.
		/// For members, this is the class that contains the member.
		/// For nested classes, this is the outer class. For top-level entities, this property returns null.
		/// </summary>
		ITypeDefinition DeclaringTypeDefinition { get; }
		
		IList<IAttribute> Attributes { get; }
		
		string Documentation { get; }
		
		/// <summary>
		/// Gets the accessibility of this entity.
		/// </summary>
		Accessibility Accessibility { get; }
		
		/// <summary>
		/// Gets whether this entity is static.
		/// Returns true if either the 'static' or the 'const' modifier is set.
		/// </summary>
		bool IsStatic { get; }
		
		/// <summary>
		/// Returns whether this entity is abstract.
		/// </summary>
		/// <remarks>Static classes also count as abstract classes.</remarks>
		bool IsAbstract { get; }
		
		/// <summary>
		/// Returns whether this entity is sealed.
		/// </summary>
		/// <remarks>Static classes also count as sealed classes.</remarks>
		bool IsSealed { get; }
		
		/// <summary>
		/// Gets whether this member is declared to be shadowing another member with the same name.
		/// </summary>
		bool IsShadowing { get; }
		
		/// <summary>
		/// Gets whether this member is generated by a macro/compiler feature.
		/// </summary>
		bool IsSynthetic { get; }
		
		/// <summary>
		/// Gets a value indicating whether this instance is private.
		/// </summary>
		/// <value>
		/// <c>true</c> if this instance is private; otherwise, <c>false</c>.
		/// </value>
		bool IsPrivate { get; }
		
		/// <summary>
		/// Gets a value indicating whether this instance is public.
		/// </summary>
		/// <value>
		/// <c>true</c> if this instance is public; otherwise, <c>false</c>.
		/// </value>
		bool IsPublic { get; }
		
		/// <summary>
		/// Gets a value indicating whether this instance is protected.
		/// </summary>
		/// <value>
		/// <c>true</c> if this instance is protected; otherwise, <c>false</c>.
		/// </value>
		bool IsProtected { get; }
		
		/// <summary>
		/// Gets a value indicating whether this instance is internal.
		/// </summary>
		/// <value>
		/// <c>true</c> if this instance is internal; otherwise, <c>false</c>.
		/// </value>
		bool IsInternal { get; }
		
		/// <summary>
		/// Gets a value indicating whether this instance is protected or internal.
		/// </summary>
		/// <value>
		/// <c>true</c> if this instance is protected or internal; otherwise, <c>false</c>.
		/// </value>
		bool IsProtectedOrInternal { get; }
		
		/// <summary>
		/// Gets a value indicating whether this instance is protected and internal.
		/// </summary>
		/// <value>
		/// <c>true</c> if this instance is protected and internal; otherwise, <c>false</c>.
		/// </value>
		bool IsProtectedAndInternal { get; }
		
		/// <summary>
		/// The assembly in which this entity is defined.
		/// This property never returns null.
		/// </summary>
		IProjectContent ProjectContent { get; }
		
		//bool IsAccessible(IClass callingClass, bool isAccessThoughReferenceOfCurrentClass);
	}
	
	#if WITH_CONTRACTS
	[ContractClassFor(typeof(IEntity))]
	abstract class IEntityContract : INamedElementContract, IEntity
	{
		EntityType IEntity.EntityType {
			get { return default(EntityType); }
		}
		
		DomRegion IEntity.Region {
			get { return DomRegion.Empty; }
		}
		
		DomRegion IEntity.BodyRegion {
			get { return DomRegion.Empty; }
		}
		
		ITypeDefinition IEntity.DeclaringTypeDefinition {
			get { return null; }
		}
		
		IList<IAttribute> IEntity.Attributes {
			get {
				Contract.Ensures(Contract.Result<IList<IAttribute>>() != null);
				return null;
			}
		}
		
		string IEntity.Documentation {
			get { return null; }
		}
		
		bool IEntity.IsStatic {
			get { return false; }
		}
		
		Accessibility IEntity.Accessibility {
			get { return default(Accessibility); }
		}
		
		bool IEntity.IsAbstract {
			get { return false; }
		}
		
		bool IEntity.IsSealed {
			get { return false; }
		}
		
		bool IEntity.IsShadowing {
			get { return false; }
		}
		
		bool IEntity.IsSynthetic {
			get { return false; }
		}
		
		IProjectContent IEntity.ProjectContent {
			get {
				Contract.Ensures(Contract.Result<IProjectContent>() != null);
				return null;
			}
		}
		
		bool IFreezable.IsFrozen {
			get { return false; }
		}
		
		void IFreezable.Freeze()
		{
		}
	}
	#endif
}
