/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	Copyright 1998-2002 Anton Vinokurov <anton@netams.com>
***	Copyright 2002-2008 NeTAMS Development Team
***	This code is GPL v3
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: flow.c,v 1.130 2008-02-23 08:35:02 anton Exp $ */

#include "netams.h"
#include "flow.h"

Flow::Flow() {
    	init();
}

Flow::Flow(u_char instance, const u_char slots, const struct attribute *new_list) {
	init();
 	ds_id = instance;	
   
	if((list = (struct attribute *)aMalloc(sizeof(struct attribute)*slots)) == NULL)
		return;
	if(new_list != NULL) {
		bcopy(new_list, list, sizeof(struct attribute)*slots);
		freeSlot = totalSlots = slots;
	} else
		totalSlots = slots;
}

void Flow::init() {
	list	= NULL;
	freeSlot	= 0;
	totalSlots	= 0;
}

Flow::~Flow() {
	removeAll();
}

const int Flow::grow_list(u_char new_size) {
	struct attribute	*new_list;

	if(new_size==0)
		new_size = totalSlots+1;

	new_list = (struct attribute *)aMalloc(sizeof(struct attribute)*new_size);
	
	if(new_list == NULL)
		return 0;

	if(list != NULL) {
		bcopy(list, new_list, sizeof(struct attribute)*totalSlots);
		aFree(list);
	}
	aDebug(DEBUG_FLOW, "Flow %p growth from %u to %u slots\n", this, totalSlots, new_size);
	totalSlots = new_size;
	list = new_list;

	return 1;
}

void* Flow::put(const u_char id, const union attribute_value *value) {
	struct attribute	*tmp;


	if((list == NULL) && !grow_list())
		return NULL;

	if((freeSlot == totalSlots) && !grow_list())
		return NULL;

	tmp = &list[freeSlot++];
	tmp->id 		= id;
	if(value) 
		tmp->value = *value;
	
	return (void*)&tmp->value;
}

const union attribute_value *Flow::get(const u_char id) {
	int		i;

	if((i = index(id)) != -1)
		return &list[i].value;
	return NULL;
}

const int Flow::contains(const u_char id) {
	return (index(id) == -1) ? 0 : 1;
}

const unsigned int Flow::elements() {
	return freeSlot;
}

const int Flow::index(const u_char id) {
	unsigned int	i;
	int		idx;

	if(list == NULL)
		return -1;

	for(i = 0, idx = -1; i < freeSlot; i++)
		if(list[i].id == id) {
			idx = i;
			break;
		}
	return idx;
}

void Flow::removeAll() {
	aFree(list);
	list		= NULL;
	freeSlot	= 0;
	totalSlots	= 0;
}

void Flow::reuse() {
    bzero(list, sizeof(struct attribute)*freeSlot);
    freeSlot	= 0;
}

void Flow::copy(Flow *src) {
	freeSlot = src->freeSlot;

	if(freeSlot > totalSlots) 
		grow_list(freeSlot);

	bcopy(src->list, list, sizeof(struct attribute)*freeSlot);
}

#ifdef DEBUG
void Flow::Debug(u_char debug) {
	union attribute_value   *value;
	char buf1[32], buf2[32];

	aDebug(debug, "ds:%u flow:%p slots:%u\n",ds_id, this, freeSlot);
	for(u_char i=0; i< freeSlot; i++) {
		value = &list[i].value;
		switch(list[i].id) {
		case ATTR_FLOW_INFO: {
			struct flow_info_value *flow_info=(struct flow_info_value *)value;
			aDebug(debug, "FLOW_INFO: packets:%lu octets:%llu flow_first:%lu flow_last:%lu direction:%u\n",
				flow_info->packets, flow_info->octets, flow_info->flow_first, flow_info->flow_last, flow_info->direction);
			}
			break;
		case ATTR_IPV4_INFO: {
			struct ipv4_info_value *ipv4_info=(struct ipv4_info_value*)value;
			inet_ntop(AF_INET, &(ipv4_info->ip_src), buf1, 32);
			inet_ntop(AF_INET, &(ipv4_info->ip_dst), buf2, 32);
			aDebug(debug,"IPV4_INFO: ip_src:%s ip_dst:%s ip_p:%u ip_tos:%u\n",
				buf1, buf2, ipv4_info->ip_p, ipv4_info->ip_tos);
			}
			break;
		case ATTR_TCP_INFO: {
			struct tcp_info_value *tcp_info=(struct tcp_info_value*)value;
			aDebug(debug, "TCP_INFO: src_port:%u dst_port:%u\n",
				ntohs(tcp_info->src_port), ntohs(tcp_info->dst_port));
			}
			break;
		case ATTR_MAC_INFO: {
			struct mac_info_value *mac_info=(struct mac_info_value*)value;
			aDebug(debug, "MAC_INFO: src:%s dst:%s\n", 
				ether_ntoa((struct ether_addr*)mac_info->src), ether_ntoa((struct ether_addr*)mac_info->dst));
			}
			break;
		case ATTR_VLAN_INFO: {
			struct vlan_info_value *vlan_info=(struct vlan_info_value*)value;
			aDebug(debug, "VLAN_INFO: dot1x:%u\n",
				ntohs(vlan_info->dot1x));
			}
			break;
		case ATTR_AS_INFO: {
			struct as_info_value *as_info=(struct as_info_value*)value;
			inet_ntop(AF_INET, &(as_info->as_src), buf1, 32);
			inet_ntop(AF_INET, &(as_info->as_dst), buf2, 32);
			aDebug(debug, "AS_INFO: as_src:%s as_dst:%s\n",
				buf1, buf2);
			}
			break;
		case ATTR_IFINDEX_INFO: {
			struct ifindex_info_value *ifindex_info=(struct ifindex_info_value*)value;
			aDebug(debug, "IFINDEX_INFO: if_in:%u if_out:%u\n",
				ntohs(ifindex_info->if_in), ntohs(ifindex_info->if_out));
			}
			break;
		case ATTR_MULTICAST_INFO: {
			struct multicast_info_value *multicast_info=(struct multicast_info_value*)value;
			aDebug(debug, "MULTICAST_INFO: dst_packets:%llu dst_bytes:%llu igmp_type:%u\n",
				multicast_info->dst_packets, multicast_info->dst_bytes, multicast_info->igmp_type);
			}
			break;
		case ATTR_NEXTHOP_INFO: {
			struct nexthop_info_value *nexthop_info=(struct nexthop_info_value*)value;
			inet_ntop(AF_INET, &(nexthop_info->ipv4_nexthop), buf1, 32);
			inet_ntop(AF_INET, &(nexthop_info->bgpv4_nexthop), buf2, 32);
			aDebug(debug, "NEXTHOP_INFO: ipv4_nexthop:%s bgpv4_nexthop:%s\n",
				buf1, buf2);
			}
			break;
		case ATTR_LAYER7_INFO: {
			struct layer7_info_value *layer7_info=(struct layer7_info_value*)value;
			aDebug(debug, "LAYER7_INFO: type:%u value:%s\n",
				layer7_info->type, layer7_info->value);
			}
			break;
		default:
			aDebug(debug, "UNKNOWN: id:%u\n", list[i].id);
		}
	}
}
#endif
