/*
 * nield.c - evnent receiver
 * Copyright (C) 2014 Tetsumune KISO <t2mune@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef _NIELD_
#define _NIELD_

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

/* ANSI C header files */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include <time.h>

/* not ANSI C header files */
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/syslog.h>
#include <arpa/inet.h>
#include <netinet/in.h> /* INET_ADDRSTRLEN, INET6_ADDRSTRLEN */
#include <sys/types.h>
#include <sys/socket.h> /* bits/socket.h, PF_*, AF_* */
#include <netdb.h>
#include <asm/types.h>
#include <net/ethernet.h> /* ETHER_ADDR_LEN */
#include <linux/types.h> /* __u8, __u16, __u32, __u64 */
#include <linux/rtnetlink.h> /* linux/if_link.h */
#include <linux/if_arp.h>
#ifdef HAVE_LINUX_FIB_RULES_H
# include <linux/fib_rules.h>
#endif
#include <linux/if.h> /* IFNAMSIZ */
#include <linux/ip.h> /* struct iphdr */
#include <linux/if_vlan.h> /* VLAN_FLAG_* */
#include <linux/if_tunnel.h> /* IFLA_GRE_* */
#include <linux/if_ether.h> /* ETH_P_* */
#include <linux/gen_stats.h>
#include <linux/pkt_cls.h> /* linux/pkt_sched.h */
#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
# include <linux/tc_act/tc_csum.h>
#endif
#include <linux/tc_act/tc_gact.h>
#include <linux/tc_act/tc_ipt.h>
#include <linux/tc_act/tc_mirred.h>
#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
# include <linux/tc_act/tc_nat.h>
#endif
# include <linux/tc_act/tc_pedit.h>
#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
# include <linux/tc_act/tc_skbedit.h>
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
#include <linux/tc_ematch/tc_em_cmp.h>
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
#include <linux/tc_ematch/tc_em_meta.h>
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_NBYTE_H
#include <linux/tc_ematch/tc_em_nbyte.h>
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_TEXT_H
#include <linux/tc_ematch/tc_em_text.h>
#endif

#include "list.h"

/* default value */
#define NIELD_USAGE        "[-vh46inar] [-p lock_file] [-l log_file] [-s buffer_size] [-L syslog_facility] [-d debug_file]"
#define LOG_FILE_DEFAULT   "/var/log/nield.log"
#define DEBUG_FILE_DEFAULT "/var/log/nield.dbg"
#define LOCK_FILE          "/var/run/nield.pid"
#define IFLIST_FILE        "/tmp/nield.iflist"
#define IFHIST_FILE        "/tmp/nield.ifhist"
#define NDLIST_FILE        "/tmp/nield.ndlist"
#define MAX_STR_SIZE       128
#define MAX_MSG_SIZE       2048
#define MODULE_NAME_LEN    (64 - sizeof(unsigned long))

/* logging option flag */
#define L_LOCAL   0x0001
#define L_SYSLOG  0x0002
#define L_DEBUG   0x0004

/* message option flag */
#define M_IPV4    0x0001
#define M_IPV6    0x0002
#define M_LINK    0x0004
#define M_NEIGH   0x0008
#define M_IFADDR  0x0010
#define M_ROUTE   0x0020
#define M_RULE    0x0040
#define M_TC      0x0080

/* interface message type flag */
#define IF_ADD    0x0001
#define IF_DEL    0x0002
#define IF_CHANGE 0x0004

/* interface list entry format */
struct iflist_entry {
	int index;
	char name[IFNAMSIZ];
	unsigned char addr[INET6_ADDRSTRLEN+1];
	unsigned char brd[INET6_ADDRSTRLEN+1];
	unsigned short type;
	unsigned flags;
	unsigned short vid;
	int mtu;
	char kind[MODULE_NAME_LEN];
	int index_master;
	char name_master[IFNAMSIZ];
	unsigned short br_attached;
	struct list_head list;
};

/* neighbor discovery cache format */
struct ndlist_entry {
	unsigned ifindex;
	char ifname[IFNAMSIZ];
	char ipaddr[INET6_ADDRSTRLEN+1];
	unsigned char lladdr[ETHER_ADDR_LEN];
	struct list_head list;
};

/* defined in net/if.h but that conflicts with linux/if.h... */
extern unsigned int if_nametoindex (const char *__ifname);
extern char *if_indextoname (unsigned int __ifindex, char *__ifname);

/* nield.c */
void close_exit(int sock, int log_flag, int ret);
int set_options(int argc, char *argv[]);
int get_log_opts(void);
int get_msg_opts(void);
int set_facility(char *facility_name);
int get_facility(void);
int open_lock(void);
int write_lock(void);
void close_lock(void);
int init_daemon(void);
int set_signal_handlers(void);
void sigterm_handler(int sig);
void sigint_handler(int sig);
void sigusr1_handler(int sig);
void sigusr2_handler(int sig);
int set_rtnetlink_groups(void);
int open_netlink_socket(unsigned groups);
int send_request(int sock, int type, int family);
int recv_reply(int sock, int type);
int recv_events(int sock);
int parse_events(struct msghdr *mhdr);

/* log.c */
int open_log(char *filename);
char *add_log(char *msg, char *mp, char *format, ...);
void rec_log(char *format, ...);
void close_log(void);

/* debug.c */
int open_dbg(char *filename);
void rec_dbg(int lev, char *format, ...);
void close_dbg(void);

/* nlmsg.c */
void debug_nlmsg(int lev, struct nlmsghdr *nlh);
const char *debug_n2s_nlmsg_type(int type);

/* ifimsg.c */
int create_iflist(struct msghdr *msg);
char *if_indextoname_from_lists(int index, char *name);
char *if_indextoname_from_iflist(int index, char *name);
char *if_indextoname_from_ifhist(int index, char *name);
unsigned short get_type_from_iflist(int index);
unsigned short get_type_from_ifhist(int index);
void print_iflist(void);
void print_ifhist(void);
int parse_ifimsg(struct nlmsghdr *nlh);
void parse_rtm_newlink(char *msg, struct iflist_entry *ifle, struct iflist_entry *ifle_tmp,
	struct rtattr *ifla[]);
void parse_rtm_dellink(char *msg, struct iflist_entry *ifle, struct iflist_entry *ifle_tmp);
int parse_ifla_ifname(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
int parse_ifla_link(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
int parse_ifla_address(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
int parse_ifla_broadcast(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
int parse_ifla_mtu(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
#if HAVE_DECL_IFLA_LINKINFO
int parse_ifla_linkinfo(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
int parse_ifla_info_kind(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle);
int parse_ifla_info_data(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle);
#endif
#if HAVE_DECL_IFLA_VLAN_UNSPEC
int parse_ifla_vlan(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle);
int parse_ifla_vlan_id(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
int parse_ifla_vlan_egress_qos(char *msg, char **mp, struct rtattr *vlan,
	struct iflist_entry *ifle);
int parse_ifla_vlan_ingress_qos(char *msg, char **mp, struct rtattr *vlan,
	struct iflist_entry *ifle);
int parse_vlan_qos_mapping(char *msg, char **mp, struct rtattr *qos, struct iflist_entry *ifle);
#endif
#if HAVE_DECL_IFLA_GRE_UNSPEC
int parse_ifla_gre(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle);
int parse_ifla_gre_local(char *msg, char **mp, struct rtattr *greinfo, struct iflist_entry *ifle);
int parse_ifla_gre_remote(char *msg, char **mp, struct rtattr *greinfo, struct iflist_entry *ifle);
#endif
#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
int parse_ifla_macvlan(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle);
int parse_ifla_macvlan_mode(char *msg, char **mp, struct rtattr *macvlan,
    struct iflist_entry *ifle);
#endif
#if HAVE_DECL_IFLA_VXLAN_UNSPEC
int parse_ifla_vxlan(char *msg, char **mp, struct rtattr *linkinfo, struct iflist_entry *ifle);
int parse_ifla_vxlan_id(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle);
int parse_ifla_vxlan_link(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle);
int parse_ifla_vxlan_local(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle);
#if HAVE_DECL_IFLA_VXLAN_GROUP
int parse_ifla_vxlan_group(char *msg, char **mp, struct rtattr *vxlan, struct iflist_entry *ifle);
#endif
#endif
int parse_ifla_master(char *msg, char **mp, struct rtattr *ifla, struct iflist_entry *ifle);
int inet_ntop_ifi(unsigned short type, struct rtattr *ifla, char *saddr, int slen);
void debug_ifimsg(int lev, struct ifinfomsg *ifim, struct rtattr *ifla[], int ifim_len);
void debug_ifla_address(int lev, struct ifinfomsg *ifim, struct rtattr *ifla);
void debug_ifla_broadcast(int lev, struct ifinfomsg *ifim, struct rtattr *ifla);
void debug_ifla_ifname(int lev, struct rtattr *ifla);
void debug_ifla_mtu(int lev, struct rtattr *ifla);
void debug_ifla_link(int lev, struct rtattr *ifla);
void debug_ifla_qdisc(int lev, struct rtattr *ifla);
void debug_ifla_stats(int lev, struct rtattr *ifla);
void debug_ifla_cost(int lev, struct rtattr *ifla);
void debug_ifla_priority(int lev, struct rtattr *ifla);
void debug_ifla_master(int lev, struct rtattr *ifla);
void debug_ifla_wireless(int lev, struct rtattr *ifla);
void debug_ifla_protoinfo(int lev, struct rtattr *ifla);
void debug_ifla_txqlen(int lev, struct rtattr *ifla);
void debug_ifla_map(int lev, struct rtattr *ifla);
void debug_ifla_weight(int lev, struct rtattr *ifla);
void debug_ifla_operstate(int lev, struct rtattr *ifla);
void debug_ifla_linkmode(int lev, struct rtattr *ifla);
#if HAVE_DECL_IFLA_LINKINFO
void debug_ifla_linkinfo(int lev, struct ifinfomsg *ifim, struct rtattr *ifla);
void debug_ifla_info_kind(int lev, struct rtattr *ifla, char *kind, int len);
void debug_ifla_info_data(int lev, struct ifinfomsg *ifim, struct rtattr *ifla,
	char *kind, int len);
void debug_ifla_info_xstats(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_VLAN_UNSPEC
void debug_ifla_vlan(int lev, struct rtattr *linkinfo);
void debug_ifla_vlan_id(int lev, struct rtattr *vlan);
void debug_ifla_vlan_flags(int lev, struct rtattr *vlan);
void debug_ifla_vlan_egress_qos(int lev, struct rtattr *vlan);
void debug_ifla_vlan_ingress_qos(int lev, struct rtattr *vlan);
void debug_vlan_qos_mapping(int lev, struct rtattr *qos);
#endif
#if HAVE_DECL_IFLA_GRE_UNSPEC
void debug_ifla_gre(int lev, struct ifinfomsg *ifim, struct rtattr *linkinfo);
void debug_ifla_gre_link(int lev, struct rtattr *gre);
void debug_ifla_gre_iflags(int lev, struct rtattr *gre);
void debug_ifla_gre_oflags(int lev, struct rtattr *gre);
void debug_ifla_gre_ikey(int lev, struct rtattr *gre);
void debug_ifla_gre_okey(int lev, struct rtattr *gre);
void debug_ifla_gre_local(int lev, struct ifinfomsg *ifim, struct rtattr *gre);
void debug_ifla_gre_remote(int lev, struct ifinfomsg *ifim, struct rtattr *gre);
void debug_ifla_gre_ttl(int lev, struct rtattr *gre);
void debug_ifla_gre_tos(int lev, struct rtattr *gre);
void debug_ifla_gre_pmtudisc(int lev, struct rtattr *gre);
#endif
#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
void debug_ifla_macvlan(int lev, struct rtattr *linkinfo);
void debug_ifla_macvlan_mode(int lev, struct rtattr *macvlan);
#if HAVE_DECL_IFLA_MACVLAN_FLAGS
void debug_ifla_macvlan_flags(int lev, struct rtattr *macvlan);
#endif
#endif
#if HAVE_DECL_IFLA_VXLAN_UNSPEC
void debug_ifla_vxlan(int lev, struct rtattr *linkinfo);
void debug_ifla_vxlan_id(int lev, struct rtattr *vxlan);
#if HAVE_DECL_IFLA_VXLAN_GROUP
void debug_ifla_vxlan_group(int lev, struct rtattr *vxlan);
#endif
void debug_ifla_vxlan_link(int lev, struct rtattr *vxlan);
void debug_ifla_vxlan_local(int lev, struct rtattr *vxlan);
void debug_ifla_vxlan_ttl(int lev, struct rtattr *vxlan);
void debug_ifla_vxlan_tos(int lev, struct rtattr *vxlan);
void debug_ifla_vxlan_learning(int lev, struct rtattr *vxlan);
void debug_ifla_vxlan_ageing(int lev, struct rtattr *vxlan);
void debug_ifla_vxlan_limit(int lev, struct rtattr *vxlan);
#if HAVE_DECL_IFLA_VXLAN_PORT_RANGE
void debug_ifla_vxlan_port_range(int lev, struct rtattr *vxlan);
#endif
#endif
#if HAVE_DECL_IFLA_NET_NS_PID
void debug_ifla_net_ns_pid(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_IFALIAS
void debug_ifla_ifalias(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_NUM_VF
void debug_ifla_num_vf(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_VFINFO_LIST
void debug_ifla_vfinfo_list(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_STATS64
void debug_ifla_stats64(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_VF_PORTS
void debug_ifla_vf_ports(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_PORT_SELF
void debug_ifla_port_self(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_AF_SPEC
void debug_ifla_af_spec(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_GROUP
void debug_ifla_group(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_NET_NS_FD
void debug_ifla_net_ns_fd(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_EXT_MASK
void debug_ifla_ext_mask(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_PROMISCUITY
void debug_ifla_promiscuity(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_NUM_TX_QUEUES
void debug_ifla_num_tx_queues(int lev, struct rtattr *ifla);
#endif
#if HAVE_DECL_IFLA_NUM_RX_QUEUES
void debug_ifla_num_rx_queues(int lev, struct rtattr *ifla);
#endif
const char *convert_af_type(int type);
const char *convert_arphrd_type(int type);
void convert_iff_flags(int flags, char *flags_list, int len);
const char *convert_if_oper_state(int state);
const char *convert_if_link_mode(int mode);
#if HAVE_DECL_IFLA_VLAN_FLAGS
void convert_vlan_flags(int flags, char *flags_list, int len);
#endif
#if HAVE_DECL_IFLA_GRE_IFLAGS
void convert_gre_flags(int flags, char *flags_list, int len);
#endif
#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
const char *convert_macvlan_mode(int mode, int debug);
#endif

/* ifamsg.c */
int parse_ifamsg(struct nlmsghdr *nlh);
int inet_ntop_ifa(int family, struct rtattr *ifa, char *saddr, int slen);
void debug_ifamsg(int lev, struct ifaddrmsg *ifam, struct rtattr *ifa[], int ifam_len);
void debug_ifa_address(int lev, struct ifaddrmsg *ifam, struct rtattr *ifa);
void debug_ifa_local(int lev, struct ifaddrmsg *ifam, struct rtattr *ifa);
void debug_ifa_label(int lev, struct rtattr *ifa);
void debug_ifa_broadcast(int lev, struct ifaddrmsg *ifam, struct rtattr *ifa);
void debug_ifa_anycast(int lev, struct ifaddrmsg *ifam, struct rtattr *ifa);
void debug_ifa_cacheinfo(int lev, struct rtattr *ifa);
void debug_ifa_multicast(int lev, struct ifaddrmsg *ifam, struct rtattr *ifa);
void convert_ifa_flags(int flags, char *flags_list, int len);
const char *convert_ifa_scope(int scope, int debug);

/* ndmsg.c */
int create_ndlist(struct msghdr *msg);
void print_ndlist(void);
int parse_ndmsg(struct nlmsghdr *nlh);
int parse_rtm_newneigh(char *ndm_type, struct ndmsg *ndm, struct ndlist_entry *ndle_tmp);
int parse_rtm_delneigh(char *ndm_type, struct ndmsg *ndm, struct ndlist_entry *ndle_tmp);
void debug_ndmsg(int lev, struct ndmsg *ndm, struct rtattr *nda[], int ndm_len);
void debug_nda_dst(int lev, struct ndmsg *ndm, struct rtattr *nda);
void debug_nda_lladdr(int lev, struct rtattr *nda);
void debug_nda_cacheinfo(int lev, struct rtattr *nda);
void debug_nda_probes(int lev, struct rtattr *nda);
const char *convert_nud_state(int state);
void convert_ntf_flags(int flags, char *flags_list, int len);

/* rtmsg.c */
int parse_rtmsg(struct nlmsghdr *nlh);
void debug_rtmsg(int lev, struct rtmsg *rtm, struct rtattr *rta[], int rtm_len);
void debug_rta_dst(int lev, struct rtmsg *rtm, struct rtattr *rta);
void debug_rta_src(int lev, struct rtmsg *rtm, struct rtattr *rta);
void debug_rta_iif(int lev, struct rtattr *rta);
void debug_rta_oif(int lev, struct rtattr *rta);
void debug_rta_gateway(int lev, struct rtmsg *rtm, struct rtattr *rta);
void debug_rta_priority(int lev, struct rtattr *rta);
void debug_rta_prefsrc(int lev, struct rtmsg *rtm, struct rtattr *rta);
void debug_rta_metrics(int lev, struct rtattr *rta);
void debug_rtax_lock(int lev, struct rtattr *rtax);
void debug_rtax_mtu(int lev, struct rtattr *rtax);
void debug_rtax_advmss(int lev, struct rtattr *rtax);
void debug_rtax_hoplimit(int lev, struct rtattr *rtax);
void debug_rtax_window(int lev, struct rtattr *rtax);
void debug_rta_multipath(int lev, struct rtmsg *rtm, struct rtattr *rta);
void debug_rta_flow(int lev, struct rtattr *rta);
void debug_rta_cacheinfo(int lev, struct rtattr *rta);
void debug_rta_table(int lev, struct rtattr *rta);
#if HAVE_DECL_RTA_MARK
void debug_rta_mark(int lev, struct rtattr *rta);
#endif
const char *convert_rt_table(int table, int debug);
const char *convert_rtprot(int protocol, int debug);
const char *convert_rt_scope(int scope);
const char *convert_rtn_type(int type, int debug);
void convert_rtm_flags(int flags, char *flags_list, int len);
void convert_rtnh_flags(int flags, char *flags_list, int len);

/* frhdr.c */
#ifdef HAVE_LINUX_FIB_RULES_H
int parse_frhdr(struct nlmsghdr *nlh);
void debug_frhdr(int lev, struct fib_rule_hdr *frh, struct rtattr *fra[], int frh_len);
void debug_fra_dst(int lev, struct fib_rule_hdr *frh, struct rtattr *fra);
void debug_fra_src(int lev, struct fib_rule_hdr *frh, struct rtattr *fra);
void debug_fra_ifname(int lev, struct rtattr *fra);
#if HAVE_DECL_FRA_GOTO
void debug_fra_goto(int lev, struct rtattr *fra);
#endif
void debug_fra_priority(int lev, struct rtattr *fra);
void debug_fra_fwmark(int lev, struct rtattr *fra);
void debug_fra_flow(int lev, struct rtattr *fra);
void debug_fra_table(int lev, struct rtattr *fra);
void debug_fra_fwmask(int lev, struct rtattr *fra);
#if HAVE_DECL_FRA_OIFNAME
void debug_fra_oifname(int lev, struct rtattr *fra);
#endif
const char *convert_fr_act(int action, int debug);
void convert_fib_rule_flags(int flags, char *flags_list, int len);
#endif

/* tcmsg_qdisc.c */
int parse_tcmsg_qdisc(struct nlmsghdr *nlh);
void parse_tc_classid(char *p, int len, unsigned id);
int parse_tca_options_prio(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_pfifo(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_bfifo(char *msg, char **mp, struct rtattr *tca);
#ifdef HAVE_STRUCT_TC_MULTIQ_QOPT_BANDS
int parse_tca_options_multiq(char *msg, char **mp, struct rtattr *tca);
#endif
#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
int parse_tca_options_plug(char *msg, char **mp, struct rtattr *tca);
#endif
int parse_tca_options_sfq(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_tbf(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_red(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_gred(char *msg, char **mp, struct rtattr *tca);
#if HAVE_DECL_TCA_CHOKE_UNSPEC
int parse_tca_options_choke(char *msg, char **mp, struct rtattr *tca);
#endif
int parse_tca_options_htb(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_hfsc(char *msg, char **mp, struct rtattr *tca);
int print_hfsc_sc(char *msg, char **mp, char *name, struct tc_service_curve *sc);
int parse_tca_options_cbq(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_dsmark(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_netem(char *msg, char **mp, struct rtattr *tca);
#if HAVE_DECL_TCA_DRR_UNSPEC
int parse_tca_options_drr(char *msg, char **mp, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_SFB_UNSPEC
int parse_tca_options_sfb(char *msg, char **mp, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_QFQ_UNSPEC
int parse_tca_options_qfq(char *msg, char **mp, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_CODEL_UNSPEC
int parse_tca_options_codel(char *msg, char **mp, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
int parse_tca_options_fq_codel(char *msg, char **mp, struct rtattr *tca);
#endif
void convert_unit_rate(char *str, int len, double num);
void convert_unit_size(char *str, int len, double num);
void convert_unit_usec(char *str, int len, double usec);
int get_us2tick(void);
double get_burst_size(unsigned rate, unsigned buffer);
double get_latency(unsigned rate, unsigned buffer, unsigned limit);
void debug_tcmsg(int lev, struct nlmsghdr *nlh, struct tcmsg *tcm, struct rtattr *tca[], int tcm_len);
void debug_tca_kind(int lev, struct rtattr *tca, char *kind, int len);
void debug_tca_options(int lev, struct tcmsg *tcm, struct rtattr *tca, char *kind, int len);
void debug_tca_stats(int lev, struct rtattr *tca);
void debug_tca_xstats(int lev, struct rtattr *tca, char *kind, int len);
void debug_tca_stats_basic(int lev, struct rtattr *stats2);
void debug_tca_stats_rate_est(int lev, struct rtattr *stats2);
void debug_tca_stats_queue(int lev, struct rtattr *stats2);
void debug_tca_stats_app(int lev, struct rtattr *stats2);
void debug_tca_rate(int lev, struct rtattr *tca);
void debug_tca_fcnt(int lev, struct rtattr *tca);
void debug_tca_stats2(int lev, struct rtattr *tca);
void debug_tca_stats2_attr(int lev, struct rtattr *tca);
#if HAVE_DECL_TCA_STAB_UNSPEC
void debug_tca_stab(int lev, struct rtattr *tca);
void debug_tca_stab_base(int lev, struct rtattr *stab);
void debug_tca_stab_data(int lev, struct rtattr *stab);
#endif
void debug_tca_options_prio(int lev, struct rtattr *tca);
void debug_tca_options_fifo(int lev, struct rtattr *tca);
#ifdef HAVE_STRUCT_TC_MULTIQ_QOPT_BANDS
void debug_tca_options_multiq(int lev, struct rtattr *tca);
#endif
#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
void debug_tca_options_plug(int lev, struct rtattr *tca);
#endif
void debug_tca_options_tbf(int lev, struct rtattr *tca);
void debug_tca_tbf_parms(int lev, struct rtattr *tbf);
void debug_tca_tbf_rtab(int lev, struct rtattr *tbf);
void debug_tca_tbf_ptab(int lev, struct rtattr *tbf);
void debug_tca_options_sfq(int lev, struct rtattr *tca);
#ifdef HAVE_STRUCT_TC_SFQ_XSTATS_ALLOT
void debug_tc_sfq_xstats(int lev, struct rtattr *tca);
#endif
void debug_tca_options_red(int lev, struct rtattr *tca);
void debug_tca_red_parms(int lev, struct rtattr *red);
void debug_tca_red_stab(int lev, struct rtattr *red);
#if HAVE_DECL_TCA_RED_MAX_P
void debug_tca_red_max_p(int lev, struct rtattr *red);
#endif
void debug_tc_red_xstats(int lev, struct rtattr *tca);
void debug_tca_options_gred(int lev, struct rtattr *tca);
void debug_tca_gred_parms(int lev, struct rtattr *gred);
void debug_tca_gred_stab(int lev, struct rtattr *gred);
void debug_tca_gred_dps(int lev, struct rtattr *gred);
#if HAVE_DECL_TCA_GRED_MAX_P
void debug_tca_gred_max_p(int lev, struct rtattr *gred);
#endif
#if HAVE_DECL_TCA_CHOKE_UNSPEC
void debug_tca_options_choke(int lev, struct rtattr *tca);
void debug_tca_choke_parms(int lev, struct rtattr *choke);
void debug_tca_choke_stab(int lev, struct rtattr *choke);
void debug_tca_choke_max_p(int lev, struct rtattr *choke);
void debug_tc_choke_xstats(int lev, struct rtattr *tca);
#endif
void debug_tca_options_htb(int lev, struct rtattr *tca);
void debug_tca_htb_parms(int lev, struct rtattr *htb);
void debug_tca_htb_init(int lev, struct rtattr *htb);
void debug_tca_htb_ctab(int lev, struct rtattr *htb);
void debug_tca_htb_rtab(int lev, struct rtattr *htb);
void debug_tc_htb_xstats(int lev, struct rtattr *tca);
void debug_tca_options_hfsc(int lev, struct rtattr *tca);
void debug_tca_hfsc_rsc(int lev, struct rtattr *hfsc);
void debug_tca_hfsc_fsc(int lev, struct rtattr *hfsc);
void debug_tca_hfsc_usc(int lev, struct rtattr *hfsc);
void debug_tca_options_cbq(int lev, struct rtattr *tca);
void debug_tca_cbq_lssopt(int lev, struct rtattr *cbq);
void debug_tca_cbq_wrropt(int lev, struct rtattr *cbq);
void debug_tca_cbq_fopt(int lev, struct rtattr *cbq);
void debug_tca_cbq_ovl_strategy(int lev, struct rtattr *cbq);
void debug_tca_cbq_rate(int lev, struct rtattr *cbq);
void debug_tca_cbq_rtab(int lev, struct rtattr *cbq);
void debug_tca_cbq_police(int lev, struct rtattr *cbq);
void debug_tc_cbq_xstats(int lev, struct rtattr *tca);
void debug_tca_options_dsmark(int lev, struct rtattr *tca);
void debug_tca_dsmark_indices(int lev, struct rtattr *dsmark);
void debug_tca_dsmark_default_index(int lev, struct rtattr *dsmark);
void debug_tca_dsmark_set_tc_index(int lev, struct rtattr *dsmark);
void debug_tca_dsmark_mask(int lev, struct rtattr *dsmark);
void debug_tca_dsmark_value(int lev, struct rtattr *dsmark);
void debug_tca_options_netem(int lev, struct rtattr *tca);
void debug_tca_netem_corr(int lev, struct rtattr *netem);
void debug_tca_netem_delay_dist(int lev, struct rtattr *netem);
void debug_tca_netem_reorder(int lev, struct rtattr *netem);
void debug_tca_netem_corrupt(int lev, struct rtattr *netem);
#if HAVE_DECL_TCA_NETEM_LOSS
void debug_tca_netem_loss(int lev, struct rtattr *netem);
void debug_netem_loss_gi(int lev, struct rtattr *loss);
void debug_netem_loss_ge(int lev, struct rtattr *loss);
#endif
#if HAVE_DECL_TCA_NETEM_RATE
void debug_tca_netem_rate(int lev, struct rtattr *netem);
#endif
#if HAVE_DECL_TCA_NETEM_ECN
void debug_tca_netem_ecn(int lev, struct rtattr *netem);
#endif
#if HAVE_DECL_TCA_DRR_UNSPEC
void debug_tca_options_drr(int lev, struct rtattr *tca);
void debug_tca_drr_quantum(int lev, struct rtattr *drr);
void debug_tc_drr_xstats(int lev, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_SFB_UNSPEC
void debug_tca_options_sfb(int lev, struct rtattr *tca);
void debug_tca_sfb_parms(int lev, struct rtattr *sfb);
void debug_tc_sfb_xstats(int lev, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_QFQ_UNSPEC
void debug_tca_options_qfq(int lev, struct rtattr *tca);
void debug_tca_qfq_weight(int lev, struct rtattr *qfq);
void debug_tca_qfq_lmax(int lev, struct rtattr *qfq);
void debug_tc_qfq_xstats(int lev, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_CODEL_UNSPEC
void debug_tca_options_codel(int lev, struct rtattr *tca);
void debug_tca_codel_target(int lev, struct rtattr *codel);
void debug_tca_codel_limit(int lev, struct rtattr *codel);
void debug_tca_codel_interval(int lev, struct rtattr *codel);
void debug_tca_codel_ecn(int lev, struct rtattr *codel);
void debug_tc_codel_xstats(int lev, struct rtattr *tca);
#endif
#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
void debug_tca_options_fq_codel(int lev, struct rtattr *tca);
void debug_tca_fq_codel_target(int lev, struct rtattr *fq_codel);
void debug_tca_fq_codel_limit(int lev, struct rtattr *fq_codel);
void debug_tca_fq_codel_interval(int lev, struct rtattr *fq_codel);
void debug_tca_fq_codel_ecn(int lev, struct rtattr *fq_codel);
void debug_tca_fq_codel_flows(int lev, struct rtattr *fq_codel);
void debug_tca_fq_codel_quantum(int lev, struct rtattr *fq_codel);
void debug_tc_fq_codel_xstats(int lev, struct rtattr *tca);
#endif
void debug_tc_ratespec(int lev, struct tc_ratespec *rate, char *name);
#ifdef HAVE_STRUCT_TC_PLUG_QOPT_ACTION
const char *convert_tcq_plug_action(int action);
#endif
void convert_tc_red_flags(int flags, char *flags_list, int len, int debug);
void convert_tcf_cbq_lss_change(int change, char *change_list, int len);
void convert_tcf_cbq_lss_flags(int flags, char *flags_list, int len);
void convert_tc_cbq_ovl_strategy(int strategy, char *strategy_list, int len);

/* tcmsg_filter.c */
int parse_tcmsg_filter(struct nlmsghdr *nlh);
int parse_tca_classid(char *msg, char **mp, struct rtattr *tca);
int parse_tca_indev(char *msg, char **mp, struct rtattr *tca);
int parse_tca_mask(char *msg, char **mp, struct rtattr *tca);
void parse_u32_handle(char *p, int len, unsigned handle);
int parse_tca_options_u32(char *msg, char **mp, struct rtattr *tca);
int parse_tca_u32_hash(char *msg, char **mp, struct rtattr *u32);
int parse_tca_u32_link(char *msg, char **mp, struct rtattr *u32);
int parse_tca_u32_divisor(char *msg, char **mp, struct rtattr *u32);
int parse_tca_u32_mark(char *msg, char **mp, struct rtattr *u32);
int parse_tca_u32_sel(char *msg, char *mp, struct rtattr *u32);
int parse_tca_options_rsvp(char *msg, char **mp, struct tcmsg *tcm, struct rtattr *tca);
int parse_tca_rsvp_dst(char *msg, char **mp, struct tcmsg *tcm, struct rtattr *rsvp);
int parse_tca_rsvp_src(char *msg, char **mp, struct tcmsg *tcm, struct rtattr *rsvp);
int parse_tca_rsvp_pinfo(char *msg, char **mp, struct rtattr *rsvp);
int parse_tca_options_route(char *msg, char **mp, struct rtattr *tca);
int parse_tca_route4_from(char *msg, char **mp, struct rtattr *route);
int parse_tca_route4_to(char *msg, char **mp, struct rtattr *route);
int parse_tca_route4_iif(char *msg, char **mp, struct rtattr *route);
int parse_tca_options_fw(char *msg, char **mp, struct rtattr *tca);
int parse_tca_options_tcindex(char *msg, char **mp, struct rtattr *tca);
int parse_tca_tcindex_hash(char *msg, char **mp, struct rtattr *tcindex);
int parse_tca_tcindex_mask(char *msg, char **mp, struct rtattr *tcindex);
int parse_tca_tcindex_shift(char *msg, char **mp, struct rtattr *tcindex);
int parse_tca_tcindex_fall_through(char *msg, char **mp, struct rtattr *tcindex);
#if HAVE_DECL_TCA_FLOW_UNSPEC
int parse_tca_options_flow(char *msg, char **mp, struct rtattr *tca);
int parse_tca_flow_keys(char *msg, char **mp, struct rtattr *flow);
int parse_tca_flow_mode(char *msg, char **mp, struct rtattr *flow);
int parse_tca_flow_xor(char *msg, char **mp, struct rtattr *flow);
int parse_tca_flow_rshift(char *msg, char **mp, struct rtattr *flow);
int parse_tca_flow_addend(char *msg, char **mp, struct rtattr *flow);
int parse_tca_flow_divisor(char *msg, char **mp, struct rtattr *flow);
int parse_tca_flow_perturb(char *msg, char **mp, struct rtattr *flow);
#endif
int parse_tca_options_basic(char *msg, char **mp, struct rtattr *tca);
#if HAVE_DECL_TCA_CGROUP_UNSPEC
int parse_tca_options_cgroup(char *msg, char **mp, struct rtattr *tca);
#endif
int parse_tca_ematch(char *msg, char *mp, struct rtattr *tca);
int parse_tca_ematch_tree_hdr(struct rtattr *em_tree, int *num);
int parse_tca_ematch_tree_list(char *msg, char *mp, struct rtattr *em_tree, int num);
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
int parse_ematch_cmp(char *msg, char *mp, void *p, int len);
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_NBYTE_H
int parse_ematch_nbyte(char *msg, char *mp, void *p, int len);
#endif
int parse_ematch_u32(char *msg, char *mp, void *p, int len);
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
int parse_ematch_meta(char *msg, char *mp, void *p, int len);
int parse_tca_em_meta_value(char *msg, char **mp, struct tcf_meta_val *val, struct rtattr *meta);
#endif
int inet_ntop_tc_addr(struct tcmsg *tcm, struct rtattr *tca, char *addrstr, int addrstrlen);
void debug_tca_options_u32(int lev, struct rtattr *tca);
void debug_tca_u32_classid(int lev, struct rtattr *u32);
void debug_tca_u32_hash(int lev, struct rtattr *u32);
void debug_tca_u32_link(int lev, struct rtattr *u32);
void debug_tca_u32_divisor(int lev, struct rtattr *u32);
void debug_tca_u32_sel(int lev, struct rtattr *u32);
void debug_tca_u32_police(int lev, struct rtattr *u32);
void debug_tca_u32_act(int lev, struct rtattr *u32);
void debug_tca_u32_indev(int lev, struct rtattr *u32);
void debug_tca_u32_pcnt(int lev, struct rtattr *u32[]);
void debug_tca_u32_mark(int lev, struct rtattr *u32);
void debug_tca_options_rsvp(int lev, struct tcmsg *tcm, struct rtattr *tca);
void debug_tca_rsvp_classid(int lev, struct rtattr *rsvp);
void debug_tca_rsvp_dst(int lev, struct tcmsg *tcm, struct rtattr *rsvp);
void debug_tca_rsvp_src(int lev, struct tcmsg *tcm, struct rtattr *rsvp);
void debug_tca_rsvp_pinfo(int lev, struct rtattr *rsvp);
void debug_tca_rsvp_police(int lev, struct rtattr *rsvp);
void debug_tca_rsvp_act(int lev, struct rtattr *rsvp);
void debug_tca_options_route(int lev, struct rtattr *tca);
void debug_tca_route4_classid(int lev, struct rtattr *route);
void debug_tca_route4_to(int lev, struct rtattr *route);
void debug_tca_route4_from(int lev, struct rtattr *route);
void debug_tca_route4_iif(int lev, struct rtattr *route);
void debug_tca_route4_police(int lev, struct rtattr *route);
void debug_tca_route4_act(int lev, struct rtattr *route);
void debug_tca_options_fw(int lev, struct rtattr *tca);
void debug_tca_fw_classid(int lev, struct rtattr *fw);
void debug_tca_fw_police(int lev, struct rtattr *fw);
void debug_tca_fw_indev(int lev, struct rtattr *fw);
void debug_tca_fw_act(int lev, struct rtattr *fw);
#if HAVE_DECL_TCA_FW_MASK
void debug_tca_fw_mask(int lev, struct rtattr *fw);
#endif
void debug_tca_options_tcindex(int lev, struct rtattr *tca);
void debug_tca_tcindex_hash(int lev, struct rtattr *tcindex);
void debug_tca_tcindex_mask(int lev, struct rtattr *tcindex);
void debug_tca_tcindex_shift(int lev, struct rtattr *tcindex);
void debug_tca_tcindex_fall_through(int lev, struct rtattr *tcindex);
void debug_tca_tcindex_classid(int lev, struct rtattr *tcindex);
void debug_tca_tcindex_police(int lev, struct rtattr *tcindex);
void debug_tca_tcindex_act(int lev, struct rtattr *tcindex);
#if HAVE_DECL_TCA_FLOW_UNSPEC
void debug_tca_options_flow(int lev, struct rtattr *tca);
void debug_tca_flow_keys(int lev, struct rtattr *flow);
void debug_tca_flow_mode(int lev, struct rtattr *flow);
void debug_tca_flow_baseclass(int lev, struct rtattr *flow);
void debug_tca_flow_rshift(int lev, struct rtattr *flow);
void debug_tca_flow_addend(int lev, struct rtattr *flow);
void debug_tca_flow_mask(int lev, struct rtattr *flow);
void debug_tca_flow_xor(int lev, struct rtattr *flow);
void debug_tca_flow_divisor(int lev, struct rtattr *flow);
void debug_tca_flow_act(int lev, struct rtattr *flow);
void debug_tca_flow_police(int lev, struct rtattr *flow);
void debug_tca_flow_ematches(int lev, struct rtattr *flow);
void debug_tca_flow_perturb(int lev, struct rtattr *flow);
#endif
void debug_tca_options_basic(int lev, struct rtattr *tca);
void debug_tca_basic_classid(int lev, struct rtattr *basic);
void debug_tca_basic_ematches(int lev, struct rtattr *basic);
void debug_tca_basic_act(int lev, struct rtattr *basic);
void debug_tca_basic_police(int lev, struct rtattr *basic);
#if HAVE_DECL_TCA_CGROUP_UNSPEC
void debug_tca_options_cgroup(int lev, struct rtattr *tca);
void debug_tca_cgroup_act(int lev, struct rtattr *cgroup);
void debug_tca_cgroup_police(int lev, struct rtattr *cgroup);
void debug_tca_cgroup_ematches(int lev, struct rtattr *cgroup);
#endif
void debug_tca_ematch(int lev, struct rtattr *tca);
int debug_tca_ematch_tree_hdr(int lev, struct rtattr *ematch);
void debug_tca_ematch_tree_list(int lev, struct rtattr *em_tree, int num);
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
void debug_ematch_cmp(int lev, void *p, int len);
#endif
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_NBYTE_H
void debug_ematch_nbyte(int lev, void *p, int len);
#endif
void debug_ematch_u32(int lev, void *p, int len);
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
void debug_ematch_meta(int lev, void *p, int len);
struct tcf_meta_hdr *debug_tca_em_meta_hdr(int lev, struct rtattr *meta);
void debug_tca_em_meta_lvalue(int lev, struct rtattr *meta, struct tcf_meta_hdr *hdr);
void debug_tca_em_meta_rvalue(int lev, struct rtattr *meta, struct tcf_meta_hdr *hdr);
#endif
void convert_tc_u32_flags(int flags, char *flags_list, int len, int debug);
const char *convert_eth_p(int proto, int debug);
#if HAVE_DECL_TCA_FLOW_UNSPEC
void convert_flow_key(int flags, char *flags_list, int len, int debug);
const char *convert_flow_mode(int mode, int debug);
#endif
const char *convert_tcf_em_kind(int kind, int debug);
const char *convert_tcf_em_flag(int flag, int debug);
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_CMP_H
const char *convert_tcf_em_align(int align, int debug);
#endif
const char *convert_tcf_em_opnd(int opnd, int debug);
#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
const char *convert_tcf_meta_type(int type, int debug);
const char *convert_tcf_meta_id(int id, int debug);
#endif

/* tcamsg.c */
int parse_tcamsg(struct nlmsghdr *nlh);
int parse_tca_acts(char *msg, char *mp, struct rtattr *tcaa);
int parse_tca_act(char *msg, char *mp, struct rtattr *acts);
int parse_tca_act_options_police(char *msg, char *mp, struct rtattr *act);
int parse_tca_act_options_gact(char *msg, char *mp, struct rtattr *act);
int parse_tca_act_options_pedit(char *msg, char *mp, struct rtattr *act);
int parse_tca_act_options_mirred(char *msg, char *mp, struct rtattr *act);
#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
int parse_tca_act_options_nat(char *msg, char *mp, struct rtattr *act);
#endif
#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
int parse_tca_act_options_skbedit(char *msg, char *mp, struct rtattr *act);
#endif
#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
int parse_tca_act_options_csum(char *msg, char *mp, struct rtattr *act);
#endif
void debug_tcamsg(int lev, struct tcamsg *tcam, struct rtattr *tcaa[], int tcam_len);
void debug_tca_act_tab(int lev, struct rtattr *tcaa);
void debug_tca_acts(int lev, struct rtattr *tcaa);
void debug_tca_act(int lev, struct rtattr *acts);
void debug_tca_act_kind(int lev, struct rtattr *act, char *kind, int len);
void debug_tca_act_options(int lev, struct rtattr *act, char *kind, int len);
void debug_tca_act_index(int lev, struct rtattr *act);
void debug_tca_act_stats(int lev, struct rtattr *act);
void debug_tca_act_options_police(int lev, struct rtattr *act);
void debug_tca_police_tbf(int lev, struct rtattr *police);
void debug_tca_police_rate(int lev, struct rtattr *police);
void debug_tca_police_peakrate(int lev, struct rtattr *police);
void debug_tca_police_avrate(int lev, struct rtattr *police);
void debug_tca_police_result(int lev, struct rtattr *police);
void debug_tca_act_options_gact(int lev, struct rtattr *act);
void debug_tca_gact_tm(int lev, struct rtattr *gact);
void debug_tca_gact_parms(int lev, struct rtattr *gact);
void debug_tca_gact_prob(int lev, struct rtattr *gact);
void debug_tca_act_options_mirred(int lev, struct rtattr *act);
void debug_tca_mirred_tm(int lev, struct rtattr *mirred);
void debug_tca_mirred_parms(int lev, struct rtattr *mirred);
#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
void debug_tca_act_options_nat(int lev, struct rtattr *act);
void debug_tca_nat_parms(int lev, struct rtattr *nat);
void debug_tca_nat_tm(int lev, struct rtattr *nat);
#endif
#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
void debug_tca_act_options_skbedit(int lev, struct rtattr *act);
void debug_tca_skbedit_tm(int lev, struct rtattr *skb);
void debug_tca_skbedit_parms(int lev, struct rtattr *skb);
void debug_tca_skbedit_priority(int lev, struct rtattr *skb);
void debug_tca_skbedit_queue_mapping(int lev, struct rtattr *skb);
#ifdef HAVE_DECL_TCA_SKBEDIT_MARK
void debug_tca_skbedit_mark(int lev, struct rtattr *skb);
#endif
#endif
void debug_tca_act_options_pedit(int lev, struct rtattr *act);
void debug_tca_pedit_tm(int lev, struct rtattr *pedit);
void debug_tca_pedit_parms(int lev, struct rtattr *pedit);
#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
void debug_tca_act_options_csum(int lev, struct rtattr *act);
void debug_tca_csum_tm(int lev, struct rtattr *csum);
void debug_tca_csum_parms(int lev, struct rtattr *csum);
#endif
void debug_tcf_t(int lev, struct tcf_t *tm);
const char *convert_tc_action(int action, int debug);
const char *convert_tc_police_action(int action, int debug);
const char *convert_pgact(int pgact, int debug);
const char *convert_tca_mirred_action(int action, int debug);
#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
void convert_tca_csum_update_flags(int flags, char *flags_list, int len, int debug);
#endif

#endif /* _NIELD_ */
