# -*- coding: utf-8 -*-
#
#  gomi.py - a gomi.dll compatible Saori module for ninix
#  Copyright (C) 2012, 2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import argparse
import shutil

import gio

from ninix.dll import SAORI

class Saori(SAORI):

    def __init__(self):
        if 'XDG_DATA_HOME' in os.environ:
            XDG_DATA_HOME = os.environ['XDG_DATA_HOME']
        else:
            XDG_DATA_HOME = os.environ['HOME'] + '/.local/share'
        self.HOME_TRASH = os.path.join(XDG_DATA_HOME, 'Trash')
        SAORI.__init__(self)

    def setup(self):
        self.parser = argparse.ArgumentParser()
        self.parser.add_argument('-e', '--empty', action='store_true')
        self.parser.add_argument('-n', '--number-of-items', action='store_true')
        self.parser.add_argument('-V', '--version', action='store_true')
        self.parser.add_argument('-a', '--asynchronous', action='store_true')
        self.parser.add_argument('-f', '--force', action='store_true')
        self.parser.add_argument('-q', '--quiet', action='store_true')
        self.parser.add_argument('-s', '--silent', action='store_true')
        self.parser.add_argument('-v', '--verbose', action='store_true')
        self.parser.add_argument('-w', '--hwnd')
        return 1 # XXX

    def get_volume_trash(self):
        vm = gio.VolumeMonitor()
        for m in vm.get_mounts():
            mp = m.get_root().get_path()
            if mp is None:
                continue
            volume_trash = os.path.join(mp, '.Trash', str(os.getuid()))
            if not os.path.exists(volume_trash):
                volume_trash = os.path.join(mp, '.Trash-' + str(os.getuid()))
            if not os.path.exists(volume_trash):
                continue
            yield volume_trash

    def get_dir_size(self, dir_name):
        file_count = 0
        dir_size = 0
        for (path, dirs, files) in os.walk(dir_name):
            for file in files:
                file_count += 1
                dir_size += os.path.getsize(os.path.join(path, file))
        return (file_count, dir_size)

    def empty_trash(self, path):
        for info in os.listdir(os.path.join(path, 'info')):
            trash = info[0:-len('.trashinfo')]
            filepath = os.path.join(path, 'files', trash)
            infopath = os.path.join(path, 'info', info)
            if os.path.isfile(filepath) or os.path.islink(filepath):
                os.remove(filepath)
                os.remove(infopath)
            elif os.path.isdir(filepath):
                shutil.rmtree(filepath)
                os.remove(infopath)

    def execute(self, argument):
        args = self.parser.parse_args(argument[0].split())
        if args.number_of_items:
            file_count, dir_size = self.get_dir_size(self.HOME_TRASH)
            for volume_trash in self.get_volume_trash():
                count, size = self.get_dir_size(volume_trash)
                file_count += count
                dir_size += size
            return ''.join(('SAORI/1.0 200 OK\r\n',
                            'Result: ',
                            str(file_count),
                            '\r\n'
                            'Reference0: ',
                            str(dir_size),
                            '\r\n\r\n'))
        elif args.empty:
            if args.force:
                self.empty_trash(self.HOME_TRASH)
                for volume_trash in self.get_volume_trash():
                    self.empty_trash(volume_trash)
            else:
                return self.RESPONSE[500]
