/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include "chattextwgt.h"
#include "globals.h"

#include <QTextCursor>
#include <QScrollBar>
#include <QFile>
#include <QDateTime>
#include <QTextFrame>

#include "message.h"
#include "smileswgt.h"
#include "smileswgt.h"

QList<Smile> ChatTextWgt::m_smiles;

ChatTextWgt::ChatTextWgt(QWidget *parent)
 : QWidget(parent),
  m_smilesFromSender(0)
{
  QGridLayout* grid  = new QGridLayout(this);
  m_text             = new QTextEdit  (this);

  m_text->setReadOnly(true);

  grid->addWidget(m_text, 0, 0);
  grid->setMargin(0);
  m_text->verticalScrollBar()->setTracking(true);

  initSmiles(Globals::prefs()->smilesThemePath());
}

ChatTextWgt::~ChatTextWgt()
{
  qDebug("[ChatTextWgt::~ChatTextWgt]\n");
}
//\*****************************************************************************
void ChatTextWgt::addSmile(const QStringList & smiles, const QString & name)
{
  m_smiles.append(Smile(smiles, name));
}
//\*****************************************************************************
void ChatTextWgt::initSmiles(const QString & path)
{
  QFile file(path + "/emoticons.xml");

  QDomDocument dom_document;
  QDomElement  root;
  QDomElement  child;
  QDomElement  emoticon;
  QDomElement  name;
  QDomNodeList elements;
  QDomNodeList elements_names;

  QStringList  list;

  m_smiles.clear();

  if(!file.open(QIODevice::ReadOnly))
  {
    Globals::addError("Couldn't open " + path + "/emoticons.xml");
    return;
  }

  if(!dom_document.setContent(&file, true))
  {
    Globals::addError("Couldn't parse " + path + "/emoticons.xml");
    return;
  }

  root  = dom_document.documentElement();

  elements = root.elementsByTagName("emoticon");

  for(uint i = 0, len = elements.length(); i < len; i++)
  {
    emoticon = elements.item(i).toElement();
    elements_names = emoticon.elementsByTagName("string");
    list.clear();

    for(uint j = 0, len = elements_names.length(); j < len; j++)
    {
      child = elements_names.item(j).toElement();
      list.append(child.text());
    }

    addSmile(list, emoticon.attribute("file"));
  }
}
//\*****************************************************************************
void ChatTextWgt::addMsg(const Message* msg)
{
  QTextDocument* doc = new QTextDocument;
  QTextCharFormat fmt;
  QDateTime date_time;
  QTextCursor cur_old;
  QTextCursor cur_new;
  QBrush brush;

  doc->setPlainText(msg->msg());

  cur_new = doc->rootFrame()->firstCursorPosition();

  date_time.setTime_t(msg->receiveTime());

  fmt = cur_new.blockCharFormat();
  brush = fmt.foreground();
  brush.setStyle(Qt::SolidPattern);

  if(msg->requested())
    fmt.setFontItalic(true);
  else
    fmt.setFontItalic(false);

  if(isSystemMsg(msg->type()))
    brush.setColor(Globals::prefs()->sysColor());
  else
    brush.setColor(msg->color());

  QString res_msg(Globals::prefs()->displayMsgFormat());
  QString tm_fmt = res_msg.section(QString("%time%"), 1, 1);

  res_msg.replace(QRegExp("%time%([^<]*)%time%"), date_time.toString(tm_fmt));
  res_msg.replace("%user", msg->userName());
  res_msg.replace("%comp", msg->compName());

  fmt.setForeground(brush);
  cur_new.setCharFormat(fmt);

  cur_new.insertText(res_msg);

  if(!(isSystemMsg(msg->type()) && !Globals::prefs()->colorWholeSysMsg() ||
     (msg->type() == Globals::MESSAGE  && !Globals::prefs()->colorWholeMsg())))
  {
    cur_new.movePosition(QTextCursor::End, QTextCursor::KeepAnchor);
    cur_new.setCharFormat(fmt);
  }

  cur_old = m_text->textCursor();
  cur_old.clearSelection();
  cur_old.setPosition(m_text->toPlainText().size());
  m_text->setTextCursor(cur_old);

  // Obrabotka soobscheniya i vstavka smailov
  int idx_end = 0;
  QString str;
  QString smile;
  QString msg_ = doc->toPlainText();

  smile   = nextSmile  (msg_);
  idx_end = msg_.indexOf(smile);

  for(; idx_end != -1 && smile != "";)
  {
    str     = msg_.left(idx_end);
    msg_    = msg_.right(msg_.size() - idx_end - smile.size());

    cur_new = doc->find(smile, cur_new);
    if(!cur_new.isNull())
    {
      while(cur_new.selectedText() != smile)
      {
        cur_new.movePosition(QTextCursor::NextCharacter, QTextCursor::KeepAnchor);
        if(cur_new.position() == doc->toPlainText().size())
          break;
      }

      if(cur_new.selectedText() == smile)
        insertSmile(cur_new, smile);

      if(cur_new.position() == doc->toPlainText().size())
        break;

    }

    if((smile = nextSmile(msg_)).isEmpty())
      break;

    idx_end = msg_.indexOf(smile);
  }

  m_text->insertHtml(doc->toHtml());
  m_text->insertPlainText("\n");

  // TODO dobavit' nastroiku autoScroll
//   if(scrl_max)

  m_text->verticalScrollBar()->setValue(m_text->verticalScrollBar()->maximum());
}
//\*****************************************************************************
void ChatTextWgt::setMsg(const QString & msg )
{
  QTextCursor cur_new = m_text->textCursor();
  cur_new.clearSelection();

  // Obrabotka soobscheniya i vstavka smailov
  int idx_end = 0;
  QString str;
  QString smile;
  QString msg_ = msg;

  cur_new.beginEditBlock();

  smile   = nextSmile  (msg_);
  idx_end = msg_.indexOf(smile);
  for(int i = 0; i < 10 && idx_end != -1 && !smile.isEmpty(); i++)
  {
    str     = msg_.left(idx_end);
    msg_    = msg_.right(msg_.size() - idx_end - smile.size());
    cur_new.insertText(str);
    insertSmile(cur_new, smile);

    if((smile = nextSmile(msg_)).isEmpty())
      break;

    idx_end = msg_.indexOf(smile);
  }

  str = msg_;
  cur_new.insertText(str);
  cur_new.endEditBlock();
}
//\*****************************************************************************
QString ChatTextWgt::nextSmile(const QString & str, Smile** smile_)
{
  int i, j;
  int idx = -1, idx1;

  int smile = -1;
  int smile_pos = -1;

  int smile_size = 0;

  for(i = 0; i < m_smiles.size(); i++)
    for(j = 0; j < m_smiles[i].smiles.size(); j++)
    {
      idx1 = str.indexOf(m_smiles[i].smiles[j]);
      if(idx1 >= 0 && (idx1 <= idx || idx < 0))
      {
        if(m_smiles[i].smiles[j].size() >= smile_size)
        {
          smile_size = m_smiles[i].smiles[j].size();
          smile      = i;
          smile_pos  = j;
          idx        = idx1;
        }
      }
    }

  if(smile >= 0 && smile_pos >= 0)
  {
    if(smile_)
      *smile_ = &m_smiles[smile];
    return m_smiles[smile].smiles[smile_pos];
  }
  else
    return QString("");


}
//\*****************************************************************************
QString ChatTextWgt::nextSmile(const QString & str) const
{
  int i, j;
  int idx = -1, idx1;

  int smile = -1;
  int smile_pos = -1;

  int smile_size = 0;

  for(i = 0; i < m_smiles.size(); i++)
    for(j = 0; j < m_smiles[i].smiles.size(); j++)
    {
      idx1 = str.indexOf(m_smiles[i].smiles[j]);
      if(idx1 >= 0 && (idx1 < idx || idx < 0 || (idx1 == idx && m_smiles[i].smiles[j].size() >= smile_size)))
      {
        smile_size = m_smiles[i].smiles[j].size();
        smile      = i;
        smile_pos  = j;
        idx        = idx1;
      }
    }

  if(m_smilesFromSender)
  {
    for(i = 0; i < m_smilesFromSender->size(); i++)
    {
      idx1 = str.indexOf((*m_smilesFromSender)[i].smiles[0]);
      if(idx1 >= 0 && (idx1 <= idx || idx < 0))
      {
        if((*m_smilesFromSender)[i].smiles[0].size() >= smile_size)
        {
          smile_size = (*m_smilesFromSender)[i].smiles[0].size();
          smile      = i;
          smile_pos  = -1;
          idx        = idx1;
        }
      }
    }
  }

  if(smile >= 0 && smile_pos >= 0)
    return m_smiles[smile].smiles[smile_pos];
  else if(smile >= 0 && smile_pos == -1)
    return (*m_smilesFromSender)[smile].smiles[0];

  return QString("");
}
//\*****************************************************************************
void ChatTextWgt::insertSmile(QTextCursor cursor, const QString & smile)
{
  QString smiles_dir =  Globals::prefs()->smilesThemePath();
  int i, j;

  if(m_smilesFromSender)
  {
    for(i = 0; i < m_smilesFromSender->size(); i++)
      if((*m_smilesFromSender)[i].smiles[0] == smile)
      {
        if(QFile((*m_smilesFromSender)[i].name).exists())
        {
          cursor.insertImage((*m_smilesFromSender)[i].name);
          return;
        }
        m_smilesFromSender->clear();
        break;
      }
  }

  for(i = 0; i < m_smiles.size(); i++)
    for(j = 0; j < m_smiles[i].smiles.size(); j++)
      if(m_smiles[i].smiles[j] == smile)
      {
        if(QFile(smiles_dir + m_smiles[i].name).exists())
          cursor.insertImage(smiles_dir + m_smiles[i].name);
        else if(QFile(smiles_dir + m_smiles[i].name + ".png").exists())
          cursor.insertImage(smiles_dir + m_smiles[i].name + ".png");
        else if(QFile(smiles_dir + m_smiles[i].name + ".jpg").exists())
           cursor.insertImage(smiles_dir + m_smiles[i].name + ".jpg");
        else if(QFile(smiles_dir + m_smiles[i].name + ".gif").exists())
           cursor.insertImage(smiles_dir + m_smiles[i].name + ".gif");
        else
          cursor.insertText(m_smiles[i].smiles[j]);
        return;
      }
}
//\*****************************************************************************
void ChatTextWgt::setSmilesFromSender(QList<Smile> * smiles)
{
  m_smilesFromSender = smiles;
}
