/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include "preferencesdlg.h"
#include "globals.h"

#include <QSpacerItem>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QHostAddress>
#include <QStringList>
#include <QFile>
#include <QCoreApplication>

#include "preferences.h"
#include "colorlabel.h"

PreferencesDlg::PreferencesDlg(QWidget *parent)
 : QDialog(parent)
{
  m_profileName = "";
  m_edited      = false;

  createWidgets();

  QVBoxLayout* btns_box          = new QVBoxLayout(0);
  QVBoxLayout* msgs_sect_vbox    = new QVBoxLayout(0);
  QGridLayout* grid              = new QGridLayout(this);

  QGridLayout* colors_grid       = new QGridLayout(m_colorsGrbx);
  QGridLayout* network_grid      = new QGridLayout(m_networkGrbx);
  QGridLayout* misc_grid         = new QGridLayout(m_miscGrbx);
  QGridLayout* nl_grid           = new QGridLayout(m_nowListeningGrbx);
  QGridLayout* smiles_grid       = new QGridLayout(m_smilesThemesGrbx);
  QGridLayout* users_list_grid   = new QGridLayout(m_usersListGrbx);
  QGridLayout* style_sheets_grid = new QGridLayout(m_styleSheetsGrbx);

  // Setting up group boxes
  //********************************************
  // Colors
  m_myColor    ->setMinimumWidth (100); m_myColor    ->setMaximumHeight(25);
  m_sysColor   ->setMinimumWidth (100); m_sysColor   ->setMaximumHeight(25);
  m_baseColor  ->setMinimumWidth (100); m_baseColor  ->setMaximumHeight(25);
  m_myColor    ->setFrameStyle(QFrame::StyledPanel | QFrame::Plain);
  m_sysColor   ->setFrameStyle(QFrame::StyledPanel | QFrame::Plain);
  m_baseColor  ->setFrameStyle(QFrame::StyledPanel | QFrame::Plain);

  colors_grid->addWidget(m_myColorLab          , 0, 0);
  colors_grid->addWidget(m_myColor             , 0, 1);
  colors_grid->addWidget(m_sysColorLab         , 1, 0);
  colors_grid->addWidget(m_sysColor            , 1, 1);
  colors_grid->addWidget(m_baseColorLab        , 2, 0);
  colors_grid->addWidget(m_baseColor           , 2, 1);
  colors_grid->addWidget(m_colorWholeMsgChbx   , 3, 0, 1, 2);
  colors_grid->addWidget(m_colorWholeSysMsgChbx, 4, 0, 1, 2);
  colors_grid->setColumnStretch(1, 1);

  // Network
  m_portOutSpbx->setMaximum(65535);
  m_portInSpbx ->setMaximum(65535);
  m_applyBtn   ->setEnabled(false);

  network_grid->addWidget(m_networkIfLab , 0, 0);
  network_grid->addWidget(m_networkIfCmbx, 0, 1);
  network_grid->addWidget(m_ipLab        , 1, 0);
  network_grid->addWidget(m_ipEdit       , 1, 1);
  network_grid->addWidget(m_broadcastLab , 2, 0);
  network_grid->addWidget(m_broadcastEdit, 2, 1);
  network_grid->addWidget(m_portInLab    , 3, 0);
  network_grid->addWidget(m_portInSpbx   , 3, 1);
  network_grid->addWidget(m_portOutLab   , 4, 0);
  network_grid->addWidget(m_portOutSpbx  , 4, 1);

  network_grid->setRowStretch(4, 1);

  //Now Listening
  nl_grid->addWidget(m_nlInStatusChbx);
  nl_grid->addWidget(m_nlWithMessageChbx);
  nl_grid->addWidget(m_nlFormatLab);
  nl_grid->addWidget(m_nlFormatEdit);

  // Misc
  misc_grid->addWidget(m_activateOnMsgInChbx, 0, 0);
  misc_grid->addWidget(m_isCommandOnIncomingChbx , 1, 0);
  misc_grid->addWidget(m_commandOnIncomingEdit, 2, 0);

  //Smiles
  smiles_grid->addWidget(m_smilesThemeChooser);

  // Users List
  users_list_grid->addWidget(m_ulRefreshIntervalLab     , 0, 0);
  users_list_grid->addWidget(m_ulRefreshIntervalSpbx    , 0, 1);
  users_list_grid->addWidget(m_ulDeepRefreshIntervalLab , 1, 0);
  users_list_grid->addWidget(m_ulDeepRefreshIntervalSpbx, 1, 1);

  // Style Sheets
  style_sheets_grid->addWidget(m_styleSheetsChooser);

  //********************************************
  btns_box->addWidget(m_okBtn     );
  btns_box->addWidget(m_applyBtn  );
  btns_box->addWidget(m_cancelBtn );

  msgs_sect_vbox->addWidget(m_colorsGrbx);
  msgs_sect_vbox->addWidget(m_miscGrbx);

  grid->addWidget(m_sectionsChooser, 0, 0, 2, 1);
  grid->addWidget(m_sections       , 0, 1, 2, 2);
  grid->addLayout(btns_box         , 0, 3);

  grid->setColumnStretch(1, 1);

  setupSections();

  setupConnections();

  retranslate();

  resize(400, 200);
}
//\*****************************************************************************
PreferencesDlg::~PreferencesDlg()
{
}

void PreferencesDlg::slot_setMsgColorMode   (int mode)
{Globals::prefs()->setMsgColorMode(mode);}

void PreferencesDlg::slot_setSysMsgColorMode(int mode)
{Globals::prefs()->setSysMsgColorMode(mode);}

void PreferencesDlg::slot_setActivateOnMsgIn(int mode)
{Globals::prefs()->setActivateOnMsgIn(mode);}

void PreferencesDlg::slot_setSoundOnMsgIn   (int mode)
{Globals::prefs()->setSoundOnMsgIn(mode);}

void PreferencesDlg::slot_setNlWithMessage  (int mode)
{Globals::prefs()->setNlMode(m_nlInStatusChbx->isChecked() * 2 | (mode == Qt::Checked));}

void PreferencesDlg::slot_setNlInStatus     (int mode)
{Globals::prefs()->setNlMode(m_nlWithMessageChbx->isChecked () | (mode == Qt::Checked) * 2);}

void PreferencesDlg::slot_setExecuteCommandMode(int mode)
{
  mode = (mode == Qt::Unchecked ? 0 : 1);
  Globals::prefs()->setIsExecuteCommandOnIncomingMsg(mode);
  m_commandOnIncomingEdit->setEnabled(mode);
}
//\*****************************************************************************
void PreferencesDlg::init()
{
  if(!Globals::prefs()->allowDiffPorts())
  {
    m_portOutLab ->hide();
    m_portOutSpbx->hide();
    m_portInLab->setText(tr("Input/Output Port : "));
    connect(m_portInSpbx, SIGNAL(valueChanged(int)), m_portOutSpbx, SLOT(setValue(int)));
  }
  else
  {
    m_portOutLab ->show();
    m_portOutSpbx->show();
    m_portInLab->setText(tr("Input Port : "));
    disconnect(m_portInSpbx, SIGNAL(valueChanged(int)), m_portOutSpbx, SLOT(setValue(int)));
  }

  m_myColor  ->setColor(Globals::prefs()->myColor());
  m_sysColor ->setColor(Globals::prefs()->sysColor());
  m_baseColor->setColor(Globals::prefs()->baseColor());

  // CheckBoxes:
  if(Globals::prefs()->colorWholeMsg())
    m_colorWholeMsgChbx      ->setCheckState(Qt::Checked);
  else
    m_colorWholeMsgChbx      ->setCheckState(Qt::Unchecked);

  if(Globals::prefs()->colorWholeSysMsg())
    m_colorWholeSysMsgChbx   ->setCheckState(Qt::Checked);
  else
    m_colorWholeSysMsgChbx   ->setCheckState(Qt::Unchecked);

  slot_setExecuteCommandMode(Globals::prefs()->isExecuteCommandOnIncomingMsg() ? Qt::Checked : Qt::Unchecked);

  // NowListening
  int mode = Globals::prefs()->nlMode();
  if(mode & 1)
    m_nlWithMessageChbx->setCheckState(Qt::Checked);
  else
    m_nlWithMessageChbx->setCheckState(Qt::Unchecked);

  if(mode & 2)
    m_nlInStatusChbx->setCheckState(Qt::Checked);
  else
    m_nlInStatusChbx->setCheckState(Qt::Unchecked);

  m_nlFormatEdit->setText(Globals::prefs()->nlFormat());

  m_oldPortIn   = Globals::prefs()->portIn ();
  m_portOutSpbx->setValue(Globals::prefs()->portOut());
  m_portInSpbx ->setValue(m_oldPortIn);
  m_ipEdit     ->setText (QHostAddress((quint32)Globals::prefs()->ip()).toString());

  if(Globals::prefs()->activateOnMsgIn())
    m_activateOnMsgInChbx   ->setCheckState(Qt::Checked);
  else
    m_activateOnMsgInChbx   ->setCheckState(Qt::Unchecked);
  if(Globals::prefs()->soundOnMsgIn())
    m_soundOnMsgInChbx   ->setCheckState(Qt::Checked);
  else
    m_soundOnMsgInChbx   ->setCheckState(Qt::Unchecked);

  //*************************
  m_broadcastEdit->clear();

  QString str       = m_ipEdit->text();
  QString broadcast = QHostAddress(Globals::prefs()->broadcast()).toString();
  QStringList list  = str.split(".");

  m_broadcastEdit->setText(broadcast);

  reloadNetworkIfs();
  //*************************

  m_commandOnIncomingEdit    ->setText(Globals::prefs()->executeCommandOnIncomingMsg());
  m_displayMsgFmtEdit        ->setText(Globals::prefs()->displayMsgFormat());
  m_msgsHistoryIntervalSpbx  ->setValue(Globals::prefs()->historyReqTimeout());
  m_msgsHistoryNumSpbx       ->setValue(Globals::prefs()->nHistoryMsgs());
  m_ulRefreshIntervalSpbx    ->setValue(Globals::prefs()->usersListRefreshInterval());
  m_ulDeepRefreshIntervalSpbx->setValue(Globals::prefs()->usersListDeepRefreshInterval());

  m_commandOnIncomingEdit->home(0);
  m_displayMsgFmtEdit->home(0);
  m_nlFormatEdit->home(0);

  m_profileName = Globals::profileName();
  m_edited = false;
  m_applyBtn->setEnabled(false);

  setWindowTitle(tr("Preferences [Profile: ") + m_profileName + "]");
}
//\*****************************************************************************
void PreferencesDlg::slot_chooseColor()
{
  QColor color = QColorDialog::getColor(((ColorLabel*)sender())->color(), this);

  if(color.isValid())
  {
    ((ColorLabel*)sender())->setColor(color);
    slot_edited();
  }
}
//\*****************************************************************************
void PreferencesDlg::slot_edited()
{
  m_edited = true;
  m_applyBtn->setEnabled(true);
}
//\*****************************************************************************
void PreferencesDlg::slot_setPrefs()
{
  int idx;

  if(Globals::profileName() == m_profileName)
  {

    switch(Sections(m_sectionsChooser->currentRow()))
    {
      case Network :
        break;

      case Messages :
        break;

      case Smiles :
        idx = m_smilesThemeChooser->currentRow();
        if(idx >= 0)
          emit wantChangeSmileTheme(m_smilesThemes[idx]);
        break;

      case Misc :
        break;

      case StyleSheets :
        idx = m_styleSheetsChooser->currentRow();
        if(idx >= 0)
          emit styleSheetChanged(m_styleSheets[idx]);
        break;
    }

    Globals::prefs()->setMyColor  (m_myColor    ->color());
    Globals::prefs()->setSysColor (m_sysColor   ->color());
    Globals::prefs()->setBaseColor(m_baseColor  ->color());
    Globals::prefs()->setPortOut  (m_portOutSpbx->value());

    QString str = "";

    if(!m_networkIfCmbx->currentIndex())
    {
      str = m_broadcastEdit->text();
      if(!QHostAddress(str).isNull())
        Globals::prefs()->setCustomBroadcast((QHostAddress(str)));

      str = m_ipEdit->text();
      if(!QHostAddress(str).isNull())
        Globals::prefs()->setCustomIp(QHostAddress(str).toIPv4Address());
    }

    str = m_broadcastEdit->text();
    if(!QHostAddress(str).isNull())
      Globals::prefs()->setBroadcast((QHostAddress(str)));

    str = m_ipEdit->text();
    if(QHostAddress(str).toString() != "")
      Globals::prefs()->setIp(QHostAddress(str).toIPv4Address());

    emit ipTextChanged(str);

    if(Globals::prefs()->portIn() != m_portInSpbx->value())
      emit portChanged(m_portInSpbx->value());

    Globals::prefs()->setExecuteCommandOnIncomingMsg(m_commandOnIncomingEdit->text());

    Globals::prefs()->setNlFormat(m_nlFormatEdit->text());
    Globals::prefs()->setDisplayMsgFormat(m_displayMsgFmtEdit->text());

    Globals::prefs()->setHistoryReqTimeout(m_msgsHistoryIntervalSpbx->value());
    Globals::prefs()->setNHistoryMsgs(m_msgsHistoryNumSpbx->value());

    Globals::prefs()->setUsersListRefreshInterval(m_ulRefreshIntervalSpbx->value());
    Globals::prefs()->setUsersListDeepRefreshInterval(m_ulDeepRefreshIntervalSpbx->value());

    emit ulRefreshIntervalChanged(m_ulRefreshIntervalSpbx->value());
    emit ulDeepRefreshIntervalChanged(m_ulDeepRefreshIntervalSpbx->value());
  }

  m_edited = false;
  m_applyBtn->setEnabled(false);
}
//\*****************************************************************************
void PreferencesDlg::slot_cancel()
{
  m_edited = false;
  m_applyBtn->setEnabled(false);
  reject();
}
//\*****************************************************************************
void PreferencesDlg::slot_validateIp(const QString & str)
{
  if((QHostAddress(str).isNull()))
    return;

  slot_edited();
}

void PreferencesDlg::retranslate()
{
  m_myColorLab           ->setText(tr("Color of My Messages : "));
  m_sysColorLab          ->setText(tr("Color of System Messages : "));
  m_baseColorLab         ->setText(tr("Base Color for Messages : "));
  m_ipLab                ->setText(tr("IP Address : "));
  m_broadcastLab         ->setText(tr("Broadcast Address : "));
  m_nlFormatLab          ->setText(tr("Now Listening string : \n(%t - Track, %a - Artist, %b - alBum)"));

  m_colorsGrbx           ->setTitle(tr("&Colors"));
  m_networkGrbx          ->setTitle(tr("&Network"));
  m_miscGrbx             ->setTitle(tr("&Misc"));
  m_nowListeningGrbx     ->setTitle(tr("Now &Listening (from Amarok1)"));
  m_smilesThemesGrbx     ->setTitle(tr("Choose smiles theme:"));
  m_usersListGrbx        ->setTitle(tr("Users List"));
  m_styleSheetsGrbx      ->setTitle(tr("Choose Style Sheet :"));

  m_okBtn                ->setText (tr("&OK"));
  m_cancelBtn            ->setText (tr("&Cancel"));
  m_applyBtn             ->setText (tr("&Apply"));

  m_colorWholeMsgChbx    ->setText (tr("Color Whole Message"));
  m_colorWholeSysMsgChbx ->setText (tr("Color Whole System Message"));
  m_activateOnMsgInChbx  ->setText (tr("Activate Window on Incomin Messages"));
  m_soundOnMsgInChbx     ->setText (tr("Play Sound on Incomig Messages"));
  m_nlWithMessageChbx    ->setText (tr("Send 'Now Listening' with messages"));
  m_nlInStatusChbx       ->setText (tr("Set 'Now Listening' in status description"));

  m_portOutLab           ->setText (tr("Output Port : "));
  m_portInLab            ->setText (tr("Input Port : "));

  m_profileSection       ->setText(tr("Profile"));
  m_miscSection          ->setText(tr("Misc"));
  m_messagesSection      ->setText(tr("Messages"));
  m_networkSection       ->setText(tr("Network"));
  m_smilesSection        ->setText(tr("Smiles"));
  m_styleSheetsSection   ->setText(tr("Style Sheets"));

  m_isCommandOnIncomingChbx ->setText(tr("Execute command on incoming messages :"));
  m_displayMsgFmtLab        ->setText(tr("Messages display format : "));
  m_msgsHistoryIntervalLab  ->setText(tr("Messages history request interval : "));
  m_msgsHistoryNumLab       ->setText(tr("Maximum number of messages in history answer : "));
  m_ulRefreshIntervalLab    ->setText(tr("Refresh Interval : "));
  m_ulDeepRefreshIntervalLab->setText(tr("Deep Refresh Interval : "));

  m_networkIfLab            ->setText(tr("Choose Network Settings :"));

  m_msgsHistoryIntervalSpbx  ->setSuffix(tr(" ms", "milliseconds"));
  m_ulRefreshIntervalSpbx    ->setSuffix(tr(" s", "seconds"));
  m_ulDeepRefreshIntervalSpbx->setSuffix(tr(" s", "seconds"));

  setWindowTitle(tr("Preferences"));
}

void PreferencesDlg::createWidgets()
{
  m_colorsGrbx                = new QGroupBox(this);
  m_networkGrbx               = new QGroupBox(this);
  m_miscGrbx                  = new QGroupBox(this);
  m_nowListeningGrbx          = new QGroupBox(this);
  m_smilesThemesGrbx          = new QGroupBox(this);
  m_usersListGrbx             = new QGroupBox(this);
  m_styleSheetsGrbx           = new QGroupBox(this);

  m_sectionsChooser           = new QListWidget(this);
  m_smilesThemeChooser        = new QListWidget(this);
  m_styleSheetsChooser        = new QListWidget(this);

  m_networkSection            = new QListWidgetItem();
  m_messagesSection           = new QListWidgetItem();
  m_smilesSection             = new QListWidgetItem();
  m_miscSection               = new QListWidgetItem();
  m_profileSection            = new QListWidgetItem();
  m_styleSheetsSection        = new QListWidgetItem();

  m_sections                  = new QStackedWidget(this);

  m_myColor                   = new ColorLabel(this, Globals::prefs()->myColor());
  m_sysColor                  = new ColorLabel(this, Globals::prefs()->sysColor());
  m_baseColor                 = new ColorLabel(this, Globals::prefs()->baseColor());
  m_okBtn                     = new QPushButton(this);
  m_cancelBtn                 = new QPushButton(this);
  m_applyBtn                  = new QPushButton(this);
  m_ipEdit                    = new QLineEdit(this);
  m_broadcastEdit             = new QLineEdit(this);
  m_portOutSpbx               = new QSpinBox (this);
  m_portInSpbx                = new QSpinBox (this);
  m_colorWholeMsgChbx         = new QCheckBox(this);
  m_colorWholeSysMsgChbx      = new QCheckBox(this);
  m_activateOnMsgInChbx       = new QCheckBox(this);
  (m_soundOnMsgInChbx         = new QCheckBox(this))->hide();
  m_nlWithMessageChbx         = new QCheckBox(this);
  m_nlInStatusChbx            = new QCheckBox(this);

  m_portOutLab                = new QLabel(this);
  m_portInLab                 = new QLabel(this);

  m_myColorLab                = new QLabel(this);
  m_sysColorLab               = new QLabel(this);
  m_baseColorLab              = new QLabel(this);
  m_ipLab                     = new QLabel(this);
  m_broadcastLab              = new QLabel(this);
  m_nlFormatLab               = new QLabel(this);
  m_displayMsgFmtLab          = new QLabel(this);
  m_msgsHistoryIntervalLab    = new QLabel(this);
  m_msgsHistoryNumLab         = new QLabel(this);
  m_ulRefreshIntervalLab      = new QLabel(this);
  m_ulDeepRefreshIntervalLab  = new QLabel(this);
  m_networkIfLab              = new QLabel(this);

  m_commandOnIncomingEdit     = new QLineEdit(this);
  m_isCommandOnIncomingChbx   = new QCheckBox(this);

  m_nlFormatEdit              = new QLineEdit(this);
  m_displayMsgFmtEdit         = new QLineEdit(this);

  m_msgsHistoryIntervalSpbx   = new QSpinBox(this);
  m_msgsHistoryNumSpbx        = new QSpinBox(this);
  m_ulRefreshIntervalSpbx     = new QSpinBox(this);
  m_ulDeepRefreshIntervalSpbx = new QSpinBox(this);

  m_networkIfCmbx             = new QComboBox(this);

  m_sectionsChooser->insertItem(Network    , m_networkSection);
  m_sectionsChooser->insertItem(Messages   , m_messagesSection);
  m_sectionsChooser->insertItem(Smiles     , m_smilesSection);
  m_sectionsChooser->insertItem(Misc       , m_miscSection);
  m_sectionsChooser->insertItem(StyleSheets, m_styleSheetsSection);

  m_msgsHistoryNumSpbx       ->setRange(-1, 100000);
  m_msgsHistoryIntervalSpbx  ->setRange(0, 100000);
  m_ulRefreshIntervalSpbx    ->setRange(0, 100000);
  m_ulDeepRefreshIntervalSpbx->setRange(0, 100000);
}

void PreferencesDlg::setupConnections()
{
  connect(m_sectionsChooser, SIGNAL(currentRowChanged(int)), this, SLOT(slot_activateSection(int)));

  connect(m_myColor  , SIGNAL(clicked()), this, SLOT(slot_chooseColor()));
  connect(m_sysColor , SIGNAL(clicked()), this, SLOT(slot_chooseColor()));
  connect(m_baseColor, SIGNAL(clicked()), this, SLOT(slot_chooseColor()));
  connect(m_okBtn    , SIGNAL(clicked()), this, SLOT(slot_setPrefs()));
  connect(m_okBtn    , SIGNAL(clicked()), this, SLOT(slot_accept  ()));
  connect(m_applyBtn , SIGNAL(clicked()), this, SLOT(slot_setPrefs()));
  connect(m_cancelBtn, SIGNAL(clicked()), this, SLOT(slot_cancel  ()));

  connect(m_colorWholeSysMsgChbx, SIGNAL(stateChanged(int)), this, SLOT(slot_setSysMsgColorMode(int)));
  connect(m_colorWholeMsgChbx   , SIGNAL(stateChanged(int)), this, SLOT(slot_setMsgColorMode   (int)));
  connect(m_activateOnMsgInChbx , SIGNAL(stateChanged(int)), this, SLOT(slot_setActivateOnMsgIn(int)));
  connect(m_soundOnMsgInChbx    , SIGNAL(stateChanged(int)), this, SLOT(slot_setSoundOnMsgIn   (int)));
  connect(m_nlInStatusChbx      , SIGNAL(stateChanged(int)), this, SLOT(slot_setNlInStatus     (int)));
  connect(m_nlWithMessageChbx   , SIGNAL(stateChanged(int)), this, SLOT(slot_setNlWithMessage  (int)));

  connect(m_portInSpbx               , SIGNAL(valueChanged(int)), this, SLOT(slot_edited()));
  connect(m_portOutSpbx              , SIGNAL(valueChanged(int)), this, SLOT(slot_edited()));
  connect(m_msgsHistoryIntervalSpbx  , SIGNAL(valueChanged(int)), this, SLOT(slot_edited()));
  connect(m_msgsHistoryNumSpbx       , SIGNAL(valueChanged(int)), this, SLOT(slot_edited()));
  connect(m_ulRefreshIntervalSpbx    , SIGNAL(valueChanged(int)), this, SLOT(slot_edited()));
  connect(m_ulDeepRefreshIntervalSpbx, SIGNAL(valueChanged(int)), this, SLOT(slot_edited()));

  connect(m_ipEdit               , SIGNAL(textChanged (QString)), this, SLOT(slot_validateIp(QString)));
  connect(m_broadcastEdit        , SIGNAL(textChanged (QString)), this, SLOT(slot_validateIp(QString)));
  connect(m_commandOnIncomingEdit, SIGNAL(textChanged (QString)), this, SLOT(slot_edited()));
  connect(m_nlFormatEdit         , SIGNAL(textChanged (QString)), this, SLOT(slot_edited()));
  connect(m_displayMsgFmtEdit    , SIGNAL(textChanged (QString)), this, SLOT(slot_edited()));

  if(!Globals::prefs()->allowDiffPorts())
    connect(m_portInSpbx, SIGNAL(valueChanged(int)), m_portOutSpbx, SLOT(setValue(int)));

  connect(m_networkIfCmbx, SIGNAL(activated(int)), this, SLOT(setNetworkIf(int)));

  connect(m_isCommandOnIncomingChbx, SIGNAL(stateChanged(int)), this, SLOT(slot_setExecuteCommandMode(int)));
}

void PreferencesDlg::setupSections()
{
  QList<QWidget*> list;

  m_sectionsChooser->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
  m_sectionsChooser->setFixedWidth(150);

  m_sections->insertWidget(Network, setupSectionLayout(m_networkGrbx));

  list.append(m_colorsGrbx);
  list.append(m_miscGrbx);
  list.append(m_displayMsgFmtLab);
  list.append(m_displayMsgFmtEdit);
  list.append(m_msgsHistoryIntervalLab);
  list.append(m_msgsHistoryIntervalSpbx);
  list.append(m_msgsHistoryNumLab);
  list.append(m_msgsHistoryNumSpbx);

  m_sections->insertWidget(Messages, setupSectionLayout(list));

  connect(m_smilesThemeChooser, SIGNAL(currentRowChanged(int)), this, SLOT(slot_edited()));
  m_sections->insertWidget(Smiles, m_smilesThemesGrbx);

  list.clear();
#if defined(Q_OS_LINUX)
  list.append(m_nowListeningGrbx);
#else
  m_nowListeningGrbx->hide();
#endif
  list.append(m_usersListGrbx);

  m_sections->insertWidget(Misc, setupSectionLayout(list));

  connect(m_styleSheetsChooser, SIGNAL(currentRowChanged(int)), this, SLOT(slot_edited()));
  m_sections->insertWidget(StyleSheets, m_styleSheetsGrbx);

  m_ulDeepRefreshIntervalSpbx->hide();
  m_ulDeepRefreshIntervalLab->hide();
}

QWidget* PreferencesDlg::setupSectionLayout(QList<QWidget*> list)
{
  QWidget*     wgt  = new QWidget(0);
  QVBoxLayout* vbox = new QVBoxLayout(wgt);
  int i;

  vbox->setMargin(0);

  for(i = 0; i < list.size(); i++)
    vbox->addWidget(list[i]);

  vbox->addStretch();

  return wgt;
}

QWidget* PreferencesDlg::setupSectionLayout(QWidget* w, bool add_stretch)
{
  QWidget*     wgt  = new QWidget(0);
  QGridLayout* grid = new QGridLayout(wgt);

  grid->setMargin(0);

  grid->addWidget(w, 0, 0);

  if(add_stretch)
    grid->setRowStretch(1, 1);

  return wgt;
}

void PreferencesDlg::slot_activateSection(int idx)
{
  int idx_;
  int edited = m_edited;

  switch(idx)
  {
    case Network :
      reloadNetworkIfs();
      break;

    case Messages :
      break;

    case Smiles :
      reloadSmileThemeList();

      if((idx_ = m_smilesThemes.indexOf(Globals::prefs()->smilesThemePath())) >= 0)
        m_smilesThemeChooser->setCurrentRow(idx_);

      m_edited = edited;
      m_applyBtn->setEnabled(edited);

      break;

    case Misc :
      break;

    case StyleSheets :
      reloadStyleSheetsList();
      m_edited = edited;
      m_applyBtn->setEnabled(edited);

  }

  m_sections->setCurrentIndex(idx);
}

void PreferencesDlg::reloadSmileThemeList()
{
  QDir dir(Globals::prefs()->settingsDir() + "/smiles");

  m_smilesThemes.clear();
  m_smilesThemeChooser->clear();

  emoticonsThemesFromDir(dir);

#if defined(Q_OS_LINUX)
  dir.setPath(QDir::homePath() + "/.kde/share/emoticons");
  emoticonsThemesFromDir(dir);
#endif

#if defined(Q_OS_LINUX)
  dir.setPath(QCoreApplication::applicationDirPath());
  if(dir.cd("../share/emoticons"))
    emoticonsThemesFromDir(dir);
  if(dir.path() != "/usr/share/emoticons" && dir.cd("/usr/share/emoticons"))
    emoticonsThemesFromDir(dir);
#else
  dir.setPath(QCoreApplication::applicationDirPath() + "/emoticons");
  emoticonsThemesFromDir(dir);
#endif
}

void PreferencesDlg::reloadStyleSheetsList()
{
  QDir dir(Globals::prefs()->settingsDir() + "/stylesheets");
  QFile file;
  QStringList list;
  bool ok = 1;

  m_styleSheets.clear();
  m_styleSheetsChooser->clear();

  list = dir.entryList();
  for(int i = 0; i < list.size(); i++)
  {
    if(list[i] == "." || list[i] == ".." || !QFileInfo(dir.absolutePath() + "/" + list[i]).isFile())
      continue;

    m_styleSheetsChooser->addItem(list[i]);
    m_styleSheets.append(dir.absolutePath() + "/" + list[i]);
  }


#if defined(Q_OS_LINUX)
  dir.setPath(QCoreApplication::applicationDirPath());
  ok = dir.cd("../share/qchat/stylesheets");
#else
  dir.setPath(QCoreApplication::applicationDirPath() + "/stylesheets");
#endif

  if(ok)
  {
    list = dir.entryList();
    for(int i = 0; i < list.size(); i++)
    {
      if(list[i] == "." || list[i] == ".." || !QFileInfo(dir.absolutePath() + "/" + list[i]).isFile())
        continue;

      m_styleSheetsChooser->addItem(list[i]);
      m_styleSheets.append(dir.absolutePath() + "/" + list[i]);
    }
  }
}

void PreferencesDlg::emoticonsThemesFromDir(QDir dir)
{
  QFile file;

  if(dir.exists())
    foreach(QString em_dir, dir.entryList())
    {
      if(em_dir == "." || em_dir == ".." || QFileInfo(dir.absolutePath() + "/" + em_dir).isFile() ||
         !QFile(dir.absolutePath() + "/" + em_dir + "/emoticons.xml").exists())
        continue;

      m_smilesThemeChooser->addItem(em_dir);
      m_smilesThemes.append(dir.absolutePath() + "/" + em_dir + "/");
    }
}

void PreferencesDlg::reloadNetworkIfs()
{
  QList<QNetworkInterface> all_i = QNetworkInterface::allInterfaces();

  m_networkIfCmbx->clear();

  m_networkIfCmbx->addItem(tr("Custom (not recommended)"));

  m_addressEntries.clear();

  foreach(QNetworkInterface i, all_i)
    foreach(QNetworkAddressEntry e, i.addressEntries())
    {
      if(e.broadcast().isNull())
        continue;

      m_addressEntries.append(e);
      m_networkIfCmbx->addItem(i.name());
    }

  for(int i = 0; i < m_addressEntries.size(); i++)
    if((m_addressEntries[i].ip() == QHostAddress(Globals::prefs()->ip())) &&
       (m_addressEntries[i].broadcast() == Globals::prefs()->broadcast()))
    {
      m_networkIfCmbx->setCurrentIndex(i + 1);
      m_ipEdit       ->setEnabled(false);
      m_broadcastEdit->setEnabled(false);
    }
}

void PreferencesDlg::setNetworkIf(int idx)
{
  m_ipEdit       ->setEnabled(!idx);
  m_broadcastEdit->setEnabled(!idx);

  if(!idx)
  {
    m_broadcastEdit->setText(QHostAddress(Globals::prefs()->customBroadcast()).toString());
    m_ipEdit->setText(QHostAddress((quint32)Globals::prefs()->customIp()).toString());
  }
  else
  {
    QNetworkAddressEntry e = m_addressEntries[idx - 1];
    m_broadcastEdit->setText(e.broadcast().toString());
    m_ipEdit->setText(e.ip().toString());
  }
}
