#include <new>
#include "tcpip.hpp"
#include "config.hpp"
#include "classdef.hpp"
#include <stdio.h>
#include <string.h>

extern "C" {
	#include <netdb.h>
	#include <netinet/in.h>
	#include "log.h"
}

using namespace std;

#define MAXLINE 1024
#define IP_QUEUE_MAXLEN_FN "/proc/sys/net/ipv4/ip_queue_maxlen"

extern volatile int log_level;

static int line_no;
static int scan_seq = 0;
static char pidfile[MAXPATH] = DEFAULT_PIDFILE;

static void chomp(char *line);
static void remove_comments(char *line);
static void remove_trailing_blanks(char *line);
static int get_port_range(char *str, char *proto, int *lo, int *hi);
static int if_check(char *str);

#ifdef WITH_IPQ
static int get_ipq_maxlen();
#endif

shaper_config::shaper_config(int opt_fw=OPT_FW_NONE)
{
	n_classes = 0;
	v_classes = 0;
	daemon = 1;
	catch_sighup = 0;
	divert_port = -1;
	packet_fw = opt_fw;
	fptr = 0;
}

shaper_config::~shaper_config()
{
	for( int i = 0 ; i<n_classes ; i++ ) delete v_classes[i];
	n_classes = 0;

	delete[] v_classes;
}

int shaper_config::add_class(classdef *new_class)
{
	classdef **new_vector;

	new_vector = new(nothrow) (classdef *)[n_classes+1];
	if( new_vector==0 ) {
		goto __err_new;
	}

	if( n_classes>0 ) {
		memcpy(new_vector, v_classes, n_classes*sizeof(classdef*));
		delete[] v_classes;
	}
	v_classes = new_vector;
	v_classes[n_classes] = new_class;
	new_class->set_id(n_classes);
	n_classes++;
	return 0;
	__err_new:
		return -1;
}

int shaper_config::check_classname(char *name)
{
	int result;

	for( int i = 0 ; i<n_classes ; i++ ) {
		result = strcmp(name, v_classes[i]->get_name());
		if( result==0 ) return 0;
	}
	return -1;
}

int shaper_config::check_lenders()
{
	int result;

	for( int i = 0 ; i<n_classes ; i++ ) {
		classdef *my_class = v_classes[i];
		list<char*>::iterator iter;

		for( iter = my_class->lenders.begin() ; iter!=my_class->lenders.end() ; iter++ ) {
			result = check_classname(*iter);
			if( result==-1 ) {
				log_info(LL_ERR, "can't find lender class "
					"[%s] for class [%s] {%s:%d}", 
					*iter, my_class->get_name(), 
					__FILE__, __LINE__);
				return -1;
			}
		}
	}
	return 0;
}

int shaper_config::check_config()
{
	int i, pckts;

	if( packet_fw==OPT_FW_NONE ) {
		log_info(LL_ALERT, "error: no packet forwarding mechanism "
			"specified {%s:%d}", __FILE__, __LINE__);
		return -1;
	}

	if( packet_fw==OPT_FW_DIVERT ) {
		if( divert_port==-1 ) {
			log_info(LL_ALERT, "error: no divert port {%s:%d}",
				__FILE__, __LINE__);
		return -1;
		}
	}

	pckts = 0;
	for( i = 0 ; i<n_classes ; i++ ) {
		int byte_lim, pckt_lim;

		classdef *pc = v_classes[i];
		if( pc->get_bw()<0 ) {
			log_info(LL_ALERT, "error: no bandwidth for class "
				"[%s] {%s:%d}", pc->get_name(), 
				__FILE__, __LINE__);
			return -1;
		}

		pc->get_queue_limits(&byte_lim, &pckt_lim);
		if( packet_fw==OPT_FW_IPQ && pckt_lim==0 )
			log_message(LL_ERR, 
				"warning: %s has no packet limit, "
				"consider setting one",
				pc->get_name());
		if( packet_fw==OPT_FW_DIVERT && byte_lim==0 )
			log_message(LL_ERR, 
				"warning: %s has no byte limit, "
				"consider setting one",
				pc->get_name());

		pckts += pckt_lim;
	}

#ifdef WITH_IPQ
	if( packet_fw==OPT_FW_IPQ ) {
		int sysctl_maxlen, result;

		result = sysctl_maxlen = get_ipq_maxlen();
		if( result<0 )
			return -1;
		if( pckts>=sysctl_maxlen )
			log_message(LL_ERR, 
				"warning: kernel's queue is too "
				"short, consider increasing %s",
				IP_QUEUE_MAXLEN_FN);
	}
#endif

	return 0;
}

char *shaper_config::get_pidfile()
{
	return pidfile;
}

classdef *shaper_config::get_classbyname(char *name)
{
	int result;

	for( int i = 0 ; i<n_classes ; i++ ) {
		result = strcmp(name, v_classes[i]->get_name());
		if( result==0 ) return v_classes[i];
	}
	return 0;
}

int shaper_config::bw_init()
{
	int i, result;

	result = check_lenders();
	if( result==-1 ) return -1;

	for( i = 0 ; i<n_classes ; i++ ) {
		classdef *my_class = v_classes[i];
		result = my_bw.add_class(my_class->get_id(), my_class->get_bw());
		if( result==-1 ) {
			log_info(LL_ERR, "can't add new class {%s:%d}",
				__FILE__, __LINE__);
			return -1;
		}
	}

	for( i = 0 ; i<n_classes ; i++ ) {
		classdef *my_class = v_classes[i];
		list<char*>::iterator iter;
		for( iter = my_class->lenders.begin() ; iter!=my_class->lenders.end() ; iter++ ) {
			classdef *lender_class = get_classbyname(*iter);
			result = my_bw.add_lender(my_class->get_id(), lender_class->get_id());
			if( result==-1 ) {
				log_info(LL_ERR, "can't add lender class "
					"[%s] to [%s] {%s:%d}", *iter, 
					my_class->get_name(), 
					__FILE__, __LINE__);
				return -1;
			}
		}

		my_class->my_sched.set_classid(my_class->get_id());
		my_class->my_sched.set_bwadm(&my_bw);
	}

	return 0;
}

int shaper_config::queues_init()
{
	int i, result;
	for( i = 0 ; i<n_classes ; i++ ) {
		result = v_classes[i]->queue_init();
		if( result==-1 ) return -1;
	}
	return 0;
}

int shaper_config::get_divert_port()
{
	return divert_port;
}

void shaper_config::set_divert_port(int port)
{
	divert_port = port;
}

void shaper_config::show_config()
{
	int i, j;
	char *str_fw;

	switch(packet_fw) {
		case OPT_FW_DIVERT:
			str_fw = "divert";
			break;
		case OPT_FW_IPQ:
			str_fw = "ipq";
			break;
		default:
			str_fw = "?";
			break;
	}

	log_info(LL_DEBUG1, "scan_seq = %d", scan_seq);
	log_info(LL_INFO, "daemon = %s", daemon ? "yes" : "no");
	log_info(LL_INFO, "pidfile = %s", pidfile);
	log_info(LL_DEBUG1, "catch sighup = %s", catch_sighup ? "yes" : "no");
	#ifdef WITH_DIVERT
	log_info(LL_INFO, "divert port = %d", divert_port);
	#endif //WITH_DIVERT
	log_info(LL_INFO, "packet forwarding = %s", str_fw);
	for( i = 0 ; i<n_classes ; i++ ) {
		classdef *my_class = v_classes[i];
		list<char*>::iterator iter;
		int bytes, pckts;

		my_class->get_queue_limits(&bytes, &pckts);
		log_info(LL_INFO, "class %s (%d) {", 
			my_class->get_name(), my_class->get_id());
		log_info(LL_INFO, "        bandwidth = %d byte/s", 
			my_class->get_bw());
		log_info(LL_INFO, "        queue limits = %d bytes, %d pckts", 
			bytes, pckts);
		if( packet_fw==OPT_FW_DIVERT ) {
			log_info(LL_INFO, "        divert reinjection = %s",
				my_class->divert_reinjection ? 
				"outbound" : "inbound");
		}
		for( iter = my_class->lenders.begin() ; 
		     iter!=my_class->lenders.end()    ; 
		     iter++ ) {
			log_info(LL_INFO, "        borrow from %s", *iter);
		}
		for( j = 0 ; j<my_class->n_classifiers ; j++ ) {
			char *str = my_class->v_classifiers[j]->log();
			log_info(LL_INFO, "        %s", str);
		}
		log_info(LL_INFO, "}");
	}
}

int shaper_config::get_line(char *line, const int bytes)
{
	int n;
	char *p = line;

	n = bytes;
	memset(p, 0, n);
	do {
		int len;
		char *p_eof = fgets(p, n, fptr);
		if (p_eof == 0)
			return +1;
		line_no++;
		if (line[bytes-2] != '\0') {
			log_message(LL_ERR, "buffer overflow in get_line()");
			return -1;
		}
		chomp(p);
		remove_comments(p);
		remove_trailing_blanks(p);
		len = strlen(p);
		if (len < 1 || p[len-1] != '\\')
			break;
		p += len - 1;
		n -= len - 1;
	} while(1);
	return 0;
}

int shaper_config::read_config(char *fn)
{
	int result;
	char ch, classid[MAXLINE], line[MAXLINE];

	fptr = fopen(fn, "r");
	if( fptr==0 ) {
		log_info(LL_ERR, "can't open %s", fn);
		return -1;
	}

	scan_seq++;

	line_no = 0;
	do {
		result = get_line(line, MAXLINE);
		if (result == -1)
			goto __err_ovfl;
		else if (result != 0)
			break;

		result = parse_opt_loglevel(line);
		if( result==-1 ) {
			goto __err_ovfl; // fix!!!
		} if( result==+1 ) {
			log_info(LL_INFO, "log_level set to %d", log_level);
			continue;
		}

		result = parse_opt_packet_fw(line);
		if( result==-1 ) {
			goto __err_ovfl; // fix!!!
		} if( result==+1 ) {
			continue;
		}

		result = parse_opt_daemon(line);
		if( result==-1 ) {
			goto __err_ovfl; // fix!!!
		} if( result==+1 ) {
			continue;
		}

		result = parse_opt_pidfile(line);
		if( result==-1 ) {
			goto __err_ovfl;
		} if( result==+1 ) {
			continue;
		}

		result = parse_opt_catch_sighup(line);
		if( result==-1 ) {
			goto __err_ovfl; // fix!!!
		} if( result==+1 ) {
			continue;
		}

		result = parse_divert_port(line);
		if( result==-1 ) {
			goto __err_ovfl; // fix!!!
		} if( result==+1 ) {
			continue;
		}

		result = is_class(line, classid);
		if( result==+1 ) {
			classdef *new_class = new(nothrow) classdef;
			if( new_class==0 ) {
				log_info(LL_ERR, "no memory for new class "
					"{%s:%d}", __FILE__, __LINE__);
				goto __err_ovfl; // fix !!!
			}

			result = parse_within_class(fptr, classid, new_class);
			if( result==-1 ) {
				log_info(LL_DEBUG1, "parse_within_class error "
					"{%s:%d}", __FILE__, __LINE__);
				delete new_class;
				goto __err_ovfl; // fix !!!
			}

			result = add_class(new_class);
			if( result==-1 ) {
				log_info(LL_ERR, "add_class() error {%s:%d}",
					__FILE__, __LINE__);
				delete new_class;
				goto __err_ovfl; // fix !!!
			}
			continue;
		} else if( result==-1 ) {
			log_info(LL_ERR, "parse error at line %d", 
				line_no, line);
			goto __err_ovfl; // fix !!!
		}

		result = sscanf(line, " %c ", &ch);
		if( result==1 ) {
			log_info(LL_ERR, "parse error at line %d [%s]", 
				line_no, line);
			goto __err_ovfl; // fix !!! handle it
		}
	} while( !::feof(fptr) );

	result = check_config();
	if( result!=0 ) {
		goto __err_ovfl;
	}

	result = bw_init();
	if( result==-1 ) {
		goto __err_ovfl;
	}

	result = queues_init();
	if( result==-1 ) {
		goto __err_ovfl;
	}

	show_config();

	fclose(fptr);
	return 0;
	__err_ovfl:
		fclose(fptr);
		log_info(LL_ERR, "parse of [%s] stopped at line %d", 
			fn, line_no);
		return -1;
}

int shaper_config::parse_opt_loglevel(char *line)
{
	int result, i;
	char str_level[64];

	result = sscanf(line, " log level = %63s ", str_level);
	if( result!=1 ) return 0;

	if( strcmp(str_level, "alert")==0 ) {
		log_level = LL_ALERT;
	} else if( strcmp(str_level, "error")==0 )   {
		log_level = LL_ERR;
	} else if( strcmp(str_level, "warning")==0 ) {
		log_level = LL_WARN;
	} else if( strcmp(str_level, "info")==0 )    {
		log_level = LL_INFO;
	} else if( strcmp(str_level, "debug1")==0 )  {
		log_level = LL_DEBUG1;
	} else if( strcmp(str_level, "debug2")==0 )  {
		log_level = LL_DEBUG2;
	} else if( sscanf(str_level, "%d", &i)==1 )  {
		log_level = i;
	} else {
		log_info(LL_ERR, "invalid log level {%s:%d}",
			__FILE__, __LINE__);
		return -1;
	}
	return +1;
}

int shaper_config::parse_opt_packet_fw(char *line)
{
	int result;
	char str_fw[64];

	result = sscanf(line, " packet forwarding = %63s ", str_fw);
	if( result!=1 ) return 0;

	if( strcmp(str_fw, "divert")==0 ) {
		#ifdef WITH_DIVERT
		if( packet_fw==OPT_FW_NONE ) {
			packet_fw = OPT_FW_DIVERT;
		} else if( packet_fw!=OPT_FW_DIVERT ) {
			log_info(LL_WARN, "warning: cannot change forwarding "
				"method in a reload {%s:%d}",
				__FILE__, __LINE__);
		}
		#else
		log_info(LL_ERR, "error: not compiled with divert sockets "
			"{%s:%d}", __FILE__, __LINE__);
		return -1;
		#endif //WITH_DIVERT
	} else if( strcmp(str_fw, "ipq")==0 ) {
		#ifdef WITH_IPQ
		if( packet_fw==OPT_FW_NONE ) {
			packet_fw = OPT_FW_IPQ;
		} else if( packet_fw!=OPT_FW_IPQ ) {
			log_info(LL_WARN, "warning: cannot change forwarding "
				"method in a reload {%s:%d}",
				__FILE__, __LINE__);
		}
		#else
		log_info(LL_ERR, "error: not compiled with ipq support "
			"{%s:%d}", __FILE__, __LINE__);
		return -1;
		#endif //WITH_IPQ
	} else {
		log_info(LL_ERR, "invalid packet forwarding spec [%s] {%s:%d}",
			str_fw, __FILE__, __LINE__);
		return -1;
	}
	return +1;
}

int shaper_config::parse_opt_pidfile(char *line)
{
	int result;
	char str_arg[1024];

	result = sscanf(line, " pid file = %1023s ", str_arg);
	if( result!=1 ) return 0;

	if( scan_seq==1 ) {
		result = snprintf(pidfile, MAXPATH, "%s", str_arg);
		if( result>=MAXPATH ) {
			log_info(LL_ALERT, "pidfile too large {%s:%d}",
				__FILE__, __LINE__);
			return -1;
		}
	} else {
		// ignore this directive on further scans
	}

	return +1;
}

int shaper_config::parse_opt_daemon(char *line)
{
	int result;
	char str[64];

	result = sscanf(line, " daemon = %63s ", str);
	if( result!=1 ) return 0;

	if( strcmp(str, "yes")==0 ) {
		daemon = 1;
	} else if( strcmp(str, "no")==0 ) {
		daemon = 0;
	} else {
		log_info(LL_ERR, "invalid daemon option (%s) {%s:%d}",
		str, __FILE__, __LINE__);
		return -1;
	}
	return +1;
}

int shaper_config::parse_opt_catch_sighup(char *line)
{
	int result;
	char str[64];

	result = sscanf(line, " catch sighup = %63s ", str);
	if( result!=1 ) return 0;

	if( strcmp(str, "yes")==0 ) {
		catch_sighup = 1;
	} else if( strcmp(str, "no")==0 ) {
		catch_sighup = 0;
	} else {
		log_info(LL_ERR, "invalid catch sighup (%s) {%s:%d}",
		str, __FILE__, __LINE__);
		return -1;
	}
	return +1;
}

int shaper_config::parse_divert_port(char *line)
{
	int result, p;

	result = sscanf(line, " divert port = %d ", &p);
	if( result!=1 ) return 0;

	if( p<=0 ) {
		log_info(LL_ALERT, "invalid divert port (%d) {%s:%d}",
			p, __FILE__, __LINE__);
		return -1;
	}

	#ifdef WITH_DIVERT
	set_divert_port(p);
	#else
	log_info(LL_WARN, "ignoring divert port value (not compiled with "
		"divert sockets support) {%s:%d}",
		__FILE__, __LINE__);
	#endif //WITH_DIVERT

	return +1;
}

//  1 -> yes
//  0 -> nope
// -1 -> parse error
int shaper_config::is_class(char *line, char *classid)
{
	int result;
	char ch1='\0', ch2='\0';

	result = sscanf(line, " class %s %c %c", classid, &ch1, &ch2);
	if( result==2 && ch1=='{' ) return +1;
	if( result==1 ) {
		do {
			char line[MAXLINE];

			result = get_line(line, MAXLINE);
			if( result!=0 ) return -1;

			result = sscanf(line, " %c %c", &ch1, &ch2);
			if( result<1 ) {
				continue;
			} else if( result==1 && ch1=='{' ) {
				return +1;
			} else {
				return -1;
			}
		} while(1);
	}
	return 0;
}

int shaper_config::parse_within_class(FILE *fp, char *name, classdef *my_class)
{
	int result;
	char ch, line[MAXLINE];

	result = my_class->set_name(name, strlen(name));
	if( result==-1 ) {
		return -1;
	}

	while(1) {
		result = get_line(line, MAXLINE);
		if (result == 1) {
			log_message(LL_ERR, "eof found within class");
			return -1;
		} else if (result != 0) {
			return -1;
		}

		result = parse_endofclass(line);
		if( result==0 ) {
			return 0;
		}

		result = parse_bandwidth(line, my_class);
		if( result==-1 ) {
			return -1;
		} else if( result==0 ) {
			continue;
		}

		result = parse_queue_limits(line, my_class);
		if( result==-1 ) {
			return -1;
		} else if( result==0 ) {
			continue;
		}

		result = parse_divert_reinjection(line, my_class);
		if( result==-1 ) {
			return -1;
		} else if( result==0 ) {
			continue;
		}

		result = parse_borrow_from(line, my_class);
		if( result==-1 ) {
			return -1;
		} else if( result==0 ) {
			continue;
		}

		result = parse_ipv4_classifier(line, my_class);
		if( result==-1 ) {
			return -1;
		} else if( result==0 ) {
			continue;
		}
#ifdef WITH_IPQ
		result = parse_nf_classifier(line, my_class);
		if( result==-1 ) {
			return -1;
		} else if( result==0 ) {
			continue;
		}
#endif
		result = sscanf(line, " %c", &ch);
		if( result==1 ) return -1;
	}
	return -1;
}

//  0 -> ok
// +1 -> not a "bandwidth=..." entry
// -1 -> error
int shaper_config::parse_bandwidth(char *line, class classdef *cur_class)
{
	int result;
	char mult[16];
	float bw = -1;

	memset(mult, 0, sizeof(mult));
	result = sscanf(line, " bandwidth = %f %15s", &bw, mult);
	if( result!=2 ) return +1;
	if( bw<0 ) {
		log_info(LL_ERR, "negative bw values arn't allowed {%s:%d}",
			__FILE__, __LINE__);
		return -1;
	}
	if( strcmp(mult, "mbyte/s")==0 ) {
		bw *= 1024*1024;
	} else if( strcmp(mult, "mbit/s")==0 )  {
		bw *= 1024*1024/8;
	} else if( strcmp(mult, "kbyte/s")==0 ) {
		bw *= 1024;
	} else if( strcmp(mult, "kbit/s")==0 )  {
		bw *= 1024/8;
	} else if( strcmp(mult, "byte/s")==0 )  {
		bw *= 1;
	} else if( strcmp(mult, "bit/s")==0 )   {
		bw /= 8;
	} else {
		return -1;
	}
	cur_class->set_bw((int)(bw));
	return 0;
}

//  0 -> ok
// +1 -> not a "ipv4 classifier ..." entry
// -1 -> error
// does dns checks -> should run in a different thread
int shaper_config::parse_ipv4_classifier(char *line, classdef *cur_class)
{
	int i, j, result, retval;
	ipv4_classifier_parser p;

	result = p.parse(line);
	if (result != 0) {
		retval = result;
		goto _err_parse;
	}

	for( i = 0 ; i<p.n_saddr ; i++ )
	for( j = 0 ; j<p.n_daddr ; j++ ) {
		ipv4_packet_classifier *pc;

		pc = new(nothrow) ipv4_packet_classifier;
		if (pc == 0) {
			log_message(LL_ERR, "no memory for new classifier");
			retval = -1;
			goto _err_new_classifier;
		}

		result = setup_ipv4_classifier(pc, &p, i, j);
		if (result != 0) {
			delete pc;
			retval = -1;
			goto _err_new_classifier;
		}

		result = cur_class->add_packet_classifier(pc);
		if (result == -1) {
			log_message(LL_ERR, "can't insert new classifier");
			delete pc;
			retval = -1;
			goto _err_new_classifier;
		}
	}

	return 0;
	_err_new_classifier:
	_err_parse:
		return retval;
}

int shaper_config::setup_ipv4_classifier(
	ipv4_packet_classifier *pc,
	ipv4_classifier_parser *p,
	int i, int j)
{
	int result;
	struct ipv4_classifier_parser::spt_list *pn;
	char saddr_str[64], daddr_str[64];

	snprintf(saddr_str, 63, "%s", 
		inet_ntoa(p->v_saddr[i]));
	snprintf(daddr_str, 63, "%s", 
		inet_ntoa(p->v_daddr[j]));

	pc->set_proto(p->proto);

	result = pc->set_saddr(saddr_str, p->smask);
	if (result != 0) {
		log_message(LL_ERR, "error setting src adress");
		return -1;
	}

	result = pc->set_daddr(daddr_str, p->dmask);
	if( result!=0 ) {
		log_message(LL_ERR, "error setting dst adress");
		return -1;
	}

	for( pn = p->spt ; pn ; pn = pn->next )
		pc->add_src_port(pn->lo, pn->hi);
	for( pn = p->dpt ; pn ; pn = pn->next )
		pc->add_dst_port(pn->lo, pn->hi);

	pc->set_user_prio(p->uprio);

	if (p->inp_if[0] != '\0') {
		result = pc->set_inp_if(p->inp_if);
		if (result != 0) {
			log_message(LL_ERR, "error setting input "
				"interface (%s) on classifier",
				p->inp_if);
			return -1;
		}
	}

	if (p->out_if[0] != '\0') {
		result = pc->set_out_if(p->out_if);
		if (result != 0) {
			log_message(LL_ERR, "error setting output "
				"interface (%s) on classifier ",
				p->out_if);
			return -1;
		}
	}

	return 0;
}

ipv4_classifier_parser::ipv4_classifier_parser()
{
	uprio = 0;
	proto = __IP_HDR_PROTO_ALL__;
	sprintf(saddr, "0.0.0.0");
	sprintf(daddr, "0.0.0.0");
	sprintf(smask, "255.255.255.255");
	sprintf(dmask, "255.255.255.255");
	memset(inp_if, 0, sizeof(inp_if));
	memset(out_if, 0, sizeof(out_if));
	spt = dpt = 0;
	n_saddr = n_daddr = 0;
	v_saddr = v_daddr = 0;
}

ipv4_classifier_parser::~ipv4_classifier_parser()
{
	struct spt_list *p;
 
	if (v_saddr) {
		delete[] v_saddr;
		v_saddr = 0;
	}
	if (v_daddr) {
		delete[] v_daddr;
		v_daddr = 0;
	}
	while ((p = spt)) {
		spt = spt->next;
		delete p;
	}
	while ((p = dpt)) {
		dpt = dpt->next;
		delete p;
	}
}

int ipv4_classifier_parser::setup()
{
	int result;

	result = resolv(saddr, &v_saddr, &n_saddr);
	if (result != 0) 
		return -1;

	result = resolv(daddr, &v_daddr, &n_daddr);
	if (result != 0) 
		return -1;

	return 0;
}

int ipv4_classifier_parser::resolv(
	char *host, 
	struct in_addr **v_addr, 
	int *n_addr)
{
	int i, n;
	struct hostent *he;
	struct in_addr *v;

	he = gethostbyname(host);
	if (he == 0) {
		log_message(LL_ERR, "gethostbyname() error (%s)",
			hstrerror(h_errno));
		goto _err_gethost;
	}

	if (he->h_addrtype != AF_INET || 
	    he->h_length != 4         || 
	    !he->h_addr_list[0])       {
		log_message(LL_ERR, "cannot resolve %s",
			host);
		goto _err_addrtype;
	}

	for( n = 0 ; he->h_addr_list[n] ; n++ ) { }
	v = new(nothrow) in_addr[n];
	if (v == 0) {
		log_message(LL_ERR, "out of memory");
		goto _err_new;
	}

	for( i = 0 ; i<n ; i++ )
		memcpy(v+i, he->h_addr_list[i], 4);

	*n_addr = n;
	*v_addr = v;

	return 0;
	_err_new:
	_err_addrtype:
	_err_gethost:
		return -1;
}

//  0 -> ok
// -1 -> error
static int if_check(char *str)
{
	int n;
	char *pc;

	assert(str!=0);

	n = strlen(str);
	if( n<2 ) return -1;

	pc = strstr(str, "+");
	if( pc && pc[1]!='\0' ) return -1;

	return 0;
}

int ipv4_classifier_parser::parse(char *line)
{
	char ch;
	int n, result;

	result = sscanf(line, " ipv4 classifier%n %c", 
		&n, &ch);
	if (result != 1)
		return +1;

	if (parse_proto(line)  != 0 ||
	    parse_sport(line)  != 0 ||
	    parse_dport(line)  != 0 ||
	    parse_saddr(line)  != 0 ||
	    parse_daddr(line)  != 0 ||
	    parse_prio(line)   != 0 ||
	    parse_inp_if(line) != 0 ||
	    parse_out_if(line) != 0)
		return -1;

	result = setup();
	if (result != 0)
		return -1;

	return 0;
}

int ipv4_classifier_parser::parse_proto(char *line)
{
	char *opt_str;
	int n, result, retval;

	if (strstr(line, "sport") && !strstr(line, "proto") || 
	    strstr(line, "dport") && !strstr(line, "proto")) {
		log_message(LL_ERR, "unknown protocol (proto=udp/tcp/icmp)");
		retval = -1;
		goto _err_port_proto;
	}

	opt_str = strstr(line, "proto");
	if (opt_str) {
		char proto_str[64];
		result = sscanf(opt_str, "proto = %63s %n", 
			proto_str, &n);
		if (result < 1) {
			retval = -1;
			goto _err_proto;
		}

		if (strcmp(proto_str, "icmp") == 0) {
			proto = __IP_HDR_PROTO_ICMP__;
		} else if (strcmp(proto_str, "udp") == 0) {
			proto = __IP_HDR_PROTO_UDP__;
		} else if (strcmp(proto_str, "tcp") == 0) {
			proto = __IP_HDR_PROTO_TCP__;
		} else if (strcmp(proto_str, "all") == 0) {
			proto = __IP_HDR_PROTO_ALL__;
		} else {
			retval = -1;
			goto _err_proto;
		}

		memset(opt_str, ' ', n);
	}

	return 0;
	_err_proto:
	_err_port_proto:
		return retval;
}

int ipv4_classifier_parser::parse_saddr(char *line)
{
	char *opt_str;
	int n, n1, n2;
	int result, retval;

	opt_str = strstr(line, "saddr");
	if (opt_str) {
		result = n = sscanf(opt_str, "saddr = %1023[^ /]%n / %1023s%n",
			saddr, &n1, smask, &n2);
		if (result < 1) {
			retval = -1;
			goto _err_saddr;
		}

		memset(opt_str, ' ', n==1 ? n1 : n2);
	} else {
		// match everything
		sprintf(saddr, "0.0.0.0");
		sprintf(smask, "0.0.0.0");
	}

	return 0;
	_err_saddr:
		return retval;
}

int ipv4_classifier_parser::parse_daddr(char *line)
{
	char *opt_str;
	int n, n1, n2;
	int result, retval;

	opt_str = strstr(line, "daddr");
	if (opt_str) {
		result = n = sscanf(opt_str, "daddr = %1023[^ /]%n / %1023s%n",
			daddr, &n1, dmask, &n2);
		if (result < 1) {
			retval = -1;
			goto _err_daddr;
		}

		memset(opt_str, ' ', n==1 ? n1 : n2);
	} else {
		// match everything
		sprintf(daddr, "0.0.0.0");
		sprintf(dmask, "0.0.0.0");
	}

	return 0;
	_err_daddr:
		return retval;
}

int ipv4_classifier_parser::parse_sport(char *line)
{
	int nb, n,result, retval;
	char *opt_str;

	nb = 0;
	opt_str = strstr(line, "sport");
	if (opt_str) {
		int lo, hi;
		char *sub, field[32];
	
		result = sscanf(opt_str, "sport = %31[^\n, ]%n", 
			field, &n);
		if (result != 1 || n >= 31) {
			retval = -1;
			goto _err_sport;
		}
		nb += n;
		sub = opt_str + n;
		do {
			char *str_proto = 0;

			if (strcmp(field, "all") == 0) {
				lo = 0;
				hi = (1<<16) - 1;
				goto _add; // shhhh ... :-)
			}

			if (proto == __IP_HDR_PROTO_TCP__)
				str_proto = "tcp";
			else if (proto == __IP_HDR_PROTO_UDP__)
				str_proto = "udp";

			result = get_port_range(field, str_proto, &lo, &hi);
			if (result != 0) {
				retval = -1;
				goto _err_sport;
			} else if (lo > hi) {
				log_message(LL_ERR, "error: empty sport range");
				retval = -1;
				goto _err_sport;
			}

_add:			struct spt_list *pn;
			pn = new spt_list;
			if (pn == 0) {
				log_message(LL_ERR, "no memory :-(");
				retval = -1;
				goto _err_sport;
			}
			pn->lo = lo;
			pn->hi = hi;
			pn->next = spt;
			spt = pn;

			sub = strstr(sub, ",");
			if (sub == 0)
				break;
			sub++;
			result = sscanf(sub, " %31[^\n, ]%n",
				field, &n);
			if (result != 1) {
				retval = -1;
				goto _err_sport;
			}
			nb += n + 1;
		} while(1);

		memset(opt_str, ' ', nb);
	}

	return 0;
	_err_sport:
		return retval;
}

int ipv4_classifier_parser::parse_dport(char *line)
{
	int nb, n,result, retval;
	char *opt_str;

	nb = 0;
	opt_str = strstr(line, "dport");
	if (opt_str) {
		int lo, hi;
		char *sub, field[32];
	
		result = sscanf(opt_str, "dport = %31[^\n, ]%n", 
			field, &n);
		if (result != 1 || n >= 31) {
			retval = -1;
			goto _err_dport;
		}
		nb += n;
		sub = opt_str + n;
		do {
			char *str_proto = 0;

			if (strcmp(field, "all") == 0) {
				lo = 0;
				hi = (1<<16) - 1;
				goto _add_dpt; // shhhh ... :-)
			}

			if (proto == __IP_HDR_PROTO_TCP__)
				str_proto = "tcp";
			else if (proto == __IP_HDR_PROTO_UDP__)
				str_proto = "udp";

			result = get_port_range(field, str_proto, &lo, &hi);
			if (result != 0) {
				retval = -1;
				goto _err_dport;
			} else if (lo > hi) {
				log_message(LL_ERR, "error: empty dport range");
				retval = -1;
				goto _err_dport;
			}

_add_dpt:		struct spt_list *pn;
			pn = new spt_list;
			if (pn == 0) {
				log_message(LL_ERR, "no memory :-(");
				retval = -1;
				goto _err_dport;
			}
			pn->lo = lo;
			pn->hi = hi;
			pn->next = dpt;
			dpt = pn;

			sub = strstr(sub, ",");
			if (sub == 0)
				break;
			sub++;
			result = sscanf(sub, " %31[^\n, ]%n",
				field, &n);
			if (result != 1) {
				retval = -1;
				goto _err_dport;
			}
			nb += n + 1;
		} while(1);

		memset(opt_str, ' ', nb);
	}

	return 0;
	_err_dport:
		return retval;
}

int ipv4_classifier_parser::parse_prio(char *line)
{
	int result, retval;
	char *opt_str;

	opt_str = strstr(line, "prio");
	if (opt_str) {
		int n, pr;
	
		result = sscanf(opt_str, "prio = %d%n", 
			&pr, &n);
		if (result!=1 || pr<0) {
			retval = -1;
			goto _err_prio;
		}
		uprio = pr;
		memset(opt_str, ' ', n);
	} else
		uprio = 0;

	return 0;
	_err_prio:
		return retval;
}

int ipv4_classifier_parser::parse_inp_if(char *line)
{
	int n, result, retval;
	char *opt_str;

	opt_str = strstr(line, "inp_if");
	if (opt_str) {
		char buf[64];

		result = sscanf(opt_str, "inp_if = %63s%n", 
			buf, &n);
		if (result != 1) {
			log_message(LL_ERR, "error: no (input) interface "
				"name supplied");
			retval = -1;
			goto _err_inp_if;
		}

		result = if_check(buf);
		if (result != 0) {
			log_message(LL_ERR, "error: invalid (input) "
				"interface name (%s)", buf);
			retval = -1;
			goto _err_inp_if;
		}

		// warning: sizeof(buf) should be less than sizeof(opt->inp_if)
		memcpy(inp_if, buf, 1+strlen(buf));

		memset(opt_str, ' ', n);
	}

	return 0;
	_err_inp_if:
		return retval;
}

int ipv4_classifier_parser::parse_out_if(char *line)
{
	int n, result, retval;
	char *opt_str;

	opt_str = strstr(line, "out_if");
	if (opt_str) {
		char buf[64];

		result = sscanf(opt_str, "out_if = %63s%n", 
			buf, &n);
		if (result != 1) {
			log_message(LL_ERR, "error: no (output) interface "
				"name supplied");
			retval = -1;
			goto _err_out_if;
		}

		result = if_check(buf);
		if (result != 0) {
			log_message(LL_ERR, "error: invalid (output) "
				"interface name (%s)", buf);
			retval = -1;
			goto _err_out_if;
		}

		// warning: sizeof(buf) should be less than sizeof(opt->inp_if)
		memcpy(out_if, buf, 1+strlen(buf));

		memset(opt_str, ' ', n);
	}

	return 0;
	_err_out_if:
		return retval;
}

static int get_port_range(char *str, char *str_proto, int *lo, int *hi)
{
	char c;
	int n, result;

	// skip leading whitespace
	result = sscanf(str, " %c%n",
		&c, &n);
	if (result != 1)
		goto _err_port;

	str += n-1;

	result = sscanf(str, "%d - %d", 
		lo, hi);
	if (result == 2 && lo > 0 && hi > 0)
		return 0;

	result = sscanf(str, "%d", 
		lo);
	if (result == 1 && lo > 0) {
		*hi = *lo;
		return 0;
	}

	if (str_proto) {
		struct servent *sptr;

		sptr = getservbyname(str, str_proto);
		if (sptr) {
			*lo = *hi = ntohs(sptr->s_port);
			return 0;
		}
	}

	_err_port:
		log_message(LL_ERR, 
			"invalid service (port) specification (%s)",
			str);
		return -1;
}

//  0 -> ok
// -1 -> error
// +1 -> not a "borrow from"
int shaper_config::parse_borrow_from(char *line, classdef *cur_class)
{
	int result;
	char ch, *pc, str[1024];

	result = sscanf(line, " borrow from %1023[^, ] %c ", str, &ch);
	if( result<1 ) return +1;
	if( result>1 && ch!=',' ) return -1;

	for( pc = line ; 1 ; ) {
		result = cur_class->add_lender(str);
		if( result==-1 ) return -1;

		pc = strstr(pc, ",");
		if( pc==0 ) {
			break;
		} else {
			pc++;
		}

		result = sscanf(pc, " %1023[^, ] %c ", str, &ch);
		if( result<1 ) return -1;
		if( result>1 && ch!=',' ) return -1;
	}

	return 0;
}

//  1
//  0 -> ok
// -1 -> parse error
int shaper_config::parse_queue_limits(char *line, classdef *cur_class)
{
	int result;
	char ch, type[64];
	int bytes = -1, pckts = -1;

	memset(type, 0, sizeof(type));
	result = sscanf(line, " queue limits = %d %63s %d packets %c",
		&bytes, type, &pckts, &ch);
	if( result<1 ) {
		return +1;
	} else if( result!=3 || bytes<0 || pckts<0 ) {
		return -1;
	}
	if( strcmp(type, "mbyte")==0 )        {
		bytes <<= 20;
	} else if( strcmp(type, "mb")==0 )    {
		bytes <<= 20;
	} else if( strcmp(type, "kbyte")==0 ) {
		bytes <<= 10;
	} else if( strcmp(type, "kb")==0 )    {
		bytes <<= 10;
	} else if( strcmp(type, "byte")==0 )  {
		bytes <<= 0;
	} else if( strcmp(type, "bytes")==0 ) {
		bytes <<= 0;
	} else {
		return -1;
	}
	cur_class->set_queue_limits(bytes, pckts);
	return 0;
}

//  1
//  0 -> ok
// -1 -> parse error
int shaper_config::parse_divert_reinjection(char *line, classdef *pc)
{
	int result;
	char ch, str[64];

	result = sscanf(line, " divert reinjection = %63s %c",
		str, &ch);
	if( result<1 ) {
		return +1;
	} else if( result>1 ) {
		return -1;
	} else if( result==1 ) {
		#ifdef WITH_DIVERT
		if( strcmp(str, "inbound")==0 ) {
			pc->divert_reinjection = 0;
		} else if( strcmp(str, "outbound")==0 ) {
			pc->divert_reinjection = 1;
		} else {
			return -1;
		}
		#else
		log_info(LL_WARN, "i'll ignore divert reijection "
			"(not compiled with divert sockets support) "
			"{%s:%d}",  __FILE__, __LINE__);
		#endif //WITH_DIVERT
	} else {
		return -1;
	}

	return 0;
}

int shaper_config::parse_endofclass(char *line)
{
	char ch1, ch2;
	int result;

	result = sscanf(line, " %c %c", &ch1, &ch2);
	if( result==1 && ch1=='}' ) return 0;
	return +1;
}

static void chomp(char *line)
{
	if( *line=='\0' ) return;
	for( ; line[1]!='\0' ; line++ ) { }
	line[0] = line[0]=='\n' ? '\0' : line[0];
}

static void remove_comments(char *line)
{
	char *p;
	for( p = line ; *p!='\0' ; p++ ) {
		if( *p=='#' ) {
			*p = '\0';
			return;
		}
	}
}

static void remove_trailing_blanks(char *p)
{
	int i, n;
	n = strlen(p);
	for( i = n-1 ; i>=0 ; i--) {
		if (p[i] != ' ' && p[i] != '\t' && p[i] != '\n')
			break;
		p[i] = '\0';
	}
}

#ifdef WITH_IPQ
static int get_ipq_maxlen()
{
	char ch;
	int result, sysctl_maxlen;
	FILE *fp = 0;

	fp = fopen(IP_QUEUE_MAXLEN_FN, "r");
	if( fp==0 ) {
		log_message(LL_ALERT, 
			"can't open %s (is ip_queue loaded? "
			"is /proc mounted? how are babies made? :)",
			IP_QUEUE_MAXLEN_FN);
		goto _err_cleanup;
	}

	result = fscanf(fp, "%d %c", 
		&sysctl_maxlen, &ch);
	if( result!=1 ) {
		log_message(LL_ALERT, "can't read %s",
			IP_QUEUE_MAXLEN_FN);
		goto _err_cleanup;
	} else if( sysctl_maxlen<0 ) {
		log_message(LL_ALERT, "can't parse %s",
			IP_QUEUE_MAXLEN_FN);
		goto _err_cleanup;
	}
 
	fclose(fp);
	return sysctl_maxlen;

	_err_cleanup:
		if( fp )
			fclose(fp);
		return -1;
}

nf_classifier_parser::nf_classifier_parser()
{
	mark = mask = 0;
}

nf_classifier_parser::~nf_classifier_parser()
{
}

int nf_classifier_parser::parse(char *line)
{
	char ch;
	int n, result;

	result = sscanf(line, " nf classifier%n %c", 
		&n, &ch);
	if (result != 1)
		return +1;

	if (parse_proto(line)  != 0 ||
	    parse_sport(line)  != 0 ||
	    parse_dport(line)  != 0 ||
	    parse_saddr(line)  != 0 ||
	    parse_daddr(line)  != 0 ||
	    parse_prio(line)   != 0 ||
	    parse_inp_if(line) != 0 ||
	    parse_out_if(line) != 0 ||
	    parse_fwmark(line) != 0)
		return -1;

	result = setup();
	if (result != 0)
		return -1;

	return 0;
}

int nf_classifier_parser::parse_fwmark(char *line)
{
	int result, retval;
	char *opt_str;

	opt_str = strstr(line, "fwmark");
	if (opt_str) {
		int n, n1, n2;
		char mark_str[32], mask_str[32];
	
		result = n = sscanf(opt_str, 
			"fwmark = %31[^/ ]%n / %31[^/ ]%n", 
			mark_str, &n1, mask_str, &n2);
		if (result < 1) {
			log_message(LL_ERR, "no fwmark argument(s)");
			retval = -1;
			goto _err_fwmark;
		}

		if (strstr(mark_str, "0x"))
			result = sscanf(mark_str, "0x%lx",
				&mark);
		else
			result = sscanf(mark_str, "%lu",
				&mark);
		if (result != 1) {
			log_message(LL_ERR, "can't read mark value");
			retval = -1;
			goto _err_fwmark;
		}

		if (n == 1) {
			mask = (unsigned long)(-1L);
			goto _nomask;
		}

		if (strstr(mask_str, "0x"))
			result = sscanf(mask_str, "0x%lx",
				&mask);
		else
			result = sscanf(mask_str, "%lu",
				&mask);
		if (result != 1) {
			log_message(LL_ERR, "can't read mask value");
			retval = -1;
			goto _err_fwmark;
		}
_nomask:
		mark &= mask;
		memset(opt_str, ' ', n==1 ? n1 : n2);
	}

	return 0;
	_err_fwmark:
		return retval;
}

//  0 -> ok
// +1 -> not a "nf classifier ..." entry
// -1 -> error
// warning: does dns checks
int shaper_config::parse_nf_classifier(char *line, classdef *cur_class)
{
	int i, j, result, retval;
	nf_classifier_parser p;

	result = p.parse(line);
	if (result != 0) {
		retval = result;
		goto _err_parse;
	}

	for( i = 0 ; i<p.n_saddr ; i++ )
	for( j = 0 ; j<p.n_daddr ; j++ ) {
		nf_packet_classifier *pc;

		pc = new(nothrow) nf_packet_classifier;
		if (pc == 0) {
			log_message(LL_ERR, "no memory for new classifier");
			retval = -1;
			goto _err_new_classifier;
		}

		result = setup_ipv4_classifier(pc, &p, i, j);
		if (result != 0) {
			delete pc;
			retval = -1;
			goto _err_new_classifier;
		}

		pc->fwmark = p.mark;
		pc->fwmask = p.mask;

		result = cur_class->add_packet_classifier(pc);
		if (result == -1) {
			log_message(LL_ERR, "can't insert new classifier");
			delete pc;
			retval = -1;
			goto _err_new_classifier;
		}
	}

	return 0;
	_err_new_classifier:
	_err_parse:
		return retval;
}

#endif
