#!/usr/bin/env python
#Copyright (C) 2009-2010 :
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
#This file is part of Shinken.
#
#Shinken is free software: you can redistribute it and/or modify
#it under the terms of the GNU Affero General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#Shinken is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU Affero General Public License for more details.
#
#You should have received a copy of the GNU Affero General Public License
#along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


#This class revolv Macro in commands by looking at the macros list
#in Class of elements. It give a propertie that call be callable or not.
#It not callable, it's a simple properties and remplace the macro with the value
#If callable, it's a method that is call for getting the value. for exemple, to
#get the number of service in a host, you call a method to get the
# len(host.services)

import re
import time

from shinken.borg import Borg


class MacroResolver(Borg):

    my_type = 'macroresolver'
    #Global macros
    macros = {
        'TOTALHOSTSUP':         'get_total_hosts_up',
        'TOTALHOSTSDOWN':       'get_total_hosts_down',
        'TOTALHOSTSUNREACHABLE': 'get_total_hosts_unreacheable',
        'TOTALHOSTSDOWNUNHANDLED': 'get_total_hosts_unhandled',
        'TOTALHOSTSUNREACHABLEUNHANDLED': 'get_total_hosts_unreacheable_unhandled',
        'TOTALHOSTPROBLEMS':    'get_total_host_problems',
        'TOTALHOSTPROBLEMSUNHANDLED': 'get_total_host_problems_unhandled',
        'TOTALSERVICESOK':      'get_total_service_ok',
        'TOTALSERVICESWARNING': 'get_total_services_warning',
        'TOTALSERVICESCRITICAL': 'get_total_services_critical',
        'TOTALSERVICESUNKNOWN': 'get_total_services_unknown',
        'TOTALSERVICESWARNINGUNHANDLED': 'get_total_services_warning_unhandled',
        'TOTALSERVICESCRITICALUNHANDLED': 'get_total_services_critical_unhandled',
        'TOTALSERVICESUNKNOWNUNHANDLED': 'get_total_services_unknown_unhandled',
        'TOTALSERVICEPROBLEMS': 'get_total_service_problems',
        'TOTALSERVICEPROBLEMSUNHANDLED': 'get_total_service_problems_unhandled',
        'LONGDATETIME':         'get_long_date_time',
        'SHORTDATETIME':        'get_short_date_time',
        'DATE':                 'get_date',
        'TIME':                 'get_time',
        'TIMET':                'get_timet',
        'PROCESSSTARTTIME':     'get_process_start_time',
        'EVENTSTARTTIME':       'get_events_start_time',
    }


    #This shall be call ONE TIME. It just put links for elements
    #by scheduler
    def init(self, conf):
        # For searching class and elements for ondemand
        #we need link to types
        self.conf = conf
        self.lists_on_demand = []
        self.hosts = conf.hosts
        #For special void host_name handling...
        self.host_class = self.hosts.inner_class
        self.lists_on_demand.append(self.hosts)
        self.services = conf.services
        self.contacts = conf.contacts
        self.lists_on_demand.append(self.contacts)
        self.hostgroups = conf.hostgroups
        self.lists_on_demand.append(self.hostgroups)
        self.commands = conf.commands
        self.servicegroups = conf.servicegroups
        self.lists_on_demand.append(self.servicegroups)
        self.contactgroups = conf.contactgroups
        self.lists_on_demand.append(self.contactgroups)
        self.illegal_macro_output_chars = conf.illegal_macro_output_chars
        self.output_macros = ['HOSTOUTPUT', 'HOSTPERFDATA', 'HOSTACKAUTHOR', 'HOSTACKCOMMENT', 'SERVICEOUTPUT', 'SERVICEPERFDATA', 'SERVICEACKAUTHOR', 'SERVICEACKCOMMENT']
        #Try cache :)
        #self.cache = {}


    #Return all macros of a string, so cut the $
    #And create a dict with it:
    #val : value, not set here
    #type : type of macro, like class one, or ARGN one
    def get_macros(self, s):
        #if s in self.cache:
        #    return self.cache[s]

        p = re.compile(r'(\$)')
        elts = p.split(s)
        macros = {}
        in_macro = False
        for elt in elts:
            if elt == '$':
                in_macro = not in_macro
            elif in_macro:
                macros[elt] = {'val' : '', 'type' : 'unknown'}

        #self.cache[s] = macros
        return macros


    #Get a value from a propertie of a element
    #Prop can ba a function or a propertie
    #So we call it or no
    def get_value_from_element(self, elt, prop):
        try:
            value = getattr(elt, prop)
            if callable(value):
                return unicode(value())
            else:
                return unicode(value)
        except AttributeError , exp:
            # Return no value
            return ''


    #For some macros, we need to delete unwanted caracters
    def delete_unwanted_caracters(self, s):
        for c in self.illegal_macro_output_chars:
            s = s.replace(c, '')
        return s


    #return a dict with all environement variable came from
    #the macros of the datas object
    def get_env_macros(self, data):
        env = {}

        clss = [d.__class__ for d in data]
        for o in data:
            for cls in clss:
                if o.__class__ == cls:
                    macros = cls.macros
                    for macro in macros:
#                        print "Macro in %s : %s" % (o.__class__, macro)
                        prop = macros[macro]
                        value = self.get_value_from_element(o, prop)
#                        print "Value: %s" % value
                        env['NAGIOS_'+macro] = value

        return env


    # This function will look at elements in data (and args if it filled)
    # to replace the macros in c_line with real value.
    def resolve_simple_macros_in_string(self, c_line, data, args=None):
        #Now we prepare the classes for looking at the class.macros
        data.append(self) #For getting global MACROS
        if hasattr(self, 'conf'):
            data.append(self.conf) # For USERN macros
        clss = [d.__class__ for d in data]

        #we should do some loops for nested macros
        #like $USER1$ hiding like a ninja in a $ARG2$ Macro. And if
        #$USER1$ is pointing to $USER34$ etc etc, we should loop
        #until we reach the botom. So the last loop is when we do
        #not still have macros :)
        still_got_macros = True
        nb_loop = 0
        while still_got_macros:
            nb_loop += 1
            #Ok, we want the macros in the command line
            macros = self.get_macros(c_line)

            #We can get out if we do not have macros this loop
            still_got_macros = (len(macros)!=0)
            #print "Still go macros:", still_got_macros

            #Put in the macros the type of macro for all macros
            self.get_type_of_macro(macros, clss)
            #Now we get values from elements
            for macro in macros:
                #If type ARGN, look at ARGN cutting
                if macros[macro]['type'] == 'ARGN' and args is not None:
                    macros[macro]['val'] = self.resolve_argn(macro, args)
                    macros[macro]['type'] = 'resolved'
                #If class, get value from properties
                if macros[macro]['type'] == 'class':
                    cls = macros[macro]['class']
                    for elt in data:
                        if elt is not None and elt.__class__ == cls:
                            prop = cls.macros[macro]
                            macros[macro]['val'] = self.get_value_from_element(elt, prop)
                            #Now check if we do not have a 'output' macro. If so, we must
                            #delete all special caracters that can be dangerous
                            if macro in self.output_macros:
                                macros[macro]['val'] = self.delete_unwanted_caracters(macros[macro]['val'])
                if macros[macro]['type'] == 'CUSTOM':
                    cls_type = macros[macro]['class']
                    macro_name = re.split('_'+cls_type, macro)[1].upper()
                    #Ok, we've got the macro like MAC_ADDRESS for _HOSTMAC_ADDRESS
                    #Now we get the element in data that have the type HOST
                    #and we check if it gots the custom value
                    for elt in data:
                        if elt is not None and elt.__class__.my_type.upper() == cls_type:
                            if '_'+macro_name in elt.customs:
                                macros[macro]['val'] = elt.customs['_'+macro_name]
                if macros[macro]['type'] == 'ONDEMAND':
                    macros[macro]['val'] = self.resolve_ondemand(macro, data)

            #We resolved all we can, now replace the macro in the command call
            for macro in macros:
                c_line = c_line.replace('$'+macro+'$', macros[macro]['val'])

            if nb_loop > 32: #too mouch loop, we exit
                still_got_macros = False

        #print "Retuning c_line", c_line.strip()
        return c_line.strip()


    #Resolve a command with macro by looking at data classes.macros
    #And get macro from item properties.
    def resolve_command(self, com, data):
        c_line = com.command.command_line
        return self.resolve_simple_macros_in_string(c_line, data, args=com.args)


    #For all Macros in macros, set the type by looking at the
    #MACRO name (ARGN? -> argn_type,
    #HOSTBLABLA -> class one and set Host in class)
    #_HOSTTOTO -> HOST CUSTOM MACRO TOTO
    #$SERVICESTATEID:srv-1:Load$ -> MACRO SERVICESTATEID of
    #the service Load of host srv-1
    def get_type_of_macro(self, macros, clss):
        for macro in macros:
            #ARGN Macros
            if re.match('ARG\d', macro):
                macros[macro]['type'] = 'ARGN'
                continue
            #USERN macros
            #are managed in the Config class, so no
            #need to look that here
            elif re.match('_HOST\w', macro):
                macros[macro]['type'] = 'CUSTOM'
                macros[macro]['class'] = 'HOST'
                continue
            elif re.match('_SERVICE\w', macro):
                macros[macro]['type'] = 'CUSTOM'
                macros[macro]['class'] = 'SERVICE'
                #value of macro : re.split('_HOST', '_HOSTMAC_ADDRESS')[1]
                continue
            elif re.match('_CONTACT\w', macro):
                macros[macro]['type'] = 'CUSTOM'
                macros[macro]['class'] = 'CONTACT'
                continue
            #On demand macro
            elif len(macro.split(':')) > 1:
                macros[macro]['type'] = 'ONDEMAND'
                continue
            #OK, classical macro...
            for cls in clss:
                if macro in cls.macros:
                    macros[macro]['type'] = 'class'
                    macros[macro]['class'] = cls
                    continue


    #Resolv MACROS for the ARGN
    def resolve_argn(self, macro, args):
        #first, get number of arg
        id = None
        r = re.search('ARG(?P<id>\d+)', macro)
        if r is not None:
            id = int(r.group('id')) - 1
            try:
                return args[id]
            except IndexError:
                return ''


    #Resolved on demande macro, quite hard on fact
    def resolve_ondemand(self, macro, data):
        #print "\nResolving macro", macro
        elts = macro.split(':')
        nb_parts = len(elts)
        macro_name = elts[0]
        #Len 3 == service, 2 = all others types...
        if nb_parts == 3:
            val = ''
            #print "Got a Service on demand asking...", elts
            (host_name, service_description) = (elts[1], elts[2])
            #host_name can be void, so it's the host in data
            #that is important. We use our self.host_class to
            #find the host in the data :)
            if host_name == '':
                for elt in data:
                    if elt is not None and elt.__class__ == self.host_class:
                        host_name = elt.host_name
            #Okn now we get service
            s = self.services.find_srv_by_name_and_hostname(host_name, service_description)
            if s is not None:
                cls = s.__class__
                prop = cls.macros[macro_name]
                val = self.get_value_from_element(s, prop)
                #print "Got val:", val
                return val
        #Ok, service was easy, now hard part
        else:
            val = ''
            elt_name = elts[1]
            #Special case : elt_name can be void
            #so it's the host where it apply
            if elt_name == '':
                for elt in data:
                    if elt is not None and elt.__class__ == self.host_class:
                        elt_name = elt.host_name
            for list in self.lists_on_demand:
                cls = list.inner_class
                #We search our type by look at the macro
                if macro_name in cls.macros:
                    prop = cls.macros[macro_name]
                    i = list.find_by_name(elt_name)
                    if i is not None:
                        val = self.get_value_from_element(i, prop)
            #print "Got val:", val
            return val
        return ''


    #Get Fri 15 May 11:42:39 CEST 2009
    def get_long_date_time(self):
        return time.strftime("%a %d %b %H:%M:%S %Z %Y", time.localtime())


    #Get 10-13-2000 00:30:28
    def get_short_date_time(self):
        return time.strftime("%d-%m-%Y %H:%M:%S", time.localtime())


    #Get 10-13-2000
    def get_date(self):
        return time.strftime("%d-%m-%Y", time.localtime())


    #Get 00:30:28
    def get_time(self):
        return time.strftime("%H:%M:%S", time.localtime())


    #Get epoch time
    def get_timet(self):
        return str(int(time.time()))


    def get_total_hosts_up(self):
        return len([h for h in self.hosts if h.state == 'UP'])

    def get_total_hosts_down(self):
        return len([h for h in self.hosts if h.state == 'DOWN'])

    def get_total_hosts_unreacheable(self):
        return len([h for h in self.hosts if h.state == 'UNREACHABLE'])

    #TODO
    def get_total_hosts_unreacheable_unhandled(self):
        return 0


    def get_total_host_problems(self):
        return len([h for h in self.hosts if h.is_problem])

    def get_total_host_problems_unhandled(self):
        return 0

    def get_total_service_ok(self):
        return len([s for s in self.services if s.state == 'OK'])

    def get_total_services_warning(self):
        return len([s for s in self.services if s.state == 'WARNING'])

    def get_total_services_critical(self):
        return len([s for s in self.services if s.state == 'CRITICAL'])

    def get_total_services_unknown(self):
        return len([s for s in self.services if s.state == 'UNKNOWN'])

    #TODO
    def get_total_services_warning_unhandled(self):
        return 0

    def get_total_services_critical_unhandled(self):
        return 0

    def get_total_services_unknown_unhandled(self):
        return 0

    def get_total_service_problems(self):
        return len([s for s in self.services if s.is_problem])

    def get_total_service_problems_unhandled(self):
        return 0

    def get_process_start_time(self):
        return 0

    def get_events_start_time(self):
        return 0
