/*
 * Copyright 2008 Klaus Triendl
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free 
 * Software Foundation, 51 Franklin Street, Fifth Floor, 
 * Boston, MA 02110-1301, USA.
*/

#include <string>
#include <iostream>
#include <sigc++/reference_wrapper.h>
#include <sigc++/functors/mem_fun.h>
#include <glibmm/thread.h>
#include <sigx/glib_auto_dispatchable.h>
#include <sigx/glib_threadable.h>
#include <sigx/request_f.h>


class string_wrapper
{
	std::string m_str;

public:
	string_wrapper(const char* s);
	string_wrapper(const string_wrapper& other);
	// implicit dtor is fine

	const std::string& str() const
	{
		return m_str;
	}

private:
	// not implemented
	string_wrapper& operator =(const string_wrapper& other);
};


class MyThread: public sigx::glib_threadable
{
public:
	MyThread();

	// async request
	sigx::request_f<const string_wrapper&> do_sg_async;
	// sync request
	void do_sg_sync(const string_wrapper& s) const;

protected:
	///	message handler for sync and async
	void on_do_sg(const string_wrapper s) const;
};



using namespace std;


string_wrapper::string_wrapper(const char* s): 
	m_str(s)
{
	cout << "ctor string_wrapper" << endl;
}

string_wrapper::string_wrapper(const string_wrapper& other): 
	m_str(other.m_str)
{
	cout << "copy ctor string_wrapper" << endl;
}


MyThread::MyThread(): 
	sigx::glib_threadable(), 
	// request api
	// note: use *this instead of this because this leads to ambiguous overloads for mem_fun
	do_sg_async(sigc::mem_fun(*this, &MyThread::on_do_sg))
{}

void MyThread::do_sg_sync(const string_wrapper& s) const
{
	// note: use *this instead of this because this leads to ambigous overloads for mem_fun
	sigx::open_sync_tunnel(sigc::mem_fun(*this, &MyThread::on_do_sg))(sigc::ref(s));
}

void MyThread::on_do_sg(const string_wrapper s) const
{
	cout << "got message: " << s.str() << endl;
}




int main(int argc, char* argv[])
{
	Glib::thread_init();

	MyThread thethread;
	thethread.run();
	// message is copied only once when MyThread::on_do_sg() is called in the context of thethread
	thethread.do_sg_sync("sync");
	cout << endl;
	// message is copied several times because it is stored inside the asynchronous tunnel_context before MyThread::on_do_sg() is called in the context of thethread
	thethread.do_sg_async("async");
	thethread.finish();

	cin.get();
	return 0;
}
