#ifndef _protocoltimer_h
#define _protocoltimer_h

#include <boost/date_time/posix_time/posix_time.hpp>


class TimerEventHandlerBase
{
public:
  virtual void operator()()=0;
};


template <class TClass>
class TimerEventHandler : public TimerEventHandlerBase
{
private:
  TClass localHandler;

public:
  TimerEventHandler(TClass handler) : localHandler(handler)
  {
  };

  virtual void operator()()
  {
    localHandler();
  };
};


class TheTimerObjectBase
{
protected:
  TimerEventHandlerBase * myHandlerBase;

public:
  virtual void startAlarmAt(const boost::posix_time::ptime & time, TimerEventHandlerBase * handler)=0;
  virtual void startAlarmAfter(const boost::posix_time::time_duration & expiry_time, TimerEventHandlerBase * handler)=0;
  virtual void stop()=0;
};


class ProtocolIO
{
public:
  virtual boost::posix_time::ptime currentTimeUTC() = 0;
  virtual TheTimerObjectBase * getTimer() = 0;
};


class ProtocolTimer
{
  // with templates  doing abstractation to  TimerEventHandlerBase
  //
  // using TheTimerObjectBase
  //           that can handle (without template, but based on object pointer) TimerEventHandlerBase
private:
  TimerEventHandlerBase * myHandlerBase;
  TheTimerObjectBase * theTimerObjectBase;

public:
  ProtocolTimer(ProtocolIO & protocolIO);
  ~ProtocolTimer();

  template <typename WaitHandler>
  void startAlarmAt(const boost::posix_time::ptime & time, WaitHandler handler)
  {
    if (myHandlerBase)
    {
      delete myHandlerBase;
      myHandlerBase=0;
    }
    myHandlerBase = new TimerEventHandler<WaitHandler>(handler);

    theTimerObjectBase->startAlarmAt(time, myHandlerBase);
  }

  template <typename WaitHandler>
  void startAlarmAfter(const boost::posix_time::time_duration & expiry_time, WaitHandler handler)
  {
    if (myHandlerBase)
    {
      delete myHandlerBase;
      myHandlerBase=0;
    }
    myHandlerBase = new TimerEventHandler<WaitHandler>(handler);

    theTimerObjectBase->startAlarmAfter(expiry_time, myHandlerBase);
  }

  void stop();
};


#endif
