/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <math.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_cblas.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_multifit_nlin.h>

#include "PSF.h"
#include "siril.h"
#include "callbacks.h"
#include "proto.h"

#define MAX_ITER_NO_ANGLE 10		//Number of iteration in the minimization with no angle
#define MAX_ITER_ANGLE 10			//Number of iteration in the minimization with angle

/* Returns the largest FWHM.
 * The optional output parameter roundness is the ratio between the two axis FWHM */
double get_fwhm(fits *fit, int layer, double *roundness) {
	Param_GAUSS *result = Get_Minimisation(fit, layer, &com.selection);
	if (result == NULL) {
		*roundness = 0.0;
		return 0.0;
	}
	double retval;
	retval = result->FWHMX;
	if (roundness) *roundness = result->FWHMY / result->FWHMX;
	free(result);
	return retval;
}

/* Computes the FWHM on data in the selection rectangle of image fit.
 * Selection rectangle is passed as third argument.
 * Return value is a structure, type Param_GAUSS, that has to be freed after use.
 */
Param_GAUSS *Get_Minimisation(fits *fit, int layer, rectangle *area){
	WORD *from;
	int stridefrom, i, j;
	Param_GAUSS *result;
	double bg = background(fit, layer, area);
	fprintf(stdout, "background: %g\n", bg);

	// create the matrix with values from the selected rectangle
	gsl_matrix *z = gsl_matrix_alloc (area->h, area->w);
	from = fit->pdata[layer] + (fit->ry - area->y - area->h) * fit->rx + area->x;
	stridefrom = fit->rx - area->w;
	
	for (i=0; i<area->h; i++){
		for (j=0; j<area->w; j++){
			gsl_matrix_set (z, i, j, (double)*from);
			from++;
		}
		from += stridefrom;
	}
	result = Global_minimisation(z, bg, layer, TRUE);
	if (result)
		update_units(fit, &result);
	gsl_matrix_free(z);
	return result;
}

/* This function is the global minimisation. Every call to the minimisation
 * must come over here. It will check if the difference between Sx and Sy is
 * larger than or equal to 0.01 pixel.
 * In this case, Dynamic PSF fits additional angle parameter wich is the
 * rotation angle of the X axis with respect to the centroid coordinates: so,
 * by design we set Sx>Sy.
 * If the difference is smaller OR if fit_Angle is equal to FALSE (in the case
 * of the star_finder algorithm), no angle parameter is fitted.
 * The function returns NULL if values look bizarre.
 */
Param_GAUSS *Global_minimisation(gsl_matrix* z, double bg, int layer, gboolean fit_Angle){
	Param_GAUSS *final_result;
		
	if ((final_result = minimiz_noAn(z, bg, layer))!=NULL){	// PROCESSING WITH NO ANGLE
		if (fit_Angle && fabs(final_result->Sx - final_result->Sy) >= 0.01) {
			Param_GAUSS *tmp_result;
			if ((tmp_result = minimiz_An(z, final_result))==NULL) { // PROCESSING WITH ANGLE
				free(final_result);
				return NULL;
			}
			free(final_result);
			final_result = tmp_result;
		}
		// Solve symmetry problem in order to have Sx>Sy in any case !!!
		if (final_result->Sy > final_result->Sx){
			swap_param(&final_result->Sx, &final_result->Sy);
			swap_param(&final_result->FWHMX, &final_result->FWHMY);
			if (fit_Angle && final_result->An!=0.){
				if (final_result->An>0) final_result->An=-90.+final_result->An;
				else final_result->An=90.+final_result->An;
			}
		}
		
		/* We normalize B, A and the RMSE for the output */
		WORD norm=get_normalized_value(&gfit);
		final_result->B  	= final_result->B/(double)norm;
		final_result->A  	= final_result->A/(double)norm;
		final_result->rmse 	= final_result->rmse/(double)norm;
		
//		/* We quicly test the result. If it is bad we return NULL */
		if (!isfinite(final_result->FWHMX) || !isfinite(final_result->FWHMY) ||
				final_result->FWHMX <= 0.0 || final_result->FWHMY <= 0.0 ) {
			free(final_result);
			final_result = NULL;
		}
	}
	/* When the first minimization gives NULL value, it's probably because the selected
	 * area was not big enough: we need more samples than parameters to fit the area */
	return final_result;
}

/* Compute initial values for the algorithm from data in the pixel value matrix */
gsl_vector* MaxInz(gsl_matrix* z, double bg){
	gsl_vector * MaxV = gsl_vector_alloc(5);
	double max = 0.0;
	size_t NbRows = z->size1;
	size_t NbCols = z->size2;
	size_t i, j;
	
	/* find maximum */
	max=gsl_matrix_max(z);
	gsl_matrix_max_index(z,&i,&j);
	gsl_vector_set(MaxV,0,j);
	gsl_vector_set(MaxV,1,i);
	gsl_vector_set(MaxV,2,max);

	size_t ii1=gsl_vector_get(MaxV,1);
	size_t ii2=gsl_vector_get(MaxV,1);
	size_t jj1=gsl_vector_get(MaxV,0);
	size_t jj2=gsl_vector_get(MaxV,0);
	size_t perm1=gsl_vector_get(MaxV,1);
	size_t perm2=gsl_vector_get(MaxV,0);
	
	while((2.*(gsl_matrix_get(z,ii1,perm2)-bg)>(gsl_matrix_get(z,perm1,perm2)-bg)) && (ii1 < NbRows - 1)){
		ii1++;
	}
	while((2.*(gsl_matrix_get(z,ii2,perm2)-bg)>(gsl_matrix_get(z,perm1,perm2)-bg)) && (ii2 > 0)){
		ii2--;
	}
	
	while((2.*(gsl_matrix_get(z,perm1,jj1)-bg)>(gsl_matrix_get(z,perm1,perm2)-bg)) && (jj1 < NbCols - 1)){
		jj1++;
	}
	while((2.*(gsl_matrix_get(z,perm1,jj2)-bg)>(gsl_matrix_get(z,perm1,perm2)-bg)) && (jj2 > 0)){
		jj2--;
	}
	gsl_vector_set(MaxV,0,(jj1+jj2+2)/2.);
	gsl_vector_set(MaxV,1,(ii1+ii2+2)/2.);
	gsl_vector_set(MaxV,3,pow(ii1-ii2,2)/4./log(2.));
	gsl_vector_set(MaxV,4,pow(jj1-jj2,2)/4./log(2.));

	return MaxV;
}

double Get_Magnitude(gsl_matrix* z, double B){
	double intensity=0.0, magnitude;
	size_t NbRows = z->size1;
	size_t NbCols = z->size2;
	size_t i, j;
	
	for (i=0;i<NbRows;i++){
		for (j=0; j<NbCols;j++)
			intensity+=gsl_matrix_get(z,i,j) - B;
	}
	magnitude = -2.5 * log10(intensity);
	return magnitude;
}

/* No angle */
int Gaussian_f (const gsl_vector * x, void *PSF_data, gsl_vector * f){
	size_t NbRows = ((struct PSF_data *) PSF_data)->NbRows;
	size_t NbCols = ((struct PSF_data *) PSF_data)->NbCols;
	size_t i, j;
	double *y     = ((struct PSF_data *) PSF_data)->y;
	double *sigma = ((struct PSF_data *) PSF_data)->sigma;
	double B  = gsl_vector_get(x,0);
	double A  = gsl_vector_get(x,1);
	double x0 = gsl_vector_get(x,2);
	double y0 = gsl_vector_get(x,3);
	double SX = gsl_vector_get(x,4);
	double SY = gsl_vector_get(x,5);
	double tmpx, tmpy, tmpc, sumres=0.;

	for (i = 0; i < NbRows; i++){
		for (j = 0; j < NbCols; j++){
			tmpx = j+1;
			tmpy = i+1;
			tmpc = exp(-(SQR(tmpx-x0)/SX + SQR(tmpy-y0)/SY));
			gsl_vector_set(f,NbCols*i+j,(B+A*tmpc - y[NbCols*i+j])/sigma[NbCols*i+j]);
			sumres+=(B+A*tmpc - y[NbCols*i+j])*(B+A*tmpc - y[NbCols*i+j]);
		}
	}
	((struct PSF_data *) PSF_data)->rmse = sqrt(sumres/(NbRows*NbCols));
	return GSL_SUCCESS;
}

int Gaussian_df (const gsl_vector * x, void *PSF_data, gsl_matrix * J){
	size_t NbRows = ((struct PSF_data *)PSF_data)->NbRows;
	size_t NbCols = ((struct PSF_data *)PSF_data)->NbCols;
	size_t i, j;
	double *sigma = ((struct PSF_data *) PSF_data)->sigma;
	double A  = gsl_vector_get(x,1);
	double x0 = gsl_vector_get(x,2);
	double y0 = gsl_vector_get(x,3);
	double SX = gsl_vector_get(x,4);
	double SY = gsl_vector_get(x,5);
	double tmpx,tmpy,tmpc,tmpd;

	for (i = 0; i < NbRows; i++){
		for (j = 0; j < NbCols; j++){
			tmpx = j+1;
			tmpy = i+1;
			double s = sigma[NbCols*i+j];
			tmpc = exp(-(SQR(tmpx-x0)/SX + SQR(tmpy-y0)/SY));
			gsl_matrix_set(J,NbCols*i+j,0,1./s);
			gsl_matrix_set(J,NbCols*i+j,1,tmpc/s);
			tmpd = A*tmpc*2*(tmpx-x0)/SX;
			gsl_matrix_set(J,NbCols*i+j,2,tmpd/s);
			tmpd = A*tmpc*2*(tmpy-y0)/SY;
			gsl_matrix_set(J,NbCols*i+j,3,tmpd/s);
			tmpd = A*tmpc*SQR(tmpx-x0)/SQR(SX);
			gsl_matrix_set(J,NbCols*i+j,4,tmpd/s);
			tmpd = A*tmpc*SQR(tmpy-y0)/SQR(SY);
			gsl_matrix_set(J,NbCols*i+j,5,tmpd/s);
		}
	}
	return GSL_SUCCESS;
}

int Gaussian_fdf (const gsl_vector * x, void *PSF_data, gsl_vector * f, gsl_matrix * J){
	Gaussian_f (x, PSF_data, f);
	Gaussian_df (x, PSF_data, J);
	return GSL_SUCCESS;
}

/* Angle */
int Gaussian_f_An (const gsl_vector * x, void *PSF_data, gsl_vector * f){
	size_t NbRows = ((struct PSF_data *) PSF_data)->NbRows;
	size_t NbCols = ((struct PSF_data *) PSF_data)->NbCols;
	size_t n      = ((struct PSF_data *) PSF_data)->n;
	size_t i, j;
	double *y     = ((struct PSF_data *) PSF_data)->y;
	double *sigma = ((struct PSF_data *) PSF_data)->sigma;
	double B  = gsl_vector_get(x,0);
	double A  = gsl_vector_get(x,1);
	double x0 = gsl_vector_get(x,2);
	double y0 = gsl_vector_get(x,3);
	double SX = gsl_vector_get(x,4);
	double SY = gsl_vector_get(x,5);
	double An = gsl_vector_get(x,6);
	double tmpx, tmpy, tmpc, sumres=0.;

	for (i = 0; i < NbRows; i++){
		for (j = 0; j < NbCols; j++){
			tmpx = cos(An)*(j+1-x0)-sin(An)*(i+1-y0)+x0;
			tmpy = sin(An)*(j+1-x0)+cos(An)*(i+1-y0)+y0;
			tmpc = exp(-(SQR(tmpx-x0)/SX + SQR(tmpy-y0)/SY));
			gsl_vector_set(f,NbCols*i+j,(B+A*tmpc - y[NbCols*i+j])/sigma[NbCols*i+j]);
			sumres+=(B+A*tmpc - y[NbCols*i+j])*(B+A*tmpc - y[NbCols*i+j]);
		}
	}
	((struct PSF_data *) PSF_data)->rmse = sqrt(sumres/n);
	return GSL_SUCCESS;
}

int Gaussian_df_An (const gsl_vector * x, void *PSF_data, gsl_matrix * J){
	size_t NbRows = ((struct PSF_data *) PSF_data)->NbRows;
	size_t NbCols = ((struct PSF_data *) PSF_data)->NbCols;
	size_t i, j;
	double *sigma = ((struct PSF_data *) PSF_data)->sigma;
	double A  = gsl_vector_get(x,1);
	double x0 = gsl_vector_get(x,2);
	double y0 = gsl_vector_get(x,3);
	double SX = gsl_vector_get(x,4);
	double SY = gsl_vector_get(x,5);
	double An = gsl_vector_get(x,6);
	double tmpx,tmpy,tmpc,tmpd,tmpderxr,tmpderyr;
	double s;

	for (i = 0; i < NbRows; i++){
		for (j = 0; j < NbCols; j++){
			tmpx = cos(An)*(j+1-x0)-sin(An)*(i+1-y0)+x0;
			tmpy = sin(An)*(j+1-x0)+cos(An)*(i+1-y0)+y0;
			s = sigma[NbCols*i+j];
			tmpc = exp(-(SQR(tmpx-x0)/SX + SQR(tmpy-y0)/SY));
			gsl_matrix_set(J,NbCols*i+j,0,1./s);
			gsl_matrix_set(J,NbCols*i+j,1,tmpc/s);
			tmpd = A*tmpc*2*(tmpx-x0)/SX*cos(An);
			gsl_matrix_set(J,NbCols*i+j,2,tmpd/s);
			tmpd = A*tmpc*2*(tmpy-y0)/SY*cos(An);
			gsl_matrix_set(J,NbCols*i+j,3,tmpd/s);
			tmpd = A*tmpc*SQR(tmpx-x0)/SQR(SX);
			gsl_matrix_set(J,NbCols*i+j,4,tmpd/s);
			tmpd = A*tmpc*SQR(tmpy-y0)/SQR(SY);
			gsl_matrix_set(J,NbCols*i+j,5,tmpd/s);
			tmpderxr = -sin(An)*(j+1-x0)-cos(An)*(i+1-y0);
			tmpderyr =  cos(An)*(j+1-x0)-sin(An)*(i+1-y0);
			tmpd = -A*tmpc*(2*(tmpx-x0)/SX*tmpderxr + 2*(tmpy-y0)/SY*tmpderyr);
			gsl_matrix_set(J,NbCols*i+j,6,tmpd/s);
		}
	}
	return GSL_SUCCESS;
}

int Gaussian_fdf_An (const gsl_vector * x, void *PSF_data, gsl_vector * f, gsl_matrix * J){
	Gaussian_f_An (x, PSF_data, f);
	Gaussian_df_An (x, PSF_data, J);
	return GSL_SUCCESS;
}

/* The function returns the fitted parameters without angle. However it 
 * returns NULL if the number of parameters is => to the pixel number. 
 */
Param_GAUSS *minimiz_noAn(gsl_matrix* z, double background, int layer){
	size_t i, j;
	size_t NbRows=z->size1;    //characteristics of the selection : height and width
	size_t NbCols=z->size2;
	const size_t p = 6;			// Number of parameters fitted
	const size_t n = NbRows*NbCols;
	Param_GAUSS *Param_Fitted = malloc(sizeof(Param_GAUSS));	
	gsl_vector *MaxV = MaxInz(z, background);
	int status;
	unsigned int iter = 0;
	gsl_matrix *covar = gsl_matrix_alloc (p, p);
	double y[n], sigma[n];
	struct PSF_data d = { n, y, sigma, NbRows, NbCols, 0};
	gsl_multifit_function_fdf f;
	double x_init[6] = { background, gsl_vector_get(MaxV,2), gsl_vector_get(MaxV,0), gsl_vector_get(MaxV,1), gsl_vector_get(MaxV,4), gsl_vector_get(MaxV,3) };
	gsl_vector_view x = gsl_vector_view_array (x_init, p);
	const gsl_rng_type * type;
	gsl_rng * r;
	const gsl_multifit_fdfsolver_type *T;
	gsl_multifit_fdfsolver *s;
	
	gsl_rng_env_setup();
	
	type = gsl_rng_default;
	r = gsl_rng_alloc (type);
	
	f.f = &Gaussian_f;
	f.df = &Gaussian_df;
	f.fdf = &Gaussian_fdf;
	f.n = n;
	if (n<=p) {
		free(Param_Fitted);
		return NULL;
	}
	f.p = p;
	f.params = &d;
	
	for(i=0; i<NbRows; i++){
		for(j=0; j<NbCols; j++){
			y[NbCols*i+j] = gsl_matrix_get(z,i,j);
			sigma[NbCols*i+j]=1;
		}
	}
	
	T = gsl_multifit_fdfsolver_lmsder;
	s = gsl_multifit_fdfsolver_alloc (T, n, p);
	gsl_multifit_fdfsolver_set (s, &f, &x.vector);

	do{
		iter++;
		status = gsl_multifit_fdfsolver_iterate (s);
		if (status) break;
		status = gsl_multifit_test_delta (s->dx, s->x, 1e-4, 1e-4);
	}while (status == GSL_CONTINUE && iter < MAX_ITER_NO_ANGLE);

	gsl_multifit_covar (s->J, 0.0, covar);

	#define FIT(i) gsl_vector_get(s->x, i)
	#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))	//for now, errors are not displayed

	/*Output structure with parameters fitted */
	Param_Fitted->B  = FIT(0);
	Param_Fitted->A  = FIT(1);
	Param_Fitted->x0 = FIT(2);
	Param_Fitted->y0 = FIT(3);
	Param_Fitted->Sx = FIT(4);
	Param_Fitted->Sy = FIT(5);
	Param_Fitted->FWHMX = sqrt(FIT(4)/2.)*2*sqrt(log(2.)*2);	//Set the real FWHMx with regards to the Sx parameter
	Param_Fitted->FWHMY = sqrt(FIT(5)/2.)*2*sqrt(log(2.)*2);	//Set the real FWHMy with regards to the Sy parameter
	Param_Fitted->An = 0;	//The angle is not fitted here
	//Units
	Param_Fitted->units = "px";
	//Magnitude
	Param_Fitted->Mag = Get_Magnitude(z, FIT(0));
	//Layer: not fitted
	Param_Fitted->layer = layer;
	//RMSE
	Param_Fitted->rmse = d.rmse;
	// absolute uncertainties
	Param_Fitted->Berr = ERR(0)/FIT(0);
	Param_Fitted->Aerr = ERR(1)/FIT(1);
	Param_Fitted->xerr = ERR(2)/FIT(2);
	Param_Fitted->yerr = ERR(3)/FIT(3);
	Param_Fitted->Sxerr = ERR(4)/FIT(4);
	Param_Fitted->Syerr = ERR(5)/FIT(5);
	Param_Fitted->Anerr = 0;	
	Param_Fitted->xpos = 0;		// will be set by the peaker
	Param_Fitted->ypos = 0;
	//we free the memory
	gsl_vector_free(MaxV);
	gsl_multifit_fdfsolver_free (s);
	gsl_matrix_free (covar);
	gsl_rng_free (r);
	return Param_Fitted;
}

/* The function returns the fitted parameters with angle. However it returns
 * NULL if the number of parameters is => to the pixel number.
 * This should not happend because this case is already treated by the
 * minimiz_noAn function */
Param_GAUSS *minimiz_An(gsl_matrix* z, Param_GAUSS *Param_Fitted){
	size_t i, j;
	size_t NbRows=z->size1;    //characteristics of the selection : height and width
	size_t NbCols=z->size2;
	const size_t p = 7;			// Number of parameters fitted
	const size_t n = NbRows*NbCols;
	Param_GAUSS *Param_Fitted_An = malloc(sizeof(Param_GAUSS));
	int status;
	unsigned int iter = 0;
	gsl_matrix *covar = gsl_matrix_alloc (p, p);
	double y[n], sigma[n];
	struct PSF_data d = { n, y, sigma, NbRows, NbCols, 0};
	gsl_multifit_function_fdf f_An;
	double x_init[7] = { Param_Fitted->B, Param_Fitted->A, Param_Fitted->x0, Param_Fitted->y0, Param_Fitted->Sx, Param_Fitted->Sy, 0 };
	gsl_vector_view x = gsl_vector_view_array (x_init, p);
	const gsl_rng_type * type;
	gsl_rng * r;
	const gsl_multifit_fdfsolver_type *T;
	gsl_multifit_fdfsolver *s;
	
	gsl_rng_env_setup();
	
	type = gsl_rng_default;
	r = gsl_rng_alloc (type);

	f_An.f = &Gaussian_f_An;
	f_An.df = &Gaussian_df_An;
	f_An.fdf = &Gaussian_fdf_An;
	f_An.n = n;
	f_An.p = p;
	f_An.params = &d;
	
	for(i=0; i<NbRows; i++){
		for(j=0; j<NbCols; j++){
			y[NbCols*i+j] = gsl_matrix_get(z,i,j);
			sigma[NbCols*i+j]=1;
		}
	}
	
	T = gsl_multifit_fdfsolver_lmsder;
	s = gsl_multifit_fdfsolver_alloc (T, n, p);
	gsl_multifit_fdfsolver_set (s, &f_An, &x.vector);

	do{
		iter++;
		status = gsl_multifit_fdfsolver_iterate (s);
		if (status) break;
		status = gsl_multifit_test_delta (s->dx, s->x, 1e-4, 1e-4);
	}while (status == GSL_CONTINUE && iter < MAX_ITER_ANGLE);

	gsl_multifit_covar (s->J, 0.0, covar);

	#define FIT(i) gsl_vector_get(s->x, i)
	#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))	//for now, errors are not displayed

	/*Output structure with parameters fitted */
	Param_Fitted_An->B  = FIT(0);
	Param_Fitted_An->A  = FIT(1);
	Param_Fitted_An->x0 = FIT(2);
	Param_Fitted_An->y0 = FIT(3);
	Param_Fitted_An->Sx = FIT(4);
	Param_Fitted_An->Sy = FIT(5);
	Param_Fitted_An->FWHMX = sqrt(FIT(4)/2.)*2*sqrt(log(2.)*2);	//Set the real FWHMx with regards to the Sx parameter
	Param_Fitted_An->FWHMY = sqrt(FIT(5)/2.)*2*sqrt(log(2.)*2);	//Set the real FWHMy with regards to the Sy parameter
	Param_Fitted_An->An = -FIT(6)*180./M_PI;
	/* The angle must be => -90 and <= 90
	 * Otherwise, the solution may be degenerate
	 * and produce an angle > 90. So we're
	 * looking for the solution between
	 * the interval we want */
	while (fabs(Param_Fitted_An->An)>90.){
		if (Param_Fitted_An->An>0)
			Param_Fitted_An->An-=90.;
		else
			Param_Fitted_An->An+=90.;
	}
	//Units
	Param_Fitted_An->units = "px";
	//Magnitude
	Param_Fitted_An->Mag = Get_Magnitude(z, FIT(0));	//we take the un-normalized value of B
	//Layer: not fitted
	Param_Fitted_An->layer = Param_Fitted->layer;
	//RMSE
	Param_Fitted_An->rmse = d.rmse;
	// absolute uncertainties
	Param_Fitted_An->Berr	= ERR(0)/FIT(0);
	Param_Fitted_An->Aerr	= ERR(1)/FIT(1);
	Param_Fitted_An->xerr	= ERR(2)/FIT(2);
	Param_Fitted_An->yerr	= ERR(3)/FIT(3);
	Param_Fitted_An->Sxerr 	= ERR(4)/FIT(4);
	Param_Fitted_An->Syerr 	= ERR(5)/FIT(5);
	Param_Fitted_An->Anerr 	= ERR(6)/FIT(6);	

	//we free the memory
	gsl_multifit_fdfsolver_free (s);
	gsl_matrix_free (covar);
	gsl_rng_free (r);
	return Param_Fitted_An;
}

void DisplayResult(Param_GAUSS *result, rectangle *area) {
	char buffer[256];
	
	sprintf(buffer, "PSF fit Result:\n");
	sprintf(buffer, "%sx0=%0.2f px, y0=%0.2f px\n", buffer,
			result->x0 + area->x, area->y + area->h - result->y0);
	sprintf(buffer, "%sFWHM X=%0.2f%s, FWHM Y=%0.2f%s\n", buffer,
			result->FWHMX, result->units, result->FWHMY, result->units);
	sprintf(buffer, "%sAngle=%0.2f deg\n", buffer, result->An);
	sprintf(buffer, "%sBackground value=%0.6f\n", buffer, result->B);
	sprintf(buffer, "%sMaximal intensity=%0.6f\n", buffer, result->A);
	sprintf(buffer, "%sMagnitude=%0.2f\n", buffer, result->Mag);
	sprintf(buffer, "%sRMSE=%.3e\n", buffer, result->rmse);
	
	siril_log_message(buffer);
}
/* If the pixel pitch and the focal length are known and filled in the 
 * setting box, we convert FWHM in pixel to arcsec by multiplying
 * the FWHM value with the sampling value */
void update_units(fits* fit, Param_GAUSS **result) {
		
	if (!fit->focal_length || !fit->pixel_size_x || !fit->pixel_size_y) return;
	
	(*result)->FWHMX*=(206.264806 * fit->pixel_size_x/fit->focal_length)*(double)fit->binning_x;
	(*result)->FWHMY*=(206.264806 * fit->pixel_size_y/fit->focal_length)*(double)fit->binning_y;
	(*result)->units="\"";
}
