/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
 * -------------------------------------------------------------------
 *
 *
 * NOTICE: this file is NOT USED ANYMORE, since FFMS2 makes it much more reliably and easily.
 * THE SEEK CODE IN THIS FILE DOES NOT WORK.
 * READING FRAMES SEQUENTIALLY IN SOME FILES DOES NOT WORK AT ALL.
 *
 *
 * This file was inspired by code from astroavibrower:
 * http://sourceforge.net/projects/astroavibrowser/
 * and from the demuxing example of ffmpeg:
 * http://ffmpeg.org/doxygen/trunk/doc_2examples_2demuxing_decoding_8c-example.html
 * Other documentation links:
 * http://ffmpeg.org/trac/ffmpeg/wiki/Using%20libav*
 *
 * It opens film files, like AVI files, using ffmpeg libraries such as
 * libavformat, libavutil, libavcodec and libswscale.
*/

/* seeking is a hard problem with videos because the don't all have the same encoding
 * and indexing system. Some have full frames on each frame, some have only full key
 * frames and diff frames on the following frames. Key frames are good checkpoints
 * where seeking could be done, but it's even hard to seek to them exactly.
 * "There are a number of discussions on the FFmpeg mailing lists about methods to
 * determine the frame number. Most of these discussions come down to an issue with
 * packet ordering. There is the presentation time stamp (PTS) and the decompression
 * time stamp (DTS). The frames may be decompressed in one order (1 2 3 4) but
 * presented in a different order (1 4 2 3)."
 *
 * "For most containers (perhaps all, I haven't checked), literally the only way to
 * find the exact frame number of any given frame is to scan through the entire file
 * from the beginning until you get to that frame. This is not a limitation of
 * FFmpeg; it's a limitation of the container. As far as I understand, based on years
 * of reading these lists, there is fundamentally no way to get around it."
 * However, most containers provide index for keyframes.
 *
 * http://stackoverflow.com/q/17546073
 *
 * "if you need to goto frame X frame you need to decode first X frames, but this can
 * be optimized if you can keep index of key frames for the first time and next time
 * you can seek to closet key frame which is exact in avformat_seek_file() and then
 * move to desired frame."
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_FFMPEG

#include <stdio.h>
#include "avi.h"
#include "proto.h"		// for fits_flip_top_to_bottom()

AVPacket flush_pkt;

int avi_read_current_frame(struct avi_struct *avi, fits *fit);

void avi_display_info(struct avi_struct *avi_file) {
	av_dump_format(avi_file->format_context, 0, avi_file->filename, 0);

	fprintf(stdout, "=========== AVI file info ==============\n");
	fprintf(stdout, "file name: %s\n", avi_file->filename);
	fprintf(stdout, "video stream number: %d\n", avi_file->stream_number);
	fprintf(stdout, "image size: %d x %d\n",
			avi_file->codec_context->width,
			avi_file->codec_context->height);
	fprintf(stdout, "number of layers: %d\n", avi_file->nb_layers);
	fprintf(stdout, "estimated frame count: %d\n", avi_file->frame_count);
	fprintf(stdout, "========================================\n");
}

/* this function should not take a long time to execute, it may be called
 * several times before the actual opening for frame reading.
 * Decoding the entire film to know the frame number is not a good idea. */
int avi_open_file(const char *filename, struct avi_struct *avi_file) {
	int i;
	avi_init_struct(avi_file);
	/* initializing a special packet used in flushing the seeks */
	av_init_packet(&flush_pkt);
	flush_pkt.data = (void*)"FLUSH";
	/*if (avi_file->format_context) {
		fprintf(stderr, "AVI: file already opened, or badly closed\n");
		return AVI_ERROR;
	}*/

	// opening file
	if (avformat_open_input(&avi_file->format_context, filename, NULL, NULL) < 0) {
		fprintf(stderr, "AVI: could not open the file\n");
		return AVI_ERROR;
	}

	// get streams
	if (avformat_find_stream_info(avi_file->format_context, NULL) < 0) {
		fprintf(stderr, "AVI: unable to get the streams\n");
		avi_close_file(avi_file);
		return AVI_ERROR;
	}

	// open streams and pick the first video
	avi_file->stream_number = -1;
	for (i = 0; i < avi_file->format_context->nb_streams; i++) {
		if (avi_file->format_context->streams[i]->codec->codec_type ==
				AVMEDIA_TYPE_VIDEO) {
			avi_file->stream_number = i;
			break;
		}
	}
	if (avi_file->stream_number == -1) {
		fprintf(stderr, "AVI: unable to find a video stream\n");
		avi_close_file(avi_file);
		return AVI_ERROR;
	}

	// load codec
	avi_file->codec_context = avi_file->format_context->streams[avi_file->stream_number]->codec;
	avi_file->codec = avcodec_find_decoder(avi_file->codec_context->codec_id);
	if (!avi_file->codec) {
		fprintf(stderr, "AVI: no codec found for the video stream\n");
		avi_close_file(avi_file);
		return AVI_ERROR;
	}
	AVDictionary *opts = NULL;
	if (avcodec_open2(avi_file->codec_context, avi_file->codec, &opts) < 0) {
		fprintf(stderr, "AVI: failed to open codec\n");
		avi_close_file(avi_file);
		return AVI_ERROR;
	}
	
	check_video_number_of_layers(avi_file);		// check format
	if ((avi_file->nb_layers != 1 && avi_file->nb_layers != 3) ||
			!sws_isSupportedOutput(avi_file->dest_fmt)) {
		avi_close_file(avi_file);
		return AVI_ERROR;
	}

	// frame count estimation
	if (avi_file->format_context->duration < 0) {
		// how to handle this, without decoding the entire file?
		avi_file->frame_count = 100;
	} else {
		AVStream *stream = avi_file->format_context->streams[avi_file->stream_number];
		avi_file->frame_count = avi_file->format_context->duration /
			AV_TIME_BASE *
			(stream->r_frame_rate.num / stream->r_frame_rate.den);
	}

#if 0
	// allocate index
	if (avi_file->frame_count > 0 && avi_file->frame_count < 5000)
		avi_file->index_size = avi_file->frame_count;
	else avi_file->index_size = 100;
	avi_file->frame_index = malloc(avi_file->index_size);
	if (avi_file->frame_index == NULL)
		avi_file->index_size = 0;
	memset(avi_file->frame_index, -1, avi_file->index_size);
#endif

	avi_file->current_frame = 0;
	avi_file->filename = strdup(filename);
	fprintf(stdout, "AVI: successfully opened the video file %s, %d frames\n", avi_file->filename, avi_file->frame_count);
	return 0;
}

void check_video_number_of_layers(struct avi_struct *avi) {
	switch (avi->codec_context->pix_fmt) {
		case PIX_FMT_GRAY8:
			avi->nb_layers = 1;
			avi->dest_fmt = PIX_FMT_GRAY8;
			break;
		case PIX_FMT_GRAY16:
		case PIX_FMT_RGB48:
			fprintf(stderr, "AVI: 16-bit pixel depth films are not supported yet.\n");
			break;
		case PIX_FMT_PAL8:	// PAL 8 is a RGB32 palette format
		default:
			avi->nb_layers = 3;
			avi->dest_fmt = PIX_FMT_RGB24;
	}
}

int allocate_frames_and_conversion(struct avi_struct *avi_file) {
	if (avi_file->frame) return 0;
	// allocate image where the decoded image will be put
	avi_file->frame = avcodec_alloc_frame();
	avi_file->frameRGB = avcodec_alloc_frame();
	if (!avi_file->frame || !avi_file->frameRGB) {
		fprintf(stderr, "AVI: could not allocate frame\n");
		return -1;
	}

	int numBytes = avpicture_get_size(avi_file->dest_fmt,
			avi_file->codec_context->width,
			avi_file->codec_context->height);
	uint8_t *buffer = (uint8_t *)av_malloc(numBytes*sizeof(uint8_t));
	avpicture_fill((AVPicture *)avi_file->frameRGB, buffer, avi_file->dest_fmt,
			avi_file->codec_context->width,
			avi_file->codec_context->height);

	avi_file->convert_context = sws_getContext(
			avi_file->codec_context->width,
			avi_file->codec_context->height,
			avi_file->codec_context->pix_fmt,
			avi_file->codec_context->width,
			avi_file->codec_context->height,
			avi_file->dest_fmt,
			SWS_FAST_BILINEAR | SWS_CPU_CAPS_MMX,
			NULL, NULL, NULL);
	if (!avi_file->convert_context) {
		fprintf(stderr, "AVI: could not allocate the convert context\n");
		return -1;
	}
	return 0;
}

void avi_close_file(struct avi_struct *avi_file) {
	fprintf(stderr, "AVI: closing current file\n");
	// close codec
	if(avi_file->codec_context != NULL) {
		avcodec_close(avi_file->codec_context);
		avi_file->codec_context = NULL;
	}
	// close file
	if(avi_file->format_context != NULL) {
		avformat_close_input(&avi_file->format_context);
		avi_file->format_context = NULL;
	}
	if (avi_file->filename) {
		free(avi_file->filename);
		avi_file->filename = NULL;
	}
	if (avi_file->convert_context)
		sws_freeContext(avi_file->convert_context);
	// TODO: should be avcodec_free_frame(&frame); below
	//avcodec_free_frame(&avi_file->frame); 
	av_freep(&avi_file->frame);
	av_freep(&avi_file->frameRGB);
	avi_init_struct(avi_file);
}

void avi_init_struct(struct avi_struct *avi_file) {
	memset(avi_file, 0, sizeof(struct avi_struct));
}

/* public function: get the fits for frame frame_no */
int avi_read_frame(struct avi_struct *avi, int frame_no, fits *fit) {
	int frame_delta;
	if (!avi || !avi->format_context || !fit || frame_no < 0) {
		// || frame_no >= avi->frame_count)
		// this ^ may happen quite often
		return AVI_ERROR;
	}

	if (allocate_frames_and_conversion(avi)) return AVI_ERROR;
	av_init_packet(&avi->pkt);
	avi->pkt.data = NULL;
	avi->pkt.size = 0;

	/* WARNING: seeking is not working properly */
	frame_delta = frame_no - avi->current_frame;
	fprintf(stderr, "FRAME DELTA: %d (current is given as %d, requested %d)\n", frame_delta, avi->current_frame, frame_no);
	if (frame_delta < 0 || frame_delta > 5) {
		fprintf(stdout, "AVI: seeking to previous key frame\n");
		if (av_seek_frame(avi->format_context, avi->stream_number,
				frame_no, AVSEEK_FLAG_BACKWARD) < 0) {
			fprintf(stderr, "AVI: seeking error!\n");
			return AVI_ERROR;
		}
	}
	while (avi->current_frame < frame_no) {
		int retval = avi_read_current_frame(avi, NULL);
		if (retval) {
			fprintf(stderr, "AVI: failed to seek to requested frame\n");
			//return AVI_END_OF_SEQUENCE;
			return AVI_ERROR;
		}
	}
	// read the requested frame, decode it, convert it, and store it in fit
	return avi_read_current_frame(avi, fit);
}

/*
static void packet_queue_flush(PacketQueue *q) {
	AVPacketList *pkt, *pkt1;

	for(pkt = q->first_pkt; pkt != NULL; pkt = pkt1) {
		pkt1 = pkt->next;
		av_free_packet(&pkt->pkt);
		av_freep(&pkt);
	}
	q->last_pkt = NULL;
	q->first_pkt = NULL;
	q->nb_packets = 0;
	q->size = 0;
}
*/

#if 0
	target_avi_frame = -1;
	if (frame_no < avi->index_size) {
		// we may have an idea of which avi frame we are going to
		if (avi->frame_index[frame_no] != -1) {
			// we have the index
			target_avi_frame = avi->frame_index[frame_no];
		} else {
			// we search for the closest known index
			int known_frame = frame_no - 1;
			while (known_frame > 0 && avi->frame_index[known_frame--] != -1);
			if (known_frame != avi->current_frame-1)
				target_avi_frame = avi->frame_index[known_frame];
		}
	}
	if (target_avi_frame != -1) {
		av_seek_frame(avi->format_context, -1, 0, AVSEEK_FLAG_ANY);
		avi->current_frame = target_;
	}

	if (frame_no == 0 || frame_no < avi->current_frame) {
		// rewind to first frame
		av_seek_frame(avi->format_context, -1, 0, AVSEEK_FLAG_ANY);
		avi->current_frame = 0;
	}
#endif

/* internal function: read and decode the current frame in the file. If fit is
 * non-NULL, the frame is also converted to RGB and stored in it. */
int avi_read_current_frame(struct avi_struct *avi, fits *fit) {
	int got_frame = 0;
	int ret = 0;
	while (!got_frame && av_read_frame(avi->format_context, &avi->pkt) >= 0) {
		AVPacket orig_pkt = avi->pkt;
		do {
			ret = decode_packet(avi, &got_frame, 0, fit);
			if (ret < 0)
				break;
			avi->pkt.data += ret;
			avi->pkt.size -= ret;
		} while (!got_frame && avi->pkt.size > 0);
		av_free_packet(&orig_pkt);
	}
#if 0
	/* flush cached frames */
	avi->pkt.data = NULL;
	avi->pkt.size = 0;
	do {
		decode_packet(avi, &got_frame, 1, NULL);
	} while (got_frame);
#endif
	return (!got_frame);
}

int decode_packet(struct avi_struct *avi_file, int *got_frame, int cached, fits *fit) {
	int ret = 0;
	int decoded = avi_file->pkt.size;
	*got_frame = 0;
	if (avi_file->pkt.stream_index != avi_file->stream_number) return -1;
	/* decode video frame */
	ret = avcodec_decode_video2(avi_file->codec_context, avi_file->frame, got_frame, &avi_file->pkt);
	if (ret < 0) {
		/* some systems don't have av_err2str */
		char buf[64];
		av_strerror(ret, buf, 64);
		fprintf(stderr, "Error decoding video frame (%s)\n", buf);
		return ret;
	}
	if (*got_frame) {
		avi_file->current_frame = avi_file->frame->coded_picture_number;
		fprintf(stdout, "AVI: video_frame%s n:%d coded_n:%d\n",
				cached ? "(cached)" : "",
				avi_file->current_frame, avi_file->frame->coded_picture_number);
		if (fit) {
			int i, j, nb_pixels;
			nb_pixels = avi_file->codec_context->width *
				avi_file->codec_context->height;

			if((fit->data = realloc(fit->data, nb_pixels * avi_file->nb_layers * sizeof(WORD)))==NULL) {
				fprintf(stderr,"AVI: NULL realloc for FITS data\n");
				return -1;
			}

			fit->naxes[0] = fit->rx = avi_file->codec_context->width;
			fit->naxes[1] = fit->ry = avi_file->codec_context->height;
			fit->naxes[2] = avi_file->nb_layers;
			if (avi_file->nb_layers == 1) {
				fit->naxis = 2;
				fit->pdata[RLAYER]=fit->data;
				fit->pdata[GLAYER]=fit->data;
				fit->pdata[BLAYER]=fit->data;
			} else {
				fit->naxis = 3;
				fit->pdata[RLAYER]=fit->data;
				fit->pdata[GLAYER]=fit->data + nb_pixels;
				fit->pdata[BLAYER]=fit->data + nb_pixels * 2;
			}
			fit->bitpix=USHORT_IMG;	// change to BYTE_IMG someday?
			fit->mini = 0;
			fit->maxi = 255;

			/* decode video if format is not natively the dest format */
			sws_scale(avi_file->convert_context,
					(const uint8_t * const*)avi_file->frame->data,
					avi_file->frame->linesize,
					0, avi_file->codec_context->height,
					avi_file->frameRGB->data,
					avi_file->frameRGB->linesize);

			for (i=0,j=0; i<nb_pixels; i++) {
				fit->pdata[RLAYER][i] = avi_file->frameRGB->data[0][j++];
				fit->pdata[GLAYER][i] = avi_file->frameRGB->data[0][j++];
				fit->pdata[BLAYER][i] = avi_file->frameRGB->data[0][j++];
			}
			fits_flip_top_to_bottom(fit);
		}
	}
	/* If we use the new API with reference counting, we own the data and need
	 * to de-reference it when we don't use it anymore */
	//if (*got_frame && api_mode == API_MODE_NEW_API_REF_COUNT)
	//	av_frame_unref(frame);
	return decoded;
}

#if 0
void update_frame_index(struct avi_struct *avi, int avi_frame, int picture) {
	if (picture > avi->index_size) {
		int *newalloc = realloc(avi->frame_index, picture + 100);
		if (!newalloc) return;
		// fill the new part with -1 values
		memset(avi_file->frame_index+avi_file->index_size, -1,
				picture + 100 - avi_file->index_size);
		avi->index_size = picture + 100;
	}
	if (avi->frame_index[picture] == -1)
		avi->frame_index[picture] = avi_frame;
	else if (avi->frame_index[picture] != avi_frame)
		fprintf(stderr, "AVI: updating to different value %d, was %d (frame %d)\n",
				avi_frame, avi->frame_index[picture], picture);
}
#endif

#endif
