/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
 *
 * Useful links about OpenCV:
 * http://docs.opencv.org/modules/core/doc/intro.html
 * http://docs.opencv.org/modules/imgproc/doc/geometric_transformations.html#resize
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#ifdef HAVE_OPENCV
# include <math.h>

#include "siril.h"
#include "opencv.h"
#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>

using namespace cv;

/* resizes image to the sizes toX * toY, and stores it back in image */
int resize_gaussian(fits *image, int toX, int toY, int interpolation) {
	assert(image->data);
	assert(image->rx);
	int type = (image->naxes[2] != 1) ? CV_16UC3 : CV_16U;
	Mat in;
	Mat red(image->ry, image->rx, CV_16U, image->pdata[RLAYER]);
	Mat green(image->ry, image->rx, CV_16U, image->pdata[GLAYER]);
	Mat blue(image->ry, image->rx, CV_16U, image->pdata[BLAYER]);
	Mat channel[] = {blue, green, red};		// in the 1-channel case, blue = green = red
	merge(channel, 3, in);
	Mat out(toY, toX, type);
	resize(in, out, out.size(), 0, 0, interpolation);
	image->rx = toX;
	image->naxes[0] = toX;
	image->ry = toY;
	image->naxes[1] = toY;
	WORD *newdata = (WORD*) realloc(image->data, toX * toY * sizeof(WORD) * image->naxes[2]);
	if (!newdata) {
		free(newdata);
		return 1;
	}
	image->data = newdata;
	split(out, channel);
	
	memcpy(image->data, channel[2].data, toX * toY * sizeof(WORD));
	if (image->naxes[2] == 3) {
		memcpy(image->data + toX * toY, channel[1].data, toX * toY * sizeof(WORD));
		memcpy(image->data + toX * toY*2, channel[0].data, toX * toY * sizeof(WORD));
	}
	
	if (image->naxes[2] == 1) {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data;
		image->pdata[2] = image->data;
	}
	else {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data + (toX * toY);
		image->pdata[2] = image->data + (toX * toY) * 2;	
	}
	red.release();
	green.release();
	blue.release();
	in.release();
	out.release();
	return 0;
}

/* Rotate an image with the angle "angle"
 * TODO allow users to choose between a cropped or not cropped result
 * cropped = 0 : no crop
 * cropped = 1 : crop
 */

int rotate_image(fits *image, double angle, int interpolation, int cropped) {
	assert(image->data);
	assert(image->rx);
	assert(image->ry);
	int type = (image->naxes[2] != 1) ? CV_16UC3 : CV_16U;
	int ndata = image->rx * image->ry;
	Mat in;
	Mat red(image->ry, image->rx, CV_16U, image->pdata[RLAYER]);
	Mat green(image->ry, image->rx, CV_16U, image->pdata[GLAYER]);
	Mat blue(image->ry, image->rx, CV_16U, image->pdata[BLAYER]);
	Mat channel[] = {blue, green, red};		// in the 1-channel case, blue = green = red
	merge(channel, 3, in);
	Mat out(image->ry, image->rx, type);
	
	if ((angle == 90.0 || angle == 270.0) && interpolation == -1) {		// fast rotation
		transpose(in, out);
		if (angle == 90.0) flip(out, out, 0);
		else flip(out, out, 1);
	}
	else {
		Point2f pt(in.cols/2.0, in.rows/2.0);		// We take the center of the image. Should we pass this in function parameters ?
		Mat r = getRotationMatrix2D(pt, angle, 1.0);
		if (cropped == 1) {
			warpAffine(in, out, r, in.size(), interpolation);
		}
		else {
		
			// determine bounding rectangle
			Rect frame = RotatedRect(pt, in.size(), angle).boundingRect();
			// adjust transformation matrix
			r.at<double>(0,2) += frame.width / 2.0 - pt.x;
			r.at<double>(1,2) += frame.height / 2.0 - pt.y;
			
			warpAffine(in, out, r, frame.size(), interpolation);
			ndata = out.cols * out.rows;
			WORD *newdata = (WORD*) realloc(image->data, ndata * image->naxes[2] * sizeof(WORD));
			if (!newdata) {
				free(newdata);
				return 1;
			}
			image->data = newdata;
		}
	}
	split(out, channel);
	
	memcpy(image->data, channel[2].data, ndata * sizeof(WORD));
	if (image->naxes[2] == 3) {
		memcpy(image->data+ndata, 	channel[1].data, ndata * sizeof(WORD));
		memcpy(image->data+ndata*2, channel[0].data, ndata * sizeof(WORD));
	}
	
	if (image->naxes[2] == 1) {
		image->pdata[RLAYER] = image->data;
		image->pdata[GLAYER] = image->data;
		image->pdata[BLAYER] = image->data;
	}
	else {
		image->pdata[RLAYER] = image->data;
		image->pdata[GLAYER] = image->data + ndata;
		image->pdata[BLAYER] = image->data + ndata * 2;	
	}
	image->rx = out.cols;
	image->ry = out.rows;
	image->naxes[0] = image->rx;
	image->naxes[1] = image->ry;
	
	/* free data */
	red.release();
	green.release();
	blue.release();
	in.release();
	out.release();
	return 0;
}

int unsharp_filter(fits* image, double sigma, double amount) {
	assert(image->data);
	assert(image->rx);
	int type =  CV_16U;
	if (image->naxes[2] != 1) 
		type = CV_16UC3;

	Mat in(image->ry, image->rx, type, image->data);
	Mat out, contrast;
	GaussianBlur(in, out, Size(), sigma);
	if (fabs(amount) > 0.0) {
		Mat sharpened = in * (1 + amount) + out * (- amount);
		out = sharpened.clone();
		sharpened.release();
	}
	
	memcpy(image->data, out.data, image->rx * image->ry * sizeof(WORD) * image->naxes[2]);
	if (image->naxes[2] == 1) {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data;
		image->pdata[2] = image->data;
	}
	else {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data + (image->rx * image->ry);
		image->pdata[2] = image->data + (image->rx * image->ry) * 2;	
	}
	in.release();
	return 0;
}

#endif
