#include <errno.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include "fio.h"
#include "signode.h"
#include "configfile.h"
#include "cycletimer.h"
#include "clock.h"
#include "sgstring.h"

typedef enum TapState {
        TapExit2DR = 0x0,
        TapExit1DR = 0x1,
        TapShiftDR = 0x2,
        TapPauseDR = 0x3,
        TapSelectIRScan = 0x4,
        TapUpdateDR = 0x5,
        TapCaptureDR = 0x6,
        TapSelectDRScan = 0x7,
        TapExit2IR = 0x8,
        TapExit1IR = 0x9,
        TapShiftIR = 0xa,
        TapPauseIR = 0xb,
        TapRunTestIdle = 0xc,
        TapUpdateIR = 0xd,
        TapCaptureIR = 0xe,
        TapTestLogicReset = 0xf
} TapState;

typedef struct JTAG_Tap {
	char *name;
	TapState tapState;
	SigNode *tdi;
	SigNode *tdo;
	SigNode *tms;
	SigNode *tck;
	SigNode *nTrst;
} JTAG_Tap;

static int 
traceNTrst(SigNode *sig,int value,void *clientData)
{
	return 0;
}
static int 
traceTck(SigNode *sig,int value,void *clientData)
{
	JTAG_Tap *tap = (JTAG_Tap *) clientData;
	int tms;
	if(value == SIG_LOW) {
		/* Sampling is done on the rising edge */
		return 0;
	}
	tms = SigNode_Val(tap->tms);
	if(tms == SIG_LOW) {
		switch(tap->tapState) {
			case TapExit2DR:
				tap->tapState = TapShiftDR;
				break;

			case TapExit1DR:
				tap->tapState = TapPauseDR;
				break;
				
			case TapShiftDR:
				break;

			case TapPauseDR:
				break; 

			case TapSelectIRScan:
				tap->tapState = TapCaptureIR;
				break;

			case TapUpdateDR:
				tap->tapState = TapRunTestIdle;
				break;

			case TapCaptureDR:
				tap->tapState = TapShiftDR;	
				break;

			case TapSelectDRScan:
				tap->tapState = TapCaptureDR;	
				break;

			case TapExit2IR:
				tap->tapState = TapShiftIR;
				break;

			case TapExit1IR:
				tap->tapState = TapPauseIR;
				break;

			case TapShiftIR:
				break;

			case TapPauseIR:
				break;

			case TapRunTestIdle:
				break;

			case TapUpdateIR:
				tap->tapState = TapRunTestIdle;	
				break;

			case TapCaptureIR:
				tap->tapState = TapShiftIR;	
				break;

			case TapTestLogicReset:
				tap->tapState = TapRunTestIdle;
				break;
		}
	} else if(tms == SIG_HIGH) {
		switch(tap->tapState) {
			case TapExit2DR:
				tap->tapState = TapUpdateDR;
				break;

			case TapExit1DR:
				tap->tapState = TapUpdateDR;
				break;	

			case TapShiftDR:
				tap->tapState = TapExit1DR;
				break;

			case TapPauseDR:
				tap->tapState = TapExit2DR;
				break;

			case TapSelectIRScan:
				tap->tapState = TapTestLogicReset;
				break;

			case TapUpdateDR:
				tap->tapState = TapSelectDRScan;
				break;

			case TapCaptureDR:
				tap->tapState = TapExit1DR;	
				break;

			case TapSelectDRScan:
				tap->tapState = TapSelectIRScan;
				break;

			case TapExit2IR:
				tap->tapState = TapUpdateIR;
				break;

			case TapExit1IR:
				tap->tapState = TapUpdateIR;
				break;
				
			case TapShiftIR:
				tap->tapState = TapExit1IR;
				break;

			case TapPauseIR:
				tap->tapState = TapExit2IR;
				break;

			case TapRunTestIdle:
				tap->tapState = TapSelectDRScan;
				break;

			case TapUpdateIR:
				tap->tapState = TapSelectDRScan;
				break;

			case TapCaptureIR:
				tap->tapState = TapExit1IR;
				break;

			case TapTestLogicReset:
				break;
		}
	}
	return 0;
}
void
JTagTap_New(const char *name) 
{
	JTAG_Tap *tap;	
	tap = sg_new(JTAG_Tap);
	tap->name = sg_strdup(name);
	tap->tdi = SigNode_New("%s.tdi",name);
	tap->tdo = SigNode_New("%s.tdo",name);
	tap->tms = SigNode_New("%s.tms",name);
	tap->tck = SigNode_New("%s.tck",name);
	tap->nTrst = SigNode_New("%s.nTrst",name);
	tap->tapState = TapTestLogicReset;
	SigNode_Trace(tap->tck,traceTck,tap);
	SigNode_Trace(tap->nTrst,traceNTrst,tap);
}
