/*
 * ----------------------------------------------------
 *
 * Standard  M16C Instruction Set and Addressing Modes
 * (C) 2005  Jochen Karrer 
 *   Author: Jochen Karrer
 *
 * State: nothing works 
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include "m16c_cpu.h"

#define ISNEG(x) ((x)&(1<<15))
#define ISNOTNEG(x) (!((x)&(1<<15)))
#define ISPOS(x) (-(x) & (1<<15))

#define ISNEGB(x) ((x)&(1<<7))
#define ISNOTNEGB(x) (!((x)&(1<<7)))
#define ISPOSB(x) (-(x) & (1<<7))


static inline uint16_t 
bcd_to_uint16(uint16_t s) {
        return (s & 0xf)+10*((s>>4)&0xf) + 100 * ((s>>8) & 0xf) + 1000 * ((s>>12) & 0xf);
}

static inline uint16_t 
uint16_to_bcd(uint16_t u) 
{
	unsigned int i;
	unsigned int digit=0;
	uint16_t bcd=0;
	for(i=0;i<4;i++) {
		digit = u % 10;	
		bcd |= (digit <<(i*4));
		u=u/10;
	}	
	return bcd;
}

static inline uint16_t 
add8_carry(uint8_t op1,uint8_t op2,uint8_t result) 
{

	if( ((ISNEGB(op1) && ISNEGB(op2))
          || (ISNEGB(op1) && ISNOTNEGB(result))
          || (ISNEGB(op2) && ISNOTNEGB(result)))) {
                        return M16C_FLG_CARRY;
        } else {
                return 0;
        }

}

static inline uint16_t
add8_overflow(uint8_t op1,uint8_t op2,uint8_t result) {
        if ((ISNEGB (op1) && ISNEGB (op2) && ISNOTNEGB (result))
          || (ISNOTNEGB (op1) && ISNOTNEGB (op2) && ISNEGB (result))) {
                return M16C_FLG_OVERFLOW;
        } else {
                return 0;
        }
}

static inline uint16_t
add16_carry(uint16_t op1,uint16_t op2,uint16_t result) {
        if( ((ISNEG(op1) && ISNEG(op2))
          || (ISNEG(op1) && ISNOTNEG(result))
          || (ISNEG(op2) && ISNOTNEG(result)))) {
                        return M16C_FLG_CARRY;
        } else {
                return 0;
        }
}

static inline uint16_t
add16_overflow(uint16_t op1,uint16_t op2,uint16_t result) {
        if ((ISNEG (op1) && ISNEG (op2) && ISNOTNEG (result))
          || (ISNOTNEG (op1) && ISNOTNEG (op2) && ISNEG (result))) {
                return M16C_FLG_OVERFLOW;
        } else {
                return 0;
        }
}

static inline void
add_flags(uint16_t op1,uint16_t op2,uint16_t result,int isword) {
	M16C_REG_FLG &= ~(M16C_FLG_OVERFLOW | M16C_FLG_SIGN 
		| M16C_FLG_ZERO | M16C_FLG_CARRY);
	if(result == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	if(isword) {
		M16C_REG_FLG |= add16_carry(op1,op2,result); 
		M16C_REG_FLG |= add16_overflow(op1,op2,result); 
		if(result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN; 
		}	
	} else {
		M16C_REG_FLG |= add8_carry(op1,op2,result); 
		M16C_REG_FLG |= add8_overflow(op1,op2,result); 
		if(result & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN; 
		}	
	}
}

static inline void
and_flags(uint16_t result,int isword) {
	M16C_REG_FLG &= ~M16C_FLG_SIGN;
	if(isword) {
		if(result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}	
	} else {
		if(result & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
}
static inline void
or_flags(uint16_t result,int isword) {
	return and_flags(result,isword);
}
static inline void
xor_flags(uint16_t result,int isword) {
	return and_flags(result,isword);
}

/*
 * ---------------------------------------------------------
 * M16C has non borrow style carry like 6502
 * It subtracts by adding the complement. 
 * ---------------------------------------------------------
 */
static inline uint16_t
sub8_carry(uint16_t op1,uint16_t op2,uint16_t result) {
        if( ((ISNEGB(op1) && ISNOTNEGB(op2))
          || (ISNEGB(op1) && ISNOTNEGB(result))
          || (ISNOTNEGB(op2) && ISNOTNEGB(result)))) {
                        return M16C_FLG_CARRY;
        } else {
                        return 0;
        }
}

static inline uint16_t
sub16_carry(uint16_t op1,uint16_t op2,uint16_t result) {
        if( ((ISNEG(op1) && ISNOTNEG(op2))
          || (ISNEG(op1) && ISNOTNEG(result))
          || (ISNOTNEG(op2) && ISNOTNEG(result)))) {
                        return M16C_FLG_CARRY;
        } else {
                        return 0;
        }
}

static inline uint16_t
sub16_overflow(uint16_t op1,uint16_t op2,uint16_t result) {
        if ((ISNEG (op1) && ISNOTNEG (op2) && ISNOTNEG (result))
          || (ISNOTNEG (op1) && ISNEG (op2) && ISNEG (result))) {
                return M16C_FLG_OVERFLOW;
        } else {
                return 0;
        }
}
static inline uint16_t
sub8_overflow(uint8_t op1,uint8_t op2,uint8_t result) {
        if ((ISNEGB (op1) && ISNOTNEGB (op2) && ISNOTNEGB (result))
          || (ISNOTNEGB (op1) && ISNEGB (op2) && ISNEGB (result))) {
                return M16C_FLG_OVERFLOW;
        } else {
                return 0;
        }
}

static inline void
sub_flags(uint16_t op1,uint16_t op2,uint16_t result,int isword) {
	M16C_REG_FLG &= ~(M16C_FLG_OVERFLOW | M16C_FLG_SIGN 
		| M16C_FLG_ZERO | M16C_FLG_CARRY);
	if(result == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	if(isword) {
		M16C_REG_FLG |= sub16_carry(op1,op2,result); 
		M16C_REG_FLG |= sub16_overflow(op1,op2,result); 
		if(result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN; 
		}	
	} else {
		M16C_REG_FLG |= sub8_carry(op1,op2,result); 
		M16C_REG_FLG |= sub8_overflow(op1,op2,result); 
		if(result & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN; 
		}	
	}
}
static int
check_condition(uint8_t cnd) {
        int result=0;
        int carry=0,zero=0,sign=0,ovl=0;
        if(M16C_REG_FLG & M16C_FLG_ZERO)
                zero=1;
        if(M16C_REG_FLG & M16C_FLG_CARRY)
                carry=1;
        if(M16C_REG_FLG & M16C_FLG_SIGN)
                sign=1;
        if(M16C_REG_FLG & M16C_FLG_OVERFLOW)
                ovl=1;
        switch (cnd & 0xf) {
                case 0:
                        result=carry;
                        break;
                case 1:
                        result= (carry && ! zero);
                        break;
                case 2:
                        result=zero;
                        break;
                case 3:
                        result=sign;
                        break;
                case 4:
                        result = !carry;
                        break;
                case 5: // LEU
                        result = !carry || zero;
                        break;
                case 6: // NE/NZ
                        result = !zero;
                        break;
                case 7:
                        result =! sign;
                        break;
                case 8:
                        //LE
                        result=(sign^ovl)||zero;
                        break;
                case 9:
                        result=ovl;
                        break;
                case 10:
                        result =! (sign^ovl);
                        break;
                case 12:
                        result =! ((sign^ovl)||zero);
                        break;

                case 13:
                        result = !ovl;
                        break;

                case 14:
                        result = sign^ovl;
                        break;
                default:
                        fprintf(stderr,"unknown condition %d\n",cnd);
                        exit(3474);

        }
        return result;
}

void
am1_set(uint16_t icode,int admode,int *arglen,int isword,uint16_t value) 
{
	switch(admode) {
		/* r0l/r0 */
		case 0:
			*arglen = 0;
			if(isword) {
				M16C_REG_R0 = value;
			} else {
				M16C_REG_R0L = value;
			}
			break;

		case 1:
			*arglen=0;
			if(isword) {
				M16C_REG_R1 = value;
			} else {
				M16C_REG_R0H = value;
			}
			break;

		case 2:
			*arglen = 0;
			if(isword) {
				M16C_REG_R2 = value;
			} else {
				M16C_REG_R1L = value;
			}
			break;

		case 3:
			*arglen = 0;
			if(isword) {
				M16C_REG_R3 = value;
			} else {
				M16C_REG_R1H = value;
			}
			break;

		case 4:
			*arglen=0;
			if(isword) {
                        	M16C_REG_A0 = value;
			} else {
				M16C_REG_A0 = value;
			}	
			break;

		case 5:
			*arglen = 0;
			if(isword) {
				M16C_REG_A1 = value;
			} else {	
				M16C_REG_A1 = value;
			}
			break;

		case 6:
			{
				uint32_t addr;
				*arglen = 0;
				addr = M16C_REG_A0;	
				if(isword) {
					M16C_Write16(value,addr);
				} else {
					M16C_Write8(value,addr);
				}
			}
			break;

		case 7:
			{
				uint32_t addr;
				*arglen = 0;
				addr = M16C_REG_A1;
				if(isword) {
					M16C_Write16(value,addr);
				} else {
					M16C_Write8(value,addr);
				}
			}
			break;

		case 8:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_A0 + dsp8;
				*arglen = 1;
				if(isword) {
					M16C_Write16(value,addr);
				} else {
					M16C_Write8(value,addr);
				}
			}
			break;

		case 9:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_A1 + dsp8;
				*arglen = 1;
				if(isword) {
					M16C_Write16(value,addr);
				} else {
					M16C_Write8(value,addr);
				}
			}
			break;

		case 10:
			 {
                                uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
                                uint32_t addr = M16C_REG_SB + dsp8;
                                *arglen = 1;
                                if(isword) {
                                        M16C_Write16(value,addr);
                                } else {
                                        M16C_Write8(value,addr);
                                }
                        }
			break;

		case 11:
			{
                                int8_t dsp8 = M16C_Read8(M16C_REG_PC);
                                uint32_t addr = M16C_REG_FB + dsp8;
                                *arglen = 1;
                                if(isword) {
                                        M16C_Write16(value,addr);
                                } else {
                                        M16C_Write8(value,addr);
                                }
                        }
			break;
		
		case 12:
			{
                                uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = M16C_REG_A0 + dsp16;
                                *arglen = 2;
                                if(isword) {
                                        M16C_Write16(value,addr);
                                } else {
                                        M16C_Write8(value,addr);
                                }
                        }
			break;

		case 13:
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = M16C_REG_A1 + dsp16;
                                *arglen = 2;
                                if(isword) {
                                        M16C_Write16(value,addr);
                                } else {
                                        M16C_Write8(value,addr);
                                }
			}			
			break;

		case 14:
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = M16C_REG_SB + dsp16;
                                *arglen = 2;
                                if(isword) {
                                        M16C_Write16(value,addr);
                                } else {
                                        M16C_Write8(value,addr);
                                }
			}			
			break;

		case 15: /* abs 16 */
			{
				uint16_t abs16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = abs16;
                                *arglen = 2;
                                if(isword) {
                                        M16C_Write16(value,addr);
                                } else {
                                        M16C_Write8(value,addr);
                                }
			}			
			break;
		default:
			break;
	}
}
/*
 * -----------------------------------------------
 * am1_get
 * 	Always cares about .b or .w
 * -----------------------------------------------
 */
uint16_t
am1_get(uint16_t icode,int admode,int *arglen,int isword) 
{
	uint16_t value;
	switch(admode) {
		/* r0l/r0 */
		case 0:
			*arglen = 0;
			if(isword) {
				value = M16C_REG_R0;
			} else {
				value = M16C_REG_R0L;
			}
			break;

		case 1:
			*arglen=0;
			if(isword) {
				value = M16C_REG_R1;
			} else {
				value = M16C_REG_R0H;
			}
			break;

		case 2:
			*arglen = 0;
			if(isword) {
				value = M16C_REG_R2;
			} else {
				value = M16C_REG_R1L;
			}
			break;

		case 3:
			*arglen = 0;
			if(isword) {
				value = M16C_REG_R3;
			} else {
				value = M16C_REG_R1H;
			}
			break;

		case 4:
			*arglen=0;
			if(isword) {
                        	value = M16C_REG_A0;
			} else {
				// modify isword P is missing here *iswordP = 1; */
				value = M16C_REG_A0 & 0xff;
			}	
			break;

		case 5:
			*arglen = 0;
			if(isword) {
				value = M16C_REG_A1;
			} else {	
				//*iswordP = 1; /* Modifying iswordp is missing here ****/
				value = M16C_REG_A1 & 0xff;
			}
			break;

		case 6:
			{
				uint32_t addr;
				*arglen = 0;
				addr = M16C_REG_A0;	
				if(isword) {
					value = M16C_Read16(addr);
				} else {
					value = M16C_Read8(addr);
				}
			}
			break;

		case 7:
			{
				uint32_t addr;
				*arglen = 0;
				addr = M16C_REG_A1;
				if(isword) {
					value = M16C_Read16(addr);
				} else {
					value = M16C_Read8(addr);
				}
			}
			break;

		case 8:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_A0 + dsp8;
				*arglen = 1;
				if(isword) {
					value = M16C_Read16(addr);
				} else {
					value = M16C_Read8(addr);
				}
			}
			break;

		case 9:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_A1 + dsp8;
				*arglen = 1;
				if(isword) {
					value = M16C_Read16(addr);
				} else {
					value = M16C_Read8(addr);
				}
			}
			break;

		case 10:
			 {
                                uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
                                uint32_t addr = M16C_REG_SB + dsp8;
                                *arglen = 1;
                                if(isword) {
                                        value = M16C_Read16(addr);
                                } else {
                                        value = M16C_Read8(addr);
                                }
                        }
			break;

		case 11:
			{
                                uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
                                uint32_t addr = M16C_REG_FB + dsp8;
                                *arglen = 1;
                                if(isword) {
                                        value = M16C_Read16(addr);
                                } else {
                                        value = M16C_Read8(addr);
                                }
                        }
			break;
		
		case 12:
			{
                                uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = M16C_REG_A0 + dsp16;
                                *arglen = 2;
                                if(isword) {
                                        value = M16C_Read16(addr);
                                } else {
                                        value = M16C_Read8(addr);
                                }
                        }
			break;

		case 13:
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = M16C_REG_A1 + dsp16;
                                *arglen = 2;
                                if(isword) {
                                        value = M16C_Read16(addr);
                                } else {
                                        value = M16C_Read8(addr);
                                }
			}			
			break;
		case 14:
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = M16C_REG_SB + dsp16;
                                *arglen = 2;
                                if(isword) {
                                        value = M16C_Read16(addr);
                                } else {
                                        value = M16C_Read8(addr);
                                }
			}			
			break;

		case 15:
			{
				uint16_t abs16 = M16C_Read16(M16C_REG_PC);
                                uint32_t addr = abs16;
                                *arglen = 2;
                                if(isword) {
                                        value = M16C_Read16(addr);
                                } else {
                                        value = M16C_Read8(addr);
                                }
			}			
			break;
		default:
			value = 0;
			break;
	}
	return value;
}

uint16_t
am1_get_eva(uint16_t icode,int admode,int *arglen,int isword) 
{
	uint32_t addr;
	switch(admode) {
		case 8:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				addr = (M16C_REG_A0 + dsp8) & 0xfffff;
				*arglen = 1;
			}
			break;

		case 9:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				addr = (M16C_REG_A1 + dsp8) & 0xfffff;
				*arglen = 1;
			}
			break;

		case 10:
			 {
                                uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
                                addr = (M16C_REG_SB + dsp8) & 0xfffff;
                                *arglen = 1;
                        }
			break;

		case 11:
			{
                                uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
                                addr = (M16C_REG_FB + dsp8) & 0xfffff;
                                *arglen = 1;
                        }
			break;
		
		case 12:
			{
                                uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                addr = (M16C_REG_A0 + dsp16) & 0xfffff;
                                *arglen = 2;
                        }
			break;

		case 13:
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                addr = (M16C_REG_A1 + dsp16) & 0xfffff;
                                *arglen = 2;
			}			
			break;
		case 14:
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
                                addr = (M16C_REG_SB + dsp16) & 0xfffff;
                                *arglen = 2;
			}			
			break;

		case 15:
			{
				uint16_t abs16 = M16C_Read16(M16C_REG_PC);
                                addr = abs16;
                                *arglen = 2;
			}			
			break;
		default:
			fprintf(stderr,"wrong address mode EVA\n");
			addr = 0;
			break;
	}
	return addr;
}
uint8_t
am2b_get(int icode,int *arglen) 
{
	uint8_t value;
	int admode = icode & 0x7;
	switch(admode) {
		case 3:
			// R0H;
			*arglen=0;
			return M16C_REG_R0H;

		case 4:
			// R0L
			*arglen=0;
			return M16C_REG_R0L;
		case 5:
			{
				// DSP:8[SB]
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_SB + dsp8;
				value = M16C_Read8(addr);
				*arglen=1;
			}
			break;
		case 6:
			{
				// DSP:8[FB]
				int8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_FB + dsp8;
				value = M16C_Read8(addr);
				*arglen=1;
			}
			break;
		case 7:
			{
				uint16_t abs16 = M16C_Read16(M16C_REG_PC);
				value = M16C_Read8(abs16);
				*arglen=2;
			}
			break;

		default:
			fprintf(stderr,"Illegal addressing mode 2: %d\n",admode);
			*arglen=0;
			value = 0;
	}
	return value;
}

void
am2b_set(uint16_t icode,int *arglen,uint8_t value) 
{
	int admode = icode & 7;
	switch(admode) {
		case 3:
			// R0H;
			M16C_REG_R0H = value;
			*arglen = 0;
			break;

		case 4:
			// R0L
			M16C_REG_R0L = value;
			*arglen = 0;
			break;

		case 5:
			{
				// DSP:8[SB]
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_SB + dsp8;
				*arglen = 1;
				M16C_Write8(value,addr);
			}
			break;
		case 6:
			{
				// DSP:8[FB]
				int8_t dsp8 = M16C_Read8(M16C_REG_PC);
				uint32_t addr = M16C_REG_FB + dsp8;
				*arglen = 1;
				M16C_Write8(value,addr);
			}
			break;
		case 7:
			{
				uint16_t abs16 = M16C_Read16(M16C_REG_PC);
				value = M16C_Read8(abs16);
				*arglen = 2;
			}
			break;

		default:
			fprintf(stderr,"write: Illegal addressing mode 2: %d\n",admode);
	}
	return;
}
/*
 */
uint8_t
am3b_get(int icode,int *arglen) 
{
	int src = icode & 3;
	uint8_t value;
	switch(src) {
		case 0:
			/* If dst is R0L src is R0H else src is R0L */
			{
				int dst = (icode >> 2) & 1;
				if(dst) {
					value =  M16C_REG_R0L; 
				} else {
					value =  M16C_REG_R0H;
				}
				*arglen = 0;
			}
			break;
		case 1:
			/* DSP:8[SB] */
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC); 
				uint32_t addr = M16C_REG_SB + dsp8;
				value = M16C_Read8(addr);
				*arglen = 1;
			}
			break;

		case 2:
			/* DSP:8[SB] */
			{
				int8_t dsp8 = M16C_Read8(M16C_REG_PC); 
				uint32_t addr = M16C_REG_SB + dsp8;
				value = M16C_Read8(addr);
				*arglen = 1;
			}
			break;
		case 3:
			/* ABS16 */
			{
				uint16_t abs16 = M16C_Read16(M16C_REG_PC);
				value = M16C_Read8(abs16);
				*arglen = 2;
			}
			break;
		default:
			/* Unreachable, make compiler quiet */
			value=0;
	}
	return value;
}
void
am3b_set(int icode,int *arglen,uint8_t value) 
{
	int src = icode & 3;
	switch(src) {
		case 0:
			/* If dst is R0L src is R0H else src is R0L */
			{
				int dst = (icode >> 2) & 1;
				if(dst) {
					M16C_REG_R0L = value; 
				} else {
					M16C_REG_R0H = value;
				}
				*arglen = 0;
			}
			break;
		case 1:
			/* DSP:8[SB] */
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC); 
				uint32_t addr = M16C_REG_SB + dsp8;
				M16C_Write8(value,addr);
				*arglen = 1;
			}
			break;

		case 2:
			/* DSP:8[SB] */
			{
				int8_t dsp8 = M16C_Read8(M16C_REG_PC); 
				uint32_t addr = M16C_REG_SB + dsp8;
				M16C_Write8(value,addr);
				*arglen = 1;
			}
			break;
		case 3:
			/* ABS16 */
			{
				uint16_t abs16 = M16C_Read16(M16C_REG_PC);
				M16C_Write8(value,abs16);
				*arglen = 2;
			}
			break;
		default:
			/* Unreachable, make compiler quiet */
			break;
	}
}
/*
 * -------------------------------------------------------------------
 * Test showed that M16C does Byte wide read-modify write
 * on memory bit operations
 * -------------------------------------------------------------------
 */
int
bitaddr_get(int admode,int *arglen)
{
	uint16_t value;
	uint16_t bitnr;
	uint16_t bytenr;
	switch(admode) {
		 case 0:
                        // bit,R0
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			value = M16C_REG_R0;
                        *arglen=1;
			break;
                case 1:
                        // bit,R1
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			value = M16C_REG_R1;
                        *arglen=1;
			break;
                case 2:
                        // bit,R2
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			value = M16C_REG_R2;
                        *arglen=1;
			break;
                case 3:
                        // bit,R3
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			value = M16C_REG_R3;
                        *arglen=1;
			break;
                case 4:
                        // bit,A0
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			value = M16C_REG_A0;
                        *arglen=1;
			break;
                case 5:
                        // bit,A1
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			value = M16C_REG_A1;
                        *arglen=1;
			break;

		case 6:
                        // bit[A0]
			bytenr = M16C_REG_A0 >> 3;
			bitnr = M16C_REG_A0 & 0x7;
                        value=M16C_Read8(bytenr);
                        *arglen=0;
			break;

                case 7:
                        // bit[A1]
			bytenr = M16C_REG_A1 >> 3;
			bitnr = M16C_REG_A1 & 0x7;
			value = M16C_Read8(bytenr);
                        *arglen=0;
			break;

                case 8:
                        // base:8[A0]
                        bitnr=M16C_Read8(M16C_REG_PC);
                        bitnr+= M16C_REG_A0;
			bytenr = bitnr >> 3;
			bitnr = bitnr & 7;
                        value=M16C_Read8(bytenr);
                        *arglen=1;
			break;

                case 9:
                        // base:8[A1]
                        bitnr=M16C_Read8(M16C_REG_PC);
                        bitnr+= M16C_REG_A1;
			bytenr = bitnr >> 3;
			bitnr = bitnr & 7;
                        value=M16C_Read8(bytenr);
                        *arglen=1;
			break;

                case 10:
                        // bit,base:8[SB]
                        bitnr=M16C_Read8(M16C_REG_PC);
                        bitnr+= M16C_REG_SB;
			bytenr = bitnr >> 3;
			bitnr = bitnr & 7;
                        value=M16C_Read8(bytenr);
                        *arglen=1;
			break;

		 case 11:
                        // bit,base:8[FB]
                        {
                                int8_t dsp=M16C_Read8(M16C_REG_PC);
                                bitnr = (M16C_REG_FB+dsp);
				bytenr = bitnr >> 8;
				bitnr = bitnr & 7;
                                value = M16C_Read8(bytenr);
                                *arglen=1;
                        }
			break;
                case 12:
                        // base:16[A0]
                        {
                                uint16_t dsp = M16C_Read16(M16C_REG_PC);
                                bitnr = (M16C_REG_A0+dsp);
				bytenr = bitnr >> 3;	
				bitnr = bitnr & 7;
                                value = M16C_Read8(bytenr);
                                *arglen=2;
                        }
			break;

                case 13:
                        // base:16[A1]
                        {
                                uint16_t dsp = M16C_Read16(M16C_REG_PC);
                                bitnr = (M16C_REG_A1+dsp);
				bytenr = bitnr >> 3;
				bitnr = bitnr & 7;
                                value =M16C_Read8(bytenr);
                                *arglen=2;
                        }
			break;
		 case 14:
                        // bit,base:16[SB]
                        {
                                uint16_t dsp = M16C_Read16(M16C_REG_PC);
                                bitnr = (M16C_REG_SB+dsp);
				bytenr = bitnr >> 3;
				bitnr = bitnr & 7;
                                value = M16C_Read8(bytenr);
                                *arglen = 2;
                        }
			break;
                case 15:
			// bit,base:16
                        {
                                uint16_t dsp = M16C_Read16(M16C_REG_PC);
				bytenr = dsp >> 3;
				bitnr = dsp & 7;
				value = M16C_Read8(bytenr);
                                *arglen=2;
                        }
			break;
		default:
			/* should not be reached, make compiler quiet */
			bitnr = 0;
			value = 0;
	}
	return (value >> bitnr) & 1;
}


void
bitaddr_set(int admode,int *arglen,int bitval)
{
	uint16_t bitnr;
	uint32_t bytenr;
	uint16_t value;
	
	switch(admode) {
		 case 0:
                        // bit,R0
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			if(bitval) {
				M16C_REG_R0  |= (1<<bitnr);
			} else {
				M16C_REG_R0  &= ~(1<<bitnr);
			}
                        *arglen=1;
			break;
                case 1:
                        // bit,R1
                        bitnr=M16C_Read8(M16C_REG_PC);
			if(bitval) {
				M16C_REG_R1 |= (1<<bitnr);
			} else {
				M16C_REG_R1 &= ~(1<<bitnr);
			}
                        *arglen=1;
			break;
                case 2:
                        // bit,R2
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			if(bitval) {
				M16C_REG_R2 |= (1<<bitnr);
			} else {
				M16C_REG_R2 &= ~(1<<bitnr);
			}
                        *arglen=1;
			break;
                case 3:
                        // bit,R3
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			if(bitval) {
				M16C_REG_R3 |= (1<<bitnr);
			} else {
				M16C_REG_R3 &= ~(1<<bitnr);
			}
                        *arglen=1;
			break;

                case 4:
                        // bit,A0
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			if(bitval) {
				M16C_REG_A0 |= (1<<bitnr);
			} else {
				M16C_REG_A0 &= ~(1<<bitnr);
			}
                        *arglen=1;
			break;
                case 5:
                        // bit,A1
                        bitnr=M16C_Read8(M16C_REG_PC) & 0xf;
			if(bitval)  {
				M16C_REG_A1 |= (1<<bitnr);
			} else {
				M16C_REG_A1 &= ~(1<<bitnr);
			}
                        *arglen=0;
			break;
		case 6:
                        // bit[A0]
                        bytenr = M16C_REG_A0 >> 3;
                        bitnr=M16C_REG_A0 & 0x7;
			value = M16C_Read8(bytenr);
			if(bitval) {
				value |= (1<<bitnr);
			} else {
				value &= ~(1<<bitnr);
			}
                        M16C_Write8(value,bytenr);
                        *arglen=0;
			break;

                case 7:
                        // bit[A1]
			bytenr = M16C_REG_A1 >> 3;
                        bitnr= M16C_REG_A1 & 0x7;
			value = M16C_Read8(bytenr);
			if(bitval) {
				value |= (1<<bitnr);
			} else {
				value &= ~(1<<bitnr);
			}
                        M16C_Write8(value,bytenr);
                        *arglen=0;
			break;

                case 8:
                        // base:8[A0]
                        bitnr=M16C_Read8(M16C_REG_PC);
                        bitnr+= M16C_REG_A0;
			bytenr = bitnr >> 3;
                        bitnr=bitnr&0x7;
                        value = M16C_Read8(bytenr);
			if(bitval) {
				value |= (1<<bitnr);
			} else {
				value &= ~(1<<bitnr);
			}
                        M16C_Write16(value,bytenr);
                        *arglen=1;
			break;

                case 9:
                        // base:8[A1]
                        bitnr=M16C_Read8(M16C_REG_PC);
                        bitnr+= M16C_REG_A1;
			bytenr = bitnr >> 3;
			bitnr &= 7;
                        value = M16C_Read8(bytenr);
			if(bitval) {
				value |= (1<<bitnr);
			} else {
				value &= ~(1<<bitnr);
			}
                        M16C_Write8(value,bytenr);
                        *arglen=1;
			break;

                case 10:
                        // bit,base:8[SB]
                        bitnr=M16C_Read8(M16C_REG_PC);
                        bitnr+= M16C_REG_SB;
			bytenr = bitnr>>3;
			bitnr &= 7;
                        value = M16C_Read8(bytenr);
			if(bitval) {
				value |= (1<<bitnr);
			} else {
				value &= ~(1<<bitnr);
			}
                        M16C_Write8(value,bytenr);
                        *arglen=1;
			break;

		 case 11:
                        // bit,base:8[FB]
                        {
                                int8_t dsp=M16C_Read8(M16C_REG_PC);
                                bitnr = (M16C_REG_FB+dsp)&0xffff;
				bytenr = bitnr >> 3;
				bitnr &= 7;
                                value = M16C_Read8(bytenr);
				if(bitval) {
					value |= (1<<bitnr);
				} else {
					value &= ~(1<<bitnr);
				}
				M16C_Write8(value,bytenr);
                                *arglen=1;
                        }
			break;
                case 12:
                        // base:16[A0]
                        {
                                uint16_t dsp=M16C_Read16(M16C_REG_PC);
                                bitnr = (M16C_REG_A0+dsp)&0xffff;
				bytenr = bitnr >> 3;
				bitnr &= 7;
                                value = M16C_Read8(bytenr);
				if(bitval) {
					value |= (1<<bitnr);
				} else {
					value &= ~(1<<bitnr);
				}
				M16C_Write8(value,bytenr);
                                *arglen=2;
                        }
			break;

                case 13:
                        // base:16[A1]
                        {
                                uint16_t dsp=M16C_Read16(M16C_REG_PC);
                                bitnr = (M16C_REG_A1+dsp)&0xffff;
				bytenr = bitnr >> 3;
				bitnr &= 7;
                                value = M16C_Read8(bytenr);
				if(bitval) {
					value |= (1<<bitnr);
				} else {
					value &= ~(1<<bitnr);
				}
				M16C_Write8(value,bytenr);
                                *arglen=2;
                        }
			break;
		 case 14:
                        // bit,base:16[SB]
                        {
                                uint16_t dsp=M16C_Read16(M16C_REG_PC);
                                bitnr = (M16C_REG_SB+dsp)&0xffff;
				bytenr = bitnr >> 3;
				bitnr &= 7;
                                value = M16C_Read8(bytenr);
				if(bitval) {
					value |= (1<<bitnr);
				} else {
					value &= ~(1<<bitnr);
				}
				M16C_Write8(value,bytenr);
                                *arglen=2;
                        }
			break;
                case 15:
			// bit,base:16
                        {
                                uint16_t dsp=M16C_Read16(M16C_REG_PC);
                                bytenr = dsp>>3;
                                bitnr = dsp&0x7;
                                value = M16C_Read8(bytenr);
				if(bitval) {
					value |= (1<<bitnr);
				} else {
					value &= ~(1<<bitnr);
				}
				M16C_Write8(value,bytenr);
                                *arglen=2;
                        }
			break;
		default:
			/* should not be reached, make compiler quiet */
			value = 0;
	}
}
void 
m16c_abs_size_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);	
	int arglen;
	uint16_t dst = am1_get(icode,icode&0xf,&arglen,isword); 
	uint16_t result;
	M16C_REG_FLG &= ~(M16C_FLG_OVERFLOW | M16C_FLG_SIGN 
		| M16C_FLG_ZERO | M16C_FLG_CARRY);
	if((isword && (dst == 0x8000)) || (!isword && (dst == 0x80))) {
		M16C_REG_FLG  |= M16C_FLG_OVERFLOW;
	} 
	result = abs(dst);		
	if(isword) {
		if(result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN; 
		} 	
	} else {
		if(result & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN; 
		} 	
	}
	if(result == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO; 
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	M16C_REG_PC+=arglen;
}

void 
m16c_adc_size_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int isword = icode & (1<<8);	
	int arglen;
	dst = am1_get(icode,icode&0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC+arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC+arglen);
	}
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = dst + src + 1;
	} else {
		result = dst + src;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	add_flags(dst,src,result,isword);
	M16C_REG_PC+=arglen;
	if(isword) {
		M16C_REG_PC+=2;
	} else {
		M16C_REG_PC++;
	}
	fprintf(stderr,"instr m16c_adc_size_immdst(%04x)\n",icode);
}

void 
m16c_adc_size_srcdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);	
	int arglen;
	uint16_t src,dst,result;
	/* ignore isword return because this is the src case */
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);	
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = dst + src + 1;
	} else {
		result = dst + src;
	}
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC+=arglen;
	add_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_adc_size_srcdst(%04x)\n",icode);
}

/*
 * ------------------------------------------------------
 * should print error message if byte dst is A0/A1 
 * ------------------------------------------------------
 */
void 
m16c_adcf_size_dst(uint16_t  icode) 
{
	uint16_t dst,result;
	int isword = icode & (1<<8);
	int arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = dst +  1;
	} else {
		result = dst;
	}
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC += arglen;
	add_flags(dst,0,result,isword);
	fprintf(stderr,"instr m16c_adcf_size_dst(%04x)\n",icode);
}

void 
m16c_add_size_g_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
	}
	result = dst + src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC += arglen;	
	if(isword) {
		M16C_REG_PC += 2;	
	} else {
		M16C_REG_PC++;	
	}
	add_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_add_size_g_immdst(%04x) not implemented\n",icode);
}

void 
m16c_add_size_q(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;	
	src = (icode>>4)&0xf; 	
	result = src + dst;
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	add_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_add_size_q(%04x) not implemented\n",icode);
}

void 
m16c_add_b_s_immdst(uint16_t  icode) 
{
	int arglen;				
	uint8_t imm; 
	uint8_t dst;
	uint8_t result;
	imm =  M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	dst = am2b_get(icode,&arglen);
	result = dst + imm;	
	am2b_set(icode,&arglen,result);
	M16C_REG_PC+=arglen;
	add_flags(dst,imm,result,0);
	fprintf(stderr,"instr m16c_add_b_s_immdst(%04x)\n",icode);
}

void 
m16c_add_size_g_srcdest(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);	
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	result = dst + src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC+=arglen;
	add_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_add_size_g_srcdest(%04x)\n",icode);
}

void 
m16c_add_b_s_srcr0l(uint16_t  icode) 
{
	uint8_t src,dst,result;	
	int arglen;
	src = am3b_get(icode,&arglen);	
	M16C_REG_PC+=arglen;
	if(icode & (1<<2)) {
		dst = M16C_REG_R0H;
		M16C_REG_R0H = result = src+dst;
	} else {
		dst = M16C_REG_R0L;
		M16C_REG_R0L = result = src+dst;
	}
	add_flags(dst,src,result,0);
	fprintf(stderr,"instr m16c_add_b_s_srcr0l(%04x)\n",icode);
}

void 
m16c_add_size_g_imm_sp(uint16_t  icode) 
{
	int size = (icode>>8) & 1;
	uint16_t result,src,dst;
	if(size) { 
		src = M16C_Read16(M16C_REG_PC);	
		M16C_REG_PC+=2;
	} else {
		src = M16C_Read8(M16C_REG_PC);	
		/* sign extend */
		if(src & 0x80) {
			src |= 0xff00;
		}
		M16C_REG_PC+=1;
	}
	dst = M16C_REG_SP;	
	M16C_REG_SP = result = src + dst;
	add_flags(dst,src,result,1 /* srcsize or dstsize ?*/);
	fprintf(stderr,"instr m16c_add_size_g_imm_sp(%04x) \n",icode);
}

void 
m16c_add_size_q_imm_sp(uint16_t  icode) 
{
	uint16_t src = icode & 0xf;	
	uint16_t dst,result;
	/* sign extend */
	if(src & (1<<3)) {
		src |= 0xfff0;
	}
	dst = M16C_REG_SP;
	result = src + dst;
	M16C_REG_SP = result = src + dst;
	add_flags(dst,src,result,1);
	fprintf(stderr,"instr m16c_add_size_q_imm_sp(%04x) \n",icode);
}

void 
m16c_adjnz_size_immdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	int8_t imm;
	int8_t lbl;
	uint16_t dst;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	imm = (icode >> 4) & 0xf;
	if(imm & 8) {
		imm |= 0xf0;
	}
	dst += imm;
	if(!isword) {
		dst = dst & 0xff;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,dst) ;
	if(!dst) {
		lbl = M16C_Read8(M16C_REG_PC+arglen);
		M16C_REG_PC+=lbl;
	} else {
		M16C_REG_PC += arglen + 1;
	}
	fprintf(stderr,"instr m16c_adjnz_size_immdst(%04x)\n",icode);
}

void 
m16c_and_size_g_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
	}
	result = dst  & src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC += arglen;	
	if(isword) {
		M16C_REG_PC += 2;	
	} else {
		M16C_REG_PC++;	
	}
	and_flags(result,isword);
	fprintf(stderr,"instr m16c_and_size_g_immdst(%04x) not implemented\n",icode);
}

void 
m16c_and_b_s_immdst(uint16_t  icode) 
{
	int arglen;				
	uint8_t imm; 
	uint8_t dst;
	uint8_t result;
	imm =  M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	dst = am2b_get(icode,&arglen);
	result = dst & imm;	
	am2b_set(icode,&arglen,result);
	M16C_REG_PC+=arglen;
	and_flags(result,0);
	fprintf(stderr,"instr m16c_and_b_s_immdst(%04x)\n",icode);
}

void 
m16c_and_size_g_srcdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);	
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	result = dst & src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC+=arglen;
	and_flags(result,isword);
	fprintf(stderr,"instr m16c_and_size_g_srcdst(%04x)\n",icode);
}

void 
m16c_and_b_s_srcr0l(uint16_t  icode) 
{
	uint8_t src,dst,result;	
	int arglen;
	src = am3b_get(icode,&arglen);	
	M16C_REG_PC+=arglen;
	if(icode & (1<<2)) {
		dst = M16C_REG_R0H;
		M16C_REG_R0H = result = src & dst;
	} else {
		dst = M16C_REG_R0L;
		M16C_REG_R0L = result = src &dst;
	}
	and_flags(result,0);
	fprintf(stderr,"instr m16c_and_b_s_srcr0l(%04x)\n",icode);
}

void 
m16c_band_src(uint16_t  icode) 
{
	int admode = icode & 0xf;		
	int arglen;
	int bit = bitaddr_get(admode,&arglen);	
	if(!(bit && (M16C_REG_FLG & M16C_FLG_CARRY))) {
		M16C_REG_FLG &= ~M16C_FLG_CARRY;	
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_band_src(%04x)\n",icode);
}

void 
m16c_bclr_g_dst(uint16_t  icode) 
{
	int admode = icode & 0xf;
	int arglen;
	bitaddr_set(admode,&arglen,0);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bclr_g_dst(%04x)\n",icode);
}

void 
m16c_bclr_s_bit_base(uint16_t  icode) 
{
	uint16_t bitnr = icode & 7;
	uint8_t value;
	int bytenr;
	bytenr = M16C_Read8(M16C_REG_PC++);
	value = M16C_Read8(bytenr);	
	value &= ~(1<<bitnr);
	M16C_Write8(value,bytenr);	
	fprintf(stderr,"instr m16c_bclr_s_bit_base(%04x)\n",icode);
}

void 
m16c_bmcnd_dst(uint16_t  icode) 
{
	int dst = icode & 0xf;
	int arglen;
	uint8_t cnd = M16C_Read8(M16C_REG_PC++);
	bitaddr_get(dst,&arglen); /* Shit , required only because of arglen */
	cnd = M16C_Read8(M16C_REG_PC+arglen);
	if(check_condition(cnd)) {
		bitaddr_set(dst,&arglen,1);
	} else {
		bitaddr_set(dst,&arglen,0);
	}
	M16C_REG_PC += arglen+1;
	fprintf(stderr,"instr m16c_bmcnd_dst(%04x)\n",icode);
}

void 
m16c_bmcnd_c(uint16_t  icode) 
{
	uint8_t cnd = icode & 0xf;
	if(check_condition(cnd))  {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	} else {
		M16C_REG_FLG &= ~M16C_FLG_CARRY;
	}
	fprintf(stderr,"instr m16c_bmcnd_c(%04x)\n",icode);
}

void 
m16c_bnand_src(uint16_t  icode) 
{
	int src;
	int arglen;
	src = bitaddr_get(icode & 0xf,&arglen);
	if(!(!src && (M16C_REG_FLG & M16C_FLG_CARRY))) {
		M16C_REG_FLG &= ~M16C_FLG_CARRY;	
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bnand_src(%04x)\n",icode);
}

void 
m16c_bnor_src(uint16_t  icode) 
{
	int src;
	int arglen;
	src = bitaddr_get(icode & 0xf,&arglen);
	if(!src) {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bnor_src(%04x)\n",icode);
}

void 
m16c_bnot_g_dst(uint16_t  icode) 
{
	int dst;	
	int arglen;
	dst = bitaddr_get(icode & 0xf,&arglen);
	dst = !dst;
	bitaddr_set(icode & 0xf,&arglen,dst);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bnot_g_dst(%04x)\n",icode);
}

void 
m16c_bnot_s_bit_base(uint16_t  icode) 
{
	uint16_t bitnr = icode & 7;
	uint8_t value;
	int bytenr;
	bytenr = M16C_Read8(M16C_REG_PC++);
	value = M16C_Read8(bytenr);	
	value ^= ~(1<<bitnr);
	M16C_Write8(value,bytenr);	
	fprintf(stderr,"instr m16c_bnot_s_bit_base(%04x)\n",icode);
}

void 
m16c_bntst_src(uint16_t  icode) 
{
	int src;
	int arglen;
	src = bitaddr_get(icode & 0xf,&arglen); 
	if(src) {
		M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_ZERO);
	} else {
		M16C_REG_FLG |= (M16C_FLG_CARRY | M16C_FLG_ZERO);
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bntst_src(%04x)\n",icode);
}

void 
m16c_bnxor_src(uint16_t  icode) 
{
	int src;
	int arglen;
	src = bitaddr_get(icode & 0xf,&arglen);
	if(!src  !=  !!(M16C_REG_FLG & M16C_FLG_CARRY)) {
		M16C_REG_FLG |= M16C_FLG_CARRY;	
	} else {
		M16C_REG_FLG &= ~M16C_FLG_CARRY;	
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bnxor_src(%04x)\n",icode);
}

void 
m16c_bor_src(uint16_t  icode) 
{
	int src;
	int arglen;
	src = bitaddr_get(icode & 0xf,&arglen);
	if(src) {
		M16C_REG_FLG |= M16C_FLG_CARRY;	
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bor_src(%04x)\n",icode);
}

void 
m16c_brk(uint16_t  icode) 
{
	fprintf(stderr,"instr m16c_brk(%04x) not implemented\n",icode);
}

void 
m16c_bset_g_dst(uint16_t  icode) 
{
	int admode = icode & 0xf;
	int arglen;
	bitaddr_set(admode,&arglen,1);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bset_g_dst(%04x)\n",icode);
}

void 
m16c_bset_s_bit_base(uint16_t  icode) 
{
	uint16_t bitnr = icode & 7;
	uint8_t value;
	int bytenr;
	bytenr = M16C_Read8(M16C_REG_PC++);
	value = M16C_Read8(bytenr);	
	value |= (1<<bitnr);
	M16C_Write8(value,bytenr);	
	fprintf(stderr,"instr m16c_bset_s_bit_base(%04x)\n",icode);
}

void 
m16c_btst_src(uint16_t  icode) 
{
	int src;
	int arglen;
	src = bitaddr_get(icode & 0xf,&arglen); 
	if(src) {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_CARRY) & ~M16C_FLG_ZERO;
	} else {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_ZERO) & ~M16C_FLG_CARRY;
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_btst_src(%04x)\n",icode);
}

void 
m16c_btst_s_bit_base(uint16_t  icode) 
{
	uint16_t bitnr = icode & 7;
	uint8_t value;
	int bytenr;
	bytenr = M16C_Read8(M16C_REG_PC++);
	value = M16C_Read8(bytenr);	
	if(value & (1<<bitnr)) {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_CARRY) & ~M16C_FLG_ZERO;
	} else {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_ZERO) & ~M16C_FLG_CARRY;
	}
	fprintf(stderr,"instr m16c_btst_s_bit_base(%04x)\n",icode);
}

void 
m16c_btstc_dst(uint16_t  icode) 
{
	int dst;
	int arglen;
	dst = bitaddr_get(icode & 0xf,&arglen); 
	if(dst) {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_CARRY) & ~M16C_FLG_ZERO;
	} else {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_ZERO) & ~M16C_FLG_CARRY;
	}
	bitaddr_set(icode & 0xf,&arglen,0);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_btstc_dst(%04x) not implemented\n",icode);
}

void 
m16c_btsts_dst(uint16_t  icode) 
{
	int dst;
	int arglen;
	dst = bitaddr_get(icode & 0xf,&arglen); 
	if(dst) {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_CARRY) & ~M16C_FLG_ZERO;
	} else {
		M16C_REG_FLG = (M16C_REG_FLG | M16C_FLG_ZERO) & ~M16C_FLG_CARRY;
	}
	bitaddr_set(icode & 0xf,&arglen,1);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_btsts_dst(%04x) not implemented\n",icode);
}

void 
m16c_bxor_src(uint16_t  icode) 
{
	int src;
	int arglen;
	src = bitaddr_get(icode & 0xf,&arglen);
	if(!!src != !!(M16C_REG_FLG & M16C_FLG_CARRY)) {
		M16C_REG_FLG |= M16C_FLG_CARRY;	
	} else {
		M16C_REG_FLG &= ~M16C_FLG_CARRY;	
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_bxor_src(%04x) not implemented\n",icode);
}

void 
m16c_cmp_size_g_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
	}
	result = dst - src;
	M16C_REG_PC += arglen;	
	if(isword) {
		M16C_REG_PC += 2;	
	} else {
		M16C_REG_PC++;	
	}
	sub_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_cmp_size_g_immdst(%04x)\n",icode);
}

void 
m16c_cmp_size_q_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;	
	src = (icode>>4)&0xf; 	
	result = dst - src;
	sub_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_cmp_size_q_immdst(%04x)\n",icode);
}

void 
m16c_cmp_b_s_immdst(uint16_t  icode) 
{
	int arglen;				
	uint8_t imm; 
	uint8_t dst;
	uint8_t result;
	imm =  M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	dst = am2b_get(icode,&arglen);
	result = dst - imm;	
	M16C_REG_PC+=arglen;
	sub_flags(dst,imm,result,0);
	fprintf(stderr,"instr m16c_cmp_b_s_immdst(%04x)\n",icode);
}

void 
m16c_cmp_size_g_srcdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);	
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	result = dst - src;
	M16C_REG_PC+=arglen;
	sub_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_cmp_size_g_srcdst(%04x)\n",icode);
}

void 
m16c_cmp_b_s_srcr0l(uint16_t  icode) 
{
	uint8_t src,dst,result;	
	int arglen;
	src = am3b_get(icode,&arglen);	
	M16C_REG_PC+=arglen;
	if(icode & (1<<2)) {
		dst = M16C_REG_R0H;
	 	result = dst - src;
	} else {
		dst = M16C_REG_R0L;
		result = dst - src;
	}
	sub_flags(dst,src,result,0);
	fprintf(stderr,"instr m16c_cmp_b_s_srcr0l(%04x)\n",icode);
}

/*
 *
 */
void 
m16c_dadc_b_imm8_r0l(uint16_t  icode) 
{
	uint8_t imm8 = M16C_Read8(M16C_REG_PC);
	uint8_t result;
	uint8_t src,dst;
	M16C_REG_PC+=1;
	src = bcd_to_uint16(imm8);
	dst = bcd_to_uint16(M16C_REG_R0L);
	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = src + dst + 1;
	} else {
		result = src + dst;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>99) {
		M16C_REG_FLG |= M16C_FLG_CARRY;
		result-=100;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dadc_b_imm8_r0l(%04x)\n",icode);
}

void 
m16c_dadc_w_imm16_r0(uint16_t  icode) 
{
	uint16_t imm16 = M16C_Read16(M16C_REG_PC);
	uint16_t result;
	uint16_t src,dst;
	M16C_REG_PC+=2;
	src = bcd_to_uint16(imm16);
	dst = bcd_to_uint16(M16C_REG_R0);
	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = src + dst + 1;
	} else {
		result = src + dst;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>9999) {
		result -= 10000;
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dadc_w_imm16_r0(%04x) not implemented\n",icode);
}

void 
m16c_dadc_b_r0h_r0l(uint16_t  icode) 
{
	uint8_t result;
	uint8_t src,dst;
	src = bcd_to_uint16(M16C_REG_R0H);
	dst = bcd_to_uint16(M16C_REG_R0L);
	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = src + dst + 1;
	} else {
		result = src + dst;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>99) {
		result-=100;
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	
	fprintf(stderr,"instr m16c_dadc_b_r0h_r0l(%04x) not implemented\n",icode);
}

void 
m16c_dadc_w_r0_r1(uint16_t  icode) 
{
	uint16_t result;
	uint16_t src,dst;
	src = bcd_to_uint16(M16C_REG_R1);
	dst = bcd_to_uint16(M16C_REG_R0);
	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = src + dst + 1;
	} else {
		result = src + dst;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>9999) {
		result -= 10000;
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dadc_w_r0_r1(%04x) not implemented\n",icode);
}

void 
m16c_dadd_b_imm8_r0l(uint16_t  icode) 
{
	uint8_t imm8 = M16C_Read8(M16C_REG_PC);
	uint8_t result;
	uint8_t src,dst;
	M16C_REG_PC+=1;
	src = bcd_to_uint16(imm8);
	dst = bcd_to_uint16(M16C_REG_R0L);
	result = src + dst;
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>99) {
		result -= 100;
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dadd_b_imm8_r0l(%04x) not implemented\n",icode);
}

void 
m16c_dadd_w_imm16_r0(uint16_t  icode) 
{
	uint16_t imm16 = M16C_Read16(M16C_REG_PC);
	uint16_t result;
	uint16_t src,dst;
	M16C_REG_PC+=2;
	src = bcd_to_uint16(imm16);
	dst = bcd_to_uint16(M16C_REG_R0);
	result = src + dst;
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>9999) {
		result -= 10000;
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dadd_w_imm16_r0(%04x) not implemented\n",icode);
}

void 
m16c_dadd_b_r0h_r0l(uint16_t  icode) 
{
	uint8_t result;
	uint8_t src,dst;
	src = bcd_to_uint16(M16C_REG_R0H);
	dst = bcd_to_uint16(M16C_REG_R0L);
	result = src + dst;
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>99) {
		result -= 100;
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dadd_b_r0h_r0l(%04x) not implemented\n",icode);
}

void 
m16c_dadd_w_r0_r1(uint16_t  icode) 
{
	uint16_t result;
	uint16_t src,dst;
	src = bcd_to_uint16(M16C_REG_R1);
	dst = bcd_to_uint16(M16C_REG_R0);
	result = src + dst;
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result>9999) {
		result -= 10000;
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dadd_w_r0_r1(%04x) not implemented\n",icode);
}

void 
m16c_dec_b_dst(uint16_t  icode) 
{
	int arglen;
	uint8_t value;
	value = am2b_get(icode,&arglen);
	value--;	
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(value == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	} else if(value & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	am2b_set(icode,&arglen,value); 
	fprintf(stderr,"instr m16c_dec_b_dst(%04x) \n",icode);
}

void 
m16c_dec_w_dst(uint16_t  icode) 
{
	uint16_t value;
	fprintf(stderr,"instr m16c_dec_w_dst(%04x) not implemented\n",icode);
	if(icode & (1<<3)) {
		M16C_REG_A1--;
		value = M16C_REG_A1;
		
	} else {
		M16C_REG_A0--;
		value = M16C_REG_A0;
	}
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(value == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	} else if(value & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	
}

void 
m16c_div_b_imm(uint16_t  icode) 
{
	int16_t r0;
	uint8_t remainder;
	int16_t quotient;
	int8_t div;
	div = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC+=1;
	if(div) {
		r0 = M16C_REG_R0;
		quotient = r0 / div;
		remainder = r0 - (quotient * div);
		M16C_REG_R0 = (int8_t)quotient  | ((uint8_t)remainder << 8);
		if((quotient < -128) || (quotient > 127)) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}
	fprintf(stderr,"instr m16c_div_size_imm(%04x) not implemented\n",icode);
}

void 
m16c_div_w_imm(uint16_t  icode) 
{
	int32_t r2r0;
	uint16_t remainder;
	int32_t quotient;
	int16_t div;
	div = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC+=2;
	if(div) {
		r2r0 = M16C_REG_R0 | ((uint32_t)M16C_REG_R1<<16);
		quotient = r2r0 / div;
		remainder = r2r0 - (quotient *div);
		M16C_REG_R0 = quotient;
		M16C_REG_R2 = remainder;
		if((quotient < -32768) || (quotient > 32767)) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}	
}

void 
m16c_div_b_src(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	int16_t r0;
	uint8_t remainder;
	int16_t quotient;
	int8_t div;
	div = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC+=arglen;
	if(div) {
		r0 = M16C_REG_R0;
		quotient = r0 / div;
		remainder = r0 - (quotient * div);
		M16C_REG_R0 = (int8_t)quotient  | ((uint16_t)remainder << 8);
		if((quotient < -128) || (quotient > 127)) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}
	fprintf(stderr,"instr m16c_div_size_src(%04x)\n",icode);
}

void 
m16c_div_w_src(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	int32_t r2r0;
	uint16_t remainder;
	int32_t quotient;
	int16_t div;
	div = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC+=arglen;
	if(div) {
		r2r0 = M16C_REG_R0 | ((uint32_t)M16C_REG_R1<<16);
		quotient = r2r0 / div;
		remainder = r2r0 - (quotient *div);
		M16C_REG_R0 = quotient;
		M16C_REG_R2 = remainder;
		if((quotient < -32768) || (quotient > 32767)) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}	
	fprintf(stderr,"instr m16c_div_size_src(%04x)\n",icode);
}

void 
m16c_divu_b_imm(uint16_t  icode) 
{
	uint16_t r0;
	uint8_t remainder;
	uint16_t quotient;
	uint8_t div;
	div = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC+=1;
	if(div) {
		r0 = M16C_REG_R0;
		quotient = r0 / div;
		remainder = r0 - (quotient * div);
		M16C_REG_R0 = (uint8_t)quotient  | ((uint16_t)remainder << 8);
		if(quotient & 0xff00) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}
	fprintf(stderr,"instr m16c_divu_size_imm(%04x)\n",icode);
}

void 
m16c_divu_w_imm(uint16_t  icode) 
{
	uint32_t r2r0;
	uint16_t remainder;
	uint32_t quotient;
	uint16_t div;
	div = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC+=2;
	if(div) {
		r2r0 = M16C_REG_R0 | ((uint32_t)M16C_REG_R1<<16);
		quotient = r2r0 / div;
		remainder = r2r0 - (quotient *div);
		M16C_REG_R0 = quotient;
		M16C_REG_R2 = remainder;
		if(quotient  & 0xffff0000) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}	
	fprintf(stderr,"instr m16c_divu_w_imm(%04x)\n",icode);
}


void 
m16c_divu_b_src(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	uint16_t r0;
	uint8_t remainder;
	uint16_t quotient;
	uint8_t div;
	div = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC+=arglen;
	if(div) {
		r0 = M16C_REG_R0;
		quotient = r0 / div;
		remainder = r0 - (quotient * div);
		M16C_REG_R0 = (int8_t)quotient  | ((uint16_t)remainder << 8);
		if(quotient & 0xff00) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}
	fprintf(stderr,"instr m16c_divu_size_src(%04x) not implemented\n",icode);
}

void 
m16c_divu_w_src(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	uint32_t r2r0;
	uint16_t remainder;
	uint32_t quotient;
	uint16_t div;
	div = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC+=arglen;
	if(div) {
		r2r0 = M16C_REG_R0 | ((uint32_t)M16C_REG_R1<<16);
		quotient = r2r0 / div;
		remainder = r2r0 - (quotient *div);
		M16C_REG_R0 = quotient;
		M16C_REG_R2 = remainder;
		if(quotient  & 0xffff0000) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		} else {
			M16C_REG_FLG &= ~M16C_FLG_OVERFLOW;
		}	
	} else {
		M16C_REG_FLG |= M16C_FLG_OVERFLOW;
	}	
	fprintf(stderr,"instr m16c_divu_size_src(%04x) not implemented\n",icode);
}

/*
 * ------------------------------------------------------
 * Where is the difference to div ?
 * ------------------------------------------------------
 */
void 
m16c_divx_b_imm(uint16_t  icode) 
{
	m16c_div_b_imm(icode);	
	fprintf(stderr,"instr m16c_divx_b_imm(%04x)\n",icode);
}

void 
m16c_divx_w_imm(uint16_t  icode) 
{
	m16c_div_w_imm(icode);	
	fprintf(stderr,"instr m16c_divx_size_imm(%04x) not implemented\n",icode);
}

void 
m16c_divx_b_src(uint16_t  icode) 
{
	m16c_div_b_src(icode);	
	fprintf(stderr,"instr m16c_divx_size_src(%04x) not implemented\n",icode);
}

void 
m16c_divx_w_src(uint16_t  icode) 
{
	m16c_div_w_src(icode);	
	fprintf(stderr,"instr m16c_divx_size_src(%04x) not implemented\n",icode);
}

/*
 * ---------------------------------------------------------
 * Decimal subtract
 * ---------------------------------------------------------
 */
void 
m16c_dsbb_b_imm8_r0l(uint16_t  icode) 
{
	uint8_t imm8 = M16C_Read8(M16C_REG_PC);
	uint8_t result;
	uint8_t src,dst;
	M16C_REG_PC+=1;
	src = bcd_to_uint16(imm8);
	dst = bcd_to_uint16(M16C_REG_R0L);
	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = src - dst;
	} else {
		result = src - dst - 1;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result & 0x80) {
		result += 100;
	} else {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsbb_b_imm8_r0l(%04x) not implemented\n",icode);
}

void 
m16c_dsbb_w_imm16_r0(uint16_t  icode) 
{
	uint16_t imm16 = M16C_Read16(M16C_REG_PC);
        uint16_t result;
        uint16_t src,dst;
        M16C_REG_PC+=2;
        src = bcd_to_uint16(imm16);
        dst = bcd_to_uint16(M16C_REG_R0);
        
        if(M16C_REG_FLG & M16C_FLG_CARRY) {
                result = src - dst;
        } else {
                result = src - dst - 1;
        }
        M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
        if(result & 0x8000) {
		result += 10000;
	} else {
                M16C_REG_FLG |= M16C_FLG_CARRY;
	}
        if(result & 0x8000) {
                M16C_REG_FLG |= M16C_FLG_SIGN;
	}
        if(!result) {
                M16C_REG_FLG |= M16C_FLG_ZERO;
        }
        M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsbb_w_imm16_r0(%04x) not implemented\n",icode);
}

void 
m16c_dsbb_b_r0h_r0l(uint16_t  icode) 
{
	uint8_t result;
	uint8_t src,dst;
	src = bcd_to_uint16(M16C_REG_R0H);
	dst = bcd_to_uint16(M16C_REG_R0L);
	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = src - dst;
	} else {
		result = src - dst - 1;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result & 0x80) {
		result += 100;
	} else {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsbb_b_r0h_r0l(%04x) not implemented\n",icode);
}

void 
m16c_dsbb_w_r1_r0(uint16_t  icode) 
{
	uint16_t result;
	uint16_t src,dst;
	src = bcd_to_uint16(M16C_REG_R1);
	dst = bcd_to_uint16(M16C_REG_R0);
	
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = src - dst;
	} else {
		result = src - dst - 1;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result & 0x8000) {
		result += 10000;
	} else {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsbb_w_r1_r0(%04x)\n",icode);
}

void 
m16c_dsub_b_imm8_r0l(uint16_t  icode) 
{
	uint8_t imm8 = M16C_Read8(M16C_REG_PC);
	uint8_t result;
	uint8_t src,dst;
	M16C_REG_PC+=1;
	src = bcd_to_uint16(imm8);
	dst = bcd_to_uint16(M16C_REG_R0L);
	result = src - dst;
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result & 0x80) {
		result = result + 100;
	} else {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsub_b_imm8_r0l(%04x) not implemented\n",icode);
}

void 
m16c_dsub_w_imm16_r0(uint16_t  icode) 
{
	uint16_t imm16 = M16C_Read16(M16C_REG_PC);
        uint16_t result;
        uint16_t src,dst;
        M16C_REG_PC+=2;
        src = bcd_to_uint16(imm16);
        dst = bcd_to_uint16(M16C_REG_R0);
        result = src - dst;
        M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
        if(result & 0x8000) {
		result += 10000;
	} else {
                M16C_REG_FLG |= M16C_FLG_CARRY;
	}
        if(result & 0x8000) {
                M16C_REG_FLG |= M16C_FLG_SIGN;
	}
        if(result == 0) {
                M16C_REG_FLG |= M16C_FLG_ZERO;
        }
        M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsub_w_imm16_r0(%04x) not implemented\n",icode);
}

void 
m16c_dsub_b_r0h_r0l(uint16_t  icode) 
{
	uint8_t result;
	uint8_t src,dst;
	src = bcd_to_uint16(M16C_REG_R0H);
	dst = bcd_to_uint16(M16C_REG_R0L);
	result = src - dst;
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result & 0x80) {
		result += 100;	
	} else {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(result == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0L = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsub_b_r0h_r0l(%04x) not implemented\n",icode);
}

void 
m16c_dsub_w_r1_r0(uint16_t  icode) 
{
	uint16_t result;
	uint16_t src,dst;
	src = bcd_to_uint16(M16C_REG_R1);
	dst = bcd_to_uint16(M16C_REG_R0);
	result = src - dst;
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(result & 0x8000) {
		result += 10000;
	} else {
		M16C_REG_FLG |= M16C_FLG_CARRY;
	}
	if(result & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	M16C_REG_R0 = uint16_to_bcd(result);
	fprintf(stderr,"instr m16c_dsub_w_r1_r0(%04x) not implemented\n",icode);
}

/*
 * --------------------------------------------------------------
 * Enter
 * --------------------------------------------------------------
 */

void 
m16c_enter(uint16_t  icode) 
{
        uint8_t imm=M16C_Read8(M16C_REG_PC++);
	M16C_REG_SP = M16C_REG_SP-2;
	M16C_Write16(M16C_REG_FB,M16C_REG_SP);
	M16C_REG_FB = M16C_REG_SP;	
	M16C_REG_SB -= imm;
	fprintf(stderr,"instr m16c_enter(%04x)\n",icode);
}

void 
m16c_exitd(uint16_t  icode) 
{
	M16C_REG_SP = M16C_REG_FB;
	M16C_REG_FB = M16C_Read16(M16C_REG_SP);
	M16C_REG_SP += 2;
	M16C_REG_PC = M16C_Read16(M16C_REG_SP);
	M16C_REG_SP += 2;
	M16C_REG_PC |= (M16C_Read16(M16C_REG_SP) & 0xff)<<16;
	M16C_REG_SP += 1;
	fprintf(stderr,"instr m16c_exitd(%04x) not implemented\n",icode);
}

void 
m16c_exts_b_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	int16_t dst;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC+=arglen;
	if(dst & 0x80) {
		dst |= 0xff00;
	} else {
		dst &= 0xff;
	}		
	/* hack */
	am1_set(icode | (1<<8),icode & 0xf,&arglen,isword,dst);
	if(dst & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	} else {
		M16C_REG_FLG &= ~M16C_FLG_SIGN;
	}
	if(dst == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	} else {
		M16C_REG_FLG &= ~M16C_FLG_ZERO;
	}
	fprintf(stderr,"instr m16c_exts_b_dst(%04x)\n",icode);
}

void 
m16c_exts_w_r0(uint16_t  icode) 
{
	uint32_t dst = M16C_REG_R0;
	if(dst & 0x8000) {
		dst |= 0xffff0000;
		M16C_REG_R2=0xffff;
	} else {
		M16C_REG_R2=0;
	}
	fprintf(stderr,"instr m16c_exts_w_r0(%04x)\n",icode);
}

void 
m16c_fclr_dst(uint16_t  icode) 
{
	int dst = (icode >> 4) & 7;
	M16C_SET_REG_FLG(M16C_REG_FLG & ~(1<<dst));
	fprintf(stderr,"instr m16c_fclr_dst(%04x)\n",icode);
}

void 
m16c_fset_dst(uint16_t  icode) 
{
	int dst = (icode >> 4) & 7;
	M16C_SET_REG_FLG(M16C_REG_FLG | (1<<dst));
	fprintf(stderr,"instr m16c_fset_dst(%04x)\n",icode);
}

void 
m16c_inc_b_dst(uint16_t  icode) 
{
	int arglen;
	uint8_t value;
	value = am2b_get(icode,&arglen);
	value++;	
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(value == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	} else if(value & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	am2b_set(icode,&arglen,value); 
	fprintf(stderr,"instr m16c_inc_b_dst(%04x)\n",icode);
}

void 
m16c_inc_w_dst(uint16_t  icode) 
{
	uint16_t value;
	if(icode & (1<<3)) {
		M16C_REG_A1++;
		value = M16C_REG_A1;
		
	} else {
		M16C_REG_A0++;
		value = M16C_REG_A0;
	}
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);
	if(value == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	} else if(value & 0x8000) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	fprintf(stderr,"instr m16c_inc_w_dst(%04x)\n",icode);
}

void 
m16c_int_imm(uint16_t  icode) 
{
	uint8_t src;
	src = M16C_Read8(M16C_REG_PC++);
	if(src<32) {
		M16C_SET_REG_FLG(M16C_REG_FLG & ~M16C_FLG_U);
	}	
	M16C_REG_SP -= 1;	
	/* Order not clear !!!! **************************************************** */
	M16C_Write8(M16C_REG_FLG,M16C_REG_SP);	
	M16C_REG_SP -= 1;	
	M16C_Write8(M16C_REG_PC >> 16,M16C_REG_SP);	
	M16C_REG_SP -= 2;	
	M16C_Write16(M16C_REG_PC,M16C_REG_SP);	
	M16C_REG_PC = M16C_REG_INTB + (((uint32_t)src)<<2);
	M16C_REG_FLG &= ~(M16C_FLG_I | M16C_FLG_D);
	fprintf(stderr,"instr m16c_int_imm(%04x) not implemented\n",icode);
}

/*
 * ------------------------------------------------
 * Interrupt on overflow
 * ------------------------------------------------
 */
void 
m16c_into(uint16_t  icode) 
{
	if(!(M16C_REG_FLG & M16C_FLG_OVERFLOW)) {
		return;
	}
	M16C_REG_SP -= 1;	
	/* Order not clear !!!! **************************************************** */
	M16C_Write8(M16C_REG_FLG,M16C_REG_SP);	
	M16C_REG_SP -= 1;	
	M16C_Write8(M16C_REG_PC >> 16,M16C_REG_SP);	
	M16C_REG_SP -= 2;	
	M16C_Write16(M16C_REG_PC,M16C_REG_SP);	
	M16C_REG_PC = M16C_Read16(0xfffe0);
	fprintf(stderr,"instr m16c_int0(%04x)\n",icode);
}

void 
m16c_jcnd1(uint16_t  icode) 
{
	uint8_t cnd = icode & 0x7;
	int8_t dsp = M16C_Read8(M16C_REG_PC);
	if(check_condition(cnd)) {
		M16C_REG_PC += dsp;	
	} else {
		M16C_REG_PC++;
	}
	fprintf(stderr,"instr m16c_jcnd1(%04x) not implemented\n",icode);
}

void 
m16c_jcnd2(uint16_t  icode) 
{
	uint8_t cnd = (icode & 0x7) | 0x8;
	int8_t dsp = M16C_Read8(M16C_REG_PC);
	if(check_condition(cnd)) {
		M16C_REG_PC += dsp;
	} else {
		M16C_REG_PC++;
	}
	fprintf(stderr,"instr m16c_jcnd2(%04x)\n",icode);
}

void 
m16c_jmp_s(uint16_t  icode) 
{
	uint8_t dsp = icode & 0x7;	
	M16C_REG_PC += dsp;
	fprintf(stderr,"instr m16c_jmp_s(%04x)\n",icode);
}

void 
m16c_jmp_b(uint16_t  icode) 
{
	int8_t dsp = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC += dsp;
	fprintf(stderr,"instr m16c_jmp_b(%04x)\n",icode);
}

void 
m16c_jmp_w(uint16_t  icode) 
{
	int16_t dsp = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC += dsp;
	fprintf(stderr,"instr m16c_jmp_w(%04x)\n",icode);
}

void 
m16c_jmp_a(uint16_t  icode) 
{
	uint32_t addr = M16C_Read16(M16C_REG_PC);
	addr |= (uint32_t)M16C_Read8(M16C_REG_PC+2) << 16;
	M16C_REG_PC = addr;
	fprintf(stderr,"instr m16c_jmp_a(%04x)\n",icode);
}

/* 
 * special addressing modes for jumps 
 * word specifier for jump width 
 */
static uint16_t 
jamw_get(uint16_t icode,int *arglen) {
	int am = icode & 0xf;		
	uint16_t retval;
	switch(am) {
		case 0:
			*arglen = 0;
			return M16C_REG_R0;
		case 1:
			*arglen = 0;
			return M16C_REG_R1;
		case 2:
			*arglen = 0;
			return M16C_REG_R2;
		case 3:
			*arglen = 0;
			return M16C_REG_R3;
		case 4:	
			*arglen = 0;
			return M16C_REG_A0;
		case 5:	
			*arglen = 0;
			return M16C_REG_A1;
		case 6:
			*arglen = 0;
			retval = M16C_Read16(M16C_REG_A0);
			break;

		case 7:
			*arglen = 0;
			retval = M16C_Read16(M16C_REG_A1);
			break;

		case 8:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				retval = M16C_Read16((M16C_REG_A0 + dsp8) & 0xffff);
				*arglen = 1;
			}
			break;

		case 9:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				retval = M16C_Read16((M16C_REG_A1 + dsp8) & 0xffff);
				*arglen = 1;
			}
			break;

		case 10:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				retval = M16C_Read16((M16C_REG_SB + dsp8) & 0xffff);
				*arglen = 1;
			}
			break;

		case 11:
			{
				int8_t dsp8 = M16C_Read8(M16C_REG_PC);
				*arglen = 1;
				retval = M16C_Read16((M16C_REG_FB + dsp8)&0xffff);
			}
			break;

		case 12:
			{
				uint32_t dsp24 = M16C_Read16(M16C_REG_PC);
				dsp24 |= M16C_Read8(M16C_REG_PC+2)<<16;
				*arglen = 3;
				retval = M16C_Read16((M16C_REG_A0 + dsp24) & 0xfffff);
			}
			break;

		case 13:
			{
				uint32_t dsp24 = M16C_Read16(M16C_REG_PC);
				dsp24 |= M16C_Read8(M16C_REG_PC+2)<<16;
				*arglen = 3;
				retval = M16C_Read16((M16C_REG_A1 + dsp24) & 0xfffff);
			}
			break;

		case 14:
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
				*arglen = 2;
				retval = M16C_Read16((M16C_REG_SB + dsp16) & 0xffff);
			}
			break;

		case 15: /* abs16 */
			{
				*arglen = 2;
				retval = M16C_Read16((M16C_REG_PC));
			}
			break;
		default:
			*arglen = 0;
			retval = 0;
			break;
	}
	return retval;
}

/* 
 * ------------------------------------------------------------
 * special addressing modes for jumps 
 * address specifier for jump width 
 * ------------------------------------------------------------
 */

static uint32_t
jama_get(uint16_t icode,int *arglen) 
{
	int am = icode & 0xf;	
	uint32_t retval;
	switch(am) {
		case 0:
			retval = M16C_REG_R0 + ((uint32_t)M16C_REG_R2 << 16);
			*arglen = 0;
			break;
		case 1:
			retval = M16C_REG_R1 + ((uint32_t)M16C_REG_R3 << 16);
			*arglen = 0;
			break;
		case 4:
			retval = M16C_REG_A0 + ((uint32_t)M16C_REG_A1 << 16);
			*arglen = 0;
			break;
		case 6:
			/* [A0] */
			retval = M16C_Read16(M16C_REG_A0);
			*arglen = 0;
			break;
		case 7:
			/* [A1] */
			retval = M16C_Read16(M16C_REG_A1);
			*arglen = 0;
			break;
		case 8:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				*arglen = 1;
				retval = M16C_Read16(M16C_REG_A0 + dsp8);
			}
			break;

		case 9:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				*arglen = 1;
				retval =  M16C_Read16(M16C_REG_A1 + dsp8);
			}
			break;

		case 10:
			{
				uint8_t dsp8 = M16C_Read8(M16C_REG_PC);
				*arglen = 1;
				retval =  M16C_Read16(M16C_REG_SB + dsp8);
			}
			break;

		case 11:
			{
				int8_t dsp8 = M16C_Read8(M16C_REG_PC);
				*arglen = 1;
				retval =  M16C_Read16(M16C_REG_FB + dsp8);
			}
			break;
		case 12:
			{
				uint32_t dsp20 = M16C_Read16(M16C_REG_PC);
				dsp20 |= (M16C_Read8(M16C_REG_PC+2) & 0xf)<<16;
				*arglen = 3;
				retval = M16C_Read16((M16C_REG_A0 + dsp20) & 0xfffff);
			}
			break;
		case 13:
			{
				uint32_t dsp20 = M16C_Read16(M16C_REG_PC);
				dsp20 |= (M16C_Read8(M16C_REG_PC+2) & 0xf)<<16;
				*arglen = 3;
				retval = M16C_Read16((M16C_REG_A1 + dsp20) & 0xfffff);
			}
			break;
		case 14: /* dsp16[SB] */
			{
				uint16_t dsp16 = M16C_Read16(M16C_REG_PC);
				*arglen = 2;
				retval = M16C_Read16((M16C_REG_SB + dsp16) & 0xfffff);
			}
			break;
		case 15: /* abs16 */
			{
				retval = M16C_Read16(M16C_REG_PC);
				*arglen = 2;
			}
			break;
		default:
			fprintf(stderr,"Unknown addressing mode %d in line %d\n",am,__LINE__);
			retval=0;

	}
	return retval;
}

void 
m16c_jmpi_w_src(uint16_t  icode) 
{
	int arglen;
	int16_t ofs;
	ofs = jamw_get(icode,&arglen); 
	M16C_REG_PC =  (M16C_REG_PC -2 + ofs) & 0xfffff;
	fprintf(stderr,"instr m16c_jmpi_w_src(%04x)\n",icode);
}

void 
m16c_jmpi_a(uint16_t  icode) 
{
	int arglen;
	uint32_t addr = jama_get(icode,&arglen);
	M16C_REG_PC = addr;
	fprintf(stderr,"instr m16c_jmpi_a(%04x)\n",icode);
}

void 
m16c_jmps_imm8(uint16_t  icode) 
{
	uint8_t imm8 = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC = 0xffffe - imm8 - imm8;	
	fprintf(stderr,"instr m16c_jmps_imm8(%04x)\n",icode);
}

void 
m16c_jsr_w(uint16_t  icode) 
{
	int16_t label = M16C_Read16(M16C_REG_PC);
	M16C_REG_SP -= 1;
	M16C_Write8((M16C_REG_PC >> 16) & 0xf,M16C_REG_SP);
	M16C_REG_SP -= 2;
	M16C_Write16(M16C_REG_PC,M16C_REG_SP);
	M16C_REG_PC += label;
	fprintf(stderr,"instr m16c_jsr_w(%04x)\n",icode);
}

void 
m16c_jsr_a(uint16_t  icode) 
{
	uint32_t abs20 = M16C_Read16(M16C_REG_PC);
	abs20 |= ((M16C_Read8(M16C_REG_PC+2) & 0xf) << 16);
	M16C_REG_SP -= 1;
	M16C_Write8(M16C_REG_PC & 0xf,M16C_REG_SP);
	M16C_REG_SP -= 2;
	M16C_Write16(M16C_REG_PC,M16C_REG_SP);
	M16C_REG_PC = abs20;
	fprintf(stderr,"instr m16c_jsr_a(%04x)\n",icode);
}

void 
m16c_jsri_w(uint16_t  icode) 
{
	int arglen;
	uint16_t label = jamw_get(icode,&arglen);	
	M16C_REG_SP -= 1;
	M16C_Write8((M16C_REG_PC >> 16) & 0xf,M16C_REG_SP);
	M16C_REG_SP -= 2;
	M16C_Write16(M16C_REG_PC & 0xf,M16C_REG_SP);
	M16C_REG_PC += label;
	fprintf(stderr,"instr m16c_jsri_w(%04x)\n",icode);
}

/*
 * Buggy in the manual
 */
void 
m16c_jsri_a(uint16_t  icode) 
{
	int arglen;
	uint32_t abs20 = jama_get(icode,&arglen);
	M16C_REG_SP -= 1;
	M16C_Write8((M16C_REG_PC >> 16) & 0xf,M16C_REG_SP);
	M16C_REG_SP -= 2;
	M16C_Write16(M16C_REG_PC,M16C_REG_SP);
	M16C_REG_PC = abs20;
	fprintf(stderr,"instr m16c_jsri_a(%04x)\n",icode);
}

void 
m16c_jsrs_imm8(uint16_t  icode) 
{
	uint8_t imm8 = M16C_Read8(M16C_REG_PC);
	M16C_REG_SP -= 1;
	M16C_Write8((M16C_REG_PC >> 16) & 0xf,M16C_REG_SP);
	M16C_REG_SP -= 2;
	M16C_Write16(M16C_REG_PC,M16C_REG_SP);
	M16C_REG_PC = 0xffffe - imm8 - imm8;	
	fprintf(stderr,"instr m16c_jsrs_imm8(%04x) not implemented\n",icode);
}

/* Set control register */
void
set_creg(int creg,uint16_t value) {
        switch(creg) {
                case 0:
                        fprintf(stderr,"unknown control register %d\n",creg);
                        exit(1349);

                case 1:
			M16C_REG_INTB = (M16C_REG_INTB & 0xf0000) | value;
			return;

                case 2:
			M16C_REG_INTB = (M16C_REG_INTB & 0xffff) | (((uint32_t)value & 0xf)<<16); 
			return;

                case 3:
			M16C_SET_REG_FLG(value); 
			return;
                case 4:
			M16C_SET_REG_ISP(value);
			return;
                case 5:
			M16C_SET_REG_USP(value);
			return;

                case 6:
			M16C_REG_SB = value;
			return;
                case 7:
			M16C_REG_FB = value;
			return;

                default:
                        return;
        }
}

static uint16_t
get_creg(int creg) {
        switch(creg) {
                case 0:
                        fprintf(stderr,"unknown control register %d\n",creg);
                        exit(1349);

                case 1:
			return M16C_REG_INTB  & 0xffff;

                case 2:
			return M16C_REG_INTB >> 16;

                case 3:
			return M16C_REG_FLG; 

                case 4:
			return M16C_REG_ISP;

                case 5:
			return M16C_REG_USP;

                case 6:
			return M16C_REG_SB;

                case 7:
			return M16C_REG_FB;

                default:
			fprintf(stderr,"unknown creg %d\n",creg);
                        return 0;
        }
}

void 
m16c_ldc_imm16_dst(uint16_t  icode) 
{
	uint16_t imm16 = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC+=2;
	set_creg((icode >> 4) & 0xf,imm16);	
	fprintf(stderr,"instr m16c_ldc_imm16_dst(%04x) not implemented\n",icode);
}

void 
m16c_ldc_srcdst(uint16_t  icode) 
{
	int isword = 1;
	int arglen;
	uint16_t src =  am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;
	set_creg((icode>>4)&0xf,src);
	fprintf(stderr,"instr m16c_ldc_srcdst(%04x) not implemented\n",icode);
}

/*
 * -------------------------------------------------------------
 * Documentations of ldctx is totally confuse
 * -------------------------------------------------------------
 */
void 
m16c_ldctx(uint16_t  icode) 
{
	uint32_t abs20;		
	uint16_t abs16;
	uint32_t addr20;
	uint8_t regset;
	uint8_t spdiff=0;
	abs16 = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC+=2;
	abs20 = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC+=2;
	abs20 |= (M16C_Read8(M16C_REG_PC) & 0xf) << 16;
	M16C_REG_PC+=1;
	addr20 = (abs20 + ((uint32_t)abs16 << 1)) & 0xfffff;
	regset = M16C_Read8(addr20);
	addr20 = (addr20+1) & 0xfffff;
	if(regset & 0x80) {
		M16C_REG_FB = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(regset & 0x40) {
		M16C_REG_SB = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(regset & 0x20) {
		M16C_REG_A1 = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(regset & 0x10) {
		M16C_REG_A0 = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(regset & 8) {
		M16C_REG_R3 = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(regset & 4) {
		M16C_REG_R2 = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(regset & 2) {
		M16C_REG_R1 = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(regset & 1) {
		M16C_REG_R0 = M16C_Read16(M16C_REG_SP);	
		M16C_REG_SP+=2;
		spdiff+=2;
	}
	if(spdiff != M16C_Read8(addr20)) {
		fprintf(stderr,"LDCTX wrong spdiff\n");
	}
	fprintf(stderr,"instr m16c_ldctx(%04x) not implemented\n",icode);
}

void 
m16c_lde_size_abs20_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint32_t abs20;
	uint16_t val;
	am1_get(icode,icode & 0xf,&arglen,isword) ; /* dummy for getting arglen ! shit */
	abs20 = M16C_Read16(M16C_REG_PC+arglen);	
	abs20 |= (M16C_Read8(M16C_REG_PC+arglen+2) & 0xf)<<16;	
	if(isword) {
		val = M16C_Read16(abs20);	
	} else {
		val = M16C_Read8(abs20);	
	}
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC+= arglen + 3;
	fprintf(stderr,"instr m16c_lde_size_abs20_dst(%04x) not implemented\n",icode);
}

void 
m16c_lde_size_dsp_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint32_t dsp20;
	uint32_t addr;
	uint16_t val;
	am1_get(icode,icode & 0xf,&arglen,isword) ; /* dummy for getting arglen ! shit */
	dsp20 = M16C_Read16(M16C_REG_PC+arglen);	
	dsp20 |= (M16C_Read8(M16C_REG_PC+arglen+2) & 0xf)<<16;	
	addr = (M16C_REG_A0 + dsp20) & 0xfffff;
	if(isword) {
		val = M16C_Read16(addr);	
	} else {
		val = M16C_Read8(addr);	
	}
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_lde_size_dsp_dst(%04x)\n",icode);
}

void 
m16c_lde_size_a1a0_dst(uint16_t  icode) 
{
	int arglen;
	int isword = (icode & (1<<8));
	uint32_t addr = M16C_REG_A0 + ((M16C_REG_A1 & 0xf) << 16);
	uint16_t val;
	if(isword) {
		val = M16C_Read16(addr);	
	} else {
		val = M16C_Read8(addr);	
	}
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_lde_size_a1a0_dst(%04x)\n",icode);
}

void 
m16c_ldipl_imm(uint16_t  icode) 
{
	int ipl = icode & 7;
	M16C_REG_IPL = ipl;
	// update_interrupts
}

void 
m16c_mov_size_g_immdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint16_t imm;
	am1_get(icode,icode & 0xf,&arglen,isword) ; /* dummy for getting arglen ! shit */
	if(isword) {
		imm = M16C_Read16(M16C_REG_PC+arglen);
	} else {
		imm = M16C_Read8(M16C_REG_PC+arglen);
	}
	am1_set(icode,icode & 0xf,&arglen,isword,imm);
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_mov_size_g_immdst(%04x)\n",icode);
}

void 
m16c_mov_size_q_immdst(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	int8_t imm = (icode >> 4) & 0xf;
	if(imm & 8) {
		imm |= 0xf0;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,imm);
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_mov_size_q_immdst(%04x)\n",icode);
}

void 
m16c_mov_b_s_imm8_dst(uint16_t  icode) 
{
	int arglen;
	uint8_t imm8 = M16C_Read8(M16C_REG_PC++);
	am2b_set(icode,&arglen,imm8) ;
	M16C_REG_PC += arglen;	
	fprintf(stderr,"instr m16c_mov_b_s_imm8_dst(%04x)\n",icode);
}

void 
m16c_mov_size_s_immdst(uint16_t  icode) 
{
	int isword = icode & (1<<6);
	uint16_t imm;
	if(isword) {
		imm = M16C_Read16(M16C_REG_PC);	
		M16C_REG_PC+=2;
	} else {
		imm = M16C_Read8(M16C_REG_PC);	
		M16C_REG_PC+=1;
	}
	if(icode & (1<<2)) {
		M16C_REG_A1 = imm;
	} else {
		M16C_REG_A0 = imm;
	}
	fprintf(stderr,"instr m16c_mov_size_s_immdst(%04x)\n",icode);
}

void 
m16c_mov_b_z_0_dst(uint16_t  icode) 
{
	int arglen;
	am2b_set(icode,&arglen,0) ;
	M16C_REG_PC += arglen;	
	fprintf(stderr,"instr m16c_mov_b_z_0_dst(%04x) not implemented\n",icode);
}

void 
m16c_mov_size_g_srcdst(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	uint16_t src;
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword) ; 
	M16C_REG_PC+=arglen;
	am1_set(icode,icode & 0xf,&arglen,isword,src);
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_mov_size_g_srcdst(%04x)\n",icode);
}

void 
m16c_mov_b_s_srcdst(uint16_t  icode) 
{
	int arglen;
	uint8_t src;
	src = am3b_get(icode,&arglen) ;
	if(icode & (1<<2)) {
		M16C_REG_A1 = src;
	} else {
		M16C_REG_A0 = src;
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_mov_b_s_srcdst(%04x)\n",icode);
}

void 
m16c_mov_b_r0dst(uint16_t  icode) 
{
	int arglen;
	uint8_t src;	
	if(icode & (1<<2)) {
		src = M16C_REG_R0H;	
	} else {
		src = M16C_REG_R0L;	
	}
	am3b_set(icode,&arglen,src);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_mov_b_r0dst(%04x)\n",icode);
}

void 
m16c_mov_b_s_r0(uint16_t  icode) 
{
	int arglen;
	uint8_t src;
	src = am3b_get(icode,&arglen);
	if(icode & (1<<2)) {
		M16C_REG_R0H = src;
	} else {
		M16C_REG_R0L = src;
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_mov_b_s_r0(%04x)\n",icode);
}

void 
m16c_mov_size_g_dspdst(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	int8_t dsp8;
	uint16_t value;
	am1_get(icode,icode & 0xf,&arglen,isword) ; /* dummy for getting arglen ! shit */
	dsp8 = M16C_Read8(M16C_REG_PC + arglen);	
	if(isword) {
		value = M16C_Read16(M16C_REG_SP + dsp8);
	} else {
		value = M16C_Read8(M16C_REG_SP + dsp8);
	}	
	am1_set(icode,icode & 0xf,&arglen,isword,value);
	M16C_REG_PC += arglen+1;
	fprintf(stderr,"instr m16c_mov_size_g_dspdst(%04x)\n",icode);
}

void 
m16c_mov_size_g_srcdsp(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	int8_t dsp8;
	uint16_t value;
	value = am1_get(icode,icode & 0xf,&arglen,isword) ;
	dsp8 = M16C_Read8(M16C_REG_PC + arglen);
	if(isword) {
		M16C_Write16(value,M16C_REG_SP + dsp8);
	} else {
		M16C_Write8(value,M16C_REG_SP + dsp8);
	}
	M16C_REG_PC += arglen+1;
	fprintf(stderr,"instr m16c_mov_size_g_srcdsp(%04x)\n",icode);
}

void 
m16c_mova_srcdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	int src = icode & 0xf;
	int dst = (icode>>4) & 0xf;
	uint16_t eva;
	eva = am1_get_eva(icode,src,&arglen,isword) ;
	M16C_REG_PC += arglen;
	if(dst>5) {
		fprintf(stderr,"bad amode\n");	
	}
	am1_set(icode,dst,&arglen,isword,eva);
	fprintf(stderr,"instr m16c_mova_srcdst(%04x)\n",icode);
}

/*
 * ------------------------------------------------------------
 * Move nibble
 * ------------------------------------------------------------
 */
void 
m16c_movdir_r0dst(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	int dir = (icode >> 4) & 3;
	uint8_t src = M16C_REG_R0L; 
	uint8_t dst = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;
	switch(dir) {
			/* src dst */
		case 0: // LL
			dst = (dst & ~0xf) | (src & 0xf);
			break;
		case 1: // LH
			dst = (dst & ~0xf0) | ((src&0xf)<<4);
			break;
		case 2: // HL
			dst = (dst & ~0x0f) | ((src&0xf0)>>4);
			break;
		case 3: // HH
			dst = (dst & ~0xf0) | (src & 0xf0);
			break;
		default:
			// unreachable;
			break;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,dst);
	fprintf(stderr,"instr m16c_movdir_r0dst(%04x) not implemented\n",icode);
}

void 
m16c_movdir_srcr0l(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	int dir = (icode >> 4) & 3;
	uint8_t src = am1_get(icode,icode & 0xf,&arglen,isword);
	uint8_t dst = M16C_REG_R0L; 
	M16C_REG_PC += arglen;
	switch(dir) {
			/* src dst */
		case 0: // LL
			dst = (dst & ~0xf) | (src & 0xf);
			break;
		case 1: // LH
			dst = (dst & ~0xf0) | ((src&0xf)<<4);
			break;
		case 2: // HL
			dst = (dst & ~0x0f) | ((src&0xf0)>>4);
			break;
		case 3: // HH
			dst = (dst & ~0xf0) | (src & 0xf0);
			break;
		default:
			// unreachable;
			break;
	}
	M16C_REG_R0L = dst;
	fprintf(stderr,"instr m16c_movdir_srcr0l(%04x)\n",icode);
}

void 
m16c_mul_size_immdst(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	int32_t result;
	int16_t dst;
	int16_t src;
	dst = am1_get(icode,icode & 0xf, &arglen,isword);	
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
		M16C_REG_PC += arglen + 2;
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
		M16C_REG_PC += arglen + 1;
	}
	result = (int32_t)src * (int32_t)dst;
	if(isword) {
		//am1_set32();	 *******
	} else {
	 	am1_set(icode,icode & 0xf, &arglen,1,result);	
	}
	fprintf(stderr,"instr m16c_mul_size_immdst(%04x)\n",icode);
}

void 
m16c_mul_size_srcdst(uint16_t  icode) 
{
	int arglen;	
	int  isword = icode & (1<<8);
	int32_t result;
	int16_t dst,src;
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	M16C_REG_PC += arglen;
	result = (int32_t)src * (int32_t)dst;
	if(isword) {
		//am1_set32();	 *******
	} else {
	 	am1_set(icode,icode & 0xf, &arglen,1,result);	
	}
	fprintf(stderr,"instr m16c_mul_size_srcdst(%04x) not implemented\n",icode);
}

void 
m16c_mulu_size_immdst(uint16_t  icode) 
{
	int arglen;
	int isword = icode & (1<<8);
	uint32_t result;
	uint16_t dst;
	uint16_t src;
	dst = am1_get(icode,icode & 0xf, &arglen,isword);	
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
		M16C_REG_PC += arglen + 2;
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
		M16C_REG_PC += arglen + 1;
	}
	result = (uint32_t)src * (uint32_t)dst;
	if(isword) {
		//am1_set32();	 *******
	} else {
	 	am1_set(icode,icode & 0xf, &arglen,1,result);	
	}
	fprintf(stderr,"instr m16c_mulu_size_immdst(%04x) not implemented\n",icode);
}

void 
m16c_mulu_size_srcdst(uint16_t  icode) 
{
	int  isword = icode & (1<<8);
	int arglen;
	uint32_t result;
	uint16_t dst,src;
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	M16C_REG_PC += arglen;
	result = (uint32_t)src * (uint32_t)dst;
	if(isword) {
		//am1_set32();	 *******
	} else {
	 	am1_set(icode,icode & 0xf, &arglen,1,result);	
	}
	fprintf(stderr,"instr m16c_mulu_size_srcdst(%04x) not implemented\n",icode);
}

void 
m16c_neg_size_dst(uint16_t  icode) 
{
	
	int isword = icode & (1<<8);
	int arglen;
	uint16_t value,dst;
	value = am1_get(icode,icode & 0xf,&arglen,isword); 
	dst = 0 - value;
	am1_set(icode,icode & 0xf,&arglen,isword,dst) ;
	M16C_REG_PC+=arglen;
	M16C_REG_FLG = M16C_REG_FLG & ~(M16C_FLG_ZERO | M16C_FLG_OVERFLOW | M16C_FLG_SIGN | M16C_FLG_CARRY);
	if(dst == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;	
		M16C_REG_FLG |= M16C_FLG_CARRY;	
	}  
	if(isword) {
		if (value == 0x8000) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;	
		} 
		if(dst & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;	
		}
	} else {
		if (value == 0x80) {
			M16C_REG_FLG |= M16C_FLG_OVERFLOW;
		}
		if(dst & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;	
		}
	}
	fprintf(stderr,"instr m16c_neg_size_dst(%04x)\n",icode);
}

void 
m16c_nop(uint16_t  icode) 
{
	fprintf(stderr,"instr m16c_nop(%04x)\n",icode);
}

void 
m16c_not_size_g_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint16_t value,dst;
	value = am1_get(icode,icode & 0xf,&arglen,isword); 
	dst = ~value;
	am1_set(icode,icode & 0xf,&arglen,isword,dst) ;
	M16C_REG_PC+=arglen;
	M16C_REG_FLG = M16C_REG_FLG & ~(M16C_FLG_ZERO |  M16C_FLG_SIGN );
	if(isword) {
		if(dst == 0) {
			M16C_REG_FLG |= M16C_FLG_ZERO;
		}
		if(dst & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	} else {
		dst = dst & 0xff;
		if(dst == 0) {
			M16C_REG_FLG |= M16C_FLG_ZERO;
		}
		if(dst & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}
	fprintf(stderr,"instr m16c_not_size_g_dst(%04x)\n",icode);
}

void 
m16c_not_b_s_dst(uint16_t  icode) 
{
	int arglen;				
	uint8_t dst;
	uint8_t result;
	dst = am2b_get(icode,&arglen);
	result = ~dst;	
	am2b_set(icode,&arglen,result);
	M16C_REG_PC+=arglen;
	M16C_REG_FLG = M16C_REG_FLG & ~(M16C_FLG_ZERO |  M16C_FLG_SIGN );
	if(dst == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	if(dst & 0x80) {
		M16C_REG_FLG |= M16C_FLG_SIGN;
	}
	fprintf(stderr,"instr m16c_not_b_s_dst(%04x) not implemented\n",icode);
}

void 
m16c_or_size_g_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
	}
	result = dst  | src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC += arglen;	
	if(isword) {
		M16C_REG_PC += 2;	
	} else {
		M16C_REG_PC++;	
	}
	or_flags(result,isword);
	fprintf(stderr,"instr m16c_or_size_g_immdst(%04x) not implemented\n",icode);
}

void 
m16c_or_b_s_immdst(uint16_t  icode) 
{
	int arglen;				
	uint8_t imm; 
	uint8_t dst;
	uint8_t result;
	imm =  M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	dst = am2b_get(icode,&arglen);
	result = dst | imm;	
	am2b_set(icode,&arglen,result);
	M16C_REG_PC+=arglen;
	or_flags(result,0);
	fprintf(stderr,"instr m16c_or_b_s_immdst(%04x)\n",icode);
}

void 
m16c_or_size_g_srcdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);	
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	result = dst | src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC+=arglen;
	or_flags(result,isword);
	fprintf(stderr,"instr m16c_or_size_g_srcdst(%04x)\n",icode);
}

void 
m16c_or_b_s_srcr0(uint16_t  icode) 
{
	uint8_t src,dst,result;	
	int arglen;
	src = am3b_get(icode,&arglen);	
	M16C_REG_PC+=arglen;
	if(icode & (1<<2)) {
		dst = M16C_REG_R0H;
		M16C_REG_R0H = result = src | dst;
	} else {
		dst = M16C_REG_R0L;
		M16C_REG_R0L = result = src | dst;
	}
	or_flags(result,0);
	fprintf(stderr,"instr m16c_or_b_s_srcr0(%04x)\n",icode);
}

void 
m16c_pop_size_g_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint16_t value;
	if(isword) {
		value = M16C_Read16(M16C_REG_SP);
		M16C_REG_SP+=2;
	} else {
		value = M16C_Read8(M16C_REG_SP);
		M16C_REG_SP+=1;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,value) ;
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_pop_size_g_dst(%04x)\n",icode);
}

void 
m16c_pop_b_s_dst(uint16_t  icode) 
{
	uint8_t dst;
	dst = M16C_Read8(M16C_REG_SP);
	M16C_REG_SP+=1;
	if(icode & (1<<3)) {
		M16C_REG_R0H = dst;
	} else {
		M16C_REG_R0L = dst;
	}
	fprintf(stderr,"instr m16c_pop_b_s_dst(%04x)\n",icode);
}

void 
m16c_pop_w_s_dst(uint16_t  icode) 
{
	uint16_t dst;
	dst = M16C_Read16(M16C_REG_SP);
	M16C_REG_SP+=2;
	if(icode & (1<<3)) {
		M16C_REG_A1 = dst;
	} else {
		M16C_REG_A0 = dst;
	}
	fprintf(stderr,"instr m16c_pop_w_s_dst(%04x)\n",icode);
}

/*
 * ----------------------------------------------------------------
 * POP control register
 * ----------------------------------------------------------------
 */
void 
m16c_popc_dst(uint16_t  icode) 
{
	uint16_t dst;
	int dstcode = (icode >> 4) & 7;
	dst = M16C_Read16(M16C_REG_SP);
	M16C_REG_SP+=2;
	set_creg(dstcode,dst); 
	fprintf(stderr,"instr m16c_popc_dst(%04x)\n",icode);
}

/* 
 * -----------------------------------------------------------------
 * POPMultiple 
 * 	Lowest bit is R0, verified with real M16C: 0xed 0x01
 *	Order on stack verified with real device
 * -----------------------------------------------------------------
 */
void 
m16c_popm_dst(uint16_t  icode) 
{
	uint8_t dst = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	if(dst & 1) {
		M16C_REG_SP+=2;
		M16C_REG_R0 = M16C_Read16(M16C_REG_SP);
	}
	if(dst & 2) {
		M16C_REG_SP+=2;
		M16C_REG_R1 = M16C_Read16(M16C_REG_SP);
	}
	if(dst & 4) {
		M16C_REG_SP+=2;
		M16C_REG_R2 = M16C_Read16(M16C_REG_SP);
	}
	if(dst & 8) {
		M16C_REG_SP+=2;
		M16C_REG_R3 = M16C_Read16(M16C_REG_SP);
	}
	if(dst & 0x10) {
		M16C_REG_SP+=2;
		M16C_REG_A0 = M16C_Read16(M16C_REG_SP);
	}
	if(dst & 0x20) {
		M16C_REG_SP+=2;
		M16C_REG_A1 = M16C_Read16(M16C_REG_SP);
	}
	if(dst & 0x40) {
		M16C_REG_SP+=2;
		M16C_REG_SB = M16C_Read16(M16C_REG_SP);
	}
	if(dst & 0x40) {
		M16C_REG_SP+=2;
		M16C_REG_FB = M16C_Read16(M16C_REG_SP);
	}
	fprintf(stderr,"instr m16c_popm_dst(%04x) not implemented\n",icode);
}

void 
m16c_push_size_g_imm(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	uint16_t value;
	if(isword) {
		value = M16C_Read16(M16C_REG_PC);
		M16C_REG_PC+=2;
		M16C_REG_SP-=2;
		M16C_Write16(value,M16C_REG_SP);
	} else {
		value = M16C_Read8(M16C_REG_PC);
		M16C_REG_SP+=1;
		M16C_Write8(value,M16C_REG_SP);
	}
	fprintf(stderr,"instr m16c_push_size_g_imm(%04x)\n",icode);
}

void 
m16c_push_size_g_src(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint16_t src;
	src = am1_get(icode,icode & 0xf,&arglen,isword);	
	if(isword) {
		M16C_REG_SP-=2;
		M16C_Write16(src,M16C_REG_SP);
	} else {
		M16C_REG_SP-=1;
		M16C_Write8(src,M16C_REG_SP);
	}
	fprintf(stderr,"instr m16c_push_size_g_src(%04x)\n",icode);
}

void 
m16c_pushb_s_src(uint16_t  icode) 
{
	uint8_t src;
	if(icode & (1<<3)) {
		src = M16C_REG_R0H;
	} else {
		src = M16C_REG_R0L;
	}
	M16C_REG_SP-=1;
	M16C_Write8(src,M16C_REG_SP);
	fprintf(stderr,"instr m16c_pushb_s_src(%04x)\n",icode);
}

void 
m16c_push_w_src(uint16_t  icode) 
{
	uint16_t src;
	if(icode & (1<<3)) {
		src = M16C_REG_A1;
	} else {
		src = M16C_REG_A0;
	}
	M16C_REG_SP-=2;
	M16C_Write16(src,M16C_REG_SP);
	fprintf(stderr,"instr m16c_push_w_src(%04x)\n",icode);
}

void 
m16c_pusha_src(uint16_t  icode) 
{
	uint16_t eva;
	int arglen;
	eva = am1_get_eva(icode,icode & 0xf,&arglen,1);
	M16C_REG_PC += arglen;
	M16C_REG_SP -= 2;
	M16C_Write16(eva,M16C_REG_SP);
	fprintf(stderr,"instr m16c_pusha_src(%04x)\n",icode);
}

/*
 * ----------------------------------------------------------------
 * Pushc src 
 * ----------------------------------------------------------------
 */
void 
m16c_pushc_src(uint16_t  icode) 
{
	uint16_t value;
	value = get_creg((icode >> 4) & 7); 
	M16C_REG_SP -= 2;
	M16C_Write16(value,M16C_REG_SP);
	fprintf(stderr,"instr m16c_pushc_src(%04x)\n",icode);
}

/*
 * -------------------------------------------------------------------
 * push multiple
 * 	Highest Bit is R0, verified with Real Device: 0xec 0x80
 *	Order on stack verified with real device
 * -------------------------------------------------------------------
 */ 
void 
m16c_pushm_src(uint16_t  icode) 
{
	uint8_t src = M16C_Read8(M16C_REG_PC);	
	M16C_REG_PC++;
	if(src & 1) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_SP);
	}
	if(src & 2) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_SB);
	}
	if(src & 4) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_A1);
	}
	if(src & 8) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_A0);
	}
	if(src & 0x10) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_R3);
	}
	if(src & 0x20) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_R2);
	}
	if(src & 0x40) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_R1);
	}
	if(src & 0x80) {
		M16C_REG_SP -= 2;
		M16C_Write16(M16C_REG_FB,M16C_REG_R0);
	}
	fprintf(stderr,"instr m16c_pushm_src(%04x) not implemented\n",icode);
}


void 
m16c_reit(uint16_t  icode) 
{
	uint16_t ml,hflg;
	uint8_t flg;
	ml = M16C_Read16(M16C_REG_SP);
	M16C_REG_SP+=2;
	hflg = M16C_Read16(M16C_REG_SP);
	M16C_REG_SP +=2;
	M16C_REG_PC = ml | ((hflg & 0xff) << 16);
	flg = hflg >> 8;
	M16C_SET_REG_FLG((M16C_REG_FLG & 0xff00) | flg);
	fprintf(stderr,"instr m16c_reit(%04x) not implemented\n",icode);
}

void 
m16c_rmpa_b(uint16_t  icode) 
{
	uint32_t r0;
	while(M16C_REG_R3) {
		r0 = M16C_REG_R0;	
		r0 = r0 + M16C_Read8(M16C_REG_A0) * M16C_Read8(M16C_REG_A1);
		M16C_REG_R0 = r0;
		M16C_REG_A0 += 1;
		M16C_REG_A1 += 1;
		M16C_REG_R3--;
		//check interrupts
	}
	fprintf(stderr,"instr m16c_rmpa(%04x) not implemented\n",icode);
}

/*
 * -------------------------------------------------------------
 * This is a hint that the M16C post increments the PC
 * -------------------------------------------------------------
 */
void 
m16c_rmpa_w(uint16_t  icode) 
{
	uint32_t r2r0;
	while(M16C_REG_R3) {
		r2r0 = M16C_REG_R0 + (((uint32_t)M16C_REG_R2)<<16);	
		r2r0 = r2r0 + M16C_Read16(M16C_REG_A0) * M16C_Read16(M16C_REG_A1);
		M16C_REG_R0 = r2r0 & 0xffff;
		M16C_REG_R2 = r2r0 >> 16;
		M16C_REG_A0 += 2;
		M16C_REG_A1 += 2;
		M16C_REG_R3--;
		//check interrupts
	}
	fprintf(stderr,"instr m16c_rmpa(%04x) not implemented\n",icode);
}

void 
m16c_rolc_size_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint16_t val = am1_get(icode,icode & 0xf,&arglen,isword);
	uint16_t carry = M16C_REG_FLG & M16C_FLG_CARRY;
	if(isword) {
		if(val & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_CARRY;		
		} else {
			M16C_REG_FLG &= ~M16C_FLG_CARRY;		
		}	
	} else {
		if(val & 0x80) {
			M16C_REG_FLG |= M16C_FLG_CARRY;		
		} else {
			M16C_REG_FLG &= ~M16C_FLG_CARRY;		
		}	
	}
	if(carry) {
		val = (val << 1) | 1;	
	} else {
		val = (val << 1);	
	}
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC+=arglen;
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);	
	if(val==0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	if(isword) {
		if(val & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	} else {
		if(val & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}
	fprintf(stderr,"instr m16c_rolc_size_dst(%04x)\n",icode);
}

void 
m16c_rorc_size_dst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	uint16_t val = am1_get(icode,icode & 0xf,&arglen,isword);
	uint16_t carry = M16C_REG_FLG & M16C_FLG_CARRY;
	if(val & 1) {
		M16C_REG_FLG |= M16C_FLG_CARRY;		
	} else {
		M16C_REG_FLG &= ~M16C_FLG_CARRY;		
	}	
	if(isword) {
		if(carry) {
			val = (val >> 1) | 0x8000;	
		} else {
			val = (val >> 1);	
		}
	} else {
		if(carry) {
			val = (val >> 1) | 0x80;	
		} else {
			val = (val >> 1);	
		}
	}
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC+=arglen;
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);	
	if(val==0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	if(isword) {
		if(val & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	} else {
		if(val & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}
	fprintf(stderr,"instr m16c_rorc_size_dst(%04x)\n",icode);
}

void 
m16c_rot_size_immdst(uint16_t  icode) 
{
	uint16_t val;
	int isword = icode & (1<<8);
	int arglen;
	uint8_t rot = ((icode >> 4) & 0x7)+1;
	int right = icode & (1<<7);	
	val = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);	
	if(right) {
		if(val & (1>>(rot-1))) {
			M16C_REG_FLG |= M16C_FLG_CARRY;
		}
		if(isword) {
			val = (val >> rot) | (val << (16-rot));
		} else {
			val = (val >> rot) | (val << (8-rot));
		}
	} else {
		if(isword) {
			if(val & (1<<(16-rot))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}
			val = (val << rot) | (val >> (16-rot));
		} else {
			if(val & (1<<(8-rot))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}
			val = (val << rot) | (val >> (8-rot));
		}
	}
	if(isword) {
		if(val & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		} 
	} else {
		if(val & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		} 
	}
	if(val == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_rot_size_immdst(%04x)\n",icode);
}

void 
m16c_rot_size_r1hdst(uint16_t  icode) 
{
	int8_t rot = M16C_REG_R1H;
	uint16_t val;
	int isword = icode & (1<<8);
	int arglen;
	val = am1_get(icode,icode & 0xf,&arglen,isword);
	if(!rot) {
		M16C_REG_PC+=arglen;
		return;
	}
	M16C_REG_FLG &= ~(M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);	
	if(rot<0) {
		rot = -rot;
		/* right */
		if(val & (1>>(rot-1))) {
			M16C_REG_FLG |= M16C_FLG_CARRY;
		}
		if(isword) {
			val = (val >> rot) | (val << (16-rot));
		} else {
			val = (val >> rot) | (val << (8-rot));
		}
	} else {
		if(isword) {
			if(val & (1<<(16-rot))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}
			val = (val << rot) | (val >> (16-rot));
		} else {
			if(val & (1<<(8-rot))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}
			val = (val << rot) | (val >> (8-rot));
		}
	}
	if(val == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	if(isword) {
		if(val & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		} 
	} else {
		if(val & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		} 
	}
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_rot_size_r1hdst(%04x)\n",icode);
}

void 
m16c_rts(uint16_t  icode) 
{
	uint16_t ml,mh;
	ml = M16C_Read16(M16C_REG_SP);
	M16C_REG_SP += 2;
	mh = M16C_Read8(M16C_REG_SP);
	M16C_REG_SP += 1;
	M16C_REG_PC = ml | (mh<<16);	
	fprintf(stderr,"instr m16c_rts(%04x)\n",icode);
}

/*
 * --------------------------------------------------
 * Subtract with borrow
 * --------------------------------------------------
 */
void 
m16c_sbb_size_immdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen; 
	uint16_t result,src,dst;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC+arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC+arglen);
	}
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = dst - src;
	} else {
		result = dst - src - 1;
	}
	M16C_REG_FLG &= ~(M16C_FLG_OVERFLOW | M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);	
	if(isword) {
		M16C_REG_FLG |= sub16_carry(dst,src,result);
		M16C_REG_FLG |= sub16_overflow(dst,src,result);
		if(result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	} else {
		M16C_REG_FLG |= sub8_carry(dst,src,result);
		M16C_REG_FLG |= sub8_overflow(dst,src,result);
		if(result & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
		result = result & 0xff;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	if(isword) {
		M16C_REG_PC += arglen + 2;
	} else {
		M16C_REG_PC += arglen + 1;
	}
	fprintf(stderr,"instr m16c_sbb_size_immdst(%04x)\n",icode);
}

void 
m16c_sbb_size_srcdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen; 
	uint16_t result,src,dst;
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC+=arglen;
	if(M16C_REG_FLG & M16C_FLG_CARRY) {
		result = dst - src;
	} else {
		result = dst - src - 1;
	}
	M16C_REG_FLG &= ~(M16C_FLG_OVERFLOW | M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);	
	if(isword) {
		M16C_REG_FLG |= sub16_carry(dst,src,result);
		M16C_REG_FLG |= sub16_overflow(dst,src,result);
		if(result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	} else {
		M16C_REG_FLG |= sub8_carry(dst,src,result);
		M16C_REG_FLG |= sub8_overflow(dst,src,result);
		if(result & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
		result = result & 0xff;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	fprintf(stderr,"instr m16c_sbb_size_srcdst(%04x)\n",icode);
}

void 
m16c_sbjnz_immdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);
	int arglen;
	int8_t imm;
	int8_t lbl;
	uint16_t dst;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	imm = (icode >> 4) & 0xf;
	if(imm & 8) {
		imm |= 0xf0;
	}
	dst += imm;
	if(!isword) {
		dst = dst & 0xff;
	}
	if(!dst) {
		lbl = M16C_Read8(M16C_REG_PC+arglen);
		M16C_REG_PC+=lbl;
	} else {
		M16C_REG_PC +=arglen+1;
	}
	fprintf(stderr,"instr m16c_sbjnz_immdst(%04x) not implemented\n",icode);
}

void 
m16c_sha_size_immdst(uint16_t  icode) 
{
	uint16_t dst,result;
	int isword = icode & (1<<8);
	int arglen;
	int8_t shift = (icode & 0x7) + 1;
	int right = icode & (1<<3);	
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_FLG &= ~(M16C_FLG_OVERFLOW | M16C_FLG_CARRY | M16C_FLG_SIGN | M16C_FLG_ZERO);	
	if(right) {
		if(dst & (1<<(shift-1))) {
			M16C_REG_FLG |= M16C_FLG_CARRY;
		}	
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(isword) {
			if((result & 0x8000) != (dst & 0x8000)) {
				M16C_REG_FLG |= M16C_FLG_OVERFLOW;
			}
			if(result & 0x8000) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(16-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		} else {
			if((result & 0x80) != (dst & 0x80)) {
				M16C_REG_FLG |= M16C_FLG_OVERFLOW;
			}
			if(result & 0x80) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(8-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		}
	}
	if(isword) {
		if(result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		} 
	} else {
		result = result & 0xff;
		if(result & 0x80) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		} 
	}
	if(result == 0) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_sha_size_immdst(%04x)\n",icode);
}

void 
m16c_sha_size_r1hdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);	
	int arglen;
	int shift = M16C_REG_R1H;
	int right = 0;
	int16_t dst,result;
	if(shift<0) {
		right = 1;
		shift = -shift;
	}
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(shift == 0) {
		M16C_REG_PC+=arglen;
		return;
	}
	if(!isword) {
		if(dst & 0x80) {
			dst |= 0xff80;
		}
	}
	M16C_REG_FLG &= ~(M16C_FLG_OVERFLOW | M16C_FLG_SIGN | M16C_FLG_ZERO | M16C_FLG_CARRY);
	if(right) {
		if(dst & (1<<(shift-1))) {
			M16C_REG_FLG |= M16C_FLG_CARRY;
		}	
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(isword) {
			if((result & 0x8000) != (dst & 0x8000)) {
				M16C_REG_FLG |= M16C_FLG_OVERFLOW;
			}
			if(result & 0x8000) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(16-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		} else {
			if((result & 0x80) != (dst & 0x80)) {
				M16C_REG_FLG |= M16C_FLG_OVERFLOW;
			}
			if(result & 0x80) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(8-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		}
	}	
	if(!isword) {
		result = result & 0xff;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	fprintf(stderr,"instr m16c_sha_size_r1hdst(%04x)\n",icode);
}

void 
m16c_sha_l_immdst(uint16_t  icode) 
{
	int right = (icode>>3) & 1;
	int shift = (icode & 7) +1;
	int dstcode = (icode >> 4) & 1;
	int32_t dst,result;
	if(dstcode) {
		dst = M16C_REG_R1 | (M16C_REG_R3 << 16);
	} else {
		dst = M16C_REG_R0 | (M16C_REG_R2 << 16);
	}
	M16C_REG_FLG &= ~(M16C_FLG_SIGN);
	if(right) {
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(result & 0x80000000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}	
	if(dstcode) {
		M16C_REG_R1 = dst & 0xffff;
		M16C_REG_R3 = dst >> 16;
	} else {
		M16C_REG_R0 = dst & 0xffff;
		M16C_REG_R2 = dst >> 16;
	}
	fprintf(stderr,"instr m16c_sha_l_immdst(%04x)\n",icode);
}

void 
m16c_sha_l_r1hdst(uint16_t  icode) 
{
	
	int shift = M16C_REG_R1H;
	int right = 0;
	int dstcode = (icode >> 4) & 1;
	int32_t dst,result;
	if(shift<0) {
		right = 1;
		shift = -shift;
	}
	if(shift == 0) {
		return;
	}
	if(dstcode) {
		dst = M16C_REG_R1 | (M16C_REG_R3 << 16);
	} else {
		dst = M16C_REG_R0 | (M16C_REG_R2 << 16);
	}
	M16C_REG_FLG &= ~(M16C_FLG_SIGN);
	if(right) {
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(result & 0x80000000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}	
	if(dstcode) {
		M16C_REG_R1 = dst & 0xffff;
		M16C_REG_R3 = dst >> 16;
	} else {
		M16C_REG_R0 = dst & 0xffff;
		M16C_REG_R2 = dst >> 16;
	}
	fprintf(stderr,"instr m16c_sha_l_r1hdst(%04x) not implemented\n",icode);
}

void 
m16c_shl_size_immdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);	
	int arglen;
	int right = (icode>>7) & 1;
	int shift = ((icode >> 4) & 7) +1;
	uint16_t dst,result;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO | M16C_FLG_CARRY);
	if(right) {
		if(dst & (1<<(shift-1))) {
			M16C_REG_FLG |= M16C_FLG_CARRY;
		}	
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(isword) {
			if(result & 0x8000) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(16-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		} else {
			if(result & 0x80) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(8-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		}
	}	
	if(!isword) {
		result = result & 0xff;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_shl_size_immdst(%04x) not implemented\n",icode);
}

void 
m16c_shl_size_r1hdst(uint16_t  icode) 
{
	int isword = icode & (1<<8);	
	int arglen;
	int shift = M16C_REG_R1H;
	int right = 0;
	uint16_t dst,result;
	if(shift<0) {
		right = 1;
		shift = -shift;
	}
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(shift == 0) {
		M16C_REG_PC+=arglen;
		return;
	}
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO | M16C_FLG_CARRY);
	if(right) {
		if(dst & (1<<(shift-1))) {
			M16C_REG_FLG |= M16C_FLG_CARRY;
		}	
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(isword) {
			if(result & 0x8000) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(16-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		} else {
			if(result & 0x80) {
				M16C_REG_FLG |= M16C_FLG_SIGN;
			}
			if(dst & (1<<(8-shift))) {
				M16C_REG_FLG |= M16C_FLG_CARRY;
			}	
		}
	}	
	if(!isword) {
		result = result & 0xff;
	}
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	am1_set(icode,icode & 0xf,&arglen,isword,result);
	fprintf(stderr,"instr m16c_shl_size_r1hdst(%04x)\n",icode);
}

void 
m16c_shl_l_immdst(uint16_t  icode) 
{
	int right = (icode>>3) & 1;
	int shift = (icode & 7) +1;
	int dstcode = (icode >> 4) & 1;
	uint32_t dst,result;
	if(dstcode) {
		dst = M16C_REG_R1 | (M16C_REG_R3 << 16);
	} else {
		dst = M16C_REG_R0 | (M16C_REG_R2 << 16);
	}
	M16C_REG_FLG &= ~(M16C_FLG_SIGN);
	if(right) {
		if(dst & (1<<(shift-1))) {
			M16C_REG_FLG |= M16C_FLG_CARRY;
		}	
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(result & 0x80000000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}	
	if(dstcode) {
		M16C_REG_R1 = dst & 0xffff;
		M16C_REG_R3 = dst >> 16;
	} else {
		M16C_REG_R0 = dst & 0xffff;
		M16C_REG_R2 = dst >> 16;
	}
	fprintf(stderr,"instr m16c_shl_immdst(%04x)\n",icode);
}

void 
m16c_shl_l_r1hdst(uint16_t  icode) 
{
	int shift = M16C_REG_R1H;
	int right = 0;
	int dstcode = (icode >> 4) & 1;
	uint32_t dst,result;
	if(shift<0) {
		right = 1;
		shift = -shift;
	}
	if(shift == 0) {
		return;
	}
	if(dstcode) {
		dst = M16C_REG_R1 | (M16C_REG_R3 << 16);
	} else {
		dst = M16C_REG_R0 | (M16C_REG_R2 << 16);
	}
	M16C_REG_FLG &= ~(M16C_FLG_SIGN);
	if(right) {
		result = dst >> shift;
	} else {
		result = dst << shift;
		if(result & 0x80000000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}	
	if(!result) {
		M16C_REG_FLG |= M16C_FLG_ZERO;
	}
	if(dstcode) {
		M16C_REG_R1 = dst & 0xffff;
		M16C_REG_R3 = dst >> 16;
	} else {
		M16C_REG_R0 = dst & 0xffff;
		M16C_REG_R2 = dst >> 16;
	}
	fprintf(stderr,"instr m16c_shl_r1hdst(%04x)\n",icode);
}

/*
 * ----------------------------------------------------
 * String move backward
 * ----------------------------------------------------
 */
void 
m16c_smovb_size(uint16_t  icode) 
{
	int isword = (icode >> 8) & 1;
	while(M16C_REG_R3) {
		uint32_t addr = (M16C_REG_R1H << 16) + M16C_REG_A0;
		if(isword) {
			uint16_t val = M16C_Read16(addr);
			M16C_Write16(val,M16C_REG_A1);
			if(M16C_REG_A0 <= 1) {
				M16C_REG_R1H--;
			}
			M16C_REG_A0 -= 2;
			M16C_REG_A1 -= 2;
		} else {
			uint8_t val = M16C_Read16(addr);
			M16C_Write8(val,M16C_REG_A1);
			if(M16C_REG_A0 < 1) {
				M16C_REG_R1H--;
			}
			M16C_REG_A0 -= 1;
			M16C_REG_A1 -= 1;
		}
		M16C_REG_R3--;
		//check interrupts here
	}
	fprintf(stderr,"instr m16c_smovb_size(%04x) not implemented\n",icode);
}

void 
m16c_smovf_size(uint16_t  icode) 
{
	int isword = (icode >> 8) & 1;
	while(M16C_REG_R3) {
		uint32_t addr = (M16C_REG_R1H << 16) + M16C_REG_A0;
		if(isword) {
			uint16_t val = M16C_Read16(addr);
			M16C_Write16(val,M16C_REG_A1);
			if(M16C_REG_A0 >= 0xfffe) {
				M16C_REG_R1H++;
			}
			M16C_REG_A0 += 2;
			M16C_REG_A1 += 2;
		} else {
			uint8_t val = M16C_Read16(addr);
			M16C_Write8(val,M16C_REG_A1);
			if(M16C_REG_A0 > 0xfffe) {
				M16C_REG_R1H++;
			}
			M16C_REG_A0 += 1;
			M16C_REG_A1 += 1;
		}
		M16C_REG_R3--;
		//check interrupts here
	}
	fprintf(stderr,"instr m16c_smovf_size(%04x) not implemented\n",icode);
}

void 
m16c_sstr_size(uint16_t  icode) 
{
	int isword = (icode >> 8) & 1;
	while(M16C_REG_R3) {
		if(isword) {
			uint16_t val = M16C_REG_R0;
			M16C_Write16(val,M16C_REG_A1);
			M16C_REG_A1 += 2;
		} else {
			uint8_t val = M16C_REG_R0;
			M16C_Write8(val,M16C_REG_A1);
			M16C_REG_A1 += 1;
		}
		M16C_REG_R3--;
		//check interrupts here
	}
	fprintf(stderr,"instr m16c_sstr_size(%04x) not implemented\n",icode);
}

/*
 * ------------------------------------------
 * Store from controlregister
 * ------------------------------------------
 */
void 
m16c_stc_srcdst(uint16_t  icode) 
{
	int creg = (icode >> 4) & 7;
	int isword = 1;
	int arglen;
	uint16_t val = get_creg(creg); 
	am1_set(icode,icode & 0xf,&arglen,isword,val);
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_stc_srcdst(%04x)\n",icode);
}

void 
m16c_stc_pcdst(uint16_t icode) 
{
	fprintf(stderr,"instr m16c_stc_pcdst(%04x)\n",icode);
}

/*
 * documented very badly
 */
void 
m16c_stctx_abs16abs20(uint16_t  icode) 
{
	uint32_t abs20;		
	uint16_t abs16;
	uint32_t addr20;
	uint8_t regset;
	uint8_t spdiff=0;
	abs16 = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC+=2;
	abs20 = M16C_Read16(M16C_REG_PC);
	M16C_REG_PC+=2;
	abs20 |= (M16C_Read8(M16C_REG_PC) & 0xf) << 16;
	M16C_REG_PC+=1;
	addr20 = (abs20 + ((uint32_t)abs16 << 1)) & 0xfffff;
	regset = M16C_Read8(addr20);
	addr20 = (addr20+1) & 0xfffff;
	if(regset & 1) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_R0,M16C_REG_SP);	
		spdiff+=2;
	}
	if(regset & 2) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_R1,M16C_REG_SP);	
		spdiff+=2;
	}
	if(regset & 4) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_R2,M16C_REG_SP);	
		spdiff+=2;
	}
	if(regset & 8) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_R3,M16C_REG_SP);	
		spdiff+=2;
	}
	if(regset & 0x10) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_A0,M16C_REG_SP);	
		spdiff+=2;
	}
	if(regset & 0x20) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_A1,M16C_REG_SP);	
		spdiff+=2;
	}
	
	if(regset & 0x40) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_SB,M16C_REG_SP);	
		spdiff+=2;
	}
	if(regset & 0x80) {
		M16C_REG_SP-=2;
		M16C_Write16(M16C_REG_FB,M16C_REG_SP);	
		spdiff+=2;
	}
	if(spdiff != M16C_Read8(addr20)) {
		fprintf(stderr,"LDCTX wrong spdiff\n");
	}
	fprintf(stderr,"instr m16c_stctx_abs16abs20(%04x)\n",icode);
}

void 
m16c_ste_size_srcabs20(uint16_t  icode) 
{
	uint16_t src;
	int arglen;
	uint32_t abs20;
	int isword = (icode >> 8) & 1;
	src = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;
	abs20 = M16C_Read16(M16C_REG_PC); 
	M16C_REG_PC+=2;
	abs20 |= M16C_Read8(M16C_REG_PC)<<16;
	M16C_REG_PC+=1;
	if(isword) {
		M16C_Write16(src,abs20);
	} else {
		M16C_Write8(src,abs20);
	}
	fprintf(stderr,"instr m16c_ste_size_srcabs20(%04x)\n",icode);
}

void 
m16c_ste_size_srcdsp20(uint16_t  icode) 
{
	uint16_t src;
	uint32_t dsp20,addr20;
	int arglen;
	int isword = (icode >> 8) & 1;
	src = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;
	dsp20 = M16C_Read16(M16C_REG_PC); 
	M16C_REG_PC+=2;
	dsp20 |= M16C_Read8(M16C_REG_PC)<<16;
	M16C_REG_PC+=1;
	addr20 = (dsp20 + M16C_REG_A0) & 0xfffff;

	if(isword) {
		M16C_Write16(src,addr20);
	} else {
		M16C_Write8(src,addr20);
	}
	fprintf(stderr,"instr m16c_ste_size_srcdsp20(%04x) not implemented\n",icode);
}

void 
m16c_ste_size_srca1a0(uint16_t  icode) 
{
	int arglen;
	uint16_t src;
	uint32_t addr20;
	int isword = (icode >> 8) & 1;
	src = am1_get(icode,icode & 0xf,&arglen,isword);
	M16C_REG_PC += arglen;
	addr20 = (M16C_REG_A0 | (M16C_REG_A1 << 16)) & 0xfffff;
	if(isword) {
		M16C_Write16(src,addr20);
	} else {
		M16C_Write8(src,addr20);
	}
	fprintf(stderr,"instr m16c_ste_size_srca1a0(%04x)\n",icode);
}

void 
m16c_stnz_immdst(uint16_t  icode) 
{
	uint8_t imm8;
	int arglen;
	imm8 = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	if(M16C_REG_FLG & M16C_FLG_ZERO) {
		am2b_get(icode,&arglen);
	} else {
		am2b_set(icode,&arglen,imm8);
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_stnz_immdst(%04x)\n",icode);
}

void 
m16c_stz_immdst(uint16_t  icode) 
{
	uint8_t imm8;
	int arglen;
	imm8 = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	if(M16C_REG_FLG & M16C_FLG_ZERO) {
		am2b_set(icode,&arglen,imm8);
	} else {
		am2b_get(icode,&arglen); // shit just for getting arglen
	}
	M16C_REG_PC += arglen;
	fprintf(stderr,"instr m16c_stz_immdst(%04x)\n",icode);
}

void 
m16c_stzx_immimmdst(uint16_t  icode) 
{
	int imm8_1,imm8_2; 
	int arglen;
	
	imm8_1 = M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	am2b_get(icode,&arglen); // shit just for getting arglen 
	imm8_2 = M16C_Read8(M16C_REG_PC+arglen);
	if(M16C_REG_FLG & M16C_FLG_ZERO) {
		am2b_set(icode,&arglen,imm8_1);
	} else {
		am2b_set(icode,&arglen,imm8_2);
	}
	M16C_REG_PC += arglen + 1;	
	fprintf(stderr,"instr m16c_stzx_immimmdst(%04x)\n",icode);
}

void 
m16c_sub_size_g_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
	}
	result = dst - src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC += arglen;	
	if(isword) {
		M16C_REG_PC += 2;	
	} else {
		M16C_REG_PC++;	
	}
	sub_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_sub_size_g_immdst(%04x) not implemented\n",icode);
}

void 
m16c_sub_b_s_immdst(uint16_t  icode) 
{
	int arglen;				
	uint8_t imm; 
	uint8_t dst;
	uint8_t result;
	imm =  M16C_Read8(M16C_REG_PC);
	M16C_REG_PC++;
	dst = am2b_get(icode,&arglen);
	result = dst - imm;	
	am2b_set(icode,&arglen,result);
	M16C_REG_PC+=arglen;
	sub_flags(dst,imm,result,0);
	fprintf(stderr,"instr m16c_sub_b_s_immdst(%04x) not implemented\n",icode);
}

void 
m16c_sub_size_g_srcdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);	
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	result = dst - src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC+=arglen;
	sub_flags(dst,src,result,isword);
	fprintf(stderr,"instr m16c_sub_size_g_srcdst(%04x)\n",icode);
}

void 
m16c_sub_b_srcr0lr0h(uint16_t  icode) 
{
	uint8_t src,dst,result;	
	int arglen;
	src = am3b_get(icode,&arglen);	
	M16C_REG_PC+=arglen;
	if(icode & (1<<2)) {
		dst = M16C_REG_R0H;
		M16C_REG_R0H = result = src-dst;
	} else {
		dst = M16C_REG_R0L;
		M16C_REG_R0L = result = src-dst;
	}
	sub_flags(dst,src,result,0);
	fprintf(stderr,"instr m16c_sub_b_srcr0lr0h(%04x)\n",icode);
}

void 
m16c_tst_size_immdst(uint16_t  icode) 
{
	uint16_t dst; 
	uint16_t imm;	
	uint16_t result;
	int isword = (icode >> 8) & 1;
	int arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	M16C_REG_PC += arglen;
	if(isword) {
		imm = M16C_Read16(M16C_REG_PC);
		M16C_REG_PC+=2;
	} else {
		imm = M16C_Read8(M16C_REG_PC);
		M16C_REG_PC+=1;
	}	
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);
	result = imm & dst;
	if(isword) {
		if(result == 0) {
			M16C_REG_FLG |= M16C_FLG_ZERO;
		} else if (result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	} else {
		if((result & 0xff) == 0) {
			M16C_REG_FLG |= M16C_FLG_ZERO;
		} else if (result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}
	fprintf(stderr,"instr m16c_tst_size_immdst(%04x)\n",icode);
}

void 
m16c_tst_size_srcdst(uint16_t  icode) 
{
	uint16_t dst; 
	uint16_t src;	
	uint16_t result;
	int isword = (icode >> 8) & 1;
	int arglen;
	src = am1_get(icode,icode & 0xf,&arglen,isword);	
	M16C_REG_PC += arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	M16C_REG_PC += arglen;
	M16C_REG_FLG &= ~(M16C_FLG_SIGN | M16C_FLG_ZERO);
	result = src & dst;
	if(isword) {
		if(result == 0) {
			M16C_REG_FLG |= M16C_FLG_ZERO;
		} else if (result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	} else {
		if((result & 0xff) == 0) {
			M16C_REG_FLG |= M16C_FLG_ZERO;
		} else if (result & 0x8000) {
			M16C_REG_FLG |= M16C_FLG_SIGN;
		}
	}
	fprintf(stderr,"instr m16c_tst_size_srcdst(%04x)\n",icode);
}

/* stack layout ? */
void 
m16c_und(uint16_t  icode) 
{
	M16C_REG_SP-=1;
	M16C_Write8(M16C_REG_FLG & 0xff,M16C_REG_SP);	
	M16C_REG_SP-=1;
	M16C_Write8(M16C_REG_PC >> 16,M16C_REG_SP);	
	M16C_REG_SP-=2;
	M16C_Write16(M16C_REG_PC & 0xffff,M16C_REG_SP);	
	M16C_REG_PC = M16C_Read16(0xfffdc);	
	M16C_SET_REG_FLG(M16C_REG_FLG & ~(M16C_FLG_U | M16C_FLG_I | M16C_FLG_D));
	fprintf(stderr,"Undefined instruction m16c_und(%04x)\n",icode);
}

/*
 * -------------------------------------------------------------------------
 * Wait for interrupt
 * -------------------------------------------------------------------------
 */
void 
m16c_wait(uint16_t  icode) 
{
	// wait for interrupt
	fprintf(stderr,"instr m16c_wait(%04x) not implemented\n",icode);
}

void 
m16c_xchg_size_srcdst(uint16_t  icode) 
{
	int isword = (icode >> 8) & 1;
	int arglen;
	uint16_t src = am1_get(icode,(icode >> 4) & 3,&arglen,isword);
	uint16_t dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	am1_set(icode,(icode>>4) & 3,&arglen,isword,dst);
	am1_set(icode,icode & 0xf,&arglen,isword,src);	
	M16C_REG_PC+=arglen;
	fprintf(stderr,"instr m16c_xchg_size_srcdst(%04x) not implemented\n",icode);
}

void 
m16c_xor_size_immdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = (1<<8);
	dst = am1_get(icode,icode & 0xf,&arglen,isword);
	if(isword) {
		src = M16C_Read16(M16C_REG_PC + arglen);
	} else {
		src = M16C_Read8(M16C_REG_PC + arglen);
	}
	result = dst  ^ src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC += arglen;	
	if(isword) {
		M16C_REG_PC += 2;	
	} else {
		M16C_REG_PC++;	
	}
	xor_flags(result,isword);
	fprintf(stderr,"instr m16c_xor_size_immdst(%04x)\n",icode);
}

void 
m16c_xor_size_srcdst(uint16_t  icode) 
{
	uint16_t src,dst,result;
	int arglen;
	int isword = icode & (1<<8);
	src = am1_get(icode,(icode>>4) & 0xf,&arglen,isword);	
	M16C_REG_PC+=arglen;
	dst = am1_get(icode,icode & 0xf,&arglen,isword);	
	result = dst ^ src;
	am1_set(icode,icode & 0xf, &arglen,isword,result);	
	M16C_REG_PC+=arglen;
	xor_flags(result,isword);
	fprintf(stderr,"instr m16c_xor_size_srcdst(%04x) not implemented\n",icode);
}
