-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (ErrorHandler)
procedure PrintErrors (Listing : in SPARK_IO.File_Type;
                       Purpose : in Error_Types.ConversionRequestSource) is
   Err_Count  : Natural;
   OK         : Boolean;
   Next_Error : Error_Types.StringError;
   Num_Err    : Error_Types.NumericError;

   Accumulator : ErrorAccumulator.T := ErrorAccumulator.Clear;

   procedure Copy_Source_Lines (To_File : in SPARK_IO.File_Type;
                                Line_No : in LexTokenManager.Line_Numbers)
   --# global in     CommandLineData.Content;
   --#        in out Error_Context_Rec;
   --#        in out SPARK_IO.File_Sys;
   --# derives Error_Context_Rec,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Error_Context_Rec,
   --#                                Line_No,
   --#                                SPARK_IO.File_Sys,
   --#                                To_File;
   is
   begin
      if CommandLineData.Content.XML then
         loop
            exit when Error_Context_Rec.Line_No >= Line_No;
            GetFileLine;
         end loop;
      else
         loop
            exit when Error_Context_Rec.Line_No >= Line_No;
            GetFileLine;
            Print_Source_Line (To_File => To_File);
         end loop;
      end if;
   end Copy_Source_Lines;

   procedure Copy_Source_Line_To_EOF (To_File : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in out Error_Context_Rec;
   --#        in out SPARK_IO.File_Sys;
   --# derives Error_Context_Rec,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Error_Context_Rec,
   --#                                SPARK_IO.File_Sys,
   --#                                To_File;
   is
   begin
      if not CommandLineData.Content.XML then
         loop
            exit when SPARK_IO.End_Of_File (Error_Context_Rec.Source);
            GetFileLine;
            Print_Source_Line (To_File => To_File);
         end loop;
      end if;
   end Copy_Source_Line_To_EOF;

   procedure Get_Error_Set (Error_Set  :    out Error_Sets;
                            Next_Error : in out Error_Types.StringError;
                            OK         :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Error_Context_Rec;
   --#        in     LexTokenManager.State;
   --#        in     Purpose;
   --#        in out Conversions.State;
   --#        in out Err_Count;
   --#        in out SPARK_IO.File_Sys;
   --# derives Conversions.State,
   --#         Next_Error,
   --#         OK,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                Error_Context_Rec,
   --#                                LexTokenManager.State,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         Error_Set,
   --#         Err_Count         from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                Error_Context_Rec,
   --#                                Err_Count,
   --#                                LexTokenManager.State,
   --#                                Next_Error,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys;
   is
      L_OK    : Boolean;
      Errors  : Error_Sets;
      Num_Err : Error_Types.NumericError;
   begin
      Errors := Empty_Error_Set;
      loop
         Errors.Length                        := Errors.Length + 1;
         Errors.Content (Errors.Length).Error := Next_Error;
         if Next_Error.ErrorType = Error_Types.NoErr
           or else ErrorAccumulator.Is_Error_Continuation (The_Error => Next_Error) then
            Errors.Content (Errors.Length).Err_Num := 0;
         else
            Err_Count                              := Err_Count + 1;
            Errors.Content (Errors.Length).Err_Num := Err_Count;
         end if;

         Error_IO.Get_Numeric_Error (Error_Context_Rec.Errs, Num_Err);
         Conversions.ToString (Num_Err, Purpose, Next_Error);
         L_OK := (Next_Error /= Error_Types.Empty_StringError);

         exit when not L_OK;
         exit when Error_Context_Rec.Line_No < Next_Error.Position.Start_Line_No;
         exit when Errors.Length = ExaminerConstants.MaxErrorSetSize;
      end loop;

      OK        := L_OK;
      Error_Set := Errors;
   end Get_Error_Set;

   procedure Process_Error_Set (Listing    : in     SPARK_IO.File_Type;
                                Next_Error : in out Error_Types.StringError;
                                OK         :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Error_Context_Rec;
   --#        in     LexTokenManager.State;
   --#        in     Purpose;
   --#        in out Accumulator;
   --#        in out Conversions.State;
   --#        in out Err_Count;
   --#        in out SPARK_IO.File_Sys;
   --# derives Accumulator,
   --#         Err_Count         from *,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                Error_Context_Rec,
   --#                                Err_Count,
   --#                                LexTokenManager.State,
   --#                                Next_Error,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         Conversions.State,
   --#         Next_Error,
   --#         OK                from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                Error_Context_Rec,
   --#                                LexTokenManager.State,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                Accumulator,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                Error_Context_Rec,
   --#                                Err_Count,
   --#                                LexTokenManager.State,
   --#                                Listing,
   --#                                Next_Error,
   --#                                Purpose;
   is
      Error_Set : Error_Sets;
   begin
      Get_Error_Set (Error_Set  => Error_Set,
                     Next_Error => Next_Error,
                     OK         => OK);
      if not CommandLineData.Content.XML then
         Put_Error_Pointers (Listing => Listing,
                             Errors  => Error_Set);
      end if;
      Put_Error_Messages (Listing     => Listing,
                          Errors      => Error_Set,
                          Start_Pos   => 31,
                          Accumulator => Accumulator);
   end Process_Error_Set;

   procedure Process_Errors_On_Line
     (Listing    : in     SPARK_IO.File_Type;
      Next_Error : in out Error_Types.StringError;
      OK         :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Error_Context_Rec;
   --#        in     LexTokenManager.State;
   --#        in     Purpose;
   --#        in out Accumulator;
   --#        in out Conversions.State;
   --#        in out Err_Count;
   --#        in out SPARK_IO.File_Sys;
   --# derives Accumulator,
   --#         Conversions.State,
   --#         Err_Count,
   --#         Next_Error,
   --#         OK,
   --#         SPARK_IO.File_Sys from Accumulator,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                Error_Context_Rec,
   --#                                Err_Count,
   --#                                LexTokenManager.State,
   --#                                Listing,
   --#                                Next_Error,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys;
   is
      L_OK                   : Boolean;
      Accumulator_Was_Active : Boolean;
   begin
      if not Error_Has_Position_Inline (Err_Type => Next_Error.ErrorType) and then not CommandLineData.Content.XML then
         New_Line (Listing, 1);
      end if;
      loop
         Process_Error_Set (Listing    => Listing,
                            Next_Error => Next_Error,
                            OK         => L_OK);
         exit when not L_OK;
         exit when Error_Context_Rec.Line_No < Next_Error.Position.Start_Line_No;
      end loop;
      OK                     := L_OK;
      Accumulator_Was_Active := ErrorAccumulator.Is_Active (This => Accumulator);
      ErrorAccumulator.Flush (Accumulator, Listing);
      if Accumulator_Was_Active then
         New_Line (Listing, 1);
      end if;
   end Process_Errors_On_Line;

   procedure Set_Up_Files (OK : out Boolean)
   --# global in out Error_Context_Rec;
   --#        in out SPARK_IO.File_Sys;
   --#           out Err_Count;
   --# derives Error_Context_Rec,
   --#         SPARK_IO.File_Sys from *,
   --#                                Error_Context_Rec &
   --#         Err_Count         from  &
   --#         OK                from Error_Context_Rec,
   --#                                SPARK_IO.File_Sys;
   is
      L_OK        : Boolean;
      Success     : SPARK_IO.File_Status;
      Source_File : SPARK_IO.File_Type;
      Error_File  : Error_IO.File_Type;
   begin
      Source_File := Error_Context_Rec.Source;
      SPARK_IO.Reset (Source_File, SPARK_IO.In_File, Success);
      Error_Context_Rec.Source  := Source_File;
      L_OK                      := Success = SPARK_IO.Ok;
      Error_Context_Rec.Line_No := 0;
      Err_Count                 := 0;
      Error_File                := Error_Context_Rec.Errs;
      Error_IO.Reset (Error_File, SPARK_IO.In_File, Success);
      Error_Context_Rec.Errs         := Error_File;
      OK                             := L_OK and Success = SPARK_IO.Ok;
      Error_Context_Rec.Current_Line := E_Strings.Empty_String;
   end Set_Up_Files;

begin
   Flush_Buffer;
   Set_Up_Files (OK => OK);
   if OK then
      if not CommandLineData.Content.XML then
         Put_Line (Listing, "Line", 0);
      end if;

      Error_IO.Get_Numeric_Error (Error_Context_Rec.Errs, Num_Err);
      Conversions.ToString (Num_Err, Purpose, Next_Error);
      OK := (Next_Error /= Error_Types.Empty_StringError);

      loop
         exit when not OK;
         Copy_Source_Lines (To_File => Listing,
                            Line_No => Next_Error.Position.Start_Line_No);
         Process_Errors_On_Line (Listing    => Listing,
                                 Next_Error => Next_Error,
                                 OK         => OK);
         --# accept Flow, 41, "Expected stable expression";
         if not CommandLineData.Content.XML then
            --# end accept;
            New_Line (Listing, 1);
         end if;
      end loop;
      Copy_Source_Line_To_EOF (To_File => Listing);
   end if;
   Justifications.Print_Justifications (Which_Table => Error_Context_Rec.Justifications_Data_Table,
                                        File        => Listing);

   WarningStatus.Report_Suppressed_Warnings (To_File => Listing,
                                             Counter => Error_Context_Rec.Counter);

end PrintErrors;
