//
// tardy - a tar post-processor
// Copyright (C) 1991-1995, 1998-2002, 2008, 2009 Peter Miller
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program. If not, see <http://www.gnu.org/licenses/>.
//

#include <libtardy/ac/ctype.h>
#include <libtardy/ac/stdio.h>
#include <libtardy/ac/string.h>
#include <libtardy/ac/stdlib.h>
#include <libtardy/ac/unistd.h>
#include <libtardy/ac/stdarg.h>

#include <libtardy/arglex.h>
#include <libtardy/error.h>
#include <libtardy/help.h>
#include <libtardy/mem.h>
#include <libtardy/progname.h>
#include <libtardy/trace.h>
#include <libtardy/version_stmp.h>


#define PAIR(a, b) ((a) * 256 + (b))


static const char *cr[] =
{
    "\\*(n) version \\*(v)",
    ".br",
    "Copyright (C) \\*(Y) Peter Miller",
    "",
    "The \\*(n) program comes with ABSOLUTELY NO WARRANTY;",
    "for details use the '\\*(n) -VERSion License' command.",
    "The \\*(n) program is free software, and you are welcome to",
    "redistribute it under certain conditions;",
    "for details use the '\\*(n) -VERSion License' command.",
};

static const char *au[] =
{
    ".nf",
    "Peter Miller   EMail: millerp@canb.auug.org.au",
    "/\\e/\\e*    WWW: http://miller.emu.id.au/pmiller/",
    ".fi",
};

static const char *so_o__rules[] =
{
#include <man1/o__rules.h>
};

static const char *so_o_help[] =
{
#include <man1/o_help.h>
};

static const char *so_z_cr[] =
{
    ".SH COPYRIGHT",
    ".so cr",
    ".SH AUTHOR",
    ".so au",
};

static const char *so_z_exit[] =
{
#include <man1/z_exit.h>
};


struct so_list_ty
{
    const char *name;
    const char **text;
    size_t length;
};

static so_list_ty so_list[] =
{
    { "o__rules.so", so_o__rules, SIZEOF(so_o__rules) },
    { "o_help.so", so_o_help, SIZEOF(so_o_help) },
    { "z_cr.so", so_z_cr, SIZEOF(so_z_cr) },
    { "z_exit.so", so_z_exit, SIZEOF(so_z_exit) },
    { "z_name.so", 0, 0 },
    { "../etc/version.so", 0, 0 },
    { "cr", cr, SIZEOF(cr), },
    { "au", au, SIZEOF(au), },
};


static int      ocol;
static int      icol;
static int      fill;       // true if currently filling
static int      in;         // current indent
static int      in_base;    // current paragraph indent
static int      ll;         // line length
static long     roff_line;
static char     *roff_file;
static int      TP_line;


static void
emit(int c)
{
    switch (c)
    {
    case ' ':
        icol++;
        break;

    case '\t':
        icol = ((icol / 8) + 1) * 8;
        break;

    case '\n':
        putchar('\n');
        icol = 0;
        ocol = 0;
        break;

    default:
        if (!isprint(c))
            break;
        while (((ocol / 8) + 1) * 8 <= icol && ocol + 1 < icol)
        {
            putchar('\t');
            ocol = ((ocol / 8) + 1) * 8;
        }
        while (ocol < icol)
        {
            putchar(' ');
            ++ocol;
        }
        putchar(c);
        ++icol;
        ++ocol;
        break;
    }
    if (ferror(stdout))
        nfatal("standard output");
}


static void
emit_word(const char *buf, long len)
{
    if (len <= 0)
        return;

    //
    // if this line is not yet indented, indent it
    //
    if (!ocol && !icol)
        icol = in;

    //
    // if there is already something on this line
    // and we are in "fill" mode
    // and this word would cause it to overflow
    // then wrap the line
    //
    if (ocol && fill && icol + len >= ll)
    {
        emit('\n');
        icol = in;
    }
    if (ocol)
        emit(' ');
    while (len-- > 0)
        emit(*buf++);
}


static void
br(void)
{
    if (ocol)
        emit('\n');
}


static void
sp(void)
{
    br();
    emit('\n');
}


static void
interpret_line_of_words(const char *line)
{
    //
    // if not filling,
    // pump the line out literrally.
    //
    if (!fill)
    {
        if (!ocol && !icol)
            icol = in;
        while (*line)
            emit(*line++);
        emit('\n');
        return;
    }

    //
    // in fill mode, a blank line means
    // finish the paragraph and emit a blank line
    //
    if (!*line)
    {
        sp();
        return;
    }

    //
    // break the line into space-separated words
    // and emit each individually
    //
    while (*line)
    {
        while (isspace(*line))
            ++line;
        if (!*line)
            break;
        const char *start = line;
        while (*line && !isspace(*line))
            ++line;
        emit_word(start, line - start);

        //
        // extra space at end of sentences
        //
        if
        (
            (line[-1] == '.' || line[-1] == '?')
        &&
            (
                !line[0]
            ||
                (
                    line[0] == ' '
                &&
                    (!line[1] || line[1] == ' ')
                )
            )
        )
            emit(' ');
    }
}


static void
roff_error(const char *s, ...)
{
    va_list ap;
    va_start(ap, s);
    char buffer[1000];
    vsnprintf(buffer, sizeof(buffer), s, ap);
    va_end(ap);

#if 0
    br();
    if (roff_file)
        emit_word(roff_file, strlen(roff_file));
    if (roff_line)
    {
        char line[20];
        snprintf(line, sizeof(line), "%ld", roff_line);
        emit_word(line, strlen(line));
    }
    interpret_line_of_words(buffer);
    br();
#else
    fatal
    (
        "%s: %ld: %s",
        (roff_file ? roff_file : "(noname)"),
        roff_line,
        buffer
    );
#endif
}


static void
get_name(const char **lp, char *name)
{
    const char *line = *lp;
    if (*line == '(')
    {
        ++line;
        if (*line)
        {
            name[0] = *line++;
            if (*line)
            {
                name[1] = *line++;
                name[2] = 0;
            }
            else
                name[1] = 0;
        }
        else
            name[0] = 0;
    }
    else if (*line == '[')
    {
        ++line;
        int n = 0;
        for (;;)
        {
            char c = *line;
            if (!c)
                break;
            ++line;
            if (c == ']')
                break;
            if (n < 3)
                name[n++] = c;
        }
        name[n] = 0;
    }
    else if (*line)
    {
        name[0] = *line++;
        name[1] = 0;
    }
    else
        name[0] = 0;
    *lp = line;
}


typedef struct string_reg_ty string_reg_ty;
struct string_reg_ty
{
    const char *name;
    const char *value;
};


static long string_reg_count;
static string_reg_ty *string_reg;



static const char *
string_find(const char *name)
{
    for (long j = 0; j < string_reg_count; ++j)
    {
        string_reg_ty *srp = &string_reg[j];
        if (!strcmp(name, srp->name))
            return srp->value;
    }
    return 0;
}


static const char *
numreg_find(const char *)
{
    return 0;
}


static void
roff_prepro(char *buffer, const char *line)
{
    const char *value;
    char    name[4];

    char *bp = buffer;
    while (*line)
    {
        int c = *line++;
        if (c != '\\')
        {
            *bp++ = c;
            continue;
        }
        c = *line++;
        if (!c)
        {
            roff_error("can't do escaped end-of-line");
            break;
        }
        switch (c)
        {
        default:
            roff_error("unknown \\%c inline directive", c);
            break;

        case '%':
            // word break info
            break;

        case '*':
            // inline string
            get_name(&line, name);
            value = string_find(name);
            if (value)
            {
                while (*value)
                    *bp++ = *value++;
            }
            break;

        case 'n':
            // inline number register
            get_name(&line, name);
            value = numreg_find(name);
            if (value)
            {
                while (*value)
                    *bp++ = *value++;
            }
            break;

        case 'e':
        case '\\':
            *bp++ = '\\';
            break;

        case '-':
            *bp++ = '-';
            break;

        case 'f':
            // ignore font directives
            get_name(&line, name);
            break;

        case '&':
        case '|':
            // ignore weird space directives
            break;

        case '(':
        case '[':
            // special characters
            // (not really strings)
            --line;
            get_name(&line, name);
            value = string_find(name);
            if (value)
            {
                while (*value)
                    *bp++ = *value++;
            }
            break;
        }
    }
    *bp = 0;
}


static void
interpret_text(const char *line)
{
    char    buffer[1000];

    roff_prepro(buffer, line);
    interpret_line_of_words(buffer);
    if (TP_line)
    {
        if (icol >= 15)
            br();
        else
            icol = 15;
        TP_line = 0;
        in = in_base + 8;
    }
}


static void
roff_sub(char *buffer, int argc, const char **argv)
{
    int     j;
    char    *bp;
    long    len;

    bp = buffer;
    for (j = 0; j < argc; ++j)
    {
        len = strlen(argv[j]);
        if (j)
            *bp++ = ' ';
        memcpy(bp, argv[j], len);
        bp += len;
    }
    *bp = 0;
}


static void
interpret_text_args(int argc, const char **argv)
{
    char    buffer[1000];

    roff_sub(buffer, argc, argv);
    interpret_text(buffer);
}


static void
concat_text_args(int argc, const char **argv)
{
    char buffer[1000];
    char *bp = buffer;
    for (int j = 0; j < argc; ++j)
    {
        size_t len = strlen(argv[j]);
        if ((bp - buffer) + len + 1 >= sizeof(buffer))
            break;
        memcpy(bp, argv[j], len);
        bp += len;
    }
    *bp = 0;
    interpret_text(buffer);
}


static void interpret(const char **, int); // forward


static void
so(int argc, const char **argv)
{
    so_list_ty      *sop;

    if (argc != 1)
    {
        roff_error(".so requires one argument");
        return;
    }
    for (sop = so_list; sop < ENDOF(so_list); ++sop)
    {
        if (!strcmp(sop->name, argv[0]))
        {
            interpret(sop->text, sop->length);
            return;
        }
    }
    roff_error("\".so %s\" not known", argv[0]);
}


static void
lf(int argc, const char **argv)
{
    if (roff_file)
        mem_free(roff_file);
    if (argc >= 1)
        roff_line = atol(argv[0]) - 1;
    else
        roff_line = 0;
    if (argc >= 2)
        roff_file = mem_copy_string(argv[1]);
    else
        roff_file = 0;
}


static void
ds_guts(const char *name, const char *value)
{
    long        j;
    string_reg_ty   *srp;

    for (j = 0; j < string_reg_count; ++j)
    {
        srp = &string_reg[j];
        if (!strcmp(name, srp->name))
        {
            mem_free((char *)srp->value);
            srp->value = mem_copy_string(value);
            return;
        }
    }

    if (string_reg_count)
    {
        string_reg =
            (string_reg_ty *)
            mem_change_size
            (
                string_reg,
                (string_reg_count + 1) * sizeof(string_reg_ty)
            );
    }
    else
        string_reg = (string_reg_ty *)mem_alloc(sizeof(string_reg_ty));
    srp = &string_reg[string_reg_count++];
    srp->name = mem_copy_string(name);
    srp->value = mem_copy_string(value);
}


static void
ds(int argc, const char **argv)
{
    char    buf1[1000];
    char    buf2[1000];

    if (!argc)
        return;
    roff_sub(buf1, argc - 1, argv + 1);
    roff_prepro(buf2, buf1);
    ds_guts(argv[0], buf2);
}


static void
dot_in(int argc, const char **argv)
{
    if (argc < 1)
        return;
    switch (argv[0][0])
    {
    case '-':
        in -= atoi(argv[0] + 1);
        break;

    case '+':
        in += atoi(argv[0] + 1);
        break;

    default:
        in = atoi(argv[0] + 1);
        break;
    }
    if (in < 0)
        in = 0;
}


static void interpret(const char **, int); // forward


static void
dot_br(int, const char **)
{
    br();
}


static void
dot_ce(int argc, const char **argv)
{
    br();
    interpret_text_args(argc, argv);
    br();
}


static void
dot_fi(int, const char **)
{
    br();
    fill = 1;
}


static void
dot_if(int argc, const char **argv)
{
    if (argc > 1 && 0 == strcmp(argv[0], "n"))
        interpret_text_args(argc - 1, argv + 1);
}


static void
dot_ip(int, const char **)
{
    in = in_base;
    sp();
    emit(' ');
    emit(' ');
}


static void
dot_nf(int, const char **)
{
    br();
    fill = 0;
}


static void
dot_pp(int, const char **)
{
    in = in_base;
    sp();
}


static void
dot_r(int, const char **)
{
    const char *cp = string_find("R)");
    if (!cp)
        cp = "";
    if (strcmp(cp, "no") != 0)
    {
        static const char *macro[] =
        {
            ".PP",
            "See also",
            ".IR \\*(n) (1)",
            "for options common to all \\*(n) commands.",
        };

        interpret(macro, SIZEOF(macro));
    }
}


static void
dot_re(int, const char **)
{
    in_base = 8;
    in = 8;
    br();
}


static void
dot_rs(int, const char **)
{
    in_base = 16;
    in = 16;
    br();
}


static void
dot_sh(int argc, const char **argv)
{
    in = 0;
    sp();
    interpret_text_args(argc, argv);
    br();
    in_base = 8;
    in = 8;
}


static void
dot_sp(int, const char **)
{
    sp();
}


static void
dot_tp(int, const char **)
{
    in = in_base;
    sp();
    TP_line = 1;
}


static void
dot_ignore(int, const char **)
{
}


struct directive_t
{
    const char *name;
    void (*perform)(int argc, const char **argv);
};

static const directive_t directive[] =
{
    { "\\\"", dot_ignore },
    { "\"", dot_ignore },
    { "B", concat_text_args },
    { "BI", concat_text_args },
    { "BR", concat_text_args },
    { "br", dot_br },
    { "ce", dot_ce },
    { "ds", ds },
    { "fi", dot_fi },
    { "I", concat_text_args },
    { "IB", concat_text_args },
    { "if", dot_if },
    { "in", dot_in },
    { "IP", dot_ip },
    { "IR", concat_text_args },
    { "lf", lf },
    { "ne", dot_ignore },
    { "nf", dot_nf },
    { "PP", dot_pp },
    { "R", concat_text_args },
    { "r)", dot_r },
    { "RB", concat_text_args },
    { "RE", dot_re },
    { "RI", concat_text_args },
    { "RS", dot_rs },
    { "SH", dot_sh },
    { "so", so },
    { "sp", dot_sp },
    { "ta", dot_ignore },
    { "TH", dot_ignore },
    { "TP", dot_tp },
    { "XX", dot_ignore },
};


static void
interpret_control(const char *line)
{
    //
    // find the directive name
    //
    line++;
    for (;;)
    {
        unsigned char c = *line;
        if (c == 0)
            return;
        if (!isspace(c))
            break;
        ++line;
    }
    char dot_name[10];
    char *dnp = dot_name;
    for (;;)
    {
        unsigned char c = *line;
        if (!c)
            break;
        ++line;
        if (isspace(c))
            break;
        if (dnp < dot_name + sizeof(dot_name) - 1)
            *dnp++ = c;
    }
    *dnp = 0;

    //
    // break the line into space-separated arguments
    //
    int argc = 0;
    char temp[1000];
    char *cp1 = temp;
    const char *argv[20];
    while (argc < (int)SIZEOF(argv))
    {
        while (isspace(*line))
            ++line;
        if (!*line)
            break;
        argv[argc++] = cp1;
        int quoting = 0;
        while (*line)
        {
            if (*line == '"')
            {
                quoting = !quoting;
                ++line;
                continue;
            }
            if (!quoting && isspace(*line))
                break;
            *cp1++ = *line++;
        }
        *cp1++ = 0;
        if (!*line)
            break;
    }

    //
    // now do something with it
    //
    for (const directive_t *dp = directive; dp < ENDOF(directive); ++dp)
    {
        if (0 == strcmp(dot_name, dp->name))
        {
            dp->perform(argc, argv);
            return;
        }
    }
    roff_error("formatting directive \".%s\" unknown", dot_name);
}


static void
interpret(const char **text, int text_len)
{
    int     j;
    long    hold_line;
    char    *hold_file;

    //
    // save position
    //
    trace(("interpret()\n{\n"));
    hold_line = roff_line;
    hold_file = roff_file ? mem_copy_string(roff_file) : (char *)0;

    //
    // interpret the text
    //
    for (j = 0; j < text_len; ++j)
    {
        const char *s = text[j];
        if (*s == '.' || *s == '\'')
            interpret_control(s);
        else
            interpret_text(s);
        ++roff_line;
        if (ferror(stdout))
            nfatal("standard output");
    }

    //
    // restore position
    //
    if (roff_file)
        mem_free(roff_file);
    roff_line = hold_line;
    roff_file = hold_file;
    trace(("}\n"));
}


void
help(const char **text, int text_len, void (*usage)(void))
{
    //
    // collect the rest of thge command line,
    // if necessary
    //
    trace(("help(text = %p, text_len = %d, usage = %p)\n{\n",
        text, text_len, usage));
    if (usage)
    {
        arglex();
        while (arglex_token != arglex_token_eoln)
            generic_argument(usage);
    }

    //
    // initialize the state of the interpreter
    //
    ds_guts("n)", progname_get());
    ds_guts("v)", version_stamp());
    ds_guts("Y)", copyright_years());
    ds_guts("lq", "\"");
    ds_guts("rq", "\"");
    ds_guts("co", "(C)");
    ds_guts("bu", "*");
    ll = 79;
    in = 0;
    in_base = 0;
    fill = 1;
    ocol = 0;
    icol = 0;
    lf(0, 0);
    TP_line = 0;

    //
    // do what they asked
    //
    interpret(text, text_len);
    br();

    //
    // close the paginator
    //
    if (fflush(stdout))
        nfatal("standard output");
    trace(("}\n"));
}


void
generic_argument(void (*usage)(void))
{
    trace(("generic_argument()\n{\n"));
    switch (arglex_token)
    {
    default:
        bad_argument(usage);
        // NOTREACHED

    case arglex_token_trace:
        if (arglex() != arglex_token_string)
            usage();
        for (;;)
        {
            trace_enable(arglex_value.alv_string);
            if (arglex() != arglex_token_string)
                break;
        }
#ifndef DEBUG
        error
        (
"Warning: the -TRace option is only effective when the %s program \
is compiled using the DEBUG define in the common/main.h include file.",
            progname_get()
        );
#endif
        break;
    }
    trace(("}\n"));
}


void
bad_argument(void (*usage)(void))
{
    trace(("bad_argument()\n{\n"));
    switch (arglex_token)
    {
    case arglex_token_string:
        error("misplaced file name (\"%s\")", arglex_value.alv_string);
        break;

    case arglex_token_number:
        error("misplaced number (%s)", arglex_value.alv_string);
        break;

    case arglex_token_option:
        error("unknown \"%s\" option", arglex_value.alv_string);
        break;

    case arglex_token_eoln:
        error("command line too short");
        break;

    default:
        error("misplaced \"%s\" option", arglex_value.alv_string);
        break;
    }
    usage();
    trace(("}\n"));
    quit(1);
    // NOTREACHED
}
