/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2012
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
#endif

/* are these values ok? */
#define HTTP_REQUEST_MAXLEN		(8 * 1024 - 1)
#define HTTP_HEADER_FIELD_MAXLEN	(2 * 1024 - 1)
#define MAX_HTTP_REDIRECT		8
#define DEFAULT_HTTP_PORT_STR		"80"
#define DEFAULT_HTTPS_PORT_STR		"443"
#define	PORT_STR_MAXLEN			PROXY_PORT_MAXLEN
#define TAB				'\x09'

/* S_MOD -> true in multiple selection mode / false otherwise */
#define S_MOD		(get_ticker_env()->selection_mode == MULTIPLE)

static void remove_chunk_info(char **response)
{
	char	*response2;
	int	i, j;

	response2 = l_str_new(*response);
	response2[0] = '\0';
	i = go_after_next_empty_line(*response);
	while ((j = (int)get_http_chunk_size(*response + i)) > 0) {
		i += go_after_next_cr_lf(*response + i);
		str_n_cat(response2, *response + i, j);
		i += j;
		i += go_after_next_cr_lf(*response + i);
	}
	free2(*response);
	*response = response2;
}

/* 'quickly check feed format' (xml/rss2.0/atom), get shift to beginnig
 * and cut off trailing part */
static int format_quick_check(char *response, int *shift)
{
	int	i;

	/* this may return false positives so disabled
	if (strcmp(get_http_header_value("Content-Type", response), "text/xml") != 0)
		return FEED_FORMAT_ERROR;*/
	i = 0;
	while (strncmp(response + i, "<rss", 4) != 0 && response[i] != '\0')	/* rss 2.0 */
		i++;
	if (response[i] != '\0') {
		*shift = i;
		while (strncmp(response + i, "/rss>", 5) != 0 && response[i] != '\0')
			i++;
		if (response[i] != '\0')
			response[i + 5] = '\0';
	 	else
			return FEED_FORMAT_ERROR;
	} else {
		i = 0;
		while (strncmp(response + i, "<feed", 5) != 0 && response[i] != '\0')	/* atom */
			i++;
		if (response[i] != '\0') {
			*shift = i;
			while (strncmp(response + i, "/feed>", 6) != 0 && response[i] != '\0')
				i++;
			if (response[i] != '\0')
				response[i + 6] = '\0';
			else
				return FEED_FORMAT_ERROR;
		} else
			return FEED_FORMAT_ERROR;
	}
	i = 0;
	while (strncmp(response + i, "<?xml", 5) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0')
	*shift = i;
	return OK;
}

/* get host from url but if connecting through proxy */
int connect_with_url(sockt *sock, char *url)
{
	char		host[FILE_NAME_MAXLEN + 1];
	char		port_num[PORT_STR_MAXLEN + 1];
	static int	connect_fail_count = 0;

	if (!get_use_proxy()) {
		str_n_cpy(host, get_host_from_url(url), FILE_NAME_MAXLEN);
		if (strcmp(get_scheme_from_url(url), "http") == 0)
			str_n_cpy(port_num, DEFAULT_HTTP_PORT_STR, PORT_STR_MAXLEN);
		/* https is not supported so far */
		else if (strcmp(get_scheme_from_url(url), "https") == 0) {
			/*str_n_cpy(port_num, DEFAULT_HTTPS_PORT_STR, PORT_STR_MAXLEN);*/
			warning(S_MOD, 2, url, ":\nHTTPS protocol not currently supported");
			return HTTP_UNSUPPORTED_SCHEME;
		} else if (strcmp(get_scheme_from_url(url), "file") == 0)
			;/* do nothing here */
		else if (get_scheme_from_url(url)[0] == '\0') {
			warning(S_MOD, 2, "No scheme found in URL: ", url);
			return HTTP_UNSUPPORTED_SCHEME;
		} else {
			warning(S_MOD, 2, "Unsupported or unknown scheme in URL: ",
				get_scheme_from_url(url));
			return HTTP_UNSUPPORTED_SCHEME;
		}
	} else {
		/* connect via proxy */
		str_n_cpy(host, get_proxy_host(), FILE_NAME_MAXLEN);
		str_n_cpy(port_num, get_proxy_port(), PORT_STR_MAXLEN);
	}
	if ((*sock = connect_to_host(host, port_num)) != SOCK_CREATE_ERROR) {
		connect_fail_count = 0;
		return OK;
	} else {
		if (get_use_proxy()) {
			warning(FALSE, 4, "Can't connect to proxy: ",
				get_proxy_host(),":", get_proxy_port());
			current_feed();
			connection_settings(PROXY_PAGE);
		} else if (get_ticker_env()->selection_mode == SINGLE)
			warning(FALSE, 2, "Can't connect to host: ", host);
		else
			fprintf(STD_ERR, "Can't connect to host: %s\n", host);
		if (++connect_fail_count >= CONNECT_FAIL_MAX) {
			connect_fail_count = 0;
			return CONNECT_TOO_MANY_ERRORS;
		} else
			return SOCK_CANT_CONNECT;
	}
}

/* ***********************************
 *   TODO: DOCUMENT BETTER THIS FUNC
 * ***********************************/
/* we replace resrc->id with modified url and file_name = downloaded resource
 * maxlen of file_name, url, new_url = FILE_NAME_MAXLEN */
int fetch_resource(const char *resrc_id, const char *file_name, char *url)
{
	sockt	sock;
#ifdef G_OS_WIN32
	gchar	*win32_filename;
#endif
	int	status, recv_status;
	char	*response;
	char	*new_url;
	char	header_field[HTTP_HEADER_FIELD_MAXLEN + 1];
	FILE	*fp;
	int	content_length;
	GError	*error = NULL;
	int	i, j;

	if (strncmp(resrc_id, "file://", strlen("file://")) == 0) {
		/* 'file' scheme */
		str_n_cpy(url, resrc_id + strlen("file://"), FILE_NAME_MAXLEN - strlen("file://"));
		if (strncmp(url, "localhost/", strlen("localhost/")) == 0)
			i = strlen("localhost/") - 1;
		else if (url[0] == '/')
			i = 0;
		else
			return RESOURCE_ERROR;

#ifndef G_OS_WIN32
		if (g_file_get_contents(url + i, &response, NULL, &error)) {
#else
		win32_filename = g_win32_locale_filename_from_utf8(url + i);
		if (g_file_get_contents(win32_filename, &response, NULL, &error)) {
			g_free(win32_filename);
#endif
			if ((j = format_quick_check(response, &i)) == OK) {
				if (g_file_set_contents(file_name, response + i, -1, &error)) {
					g_free(response);
					return OK;
				} else {
					g_free(response);
					if (error != NULL) {
						warning(FALSE, 1, error->message);
						g_error_free(error);
					} else
						warning(FALSE, 2, "Can't create file: ", file_name);
					return CREATE_FILE_ERROR;
				}
			} else {
				g_free(response);
				return j;
			}
		} else {
#ifdef G_OS_WIN32
			g_free(win32_filename);
#endif
			if (error != NULL) {
				warning(S_MOD, 1, error->message);
				g_error_free(error);
			} else
				warning(S_MOD, 2, "Can't open file: ", url + i);
			return OPEN_FILE_ERROR;
		}
	} else
		/* 'http' scheme ('https' not supported so far) */
		str_n_cpy(url, resrc_id, FILE_NAME_MAXLEN);
	if ((i = connect_with_url(&sock, url)) == OK) {
		if ((status = get_http_response(sock, "GET", "", url, &new_url,
				&response, &recv_status)) == SEND_ERROR || status == RECV_ERROR) {
			if (recv_status == CONNECTION_CLOSED_BY_SERVER || recv_status == SOCK_SHOULD_BE_CLOSED)
				CLOSE_SOCK(sock);
			return i;
		}
	} else
		return i;
	while (TRUE) {
		/* status checked here are those returned by get_http_response()
		 * so we must make sure they all match */
		if (status == HTTP_CONTINUE) {
			free2(response);
			status = get_http_response(sock, "GET", "", url, &new_url, &response, &recv_status);
			continue;
		} else if (status == HTTP_SWITCH_PROTO) {
			if (strcmp(get_http_header_value("Upgrade", response), "HTTP") == 0)
				warning(S_MOD, 2, url, ":\nSwitching to HTTP protocol requested for this connection"
					" - Request currently not supported");
			else if (strcmp(get_http_header_value("Upgrade", response), "HTTPS") == 0)
				warning(S_MOD, 2, url, ":\nSwitching to HTTPS protocol requested for this connection"
					" - Request currently not supported");
			return status;
		} else if (status == OK) {
			if (strcmp(get_http_header_value("Transfer-Encoding", response), "chunked") == 0) {
				/* chunked transfer encoding */
				remove_chunk_info(&response);
			} else if ((content_length = atoi(get_http_header_value("Content-Length", response))) > 0) {
				/* 'Content-Length' = length of entity body is mandatory
				 * but if 'Transfer-Encoding: chunked' */
				/* do nothing */
			} else {
#ifdef VERBOSE_OUTPUT		/* only if verbose because notification only, nothing done */
				fprintf(STD_ERR, "No 'Transfer-Encoding' nor 'Content-Length' "
					"header field in HTTP response\n");
#endif
			}
			/* quickly 'check format' (xml and rss2.0/atom stuff),
			 * shift to beginnig and cut off trailing part */
			if ((j = format_quick_check(response, &i)) != OK) {
				free2(response);
				return j;
			}
			/* if ok, save to dump file */
			if ((fp = g_fopen(file_name, "wb+")) != NULL) {
				fprintf(fp, "%s", response + i);
				fclose(fp);
				free2(response);
				CLOSE_SOCK(sock);
				return OK;
			} else {
				warning(FALSE, 2, "Can't create file: ", file_name);
				fclose(fp);
				free2(response);
				CLOSE_SOCK(sock);
				return CREATE_FILE_ERROR;
			}
		} else if (status == HTTP_MOVED) {
			i = 0;
			do {
				str_n_cpy(url, new_url, FILE_NAME_MAXLEN);
				if (strcmp(get_http_header_value("Connection", response), "close") == 0 ||\
						strcmp(get_scheme_from_url(url), "https") == 0) {
					free2(response);
					CLOSE_SOCK(sock);
					if (strcmp(get_scheme_from_url(url), "https") == 0) {
						warning(S_MOD, 2, resrc_id, ":\nHTTPS protocol requested for this connection"
							" (following HTTP moved) - Not currently supported");
						return HTTP_ERROR;	/* ???? */
					}
					if ((j = connect_with_url(&sock, url)) != OK)
						return j;
				} else
					free2(response);
			} while ((status = get_http_response(sock, "GET", "", url,
				&new_url, &response, &recv_status)) == HTTP_MOVED &&\
				++i < MAX_HTTP_REDIRECT);
			if (status != HTTP_MOVED && i <= MAX_HTTP_REDIRECT)
				continue;
			else {
				warning(S_MOD, 2, resrc_id, ":\nToo many HTTP redirects");
				free2(response);
				CLOSE_SOCK(sock);
				return HTTP_TOO_MANY_REDIRECTS;
			}
		} else if (status == HTTP_USE_PROXY) {
			warning(FALSE, 4, "Resource: ", url, "\nmust be accessed through proxy.\nProxy host: ",
				new_url);
			str_n_cpy(get_proxy_host(), new_url, PROXY_HOST_MAXLEN);
			free2(response);
			CLOSE_SOCK(sock);
			connection_settings(PROXY_PAGE);
			current_feed();
			return status;
		} else if (status == HTTP_PROXY_AUTH_REQUIRED) {
			/* proxy authentication */
			free2(response);
			CLOSE_SOCK(sock);
			if (get_use_proxy_auth()) {
				if ((i = connect_with_url(&sock, url)) != OK)
					return i;
				snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
					"Proxy-Authorization: Basic %s\r\n\r\n",
					get_proxy_auth_str());
				if ((status = get_http_response(sock, "GET", header_field, url, &new_url,
						&response, &recv_status)) == HTTP_PROXY_AUTH_REQUIRED) {
					warning(FALSE, 2, "Proxy authentication failed for: ", get_proxy_host());
					free2(response);
					CLOSE_SOCK(sock);
					if (connection_settings(PROXY_PAGE) == GTK_RESPONSE_OK) {
						if ((i = connect_with_url(&sock, url)) != OK)
							return i;
						status = get_http_response(sock, "GET", header_field, url,
							&new_url, &response, &recv_status);
					} else
						return HTTP_NO_PROXY_AUTH_CREDENTIALS;
				}
				continue;
			} else {
				warning(FALSE, 2, "Proxy authentication required for: ", get_proxy_host());
				if (connection_settings(PROXY_PAGE) == GTK_RESPONSE_OK &&\
					get_use_proxy_auth()) {
					if ((i = connect_with_url(&sock, url)) != OK)
						return i;
					snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
						"Proxy-Authorization: Basic %s\r\n\r\n",
						get_proxy_auth_str());
					status = get_http_response(sock, "GET", header_field, url,
						&new_url, &response, &recv_status);
					continue;
				} else
					return HTTP_NO_PROXY_AUTH_CREDENTIALS;
			}
		} else if (status == HTTP_UNAUTHORIZED) {
			/* http authentication - only basic so far */
			free2(response);
			CLOSE_SOCK(sock);
			if (get_use_authentication()) {
				if ((i = connect_with_url(&sock, url)) != OK)
					return i;
				snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
					"Authorization: Basic %s\r\n\r\n",
					get_http_auth_str());
				if ((status = get_http_response(sock, "GET", header_field,
						url, &new_url, &response, &recv_status)) == HTTP_UNAUTHORIZED) {
					warning(FALSE, 2, "HTTP authentication failed for: ", url);
					free2(response);
					CLOSE_SOCK(sock);
					if (connection_settings(AUTH_PAGE) == GTK_RESPONSE_OK) {
						if ((i = connect_with_url(&sock, url)) != OK)
							return i;
						status = get_http_response(sock, "GET", header_field,
							url, &new_url, &response, &recv_status);
					} else
						return HTTP_NO_AUTH_CREDENTIALS;
				}
				continue;
			} else {
				warning(FALSE, 2, "HTTP authentication required for: ", url);
				if (connection_settings(AUTH_PAGE) == GTK_RESPONSE_OK &&\
					get_use_authentication()) {
					if ((i = connect_with_url(&sock, url)) != OK)
						return i;
					snprintf(header_field, HTTP_HEADER_FIELD_MAXLEN + 1,
						"Authorization: Basic %s\r\n\r\n",
						get_http_auth_str());
					status = get_http_response(sock, "GET", header_field,
						url, &new_url, &response, &recv_status);
					continue;
				} else
					return HTTP_NO_AUTH_CREDENTIALS;
			}
		} else if (status == HTTP_BAD_REQUEST) {
			warning(S_MOD, 2, url, ":\nBad Request");
			free2(response);
			CLOSE_SOCK(sock);
			return status;
		} else if (status == HTTP_FORBIDDEN) {
			warning(S_MOD, 2, url, ":\nForbidden");
			free2(response);
			CLOSE_SOCK(sock);
			return status;
		} else if (status == HTTP_NOT_FOUND) {
			warning(S_MOD, 2, url, ":\nNot found");
			free2(response);
			CLOSE_SOCK(sock);
			return status;
		} else if (status == HTTP_INT_SERVER_ERROR) {
			warning(S_MOD, 2, url, ":\nInternal server error");
			free2(response);
			CLOSE_SOCK(sock);
			return status;
		} else if (status == HTTP_NO_STATUS_CODE) {
			warning(S_MOD, 2, url, ":\nNo HTTP status code returned");
			free2(response);
			CLOSE_SOCK(sock);
			return status;
		} else {
			warning(S_MOD, 3, url, ":\nHTTP response status code: ",
				itoa2(status - 1000));
			if (response != NULL)
				free2(response);
			CLOSE_SOCK(sock);
			return HTTP_ERROR;
		}
	}
}

/* rq_str may contain header fields(s) separated and ended by "\r\n" */
const char *build_http_request(const char *method, const char *path,\
		const char *host, const char* rq_str)
{
	static char	str[HTTP_REQUEST_MAXLEN + 1];

	snprintf(str, HTTP_REQUEST_MAXLEN,
		"%s %s HTTP/1.1\r\n"	/* start line with method and path */
		"Host: %s\r\n"		/* mandatory host header field */
		"User-Agent: "APP_NAME"-"APP_VERSION_NUMBER"\r\n"
		"%s\r\n",		/* optional extra header field(s) */
		method, path, host, rq_str);
	return (const char *)str;
}

/*
 * rq_str may contain header field(s) separated and ended by "\r\n"
 * must 'free2' response afterwards
 */
int get_http_response(sockt sock, const char *rq_method, const char *rq_str,
	const char *rq_url, char **new_rq_url, char **response, int *recv_status)
{
	char		*str;
	static char	location[FILE_NAME_MAXLEN + 1];
	int		bytes_sent, bytes_received, status_code;

	location[0] = '\0';
	*new_rq_url = (char *)location;
	*response = NULL;
	str = (char *)build_http_request(
		rq_method,
		(get_use_proxy() ? rq_url : get_path_from_url(rq_url)),	/* path or full (absolute) url if using proxy */
		/* is that correct when using proxy? */
		get_host_from_url(rq_url),
		rq_str);
	if ((bytes_sent = send_full(sock, (const char *)str)) >= 0) {
		if ((*response = recv_full(sock, &bytes_received, recv_status)) != NULL) {
			if ((status_code = get_http_status_code(*response)) == 100)
				return HTTP_CONTINUE;
			else if (status_code == 101)
				return HTTP_SWITCH_PROTO;
			else if (status_code == 200)
				return OK;
			else if (	/* 'multiple choices' */
					status_code == 300 ||\
					/* 'moved permanently' */
					status_code == 301 ||\
					/* 'found' */
					status_code == 302 ||\
					/* 'see other' */
					status_code == 303 ||\
					/* 'not modified' */
					status_code == 304 ||\
					/* 'moved temporarily' */
					status_code == 307) {
				str_n_cpy(location, get_http_header_value("Location", *response),
					FILE_NAME_MAXLEN);
				return HTTP_MOVED;	/* must use the new url in 'Location' */
			} else if (status_code == 305)
				return HTTP_USE_PROXY;
			else if (status_code == 400)
				return HTTP_BAD_REQUEST;
			else if (status_code == 401)
				return HTTP_UNAUTHORIZED;
			else if (status_code == 403)
				return HTTP_FORBIDDEN;
			else if (status_code ==  404)
				return HTTP_NOT_FOUND;
			else if (status_code == 407)
				return HTTP_PROXY_AUTH_REQUIRED;
			else if (status_code == 500)
				return HTTP_INT_SERVER_ERROR;
			else if (status_code == -1)
				return HTTP_NO_STATUS_CODE;
			else
				return status_code + 1000;
		} else
			return RECV_ERROR;
	} else
		return SEND_ERROR;
}

/* return -1 if error */
int get_http_status_code(const char *response)
{
	char	status_code[4];
	int	i = 0;

	while (response[i] != ' ' && response[i] != TAB) {
		if (response[i] == '\n' || response[i] == '\r' || response[i] == '\0')
			return -1;
		else
			i++;
	}
	while (response[i] == ' ' || response[i] == TAB)
		i++;
	if (response[i] == '\n' || response[i] == '\r' || response[i] == '\0')
		status_code[0] = '\0';
	else
		str_n_cpy(status_code, response + i, 3);
	return atoi(status_code);
}

/* get header_value as string (empty one if header name not found) */
const char *get_http_header_value(const char *header_name, const char *response)
{
	static char	header_value[1024];
	int		len = strlen(header_name), i = 0;

	while (strncasecmp(response + i, header_name, len) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0') {
		i += len;
		if (response[i] != ':') {
				header_value[0] = '\0';
				return (const char *)header_value;
		} else
			i++;
		while (response[i] == ' ' || response[i] == TAB) {
			if (response[i] == '\n' || response[i] == '\r' ||\
					response[i] == '\0') {
				header_value[0] = '\0';
				return (const char *)header_value;
			} else
				i++;
		}
		str_n_cpy(header_value, response + i, 1023);
		i = 0;
		while (header_value[i] != '\0' && header_value[i] != ' ' &&\
			header_value[i] != TAB && header_value[i] != '\n' &&
			header_value[i] != '\r' && i < 1023)
			i++;
		header_value[i] = '\0';
	} else
		header_value[0] = '\0';
	return (const char *)header_value;
}

/* return 0 if none found */
int go_after_next_cr_lf(const char *response)
{
	int	i = 0;

	while (strncmp(response + i, "\r\n", 2) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0')
		return i + 2;
	else
		return 0;
}

/* return 0 if none found */
int go_after_next_empty_line(const char *response)
{
	int	i = 0;

	while (strncmp(response + i, "\r\n\r\n", 4) != 0 && response[i] != '\0')
		i++;
	if (response[i] != '\0')
		return i + 4;
	else
		return 0;
}

/* response must point to chunk size hexa str or preceeding space(s)
 * return -1 if invalid chunk size format (ie not an hexa str) */
int get_http_chunk_size(const char *response)
{
	char	size_str[32], *tailptr;
	int	size, i = 0;

	while (response[i] == ' ' || response[i] == TAB ||\
			response[i] == '\n' || response[i] == '\r')
		i++;
	str_n_cpy(size_str, response + i, 31);
	i = 0;
	while (size_str[i] != ' ' && size_str[i] != TAB &&\
			size_str[i] != '\n' && size_str[i] != '\r' &&\
			size_str[i] != ';' && size_str[i] != '\0' &&\
			i < 31)
		i++;
	size_str[i] = '\0';
	size = (int)strtoul(size_str, &tailptr, 16);
	if (tailptr == size_str) {
		fprintf(STD_ERR, "Invalid hexadecimal value in HTTP chunk size: %s\n",
			size_str);
		return -1;
	} else
		return size;
}

/*
 * 'http://www.sth1.org/sth2/sth2.xml' -> 'http'
 * (expect one scheme in url, return empty str if none)
 */
const char *get_scheme_from_url(const char* url)
{
	static char	str[16];
	int		i = 0;

	while (strncmp(url + i, "://", 3) != 0 && url[i] != '\0')
		i++;
	if (url[i] != '\0')
		str_n_cpy(str, url, MIN(i, 15));
	else
		str[0] = '\0';
	return (const char *)str;
}

/*
 * 'http://www.sth1.org/sth2/sth2.xml' -> 'www.sth1.org'
 * (expect one scheme in url, return empty str if none)
 */
const char *get_host_from_url(const char* url)
{
	static char	str[FILE_NAME_MAXLEN + 1];
	int		i = 0;

	while (strncmp(url + i, "://", 3) != 0 && url[i] != '\0')
		i++;
	if (url[i] != '\0') {
		str_n_cpy(str, url + i + 3, FILE_NAME_MAXLEN);
		i = 0;
		while (str[i] != '\0' && str[i] != '/' && i < FILE_NAME_MAXLEN)
			i++;
		str[i] = '\0';
	} else
		str[0] = '\0';
	return (const char *)str;
}

/*
 * 'http://www.sth1.org/sth2/sth2.xml' -> '/sth2/sth2.xml'
 * (expect one scheme in url, return empty str if none)
 */
const char *get_path_from_url(const char* url)
{
	static char	str[FILE_NAME_MAXLEN + 1];
	int		i = 0;

	while (strncmp(url + i, "://", 3) != 0 && url[i] != '\0')
		i++;
	if (url[i] != '\0') {
		i += 3;
		while (url[i] != '\0' && url[i] != '/')
			i++;
		if (url[i] != '\0')
			str_n_cpy(str, url + i, FILE_NAME_MAXLEN);
		else
			str[0] = '\0';
	} else
		str[0] = '\0';
	return (const char *)str;
}
