"""
main_body.py

Copyright 2007 Andres Riancho

This file is part of w3af, http://w3af.org/ .

w3af is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation version 2 of the License.

w3af is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with w3af; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
"""
import gtk

from w3af.core.ui.gui import entries
from w3af.core.ui.gui.pluginconfig import ConfigPanel
from w3af.core.ui.gui.tabs.exploit.exploits import ExploitTree
from w3af.core.ui.gui.tabs.exploit.vuln_list import VulnerabList
from w3af.core.ui.gui.tabs.exploit.shell import Shells
from w3af.core.ui.gui.tabs.exploit.proxy import Proxies
from w3af.core.ui.gui.tabs.exploit.vuln_add import VulnAddDialog
from w3af.core.ui.gui.tabs.exploit.exploit_all import (effectively_exploit_all,
                                                  ExploitAllDialog)


class ExploitBody(entries.RememberingHPaned):
    """Body of the exploit tab.

    :param w3af: the Core instance.

    :author: Facundo Batista <facundobatista =at= taniquetil.com.ar>
    """
    def __init__(self, w3af):
        super(ExploitBody, self).__init__(w3af, "pane-exploitbody", 750)
        self.w3af = w3af
        self.panels = {}

        # left & right
        exploitvuln = self._buildExplVuln()
        interac = self._buildInteraction()
        self.panels["exploitvuln"] = exploitvuln
        self.panels["interac"] = interac

        # pack it all and show
        self.pack1(exploitvuln)
        self.pack2(interac)

        self.panactiv = dict((x, True) for x in self.panels)
        self.show()

    def _buildExplVuln(self):
        """The pane with the exploit list and vulnerabilities tree."""
        hpan = entries.RememberingHPaned(self.w3af, "pane-epxlvuln", 300)
        vpan = entries.RememberingVPaned(self.w3af, "pane-expl-desc", 300)
        
        # left-bottom
        plugin_cfg = ConfigPanel()
        
        # left-top
        exploitlist = ExploitTree(self.w3af, plugin_cfg)
        scrollwin1 = gtk.ScrolledWindow()
        scrollwin1.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrollwin1.add_with_viewport(exploitlist)
        scrollwin1.show()

        vpan.pack1(scrollwin1)
        vpan.pack2(plugin_cfg)
        vpan.show()
        
        # Right
        vuln_vbox = gtk.VBox()
        interac = VulnerabList(self.w3af, exploitlist)
        exploitlist.vulnerabs = interac
        scrollwin2 = gtk.ScrolledWindow()
        scrollwin2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrollwin2.add_with_viewport(interac)
        scrollwin2.show()
        
        add_button = entries.SemiStockButton("Add...", gtk.STOCK_ADD,
                                             "Manually add a new vulnerability")
        add_button.show()
        add_button.connect("clicked", self.add_vuln)
        
        vuln_vbox.pack_start(scrollwin2)
        vuln_vbox.pack_start(add_button, False, False)
        vuln_vbox.show()
        
        # pack it all and show
        hpan.pack1(vpan)
        hpan.pack2(vuln_vbox)
        hpan.show()
        return hpan

    def _buildInteraction(self):
        """The pane with the shells and proxies list."""
        pan = entries.RememberingVPaned(self.w3af, "pane-explinteraction")

        # left
        shells = Shells(self.w3af)
        scrollwin1 = gtk.ScrolledWindow()
        scrollwin1.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrollwin1.add_with_viewport(shells)
        scrollwin1.show()

        # rigth
        proxies = Proxies()
        scrollwin2 = gtk.ScrolledWindow()
        scrollwin2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrollwin2.add_with_viewport(proxies)
        scrollwin2.show()

        # pack it all and show
        pan.pack1(scrollwin1)
        pan.pack2(scrollwin2)
        pan.show()
        return pan

    def add_vuln(self, *args):
        vad = VulnAddDialog(self.w3af)
        vad.run()

    def toggle_panels(self, panel, active):
        """Turn on and off the panels.

        :param panel: The panel to turn on and off
        :param active: If it should be activated or deactivated
        """
        widg = self.panels[panel]
        if active:
            widg.show()
        else:
            widg.hide()
        self.panactiv[panel] = active

    def exploit_all(self):
        """Exploit all vulns with all plugins."""

        ea = ExploitAllDialog(self.w3af)
        resp = ea.run()
        if resp != gtk.RESPONSE_OK:
            return

        effectively_exploit_all(self.w3af, ea.enabled_plugins, ea.stopOnFirst)
        return
