# 34 "src/lexer.mll"
 
  open Lexing
  open Logic
  open Parser

  let keywords = Hashtbl.create 97
  let () = 
    List.iter 
      (fun (x,y) -> Hashtbl.add keywords x y)
      [ "absurd", ABSURD;
	"and", AND;
        "array", ARRAY;
	"as", AS;
	"assert", ASSERT;
	"axiom", AXIOM;
	"begin", BEGIN;
        "bool", BOOL;
        "check", CHECK;
	"do", DO;
	"done", DONE;
        "else", ELSE;
	"end", END;
	"exception", EXCEPTION; 
	"exists", EXISTS;
	"external", EXTERNAL;
        "false", FALSE;
	"for", FOR;
	"forall", FORALL;
	"fun", FUN;
	"function", FUNCTION;
	"goal", GOAL;
	"if", IF;
	"in", IN;
	"include", INCLUDE;
	"inductive", INDUCTIVE;
	"int", INT;
	"invariant", INVARIANT;
        "lemma", LEMMA;
	"let", LET;
	"logic", LOGIC;
	"match", MATCH;
	"not", NOT;
	"of", OF;
	"or", OR;
	"parameter", PARAMETER;
	"predicate", PREDICATE;
	"prop", PROP;
	"raise", RAISE;
	"raises", RAISES;
	"reads", READS;
	"real", REAL;
	"rec", REC;
	"ref", REF;
	"returns", RETURNS;
	"then", THEN;
	"true", TRUE;
	"try", TRY;
	"type", TYPE;
	"unit", UNIT;
	"variant", VARIANT;
	"void", VOID;
	"while", WHILE;
	"with", WITH;
        "writes", WRITES ]
	       
  let newline lexbuf =
    let pos = lexbuf.lex_curr_p in
    lexbuf.lex_curr_p <- 
      { pos with pos_lnum = pos.pos_lnum + 1; pos_bol = pos.pos_cnum }

  let string_buf = Buffer.create 1024

  exception Lexical_error of string

  let char_for_backslash = function
    | 'n' -> '\n'
    | 't' -> '\t'
    | c -> c

  let update_loc lexbuf file line chars =
    let pos = lexbuf.lex_curr_p in
    let new_file = match file with None -> pos.pos_fname | Some s -> s in
    lexbuf.lex_curr_p <- 
      { pos with
	  pos_fname = new_file;
	  pos_lnum = int_of_string line;
	  pos_bol = pos.pos_cnum - int_of_string chars;
      }

  let remove_leading_plus s =
    let n = String.length s in 
    if n > 0 && s.[0] = '+' then String.sub s 1 (n-1) else s

  let option_app f = function None -> None | Some x -> Some (f x)


# 99 "src/lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\213\255\214\255\215\255\025\000\002\000\002\000\005\000\
    \224\255\225\255\227\255\228\255\229\255\231\255\031\000\032\000\
    \084\000\034\000\241\255\069\000\243\255\244\255\246\255\247\255\
    \089\000\084\000\101\000\123\000\196\000\005\000\254\255\054\001\
    \177\000\002\000\064\001\113\001\123\001\172\001\182\001\255\255\
    \231\001\241\001\150\000\034\002\044\002\255\255\093\002\103\002\
    \152\000\115\002\138\002\165\000\148\002\170\002\192\002\202\002\
    \224\002\234\002\244\002\000\003\010\003\020\003\045\003\081\003\
    \110\003\143\003\175\003\185\003\219\003\250\003\004\004\017\004\
    \248\255\240\255\239\255\232\255\235\255\080\000\238\255\216\255\
    \233\255\221\255\220\255\218\255\217\255\154\000\251\255\252\255\
    \253\255\101\000\103\000\255\255\254\255\055\001\251\255\252\255\
    \253\255\001\000\255\255\254\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\255\255\042\000\036\000\033\000\032\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\021\000\018\000\
    \019\000\025\000\255\255\013\000\255\255\255\255\255\255\255\255\
    \010\000\029\000\004\000\004\000\003\000\002\000\255\255\042\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\005\000\255\255\005\000\255\255\255\255\005\000\
    \255\255\005\000\255\255\005\000\255\255\255\255\255\255\255\255\
    \255\255\006\000\255\255\255\255\005\000\255\255\005\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\004\000\004\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\004\000\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\033\000\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\086\000\000\000\000\000\
    \000\000\255\255\255\255\000\000\000\000\094\000\000\000\000\000\
    \000\000\099\000\000\000\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\029\000\030\000\000\000\255\255\029\000\029\000\255\255\
    \000\000\000\000\029\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \029\000\020\000\003\000\031\000\035\000\029\000\004\000\023\000\
    \024\000\021\000\012\000\013\000\022\000\017\000\025\000\011\000\
    \027\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\019\000\018\000\016\000\015\000\014\000\084\000\
    \010\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\009\000\080\000\008\000\079\000\028\000\
    \074\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\007\000\005\000\006\000\083\000\082\000\
    \081\000\077\000\073\000\072\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\078\000\092\000\
    \091\000\076\000\075\000\050\000\000\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\042\000\
    \000\000\048\000\000\000\042\000\088\000\048\000\000\000\000\000\
    \000\000\050\000\051\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\000\000\042\000\000\000\
    \048\000\039\000\043\000\045\000\000\000\000\000\043\000\000\000\
    \051\000\000\000\089\000\000\000\090\000\000\000\000\000\000\000\
    \000\000\000\000\051\000\000\000\000\000\000\000\000\000\000\000\
    \053\000\043\000\053\000\049\000\000\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\000\000\
    \051\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\028\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\049\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\000\000\000\000\
    \002\000\255\255\255\255\000\000\000\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\000\000\
    \000\000\000\000\000\000\028\000\000\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\034\000\
    \000\000\096\000\000\000\034\000\000\000\000\000\000\000\000\000\
    \000\000\034\000\000\000\000\000\000\000\034\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\034\000\000\000\
    \033\000\098\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \034\000\000\000\033\000\000\000\000\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\035\000\000\000\000\000\000\000\035\000\000\000\
    \000\000\000\000\000\000\000\000\037\000\000\000\000\000\000\000\
    \037\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\035\000\000\000\097\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\087\000\037\000\000\000\000\000\000\000\000\000\
    \000\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\037\000\000\000\000\000\
    \000\000\037\000\000\000\000\000\000\000\000\000\000\000\040\000\
    \000\000\000\000\000\000\040\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\037\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\040\000\000\000\
    \000\000\039\000\000\000\000\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \040\000\000\000\000\000\000\000\040\000\000\000\000\000\000\000\
    \000\000\000\000\042\000\000\000\000\000\000\000\042\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\040\000\
    \000\000\000\000\039\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\042\000\000\000\000\000\039\000\000\000\000\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\043\000\000\000\000\000\000\000\043\000\
    \000\000\000\000\000\000\000\000\000\000\046\000\000\000\095\000\
    \000\000\046\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\043\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\046\000\000\000\000\000\045\000\
    \000\000\000\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\046\000\000\000\
    \000\000\000\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \048\000\000\000\000\000\000\000\048\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\046\000\000\000\000\000\
    \045\000\000\000\000\000\000\000\000\000\000\000\000\000\048\000\
    \000\000\000\000\045\000\000\000\000\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\061\000\000\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\000\000\054\000\
    \000\000\000\000\000\000\000\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\060\000\000\000\060\000\000\000\054\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\058\000\000\000\058\000\000\000\056\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\000\000\056\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\064\000\000\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\000\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\066\000\063\000\066\000\000\000\
    \000\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\000\000\000\000\000\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\063\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\063\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\000\000\000\000\000\000\000\000\000\000\000\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\063\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\063\000\000\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \069\000\000\000\000\000\000\000\000\000\071\000\000\000\071\000\
    \000\000\063\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\000\000\000\000\
    \069\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\033\000\000\000\029\000\033\000\
    \255\255\255\255\029\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\033\000\029\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\004\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\014\000\000\000\015\000\000\000\
    \017\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\005\000\006\000\
    \007\000\016\000\019\000\024\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\077\000\089\000\
    \090\000\016\000\016\000\026\000\255\255\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\042\000\
    \255\255\048\000\255\255\042\000\085\000\048\000\255\255\255\255\
    \255\255\027\000\026\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\255\255\042\000\255\255\
    \048\000\042\000\032\000\048\000\255\255\255\255\032\000\255\255\
    \027\000\255\255\085\000\255\255\085\000\255\255\255\255\255\255\
    \255\255\255\255\026\000\255\255\255\255\255\255\255\255\255\255\
    \051\000\032\000\051\000\027\000\255\255\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\255\255\
    \027\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\028\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\027\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\255\255\255\255\
    \000\000\097\000\033\000\255\255\255\255\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\255\255\
    \255\255\255\255\255\255\028\000\255\255\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\031\000\
    \255\255\093\000\255\255\031\000\255\255\255\255\255\255\255\255\
    \255\255\034\000\255\255\255\255\255\255\034\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\031\000\255\255\
    \031\000\093\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \034\000\255\255\034\000\255\255\255\255\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\035\000\255\255\255\255\255\255\035\000\255\255\
    \255\255\255\255\255\255\255\255\036\000\255\255\255\255\255\255\
    \036\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\255\255\093\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\085\000\036\000\255\255\255\255\255\255\255\255\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\037\000\255\255\255\255\
    \255\255\037\000\255\255\255\255\255\255\255\255\255\255\038\000\
    \255\255\255\255\255\255\038\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\038\000\255\255\
    \255\255\038\000\255\255\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \040\000\255\255\255\255\255\255\040\000\255\255\255\255\255\255\
    \255\255\255\255\041\000\255\255\255\255\255\255\041\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\040\000\
    \255\255\255\255\040\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\041\000\255\255\255\255\041\000\255\255\255\255\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\043\000\255\255\255\255\255\255\043\000\
    \255\255\255\255\255\255\255\255\255\255\044\000\255\255\093\000\
    \255\255\044\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\043\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\044\000\255\255\255\255\044\000\
    \255\255\255\255\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\046\000\255\255\
    \255\255\255\255\046\000\255\255\255\255\255\255\255\255\255\255\
    \047\000\255\255\255\255\255\255\047\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\046\000\255\255\255\255\
    \046\000\255\255\255\255\255\255\255\255\255\255\255\255\047\000\
    \255\255\255\255\047\000\255\255\255\255\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\049\000\255\255\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\049\000\049\000\049\000\049\000\
    \049\000\049\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\255\255\050\000\
    \255\255\255\255\255\255\255\255\049\000\049\000\049\000\049\000\
    \049\000\049\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\054\000\255\255\054\000\255\255\050\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\056\000\255\255\056\000\255\255\055\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\255\255\055\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\061\000\061\000\061\000\
    \061\000\061\000\061\000\062\000\255\255\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\062\000\062\000\
    \062\000\062\000\062\000\062\000\255\255\061\000\061\000\061\000\
    \061\000\061\000\061\000\255\255\063\000\062\000\063\000\255\255\
    \255\255\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\255\255\255\255\255\255\062\000\062\000\
    \062\000\062\000\062\000\062\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\062\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\064\000\
    \064\000\064\000\064\000\064\000\064\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\064\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\255\255\255\255\255\255\255\255\255\255\255\255\064\000\
    \064\000\064\000\064\000\064\000\064\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\064\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\067\000\067\000\067\000\067\000\067\000\067\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\067\000\255\255\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\255\255\255\255\255\255\
    \255\255\255\255\067\000\067\000\067\000\067\000\067\000\067\000\
    \068\000\255\255\255\255\255\255\255\255\069\000\255\255\069\000\
    \255\255\067\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\255\255\255\255\
    \068\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\022\000\044\000\000\000\000\000\000\000\099\000\
    \125\000\000\000\152\000\178\000\209\000\235\000\005\001\096\000\
    \031\001\016\001\001\000\065\001\080\001\117\000\091\001\103\001\
    \002\000\115\001\138\001\032\000\148\001\170\001\180\001\192\001\
    \202\001\214\001\224\001\234\001\246\001\000\002\025\002\000\000\
    \063\002\000\000\000\000\101\002\135\002\166\002\176\002\189\002\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\138\000\000\000\152\000\000\000\000\000\138\000\
    \000\000\152\000\000\000\152\000\000\000\000\000\000\000\000\000\
    \000\000\183\000\000\000\000\000\138\000\000\000\152\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\062\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\001\000\112\000\112\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\004\000\000\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\017\000\000\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\032\000\029\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\001\000\000\000\000\000\000\000\
    \001\000\029\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\029\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\001\000\037\000\042\000\052\000\000\000\
    \000\000\000\000\052\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\029\000\000\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\052\000\000\000\000\000\
    \000\000\001\000\000\000\000\000\037\000\001\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \001\000\000\000\065\000\001\000\000\000\000\000\000\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\052\000\000\000\000\000\000\000\052\000\000\000\
    \000\000\000\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\000\000\000\000\000\000\000\000\
    \000\000\052\000\000\000\000\000\052\000\000\000\000\000\000\000\
    \052\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\052\000\000\000\000\000\082\000\000\000\
    \000\000\000\000\082\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\082\000\000\000\000\000\
    \082\000\000\000\000\000\000\000\082\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\112\000\000\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\082\000\
    \109\000\109\000\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\052\000\000\000\000\000\000\000\052\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\082\000\000\000\000\000\000\000\082\000\000\000\000\000\
    \000\000\052\000\000\000\000\000\082\000\000\000\000\000\000\000\
    \082\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \082\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\082\000\000\000\000\000\000\000\000\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\112\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\000\000\000\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\128\000\000\000\131\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\131\000\131\000\131\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\000\000\029\000\
    \000\000\000\000\000\000\000\000\131\000\131\000\131\000\131\000\
    \131\000\131\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\000\000\029\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\000\000\029\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\000\000\029\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \165\000\165\000\165\000\165\000\165\000\165\000\165\000\165\000\
    \165\000\165\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\165\000\165\000\165\000\165\000\165\000\165\000\168\000\
    \000\000\173\000\173\000\173\000\173\000\173\000\173\000\173\000\
    \173\000\173\000\173\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\173\000\173\000\173\000\173\000\173\000\173\000\
    \000\000\165\000\165\000\165\000\165\000\165\000\165\000\000\000\
    \000\000\180\000\000\000\000\000\000\000\000\000\000\000\165\000\
    \165\000\165\000\165\000\165\000\165\000\165\000\165\000\165\000\
    \165\000\000\000\173\000\173\000\173\000\173\000\173\000\173\000\
    \165\000\165\000\165\000\165\000\165\000\165\000\000\000\000\000\
    \000\000\180\000\000\000\000\000\000\000\000\000\000\000\180\000\
    \000\000\000\000\000\000\000\000\000\000\165\000\165\000\165\000\
    \165\000\165\000\165\000\165\000\165\000\165\000\165\000\000\000\
    \165\000\165\000\165\000\165\000\165\000\165\000\165\000\165\000\
    \165\000\165\000\165\000\165\000\000\000\000\000\000\000\180\000\
    \000\000\000\000\000\000\000\000\000\000\180\000\000\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\000\000\000\000\000\000\000\000\000\000\165\000\165\000\
    \165\000\165\000\165\000\165\000\029\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\180\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\149\000\
    \149\000\149\000\000\000\000\000\029\000\149\000\149\000\149\000\
    \149\000\149\000\149\000\149\000\149\000\149\000\149\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\033\000\255\255\255\255\033\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\033\000\000\000\042\000\048\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\026\000\255\255\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\027\000\026\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\031\000\255\255\255\255\255\255\
    \031\000\027\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\026\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\031\000\027\000\031\000\032\000\255\255\
    \255\255\255\255\032\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\027\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\032\000\255\255\255\255\
    \255\255\034\000\255\255\255\255\027\000\034\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \034\000\255\255\034\000\035\000\255\255\255\255\255\255\035\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\035\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\036\000\255\255\255\255\255\255\036\000\255\255\
    \255\255\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\255\255\255\255\255\255\
    \255\255\036\000\255\255\255\255\037\000\255\255\255\255\255\255\
    \037\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \033\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\037\000\255\255\255\255\038\000\255\255\
    \255\255\255\255\038\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\038\000\255\255\255\255\
    \040\000\255\255\255\255\255\255\040\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\041\000\255\255\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\040\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\043\000\255\255\255\255\255\255\043\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\044\000\255\255\255\255\255\255\044\000\255\255\255\255\
    \255\255\043\000\255\255\255\255\046\000\255\255\255\255\255\255\
    \046\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \044\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\046\000\255\255\255\255\255\255\255\255\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\047\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\255\255\255\255\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\049\000\255\255\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\049\000\049\000\049\000\049\000\
    \049\000\049\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\255\255\050\000\
    \255\255\255\255\255\255\255\255\049\000\049\000\049\000\049\000\
    \049\000\049\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\255\255\050\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\255\255\055\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\255\255\055\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\061\000\061\000\061\000\061\000\061\000\061\000\062\000\
    \255\255\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\062\000\062\000\062\000\062\000\062\000\062\000\
    \255\255\061\000\061\000\061\000\061\000\061\000\061\000\255\255\
    \255\255\062\000\255\255\255\255\255\255\255\255\255\255\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\255\255\062\000\062\000\062\000\062\000\062\000\062\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\255\255\255\255\
    \255\255\062\000\255\255\255\255\255\255\255\255\255\255\064\000\
    \255\255\255\255\255\255\255\255\255\255\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\255\255\
    \064\000\064\000\064\000\064\000\064\000\064\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\255\255\255\255\255\255\064\000\
    \255\255\255\255\255\255\255\255\255\255\067\000\255\255\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\255\255\255\255\255\255\255\255\255\255\067\000\067\000\
    \067\000\067\000\067\000\067\000\068\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\067\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\255\255\255\255\068\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\010\255\255\008\255\255\009\255\008\255\007\255\255\009\255\
    \255\008\255\009\255\255\007\255\008\255\009\255\255\020\255\255\
    \009\255\008\255\255\019\255\018\255\255\014\255\013\255\255\012\
    \255\011\255\255\012\255\255\017\255\016\255\015\255\255\014\255\
    \255\013\255\014\255\255\011\255\012\255\255\015\011\016\012\017\
    \255\255\016\255\255\011\015\012\016\015\255\016\255\017\255\255\
    \000\010\001\011\005\013\004\014\002\012\003\017\255\017\255\255\
    \011\015\012\016\255\005\255\000\010\001\011\002\012\003\017\255\
    \022\255\255\023\255\022\255\019\255\255\001\255\005\006\004\007\
    \003\008\002\009\255\021\255\255\005\006\004\007\003\008\002\009\
    \001\020\000\021\255\023\255\255\023\255\022\255\255\019\255\022\
    \255\023\255\255\024\255\255\004\018\003\019\002\022\001\023\000\
    \024\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 25 (-1) ; (* L=2 [7] <- p ; [6] <- p ;  *)
  lexbuf.Lexing.lex_mem.(7) <- lexbuf.Lexing.lex_curr_pos ;
  lexbuf.Lexing.lex_mem.(6) <- lexbuf.Lexing.lex_curr_pos ;
  __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 143 "src/lexer.mll"
                                                 file
# 782 "src/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(5) lexbuf.Lexing.lex_mem.(4)
and
# 144 "src/lexer.mll"
                      line
# 787 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 144 "src/lexer.mll"
                                              char
# 792 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(3) in
# 145 "src/lexer.mll"
      ( update_loc lexbuf file line char; token lexbuf )
# 796 "src/lexer.ml"

  | 1 ->
# 147 "src/lexer.mll"
      ( newline lexbuf; token lexbuf )
# 801 "src/lexer.ml"

  | 2 ->
# 149 "src/lexer.mll"
      ( token lexbuf )
# 806 "src/lexer.ml"

  | 3 ->
let
# 150 "src/lexer.mll"
             id
# 812 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 151 "src/lexer.mll"
      ( try Hashtbl.find keywords id with Not_found -> IDENT id )
# 816 "src/lexer.ml"

  | 4 ->
let
# 152 "src/lexer.mll"
              s
# 822 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 153 "src/lexer.mll"
      ( INTEGER s )
# 826 "src/lexer.ml"

  | 5 ->
let
# 154 "src/lexer.mll"
               i
# 832 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(5) lexbuf.Lexing.lex_mem.(4)
and
# 154 "src/lexer.mll"
                         f
# 837 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2)
and
# 154 "src/lexer.mll"
                                                            e
# 842 "src/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0) in
# 157 "src/lexer.mll"
      ( FLOAT (RConstDecimal (i, f, option_app remove_leading_plus e)) )
# 846 "src/lexer.ml"

  | 6 ->
let
# 158 "src/lexer.mll"
                                  i
# 852 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(4) lexbuf.Lexing.lex_mem.(3)
and
# 158 "src/lexer.mll"
                                                        f
# 857 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(1)
and
# 161 "src/lexer.mll"
                                    e
# 862 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 162 "src/lexer.mll"
      ( FLOAT (RConstHexa (i, f, remove_leading_plus e)) )
# 866 "src/lexer.ml"

  | 7 ->
# 164 "src/lexer.mll"
      ( comment lexbuf; token lexbuf )
# 871 "src/lexer.ml"

  | 8 ->
# 166 "src/lexer.mll"
      ( QUOTE )
# 876 "src/lexer.ml"

  | 9 ->
# 168 "src/lexer.mll"
      ( COMMA )
# 881 "src/lexer.ml"

  | 10 ->
# 170 "src/lexer.mll"
      ( LEFTPAR )
# 886 "src/lexer.ml"

  | 11 ->
# 172 "src/lexer.mll"
      ( RIGHTPAR )
# 891 "src/lexer.ml"

  | 12 ->
# 174 "src/lexer.mll"
      ( BANG )
# 896 "src/lexer.ml"

  | 13 ->
# 176 "src/lexer.mll"
      ( COLON )
# 901 "src/lexer.ml"

  | 14 ->
# 178 "src/lexer.mll"
      ( SEMICOLON )
# 906 "src/lexer.ml"

  | 15 ->
# 180 "src/lexer.mll"
      ( COLONEQUAL )
# 911 "src/lexer.ml"

  | 16 ->
# 182 "src/lexer.mll"
      ( ARROW )
# 916 "src/lexer.ml"

  | 17 ->
# 184 "src/lexer.mll"
      ( LRARROW )
# 921 "src/lexer.ml"

  | 18 ->
# 186 "src/lexer.mll"
      ( EQUAL )
# 926 "src/lexer.ml"

  | 19 ->
# 188 "src/lexer.mll"
      ( LT )
# 931 "src/lexer.ml"

  | 20 ->
# 190 "src/lexer.mll"
      ( LE )
# 936 "src/lexer.ml"

  | 21 ->
# 192 "src/lexer.mll"
      ( GT )
# 941 "src/lexer.ml"

  | 22 ->
# 194 "src/lexer.mll"
      ( GE )
# 946 "src/lexer.ml"

  | 23 ->
# 196 "src/lexer.mll"
      ( NOTEQ )
# 951 "src/lexer.ml"

  | 24 ->
# 198 "src/lexer.mll"
      ( PLUS )
# 956 "src/lexer.ml"

  | 25 ->
# 200 "src/lexer.mll"
      ( MINUS )
# 961 "src/lexer.ml"

  | 26 ->
# 202 "src/lexer.mll"
      ( TIMES )
# 966 "src/lexer.ml"

  | 27 ->
# 204 "src/lexer.mll"
      ( SLASH )
# 971 "src/lexer.ml"

  | 28 ->
# 206 "src/lexer.mll"
      ( AT )
# 976 "src/lexer.ml"

  | 29 ->
# 208 "src/lexer.mll"
      ( DOT )
# 981 "src/lexer.ml"

  | 30 ->
# 210 "src/lexer.mll"
      ( LEFTSQ )
# 986 "src/lexer.ml"

  | 31 ->
# 212 "src/lexer.mll"
      ( RIGHTSQ )
# 991 "src/lexer.ml"

  | 32 ->
# 214 "src/lexer.mll"
      ( LEFTB )
# 996 "src/lexer.ml"

  | 33 ->
# 216 "src/lexer.mll"
      ( RIGHTB )
# 1001 "src/lexer.ml"

  | 34 ->
# 218 "src/lexer.mll"
      ( LEFTBLEFTB )
# 1006 "src/lexer.ml"

  | 35 ->
# 220 "src/lexer.mll"
      ( RIGHTBRIGHTB )
# 1011 "src/lexer.ml"

  | 36 ->
# 222 "src/lexer.mll"
      ( BAR )
# 1016 "src/lexer.ml"

  | 37 ->
# 224 "src/lexer.mll"
      ( BARBAR )
# 1021 "src/lexer.ml"

  | 38 ->
# 226 "src/lexer.mll"
      ( AMPAMP )
# 1026 "src/lexer.ml"

  | 39 ->
# 228 "src/lexer.mll"
      ( BIGARROW )
# 1031 "src/lexer.ml"

  | 40 ->
# 230 "src/lexer.mll"
      ( Buffer.clear string_buf; string lexbuf )
# 1036 "src/lexer.ml"

  | 41 ->
# 232 "src/lexer.mll"
      ( EOF )
# 1041 "src/lexer.ml"

  | 42 ->
let
# 233 "src/lexer.mll"
         c
# 1047 "src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 234 "src/lexer.mll"
      ( raise (Lexical_error ("illegal character: " ^ String.make 1 c)) )
# 1051 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 85
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 238 "src/lexer.mll"
      ( () )
# 1062 "src/lexer.ml"

  | 1 ->
# 240 "src/lexer.mll"
      ( comment lexbuf; comment lexbuf )
# 1067 "src/lexer.ml"

  | 2 ->
# 242 "src/lexer.mll"
      ( newline lexbuf; comment lexbuf )
# 1072 "src/lexer.ml"

  | 3 ->
# 244 "src/lexer.mll"
      ( raise (Lexical_error "unterminated comment") )
# 1077 "src/lexer.ml"

  | 4 ->
# 246 "src/lexer.mll"
      ( comment lexbuf )
# 1082 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and string lexbuf =
    __ocaml_lex_string_rec lexbuf 93
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 250 "src/lexer.mll"
      ( STRING (Buffer.contents string_buf) )
# 1093 "src/lexer.ml"

  | 1 ->
let
# 251 "src/lexer.mll"
               c
# 1099 "src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 252 "src/lexer.mll"
      ( Buffer.add_char string_buf (char_for_backslash c); string lexbuf )
# 1103 "src/lexer.ml"

  | 2 ->
# 254 "src/lexer.mll"
      ( newline lexbuf; Buffer.add_char string_buf '\n'; string lexbuf )
# 1108 "src/lexer.ml"

  | 3 ->
# 256 "src/lexer.mll"
      ( raise (Lexical_error "unterminated string") )
# 1113 "src/lexer.ml"

  | 4 ->
let
# 257 "src/lexer.mll"
         c
# 1119 "src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 258 "src/lexer.mll"
      ( Buffer.add_char string_buf c; string lexbuf )
# 1123 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_string_rec lexbuf __ocaml_lex_state

;;

# 262 "src/lexer.mll"
 

  let loc lb = (lexeme_start_p lb, lexeme_end_p lb)

  let with_location f lb =
    try f lb with e -> raise (Loc.Located (loc lb, e))

  let parse_lexpr = with_location (lexpr token)
  let parse_file = with_location (file token)

  let lexpr_of_string s = parse_lexpr (from_string s)

# 1142 "src/lexer.ml"
