// WideMargin. Simple fast bible software.
// Copyright (C) 2011  Daniel Hughes
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

using System;
using System.Linq;
using Gtk;
using WideMargin.MVCInterfaces;
using System.Collections.Generic;
using WideMargin.Utilities;

namespace WideMargin.GUI
{
	/// <summary>
	/// TextView which supports basic XML style markup.
	/// </summary>
	[System.ComponentModel.ToolboxItem(true)]
	public partial class RichTextView : Gtk.Bin
	{
		private EventLocker _eventLocker = new EventLocker();
		private EventHandler<LinkClickedEventArgs<IVerseIdentifier>> _verseLinkClicked;
		private TextView _textView;
		private List<XmlTag> _xmlTags;
		private Dictionary<string, TextTag> _tags = new Dictionary<string, TextTag>();
		
		/// <summary>
		/// Constructor
		/// </summary>
		public RichTextView ()
		{
			this.Build ();
			GaplessScrolledWindow scrolledWindow = new GaplessScrolledWindow();
			scrolledWindow.ShadowType = ShadowType.In;
			_textView = new TextView();
			_textView.WrapMode = WrapMode.Word;
			_textView.MotionNotifyEvent += HandleTextViewMotionNotifyEvent;
			scrolledWindow.Add(_textView);
			this.Add(scrolledWindow);
		}
		
		/// <summary>
		/// Occurs when the mouse moves over the text
		/// If its over a link show the hand cursor.
		/// </summary>
		/// <param name="o">sender</param>
		/// <param name="args">Movement arguments</param>
		private void HandleTextViewMotionNotifyEvent (object o, MotionNotifyEventArgs args)
		{
			int x;
			int	y;
			_textView.WindowToBufferCoords (Gtk.TextWindowType.Widget,
		                                    (int)args.Event.X,
		                                    (int)args.Event.Y,
		                                    out x,
		                                    out y);

			Gtk.TextIter iter = _textView.GetIterAtLocation (x, y);
			bool foundLink = false;
			foreach(TextTag tag in iter.Tags)
			{
				if(tag is LinkTag<IVerseIdentifier>)
				{
					foundLink = true;
					break;
				}
			}
			
			Gdk.Window window = _textView.GetWindow (Gtk.TextWindowType.Text);
			if(!foundLink)
			{
				window.Cursor = new Gdk.Cursor (Gdk.CursorType.Xterm);
				return;
			}
			window.Cursor = new Gdk.Cursor (Gdk.CursorType.Hand2);
		}
		
		/// <summary>
		/// Text including markup
		/// </summary>
		public string Text
		{
			set
			{
				_textView.Buffer.Text = new string(ParseMarkup(value).ToArray());
				ParseTags();
			}
		}
		
		/// <summary>
		/// Adds the tags to the TextView so to display the markup
		/// </summary>
		private void ParseTags()
		{
			foreach(XmlTag tag in _xmlTags)
			{
				if(tag.Name == "VerseLink")
				{
					string target = tag.Attributes["target"];
					TextTag textTag;
					if(!_tags.TryGetValue(target, out textTag))
					{
						IVerseIdentifier verse = new VerseIdentifier(target);
						LinkTag<IVerseIdentifier> linkTag = new LinkTag<IVerseIdentifier>(target, verse);
						linkTag.LinkClicked += HandleTagLinkClicked;
						textTag = linkTag;
						AddTag(textTag, target);
					}
					
					ApplyTag(textTag, tag.StartingIndex, tag.EndIndex);
				}
				if(tag.Name == "Center")
				{
					TextTag textTag;
					if(!_tags.TryGetValue("Center", out textTag))
					{
						textTag = new TextTag("Center");
						textTag.Justification = Justification.Center;
						AddTag(textTag, "Center");
					}
					
					ApplyTag(textTag, tag.StartingIndex, tag.EndIndex);
				}
				if(tag.Name == "Size")
				{
					TextTag textTag;
					string val = tag.Attributes["value"];
					string name = String.Format("Size{0}", val);
					if(!_tags.TryGetValue(name, out textTag))
					{
						textTag = new TextTag(name);
						//textTag.Size = int.Parse(val);
						textTag.SizePoints = double.Parse(val);
						AddTag(textTag, name);
					}
					
					ApplyTag(textTag, tag.StartingIndex, tag.EndIndex);
				}
				if(tag.Name == "Bold")
				{
					TextTag textTag;
					if(!_tags.TryGetValue("Bold", out textTag))
					{
						textTag = new TextTag("Bold");
						textTag.Weight = Pango.Weight.Bold;
						AddTag(textTag, "Bold");
					}
					
					ApplyTag(textTag, tag.StartingIndex, tag.EndIndex);	
				}
			}
		}

		/// <summary>
		/// A verse link has been cliked, Passes the event up to the controller.
		/// </summary>
		private void HandleTagLinkClicked (object sender, LinkClickedEventArgs<IVerseIdentifier> e)
		{
			_verseLinkClicked.Fire(this, e);
		}
		
		/// <summary>
		/// Parses the markup extracting the tags and returning the raw text
		/// </summary>
		/// <param name="markup">text containing markup</param>
		/// <returns>markup free text</returns>
		private IEnumerable<char> ParseMarkup(string markup)
		{
			_xmlTags = new List<XmlTag>();
			Stack<XmlTag> openTags = new Stack<XmlTag>();
			IEnumerator<char> enumerator = markup.GetEnumerator();
			int index = -1;
			while(enumerator.MoveNext())
			{
				
				if(enumerator.Current != '<')
				{
					index++;
					yield return enumerator.Current;	
				}
				else
				{
					string rawTag = ReadTag(enumerator);
					if(rawTag[0] == '/')
					{
						XmlTag tagToClose = openTags.Pop();
						tagToClose.EndIndex = index + 1;	
					}
					else
					{
						var tag = new XmlTag(rawTag);
						_xmlTags.Add(tag);
						tag.StartingIndex = index + 1;
						openTags.Push(tag);
					}
				}
			}
		}
		
		/// <summary>
		/// Adds a tag
		/// </summary>
		/// <param name="tag">tag to add</param>
		/// <param name="name">unique name for tag</param>
		private void AddTag(TextTag tag, string name)
		{
			_tags.Add(name, tag);
			_textView.Buffer.TagTable.Add(tag);
		}
		
		/// <summary>
		/// Applies a tag at the given index
		/// </summary>
		/// <param name="tag">tag to apply</param>
		/// <param name="startIndex">start index to apply at</param>
		/// <param name="endIndex">end index to apply at</param>
		private void ApplyTag(TextTag tag, int startIndex, int endIndex)
		{
			TextIter iter1 = _textView.Buffer.GetIterAtOffset (startIndex); 
			TextIter iter2 = _textView.Buffer.GetIterAtOffset (endIndex); 
			_textView.Buffer.ApplyTag(tag, iter1, iter2);
		}
		
		/// <summary>
		/// Reads an xml tag from the enumeration and returns in with the '<' and '>'
		/// </summary>
		/// <param name="enumerator">chacters starting with and opening '<'</param>
		/// <returns>Tag, stuff between '<' and '>'</returns>
		private string ReadTag(IEnumerator<char> enumerator)
		{
			List<char> list = new List<char>();
			while(enumerator.MoveNext())
			{
				char current = enumerator.Current;
				if(current != '>')
				{
					list.Add(current);
				}
				else
				{
					return	new string(list.ToArray());
				}
			}
			throw new FormatException("Tag doesn't end");
		}
		
		/// <summary>
		/// Occurs when a verse link is clicked.
		/// </summary>
		public event EventHandler<LinkClickedEventArgs<IVerseIdentifier>> VerseLinkClicked
		{
			add
			{
				_eventLocker.Add(ref _verseLinkClicked, value);
			}
			remove
			{
				_eventLocker.Remove(ref _verseLinkClicked, value);
			}
		}
	}
}

